console.log('🔥 POPUP.JS LOADED - VERSION: LOCAL-SIGNER-v1 - 2024-12-18');
/**
 * 🔥 MyWallet Extension - Popup Script
 * Gerencia a interface da extensão
 * 
 * 🔐 LEI PRIMORDIAL: Mnemonic NEVER leaves the device!
 *    Local signing is done in this popup (WASM works here!)
 */

// 🔐 Check if TaprootSignerPure is available (loaded from taproot-signer-pure.bundle.js)
// This is the PURE JS version - NO WASM! Works everywhere!
if (typeof TaprootSignerPure !== 'undefined' && TaprootSignerPure.signPsbtTaprootLocal) {
    console.log('🔐 TaprootSignerPure loaded successfully!');
    console.log('   ✅ signPsbtTaprootLocal available (PURE JS - NO WASM!)');
    console.log('   ✅ LEI PRIMORDIAL: Local signing enabled!');
} else {
    console.warn('⚠️ TaprootSignerPure not available - checking TaprootSigner fallback');
    if (typeof TaprootSigner !== 'undefined' && TaprootSigner.signPsbtTaprootLocal) {
        console.log('   ✅ TaprootSigner (WASM) available as fallback');
    } else {
        console.warn('   ❌ No local signer available');
    }
}

// 🔐 SECURITY: HTML/URL sanitization helpers (2026-01-31)
// Prevents XSS from API data injected into innerHTML
function sanitizeHTML(str) {
    if (str === null || str === undefined) return '';
    return String(str)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}

function sanitizeURL(url) {
    if (!url) return '';
    const str = String(url).trim();
    // Only allow https:// and data:image/ URLs
    if (str.startsWith('https://') || str.startsWith('data:image/')) {
        return str;
    }
    // For ordinals.com content, allow relative /api/ paths from our backend
    if (str.startsWith('/api/')) {
        return str;
    }
    console.warn('🔐 Blocked unsafe URL:', str.substring(0, 50));
    return '';
}

// Estado global
let currentScreen = 'no-wallet';
let wallet = null;

// 💾 CACHE SYSTEM - Evita recarregar dados toda vez que troca de tab
const dataCache = {
    ordinals: {
        data: null,
        timestamp: null,
        loaded: false
    },
    runes: {
        data: null,
        timestamp: null,
        loaded: false
    },
    activity: {
        data: null,
        timestamp: null,
        loaded: false,
        lastTxid: null  // 🔄 Último TXID para detectar mudanças
    }
};

// Tempo de validade do cache
const CACHE_VALIDITY = {
    ordinals: 5 * 60 * 1000,   // 5 minutos (dados estáticos)
    runes: 60 * 1000,          // 1 minuto (updates frequentes)
    activity: 60 * 1000        // 1 minuto (updates frequentes)
};

// Verificar se cache é válido
function isCacheValid(cacheType) {
    const cache = dataCache[cacheType];
    if (!cache.loaded || !cache.timestamp) return false;
    
    const age = Date.now() - cache.timestamp;
    return age < CACHE_VALIDITY[cacheType];
}

// Limpar cache (usado quando unlock wallet ou refresh)
function clearAllCache() {
    console.log('🗑️ Clearing all cache...');
    dataCache.ordinals = { data: null, timestamp: null, loaded: false };
    dataCache.runes = { data: null, timestamp: null, loaded: false };
    dataCache.activity = { data: null, timestamp: null, loaded: false, lastTxid: null };
}

// 🔄 Limpar apenas cache de Activity (para forçar refresh após transação)
function clearActivityCache() {
    console.log('🔄 Clearing activity cache (transaction detected)...');
    dataCache.activity = { data: null, timestamp: null, loaded: false, lastTxid: null };
}

// 🔑 Helper function to get wallet address from background script
async function getWalletAddress() {
    try {
        const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (response && response.success && response.data && response.data.address) {
            return response.data.address;
        }
        console.error('❌ getWalletAddress: No address in response');
        return null;
    } catch (error) {
        console.error('❌ getWalletAddress error:', error);
        return null;
    }
}

// 🔑 Helper function to get wallet public key from background script
async function getWalletPublicKey() {
    try {
        const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (response && response.success && response.data && response.data.publicKey) {
            return response.data.publicKey;
        }
        console.error('❌ getWalletPublicKey: No publicKey in response');
        return null;
    } catch (error) {
        console.error('❌ getWalletPublicKey error:', error);
        return null;
    }
}

// 🔒 SECURITY: Clear ALL password fields after use
function clearAllPasswordFields() {
    console.log('🔒 Security: Clearing all password fields...');
    const passwordFields = [
        'create-password',
        'confirm-password',
        'restore-password',
        'view-mnemonic-password',
        'view-key-password',
        'psbt-confirm-password',
        'message-sign-password',
        'lightning-payment-password',
        'confirm-password-input',
        'l2-confirm-password',
        'l2-withdraw-confirm-password',
        'listing-confirm-password'
    ];
    
    passwordFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.value = '';
        }
    });
}

// 🔍 Verificar se há novas transações (RÁPIDO - só busca última TX)
async function checkForNewTransactions(address) {
    try {
        // Buscar apenas a última transação (muito mais rápido!)
        const response = await fetch(`https://mempool.space/api/address/${address}/txs/chain/0`);
        if (!response.ok) return false;
        
        const txs = await response.json();
        if (!txs || txs.length === 0) return false;
        
        const latestTxid = txs[0].txid;
        
        // Comparar com último TXID conhecido
        if (!dataCache.activity.lastTxid) {
            dataCache.activity.lastTxid = latestTxid;
            return false;
        }
        
        // Se TXID mudou = nova transação!
        if (latestTxid !== dataCache.activity.lastTxid) {
            console.log('   🆕 New TXID detected:', latestTxid.substring(0, 16) + '...');
            return true;
        }
        
        return false;
    } catch (error) {
        console.warn('   ⚠️ Check failed:', error.message);
        return false;
    }
}

// Helper function para enviar mensagens ao background
async function sendMessage(message) {
    return new Promise((resolve, reject) => {
        chrome.runtime.sendMessage(message, (response) => {
            if (chrome.runtime.lastError) {
                reject(new Error(chrome.runtime.lastError.message));
            } else {
                resolve(response);
            }
        });
    });
}

// 🔄 RESET AUTO-LOCK TIMER EM QUALQUER INTERAÇÃO (throttle)
let lastResetTime = 0;
const RESET_INTERVAL = 30000; // 30 segundos (só chama a cada 30s)

function resetAutolockTimer() {
    const now = Date.now();
    
    // Só chama se passou mais de 30 segundos desde última chamada
    if (now - lastResetTime < RESET_INTERVAL) {
        return;
    }
    
    lastResetTime = now;
    
    chrome.runtime.sendMessage({ action: 'resetAutolockTimer' }).catch(() => {
        // Ignore error if background is busy
    });
    
    console.log('⏰ Auto-lock timer reset');
}

// ⏰ DETECTAR INTERAÇÕES DO USUÁRIO (só click e keypress)
['click', 'keypress'].forEach(eventType => {
    document.addEventListener(eventType, () => {
        resetAutolockTimer();
    }, { passive: true, capture: true });
});

// EVENT DELEGATION GLOBAL - Configurar ANTES de tudo!
console.log('🔥 Setting up GLOBAL event delegation...');
document.addEventListener('click', (e) => {
    const target = e.target.closest('button');
    if (!target) return;
    
    console.log('🎯 Button clicked:', target.id);
    
    if (target.id === 'copy-address-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Copy Address clicked!');
        handleCopyAddress();
        return;
    }
    
    if (target.id === 'send-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Send clicked!');
        showScreen('send');
        return;
    }
    
    if (target.id === 'receive-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Receive clicked!');
        showScreen('receive');
        return;
    }
    
    if (target.id === 'settings-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Settings clicked!');
        showScreen('settings');
        return;
    }
    
    if (target.id === 'send-confirm-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Send Transaction clicked!');
        handleSend();
        return;
    }
    
    if (target.id === 'send-inscription-confirm-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Send Inscription clicked!');
        handleSendInscription();
        return;
    }
    
    if (target.id === 'close-password-modal' || target.id === 'cancel-send-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Cancel/Close modal clicked!');
        closePasswordModal();
        return;
    }
    
    if (target.id === 'confirm-send-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🔘 Confirm Send clicked!');
        confirmAndSend();
        return;
    }
    
    // 🪙 List Rune from Market Tab
    if (target.id === 'list-rune-market-btn') {
        e.preventDefault();
        e.stopPropagation();
        console.log('🪙 List Rune from Market clicked!');
        showRuneSelector();
        return;
    }
}, true);
console.log('✅ GLOBAL event delegation configured!');

// ═══════════════════════════════════════════════════════════════
// 🎨 FEATURE SLIDER - Welcome Screen (Mobile Style)
// ═══════════════════════════════════════════════════════════════
const WELCOME_FEATURES = [
    { icon: 'flash', text: 'Taproot Native' },
    { icon: 'shield', text: 'Self-Custodial' },
    { icon: 'layers', text: 'Ordinals & Runes' },
    { icon: 'store', text: 'Built-in Marketplace' },
    { icon: 'swap', text: 'Atomic Swaps' },
    { icon: 'zap', text: 'Lightning Ready' },
];

const FEATURE_ICONS = {
    flash: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"></polygon></svg>',
    shield: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"></path></svg>',
    layers: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="12 2 2 7 12 12 22 7 12 2"></polygon><polyline points="2 17 12 22 22 17"></polyline><polyline points="2 12 12 17 22 12"></polyline></svg>',
    store: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path><polyline points="9 22 9 12 15 12 15 22"></polyline></svg>',
    swap: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="17 1 21 5 17 9"></polyline><path d="M3 11V9a4 4 0 0 1 4-4h14"></path><polyline points="7 23 3 19 7 15"></polyline><path d="M21 13v2a4 4 0 0 1-4 4H3"></path></svg>',
    zap: '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"></polygon></svg>',
};

let featureSliderInterval = null;
let currentFeatureIndex = 0;

function initFeatureSlider() {
    const featureText = document.getElementById('feature-text');
    const featureIcon = document.getElementById('feature-icon');
    const sliderCard = document.querySelector('.feature-slider-card');
    const dots = document.querySelectorAll('.feature-dot');
    
    if (!featureText || !featureIcon || !sliderCard) return;
    
    if (featureSliderInterval) clearInterval(featureSliderInterval);
    
    function updateFeature(index) {
        const feature = WELCOME_FEATURES[index];
        sliderCard.style.opacity = '0';
        
        setTimeout(() => {
            featureText.textContent = feature.text;
            featureIcon.innerHTML = FEATURE_ICONS[feature.icon];
            dots.forEach((dot, i) => dot.classList.toggle('active', i === index));
            sliderCard.style.opacity = '1';
        }, 250);
    }
    
    featureSliderInterval = setInterval(() => {
        currentFeatureIndex = (currentFeatureIndex + 1) % WELCOME_FEATURES.length;
        updateFeature(currentFeatureIndex);
    }, 2500);
    
    updateFeature(0);
    sliderCard.style.transition = 'opacity 0.25s ease';
}

function stopFeatureSlider() {
    if (featureSliderInterval) {
        clearInterval(featureSliderInterval);
        featureSliderInterval = null;
    }
}

// Inicialização
document.addEventListener('DOMContentLoaded', async () => {
    try {
        console.log('🔥 MyWallet Extension initialized');
        
        // 🧹 LIMPEZA AGRESSIVA: Verificar idade de todos PSBTs no storage
        console.log('🧹 Checking for old/corrupted PSBTs in storage...');
        const storageCheck = await chrome.storage.local.get(['pendingPsbtRequest', 'psbtSignResult']);
        
        if (storageCheck.pendingPsbtRequest) {
            const age = Date.now() - (storageCheck.pendingPsbtRequest.timestamp || 0);
            const maxAge = 2 * 60 * 1000; // 2 minutos
            
            if (age > maxAge || !storageCheck.pendingPsbtRequest.timestamp) {
                console.warn('⚠️  Found OLD/CORRUPTED PSBT in storage (age:', Math.round(age / 1000), 's)');
                console.warn('   Deleting old PSBT data...');
                await chrome.storage.local.remove(['pendingPsbtRequest', 'psbtSignResult', 'pendingMarketListing']);
                await chrome.runtime.sendMessage({ action: 'cancelPsbtSign' });
                console.log('✅ Old PSBT data DELETED');
            }
        }
        
        // 🔍 PRIORIDADE: Verificar se há PSBT pendente via chrome.storage
        // (URL params não funcionam com chrome.windows.create popup)
        // Tentar múltiplas vezes para evitar race condition
        let response = await chrome.runtime.sendMessage({ action: 'getPendingPsbt' });
        
        console.log('   Pending PSBT check (1st try):', response);
        
        // Se não encontrou na primeira tentativa, esperar 100ms e tentar novamente
        if (!response.pending) {
            console.log('   Waiting 100ms and retrying...');
            await new Promise(resolve => setTimeout(resolve, 100));
            response = await chrome.runtime.sendMessage({ action: 'getPendingPsbt' });
            console.log('   Pending PSBT check (2nd try):', response);
        }
        
        // Se ainda não encontrou, esperar mais 200ms e tentar uma última vez
        if (!response.pending) {
            console.log('   Waiting 200ms and retrying...');
            await new Promise(resolve => setTimeout(resolve, 200));
            response = await chrome.runtime.sendMessage({ action: 'getPendingPsbt' });
            console.log('   Pending PSBT check (3rd try):', response);
        }
        
        if (response.success && response.pending) {
            console.log('🔏 PSBT signing request detected via storage!');
            console.log('   PSBT details:', {
            inputs: response.pending.inputsToSign?.length || 'all',
            sighash: response.pending.sighashType,
            autoFinalized: response.pending.autoFinalized,
            timestamp: response.pending.timestamp,
            psbtLength: response.pending.psbt?.length || 0
        });
        
        // ✅ Verificar se PSBT é muito antigo (mais de 2 minutos) ou inválido
        const age = Date.now() - (response.pending.timestamp || 0);
        const maxAge = 2 * 60 * 1000; // 2 minutos (reduzido para ser mais agressivo)
        
        if (age > maxAge || !response.pending.timestamp || !response.pending.psbt) {
            console.warn('⚠️  PSBT is INVALID:');
            console.warn('   - Age:', Math.round(age / 1000), 's');
            console.warn('   - Has timestamp:', !!response.pending.timestamp);
            console.warn('   - Has PSBT data:', !!response.pending.psbt);
            console.warn('   → DELETING...');
            
            // 🔥 LIMPAR TUDO (agressivo)
            await chrome.storage.local.remove([
                'pendingPsbtRequest', 
                'psbtSignResult',
                'pendingMarketListing'
            ]);
            await chrome.runtime.sendMessage({ action: 'cancelPsbtSign' });
            
            console.log('✅ Invalid PSBT DELETED from storage, showing normal wallet');
            // NÃO mostrar PSBT, continuar para wallet normal
        } else {
            // PSBT é válido (< 2 minutos), esconder loading e mostrar PSBT
            console.log('✅ PSBT is VALID:');
            console.log('   - Age:', Math.round(age / 1000), 's');
            console.log('   - PSBT length:', response.pending.psbt?.length || 0);
            console.log('   → Showing sign screen...');
            
            // 🎯 MOSTRAR TELA DE ASSINATURA IMEDIATAMENTE
            document.getElementById('loading-screen')?.classList.add('hidden');
            
            console.log('🔄 Calling showScreen(confirm-psbt)...');
            showScreen('confirm-psbt');
            console.log('✅ Screen changed to confirm-psbt');
            
            try {
                console.log('🔄 Calling showPsbtConfirmation()...');
                await showPsbtConfirmation();
                console.log('✅ showPsbtConfirmation() completed successfully');
            } catch (psbtError) {
                console.error('❌ ERROR in showPsbtConfirmation():', psbtError);
                console.error('   Stack:', psbtError.stack);
                // Em caso de erro, mostrar wallet normal
                showScreen('wallet');
                await loadWalletData();
            }
            return;
        }
        } else {
            console.log('ℹ️  No pending PSBT found after 3 attempts, showing normal wallet');
        }
        
        // 🔍 VERIFICAR SE HÁ MESSAGE PENDENTE (para likes)
        const messageCheck = await chrome.storage.local.get(['pendingMessageRequest']);
        if (messageCheck.pendingMessageRequest) {
            const messageAge = Date.now() - (messageCheck.pendingMessageRequest.timestamp || 0);
            const maxAge = 60 * 1000; // 1 minuto
            
            if (messageAge < maxAge && messageCheck.pendingMessageRequest.message) {
                console.log('✅ Found pending message request, showing sign message screen');
                document.getElementById('loading-screen')?.classList.add('hidden');
                // Pass display metadata if available (from signL2Transaction)
                const displayMeta = messageCheck.pendingMessageRequest.display || null;
                showSignMessageScreen(messageCheck.pendingMessageRequest.message, displayMeta);
                return;
            } else {
                console.log('⚠️  Old message request found, deleting...');
                await chrome.storage.local.remove('pendingMessageRequest');
            }
        }
        
        // 🔍 VERIFICAR SE HÁ LIGHTNING PAYMENT PENDENTE
        console.log('🔍 Checking for pending Lightning payment...');
        const lightningCheck = await chrome.storage.local.get(['pendingPaymentRequest']);
        if (lightningCheck.pendingPaymentRequest) {
            const paymentAge = Date.now() - (lightningCheck.pendingPaymentRequest.timestamp || 0);
            const maxAge = 2 * 60 * 1000; // 2 minutos
            
            if (paymentAge < maxAge && lightningCheck.pendingPaymentRequest.invoice) {
                console.log('✅ Found pending Lightning payment, showing confirmation screen');
                console.log('   Payment details:', {
                    amount: lightningCheck.pendingPaymentRequest.decoded?.amount,
                    description: lightningCheck.pendingPaymentRequest.decoded?.description
                });
                document.getElementById('loading-screen')?.classList.add('hidden');
                showScreen('confirm-lightning-payment');
                await showLightningPaymentConfirmation(lightningCheck.pendingPaymentRequest);
                return;
            } else {
                console.log('⚠️  Old Lightning payment request found, deleting...');
                await chrome.storage.local.remove(['pendingPaymentRequest', 'paymentResult']);
            }
        }
        
        // Verificar status da wallet (exists, unlocked)
        const walletStatus = await sendMessage({ action: 'checkWalletStatus' });
        console.log('Wallet status:', walletStatus);
        
        // Esconder loading screen
        document.getElementById('loading-screen').classList.add('hidden');
        
        if (walletStatus.success && walletStatus.exists) {
            if (walletStatus.unlocked) {
                // Wallet desbloqueada, mostrar tela normal
                console.log('✅ Wallet is unlocked');
                showScreen('wallet');
                await loadWalletData();
                
                // ⚡ Initialize KRAY L2 (new!)
                if (window.krayL2) {
                    // Register showScreen function with L2 module
                    if (typeof window.krayL2.setShowScreen === 'function') {
                        window.krayL2.setShowScreen(showScreen);
                    }
                    // Initialize L2
                    if (typeof window.krayL2.initL2 === 'function') {
                        setTimeout(() => window.krayL2.initL2(), 1000);
                    }
                }
            } else {
                // Wallet bloqueada, mostrar tela de unlock
                console.log('🔒 Wallet is locked, showing unlock screen');
                showScreen('unlock');
            }
        } else {
            // Nenhuma wallet, mostrar tela de boas-vindas
            console.log('ℹ️  No wallet found');
            showScreen('no-wallet');
        }
        
        // Event Listeners
        console.log('🔗 Calling setupEventListeners...');
        try {
            setupEventListeners();
            console.log('✅ Event listeners setup complete');
        } catch (listenerError) {
            console.error('❌ Error setting up event listeners:', listenerError);
            console.error('   Stack:', listenerError.stack);
        }
        
        // GARANTIR que botões da wallet sejam vinculados após 1 segundo
        console.log('⏰ Setting up delayed rebind as backup...');
        setTimeout(() => {
            console.log('⏰ Delayed rebind executing now...');
            try {
                rebindWalletButtons();
            } catch (rebindError) {
                console.error('❌ Error in delayed rebind:', rebindError);
            }
        }, 1000);
        
        // 🔥 LISTENER PARA pendingMessageRequest (detecta quando cancel/update price é clicado)
        chrome.storage.onChanged.addListener((changes, namespace) => {
            if (namespace === 'local' && changes.pendingMessageRequest) {
                const newValue = changes.pendingMessageRequest.newValue;
                
                if (newValue && newValue.message) {
                    console.log('🔔 pendingMessageRequest detected!', newValue.action);
                    console.log('   Message:', newValue.message);
                    
                    // Verificar idade (max 1 minuto)
                    const messageAge = Date.now() - (newValue.timestamp || 0);
                    const maxAge = 60 * 1000;
                    
                    if (messageAge < maxAge) {
                        console.log('✅ Opening sign message screen automatically!');
                        // Pass display metadata if available (from signL2Transaction)
                        const displayMeta = newValue.display || null;
                        showSignMessageScreen(newValue.message, displayMeta);
                    } else {
                        console.log('⚠️  Message too old, ignoring');
                    }
                }
            }
        });
        console.log('🎧 pendingMessageRequest listener active!');
        
    } catch (error) {
        console.error('❌ Initialization error:', error);
        showNotification('Failed to initialize wallet: ' + error.message, 'error');
    }
});

// Setup Event Listeners
function setupEventListeners() {
    console.log('🔗 ========== SETUP EVENT LISTENERS ==========');
    console.log('   Function called at:', new Date().toISOString());
    
    // No Wallet Screen
    console.log('🔍 Setting up No Wallet Screen buttons...');
    const createBtn = document.getElementById('create-wallet-btn');
    console.log('   create-wallet-btn element:', createBtn);
    console.log('   createBtn exists:', !!createBtn);
    console.log('   createBtn visible:', createBtn?.offsetParent !== null);
    console.log('   createBtn disabled:', createBtn?.disabled);
    
    if (createBtn) {
        // Remover listeners antigos se existirem
        createBtn.replaceWith(createBtn.cloneNode(true));
        const newCreateBtn = document.getElementById('create-wallet-btn');
        
        // Usar onclick direto (mais confiável)
        newCreateBtn.onclick = function(e) {
            console.log('🔘 CREATE WALLET BUTTON CLICKED!');
            console.log('   Event:', e);
            e.preventDefault();
            e.stopPropagation();
            showScreen('create-wallet');
        };
        console.log('✅ Create wallet button listener added (onclick)');
    } else {
        console.error('❌ create-wallet-btn NOT FOUND!');
    }
    
    // Restore Wallet Button
    const restoreWalletBtn = document.getElementById('restore-wallet-btn');
    if (restoreWalletBtn) {
        restoreWalletBtn.onclick = () => showScreen('restore-wallet');
        console.log('✅ Restore wallet button listener added');
    }
    
    // Unlock Screen
    console.log('🔍 Setting up Unlock Screen button...');
    const unlockBtn = document.getElementById('unlock-wallet-btn');
    if (unlockBtn) {
        unlockBtn.onclick = async function(e) {
            console.log('🔘 Unlock button clicked!');
            e.preventDefault();
            e.stopPropagation();
            await handleUnlockWallet();
        };
        console.log('✅ Unlock wallet button listener added');
    }
    
    // Unlock password Enter key
    const unlockPasswordInput = document.getElementById('unlock-password');
    if (unlockPasswordInput) {
        unlockPasswordInput.addEventListener('keypress', async function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                await handleUnlockWallet();
            }
        });
        console.log('✅ Unlock password Enter key listener added');
    }
    
    // 🔒 SECURITY: Forgot password link removed
    // Users must reinstall extension to reset wallet
    console.log('✅ No forgot password link (security measure)');
    
    // Create Wallet
    const generateBtn = document.getElementById('generate-mnemonic-btn');
    console.log('🔍 Looking for generate button...');
    console.log('   Button element:', generateBtn);
    console.log('   Button visible:', generateBtn?.offsetParent !== null);
    
    if (generateBtn) {
        // Adicionar onclick direto no elemento via JavaScript (não viola CSP!)
        generateBtn.onclick = async function(e) {
            console.log('🔘 Generate button clicked!');
            e.preventDefault();
            e.stopPropagation();
            await handleGenerateMnemonic();
        };
        
        console.log('✅ Generate mnemonic button listener added (onclick)');
    } else {
        console.warn('⚠️  Generate mnemonic button not found!');
    }
    
    const backFromCreateBtn = document.getElementById('back-from-create-btn');
    if (backFromCreateBtn) {
        backFromCreateBtn.addEventListener('click', () => {
            showScreen('no-wallet');
        });
        console.log('✅ Back from create button listener added');
    }
    
    // Show Mnemonic
    document.getElementById('mnemonic-saved-check')?.addEventListener('change', (e) => {
        document.getElementById('mnemonic-saved-btn').disabled = !e.target.checked;
    });
    
    document.getElementById('mnemonic-saved-btn')?.addEventListener('click', handleMnemonicSaved);
    
    // Restore Wallet
    document.getElementById('restore-btn')?.addEventListener('click', handleRestoreWallet);
    document.getElementById('back-from-restore-btn')?.addEventListener('click', () => {
        showScreen('no-wallet');
    });
    
    // Wallet Screen
    console.log('🔍 Setting up Wallet Screen buttons...');
    
    const copyAddressBtn = document.getElementById('copy-address-btn');
    console.log('   copy-address-btn:', !!copyAddressBtn);
    if (copyAddressBtn) {
        copyAddressBtn.onclick = async () => {
            console.log('🔘 Copy Address clicked!');
            await handleCopyAddress();
        };
        console.log('✅ Copy address listener added');
    }
    
    const sendBtn = document.getElementById('send-btn');
    console.log('   send-btn:', !!sendBtn);
    if (sendBtn) {
        sendBtn.onclick = () => {
            console.log('🔘 Send clicked!');
            showScreen('send');
        };
        console.log('✅ Send listener added');
    }
    
    const receiveBtn = document.getElementById('receive-btn');
    console.log('   receive-btn:', !!receiveBtn);
    if (receiveBtn) {
        receiveBtn.onclick = () => {
            console.log('🔘 Receive clicked!');
            showScreen('receive');
        };
        console.log('✅ Receive listener added');
    }
    
    const settingsBtn = document.getElementById('settings-btn');
    console.log('   settings-btn:', !!settingsBtn);
    if (settingsBtn) {
        settingsBtn.onclick = () => {
            console.log('🔘 Settings clicked!');
            showScreen('settings');
        };
        console.log('✅ Settings listener added');
    }
    
    // 🆕 Inscribe Ordinal button
    const inscribeOrdinalBtn = document.getElementById('inscribe-ordinal-btn');
    console.log('   inscribe-ordinal-btn:', !!inscribeOrdinalBtn);
    if (inscribeOrdinalBtn) {
        inscribeOrdinalBtn.onclick = () => {
            console.log('💎 Inscribe Ordinal clicked!');
            openInscribeModal();
        };
    }
    
    // 🆕 Etch Rune button (renamed from create-rune-btn)
    const etchRuneBtn = document.getElementById('etch-rune-btn');
    console.log('   etch-rune-btn:', !!etchRuneBtn);
    if (etchRuneBtn) {
        etchRuneBtn.onclick = () => {
            console.log('✨ Etch Rune clicked!');
            openEtchRuneModal();
        };
    }
    
    // Legacy: Keep for backward compatibility
    const createRuneBtn = document.getElementById('create-rune-btn');
    console.log('   create-rune-btn:', !!createRuneBtn);
    if (createRuneBtn) {
        createRuneBtn.onclick = () => {
            console.log('🎨 Create Rune clicked!');
            openEtchRuneModal(); // Use new modal
        };
        console.log('✅ Create Rune listener added');
    }
    
    // Send Screen
    document.getElementById('send-confirm-btn')?.addEventListener('click', handleSend);
    document.getElementById('back-from-send-btn')?.addEventListener('click', () => showScreen('wallet'));
    
    // Send Inscription Screen
    document.getElementById('send-inscription-confirm-btn')?.addEventListener('click', handleSendInscription);
    document.getElementById('back-from-send-inscription-btn')?.addEventListener('click', () => showScreen('wallet'));
    
    // Receive Screen
    document.getElementById('copy-receive-address-btn')?.addEventListener('click', handleCopyAddress);
    document.getElementById('back-from-receive-btn')?.addEventListener('click', () => showScreen('wallet'));
    
    // Settings Screen
    document.getElementById('view-mnemonic-btn')?.addEventListener('click', () => {
        showScreen('view-mnemonic-security');
        // 🛡️ Initialize security check when screen opens
        setTimeout(initMnemonicSecurityScreen, 100);
    });
    document.getElementById('view-private-key-btn')?.addEventListener('click', () => showScreen('view-private-key'));
    document.getElementById('export-wallet-btn')?.addEventListener('click', handleExportWallet);
    document.getElementById('lock-wallet-btn')?.addEventListener('click', handleLockWallet);
    document.getElementById('autolock-timeout')?.addEventListener('change', handleAutolockTimeoutChange);
    document.getElementById('reset-wallet-btn')?.addEventListener('click', handleResetWallet);
    document.getElementById('back-from-settings-btn')?.addEventListener('click', () => showScreen('wallet'));
    
    // Reset confirmation modal listeners
    document.getElementById('cancel-reset-btn')?.addEventListener('click', closeResetConfirmModal);
    document.getElementById('confirm-reset-btn')?.addEventListener('click', executeResetWallet);
    document.getElementById('reset-confirm-input')?.addEventListener('input', handleResetConfirmInput);
    document.getElementById('reset-confirm-input')?.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') executeResetWallet();
    });
    document.getElementById('reset-confirm-modal')?.addEventListener('click', (e) => {
        if (e.target.id === 'reset-confirm-modal') closeResetConfirmModal();
    });
    
    // View Mnemonic Security Screen
    document.getElementById('reveal-mnemonic-btn')?.addEventListener('click', handleRevealMnemonic);
    document.getElementById('copy-mnemonic-btn')?.addEventListener('click', handleCopyMnemonic);
    document.getElementById('back-from-mnemonic-view-btn')?.addEventListener('click', () => {
        // 🔒 Security: Clear mnemonic and reset screen
        currentMnemonicWords = [];
        document.getElementById('view-mnemonic-password').value = '';
        document.getElementById('mnemonic-password-step')?.classList.remove('hidden');
        document.getElementById('mnemonic-revealed-step')?.classList.add('hidden');
        document.getElementById('internet-security-warning')?.classList.remove('hidden');
        const grid = document.getElementById('mnemonic-grid-security');
        if (grid) grid.innerHTML = '';
        showScreen('settings');
    });
    
    // View Private Key Screen
    document.getElementById('reveal-key-btn')?.addEventListener('click', handleRevealPrivateKey);
    document.getElementById('copy-key-btn')?.addEventListener('click', handleCopyPrivateKey);
    document.getElementById('back-from-key-view-btn')?.addEventListener('click', () => showScreen('settings'));
    
    // ⚡ Network Selector (Mainnet/Lightning/Testnet)
    console.log('⚡ Setting up Network Selector...');
    const networkDropdownBtn = document.getElementById('network-dropdown-btn');
    const networkDropdownMenu = document.getElementById('network-dropdown-menu');
    const networkOptions = document.querySelectorAll('.network-option');
    
    console.log('   🔍 networkDropdownBtn:', networkDropdownBtn);
    console.log('   🔍 networkDropdownMenu:', networkDropdownMenu);
    console.log('   🔍 networkOptions count:', networkOptions.length);
    
    if (!networkDropdownBtn) {
        console.error('   ❌ Network dropdown button NOT FOUND!');
        console.error('   📍 Checking if element exists in DOM...');
        const allButtons = document.querySelectorAll('button');
        console.error('   📊 Total buttons in DOM:', allButtons.length);
    }
    
    if (networkDropdownBtn) {
        console.log('   ✅ Network dropdown button found!');
        console.log('   🎯 Network dropdown menu found:', networkDropdownMenu);
        
        // Toggle dropdown
        networkDropdownBtn.addEventListener('click', (e) => {
            console.log('   🖱️  Network dropdown clicked!');
            e.stopPropagation();
            
            const isOpen = networkDropdownBtn.classList.contains('open');
            console.log('   📊 Current state - isOpen:', isOpen);
            
            networkDropdownBtn.classList.toggle('open');
            networkDropdownMenu.classList.toggle('hidden');
            
            console.log('   📊 New state - open:', networkDropdownBtn.classList.contains('open'));
            console.log('   📊 New state - hidden:', networkDropdownMenu.classList.contains('hidden'));
        });
        
        // Close dropdown when clicking outside
        document.addEventListener('click', (e) => {
            if (!networkDropdownBtn.contains(e.target) && !networkDropdownMenu.contains(e.target)) {
                networkDropdownBtn.classList.remove('open');
                networkDropdownMenu.classList.add('hidden');
            }
        });
        
        console.log('   ✅ Network dropdown configured');
    }
    
    // Network option clicks
    networkOptions.forEach(option => {
        option.addEventListener('click', () => {
            const network = option.dataset.network;
            console.log(`🔗 Switching to ${network}...`);
            switchNetwork(network);
            
            // Close dropdown
            networkDropdownBtn.classList.remove('open');
            networkDropdownMenu.classList.add('hidden');
        });
    });
    
    console.log(`   ✅ ${networkOptions.length} network options configured`);
    
    // Lightning Action Buttons
    const openChannelBtn = document.getElementById('open-channel-btn');
    const depositLightningBtn = document.getElementById('deposit-lightning-btn');
    const withdrawLightningBtn = document.getElementById('withdraw-lightning-btn');
    
    if (openChannelBtn) {
        openChannelBtn.addEventListener('click', handleOpenChannel);
        console.log('   ✅ Open Channel button configured');
    }
    
    if (depositLightningBtn) {
        depositLightningBtn.addEventListener('click', handleDepositToLightning);
        console.log('   ✅ Deposit button configured');
    }
    
    if (withdrawLightningBtn) {
        withdrawLightningBtn.addEventListener('click', handleWithdrawFromLightning);
        console.log('   ✅ Withdraw button configured');
    }
    
    // Tabs
    console.log('🔗 Setting up tab listeners...');
    const tabs = document.querySelectorAll('.tab');
    console.log('   Found tabs:', tabs.length);
    
    tabs.forEach((tab, index) => {
        const tabName = tab.dataset.tab;
        console.log(`   Tab ${index}:`, tabName, tab);
        
        tab.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            
            const clickedTabName = e.target.dataset.tab || e.currentTarget.dataset.tab;
            console.log('🔘 Tab clicked:', clickedTabName);
            
            if (clickedTabName) {
                switchTab(clickedTabName);
            } else {
                console.error('❌ No tab name found on click!');
            }
        });
    });
    
    console.log('✅ Tab listeners configured');
}

// Re-bind Tab Listeners (chamado após mostrar wallet screen)
function rebindTabListeners() {
    console.log('🔄 Re-binding tab listeners...');
    
    const tabs = document.querySelectorAll('.tab');
    console.log('   Found tabs:', tabs.length);
    
    tabs.forEach((tab, index) => {
        // Remover listeners antigos clonando o elemento
        const newTab = tab.cloneNode(true);
        tab.parentNode.replaceChild(newTab, tab);
        
        const tabName = newTab.dataset.tab;
        console.log(`   Re-binding Tab ${index}:`, tabName);
        
        newTab.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            
            const clickedTabName = e.target.dataset.tab || e.currentTarget.dataset.tab;
            console.log('🔘 Tab clicked:', clickedTabName);
            
            if (clickedTabName) {
                switchTab(clickedTabName);
            } else {
                console.error('❌ No tab name found on click!');
            }
        });
    });
    
    console.log('✅ Tab listeners re-bound');
}

// Screen Management
function showScreen(screenName) {
    console.log('📱 Switching to screen:', screenName);
    
    try {
        // Esconder todas as screens
        document.querySelectorAll('.screen').forEach(screen => {
            screen.classList.add('hidden');
        });
        
        // Mostrar screen desejada
        const screen = document.getElementById(`${screenName}-screen`);
        if (screen) {
            screen.classList.remove('hidden');
            currentScreen = screenName;
            console.log('✅ Screen shown:', screenName);
            
            // Feature slider para tela welcome
            if (screenName === 'no-wallet') {
                setTimeout(() => initFeatureSlider(), 100);
            } else {
                stopFeatureSlider();
            }
            
            // Se é a wallet screen, re-vincular listeners das tabs e inicializar Network Selector
            if (screenName === 'wallet') {
                console.log('🔄 Wallet screen shown, re-binding tab listeners...');
                setTimeout(() => {
                    rebindTabListeners();
                    initializeNetworkSelector(); // ⚡ Initialize Network Selector
                }, 100); // Pequeno delay para garantir que DOM está pronto
            }
            
            // Se é a send screen, carregar fees dinâmicas
            if (screenName === 'send') {
                console.log('📊 Loading fees for Send Bitcoin screen...');
                loadBitcoinSendFees();
            }
        } else {
            console.error('❌ Screen not found:', `${screenName}-screen`);
        }
    } catch (error) {
        console.error('❌ Error showing screen:', error);
    }
}

// Tab Management
async function switchTab(tabName) {
    console.log('🔄 switchTab called:', tabName);
    
    // Atualizar tabs
    document.querySelectorAll('.tab').forEach(tab => {
        tab.classList.remove('active');
    });
    document.querySelector(`[data-tab="${tabName}"]`)?.classList.add('active');
    
    // Atualizar conteúdo
    document.querySelectorAll('.tab-content').forEach(content => {
        content.classList.add('hidden');
    });
    
    const targetTab = document.getElementById(`${tabName}-tab`);
    console.log('  Target element:', targetTab ? 'Found' : 'NOT FOUND', `(#${tabName}-tab)`);
    
    if (targetTab) {
        targetTab.classList.remove('hidden');
        console.log('  ✅ Tab switched to:', tabName);
        
        // Se for a tab de Activity, carregar transações
        if (tabName === 'activity') {
            console.log('  🔄 Activity tab selected, loading transactions...');
            
            // 🔄 MOSTRAR LOADING IMEDIATAMENTE
            const activityList = document.getElementById('activity-list');
            if (activityList) {
                activityList.innerHTML = `
                    <div class="loading-container">
                        <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
                        <div class="loading-spinner"></div>
                        <p class="loading-text">Loading transactions...</p>
                    </div>
                `;
            }
            
            // Buscar endereço da wallet
            try {
                const response = await chrome.runtime.sendMessage({
                    action: 'getWalletInfo'
                });
                
                if (response && response.success && response.data) {
                    const address = response.data.address;
                    console.log('  📍 Got wallet address:', address);
                    await loadActivity(address);
                } else {
                    console.error('  ❌ Failed to get wallet address');
                    if (activityList) {
                        activityList.innerHTML = `
                            <div class="empty-state">
                                <span class="empty-icon">⚠️</span>
                                <p>Failed to load wallet address</p>
                            </div>
                        `;
                    }
                }
            } catch (error) {
                console.error('  ❌ Error getting wallet info:', error);
                if (activityList) {
                    activityList.innerHTML = `
                        <div class="empty-state">
                            <span class="empty-icon">⚠️</span>
                            <p>Error loading transactions</p>
                        </div>
                    `;
                }
            }
        }
        
        // Se for a tab de Ordinals, carregar inscriptions
        if (tabName === 'ordinals') {
            console.log('  🖼️  Ordinals tab selected, loading inscriptions...');
            
            // 🔄 MOSTRAR LOADING IMEDIATAMENTE
            const ordinalsList = document.getElementById('ordinals-list');
            if (ordinalsList) {
                ordinalsList.innerHTML = `
                    <div class="loading-container">
                        <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
                        <div class="loading-spinner"></div>
                        <p class="loading-text">Loading inscriptions...</p>
                    </div>
                `;
            }
            
            // Buscar endereço da wallet
            try {
                const response = await chrome.runtime.sendMessage({
                    action: 'getWalletInfo'
                });
                
                if (response && response.success && response.data) {
                    const address = response.data.address;
                    console.log('  📍 Got wallet address:', address);
                    await loadOrdinals(address);
                } else {
                    console.error('  ❌ Failed to get wallet address');
                    if (ordinalsList) {
                        ordinalsList.innerHTML = `
                            <div class="empty-state">
                                <span class="empty-icon">⚠️</span>
                                <p>Failed to load wallet address</p>
                            </div>
                        `;
                    }
                }
            } catch (error) {
                console.error('  ❌ Error getting wallet info:', error);
                if (ordinalsList) {
                    ordinalsList.innerHTML = `
                        <div class="empty-state">
                            <span class="empty-icon">⚠️</span>
                            <p>Error loading inscriptions</p>
                        </div>
                    `;
                }
            }
        }
        
        // Se for a tab de Runes, carregar runes
        if (tabName === 'runes') {
            console.log('  🪙 Runes tab selected, loading runes...');
            
            // 🔄 MOSTRAR LOADING IMEDIATAMENTE
            const runesList = document.getElementById('runes-list');
            if (runesList) {
                runesList.innerHTML = `
                    <div class="loading-container">
                        <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
                        <div class="loading-spinner"></div>
                        <p class="loading-text">Loading runes...</p>
                    </div>
                `;
            }
            
            // Buscar endereço da wallet
            try {
                const response = await chrome.runtime.sendMessage({
                    action: 'getWalletInfo'
                });
                
                if (response && response.success && response.data) {
                    const address = response.data.address;
                    console.log('  📍 Got wallet address:', address);
                    await loadRunes(address);
                } else {
                    console.error('  ❌ Failed to get wallet address');
                    if (runesList) {
                        runesList.innerHTML = `
                            <div class="empty-state">
                                <span class="empty-icon">⚠️</span>
                                <p>Failed to load wallet address</p>
                            </div>
                        `;
                    }
                }
            } catch (error) {
                console.error('  ❌ Error getting wallet info:', error);
                if (runesList) {
                    runesList.innerHTML = `
                        <div class="empty-state">
                            <span class="empty-icon">⚠️</span>
                            <p>Error loading runes</p>
                        </div>
                    `;
                }
            }
        }
        
        // Se for a tab de Market, carregar listings automaticamente
        if (tabName === 'market') {
            console.log('  🏪 Market tab selected, loading listings...');
            loadMarketData();
        }
    } else {
        console.error('  ❌ Tab element not found:', `${tabName}-tab`);
    }
}

// Verificar se tem wallet
async function checkWallet() {
    const result = await chrome.storage.local.get(['walletEncrypted']);
    return !!result.walletEncrypted;
}

// Gerar Mnemonic
async function handleGenerateMnemonic() {
    console.log('🔥 handleGenerateMnemonic called!');
    
    try {
        const wordCountEl = document.getElementById('mnemonic-length');
        const passwordEl = document.getElementById('create-password');
        const confirmPasswordEl = document.getElementById('confirm-password');
        
        console.log('   Elements found:', {
            wordCountEl: !!wordCountEl,
            passwordEl: !!passwordEl,
            confirmPasswordEl: !!confirmPasswordEl
        });
        
        const wordCount = parseInt(wordCountEl?.value || '12');
        const password = passwordEl?.value || '';
        const confirmPassword = confirmPasswordEl?.value || '';
        
        console.log('   Values:', {
            wordCount,
            password: password ? `***${password.length} chars` : '(empty)',
            confirmPassword: confirmPassword ? `***${confirmPassword.length} chars` : '(empty)'
        });
        
        if (!password || password.length < 6) {
            console.error('❌ Password too short');
            showNotification('Password must be at least 6 characters', 'error');
            return;
        }
        
        if (password !== confirmPassword) {
            console.error('❌ Passwords do not match');
            showNotification('Passwords do not match', 'error');
            return;
        }
        
        console.log('✅ Validation passed, generating wallet...');
        console.log('   Showing loading...');
        showLoading('Generating wallet...');
        
        console.log('📨 Sending message to background script...');
        console.log('   Action: generateWallet');
        console.log('   Data:', { wordCount, password: '***' });
        
        // Enviar mensagem para background script gerar wallet
        const response = await chrome.runtime.sendMessage({
            action: 'generateWallet',
            data: { wordCount, password }
        });
        
        console.log('📨 Response received:', response);
        
        if (response && response.success) {
            console.log('✅ Wallet generated successfully!');
            console.log('   Mnemonic length:', response.mnemonic?.split(' ').length, 'words');
            console.log('   Address:', response.address);
            
            // Salvar mnemonic temporariamente para usar após mudar de tela
            const mnemonicToShow = response.mnemonic;
            
            console.log('   Switching to show-mnemonic screen...');
            showScreen('show-mnemonic');
            console.log('✅ Screen switched!');
            
            // Usar requestAnimationFrame + setTimeout para garantir que o DOM foi atualizado
            requestAnimationFrame(() => {
                setTimeout(() => {
                    console.log('🔍 Trying to find mnemonic-display element...');
                    const mnemonicDisplay = document.getElementById('mnemonic-display');
                    console.log('   Element found:', !!mnemonicDisplay);
                    console.log('   Element visible:', mnemonicDisplay?.offsetParent !== null);
                    
                    if (mnemonicDisplay) {
                        // Quebrar em palavras e criar grid
                        const words = mnemonicToShow.split(' ');
                        mnemonicDisplay.innerHTML = '';
                        
                        words.forEach((word, index) => {
                            const wordCard = document.createElement('div');
                            wordCard.className = 'mnemonic-word';
                            wordCard.innerHTML = `<span class="mnemonic-word-number">${index + 1}.</span>${word}`;
                            mnemonicDisplay.appendChild(wordCard);
                        });
                        
                        console.log('✅ Mnemonic set in display:', words.length, 'words');
                    } else {
                        console.error('❌ mnemonic-display element still not found!');
                        console.error('   Available screens:', 
                            Array.from(document.querySelectorAll('.screen'))
                                .map(s => s.id));
                    }
                }, 200);
            });
        } else {
            console.error('❌ Generate failed:', response?.error || 'No response');
            showNotification(response?.error || 'Failed to generate wallet', 'error');
        }
    } catch (error) {
        console.error('❌ Error generating wallet:', error);
        console.error('   Error stack:', error.stack);
        showNotification('Failed to generate wallet: ' + error.message, 'error');
    } finally {
        console.log('🔄 Hiding loading...');
        hideLoading();
        console.log('✅ handleGenerateMnemonic completed');
    }
}

// Mnemonic Saved
async function handleMnemonicSaved() {
    showScreen('wallet');
    await loadWalletData();
    
    // 📊 Track wallet creation
    try {
        const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (response?.success) {
            trackUserAction(response.data.address, 'wallet_created', {
                timestamp: Date.now()
            });
        }
    } catch (error) {
        console.warn('⚠️  Failed to track wallet creation:', error);
    }
}

// Restore Wallet
async function handleRestoreWallet() {
    const mnemonic = document.getElementById('restore-mnemonic').value.trim();
    const password = document.getElementById('restore-password').value;
    
    if (!mnemonic) {
        showNotification('Please enter your mnemonic phrase', 'error');
        return;
    }
    
    if (!password || password.length < 6) {
        showNotification('Password must be at least 6 characters', 'error');
        return;
    }
    
    showLoading('Restoring wallet...');
    
    try {
        const response = await chrome.runtime.sendMessage({
            action: 'restoreWallet',
            data: { mnemonic, password }
        });
        
        if (response.success) {
            // 🔒 SECURITY: Clear password field immediately
            clearAllPasswordFields();
            showNotification('Wallet restored successfully!', 'success');
            showScreen('wallet');
            await loadWalletData();
            
            // 📊 Track wallet restore
            try {
                const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
                if (walletInfo?.success) {
                    trackUserAction(walletInfo.data.address, 'wallet_restored', {
                        timestamp: Date.now()
                    });
                }
            } catch (error) {
                console.warn('⚠️  Failed to track wallet restore:', error);
            }
        } else {
            showNotification(response.error || 'Failed to restore wallet', 'error');
        }
    } catch (error) {
        console.error('Error restoring wallet:', error);
        showNotification('Failed to restore wallet', 'error');
    } finally {
        hideLoading();
    }
}

// Re-bind Wallet Buttons (chamado após carregar dados)
function rebindWalletButtons() {
    console.log('🔗 ========== REBINDING WALLET BUTTONS (EVENT DELEGATION) ==========');
    
    // Usar EVENT DELEGATION no documento
    // Isso captura cliques nos botões E seus filhos!
    
    document.addEventListener('click', (e) => {
        // Encontrar o botão clicado (pode ser filho do botão)
        const target = e.target.closest('button');
        
        if (!target) return;
        
        // Copy Address
        if (target.id === 'copy-address-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Copy Address clicked!');
            handleCopyAddress();
            return;
        }
        
        // Send
        if (target.id === 'send-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Send clicked!');
            showScreen('send');
            return;
        }
        
        // Receive
        if (target.id === 'receive-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Receive clicked!');
            showScreen('receive');
            return;
        }
        
        // Settings
        if (target.id === 'settings-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Settings clicked!');
            showScreen('settings');
            return;
        }
        
        // ⚡ L2 Buttons (NEW!)
        if (target.id === 'l2-deposit-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Deposit clicked!');
            if (window.krayL2 && window.krayL2.showL2DepositScreen) {
                window.krayL2.showL2DepositScreen();
            }
            return;
        }
        
        if (target.id === 'l2-withdraw-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Withdraw clicked!');
            if (window.krayL2 && window.krayL2.showL2WithdrawScreen) {
                window.krayL2.showL2WithdrawScreen();
            }
            return;
        }
        
        if (target.id === 'l2-transfer-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Transfer clicked!');
            if (window.krayL2 && window.krayL2.showL2TransferScreen) {
                window.krayL2.showL2TransferScreen();
            }
            return;
        }
        
        if (target.id === 'l2-swap-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Swap clicked!');
            if (window.krayL2 && window.krayL2.showL2SwapScreen) {
                window.krayL2.showL2SwapScreen();
            }
            return;
        }
        
        if (target.id === 'l2-marketplace-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Marketplace clicked!');
            if (window.krayL2 && window.krayL2.showL2MarketplaceScreen) {
                window.krayL2.showL2MarketplaceScreen();
            }
            return;
        }
        
        if (target.id === 'l2-rewards-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 L2 Rewards clicked!');
            if (window.krayL2 && window.krayL2.showL2RewardsScreen) {
                window.krayL2.showL2RewardsScreen();
            }
            return;
        }
        
        // ⚡ L2 BACK Buttons (NEW!) - Return to wallet screen in L2 mode
        if (target.id === 'back-from-l2-deposit-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Back from L2 Deposit');
            showScreen('wallet'); // Volta para wallet screen
            // Reativar L2 content
            document.getElementById('kray-l2-content')?.classList.remove('hidden');
            return;
        }
        
        if (target.id === 'back-from-l2-transfer-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Back from L2 Transfer');
            showScreen('wallet');
            document.getElementById('kray-l2-content')?.classList.remove('hidden');
            return;
        }
        
        if (target.id === 'back-from-l2-swap-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Back from L2 Swap');
            showScreen('wallet');
            document.getElementById('kray-l2-content')?.classList.remove('hidden');
            return;
        }
        
        if (target.id === 'back-from-l2-withdraw-btn') {
            e.preventDefault();
            e.stopPropagation();
            console.log('🔘 Back from L2 Withdraw');
            showScreen('wallet');
            document.getElementById('kray-l2-content')?.classList.remove('hidden');
            return;
        }
    }, true); // true = capture phase (pega antes de qualquer outro)
    
    console.log('✅ Event delegation configured for all wallet buttons!');
}

// Global flag para prevenir chamadas duplicadas
let isLoadingActivity = false;
const ACTIVITY_DEBUG = true; // 🔧 Toggle para debug detalhado - ENABLED FOR TROUBLESHOOTING
const ACTIVITY_VERSION = '2024-12-11-INSCRIPTION-CLICK-v1'; // Version for debugging

// ═══════════════════════════════════════════════════════════════════════════
// 🚀 ACTIVITY TAB V3 - EXACT MOBILE CLONE
// ═══════════════════════════════════════════════════════════════════════════
// Optimizations:
// 1. Show only 10 transactions initially + "View More" button
// 2. Enrich only first 5 visible transactions
// 3. Market detection: Bought/Sold badges (exactly like mobile)
// 4. Thumbnails, rune names, symbols - everything!
// ═══════════════════════════════════════════════════════════════════════════

const ACTIVITY_PAGE_SIZE = 10;
let activityEnrichments = new Map();
let activityEnrichedTxIds = new Set();
let allActivityTransactions = [];
let activityVisibleCount = ACTIVITY_PAGE_SIZE;
let currentActivityAddress = '';

async function loadActivity(address) {
    const activityList = document.getElementById('activity-list');
    
    if (!activityList) return;
    
    if (!address) {
        activityList.innerHTML = `<div class="empty-state"><p>No wallet connected</p></div>`;
        return;
    }
    
    if (isLoadingActivity) return;
    isLoadingActivity = true;
    currentActivityAddress = address;
    
    console.log('📜 Loading Activity V3...');
    
    // Show loading
    activityList.innerHTML = `
        <div style="display: flex; flex-direction: column; align-items: center; padding: 40px 20px; gap: 12px;">
            <img src="../assets/logo.png" alt="Loading" style="width: 48px; height: 48px; opacity: 0.6; animation: pulse 2s infinite;">
            <span style="color: #888; font-size: 13px;">Loading transactions...</span>
        </div>
    `;
    
    try {
        // Fetch transactions
        const [mempoolRes, chainRes] = await Promise.all([
            fetch(`https://mempool.space/api/address/${address}/txs/mempool`, { 
                signal: AbortSignal.timeout(5000) 
            }).catch(() => ({ ok: false })),
            fetch(`https://mempool.space/api/address/${address}/txs/chain`, { 
                signal: AbortSignal.timeout(5000) 
            })
        ]);
        
        const mempoolTxs = mempoolRes.ok ? await mempoolRes.json() : [];
        if (!chainRes.ok) throw new Error('Failed to fetch transactions');
        const chainTxs = await chainRes.json();
        
        // Combine and deduplicate
        const seenTxids = new Set();
        allActivityTransactions = [];
        
        for (const tx of mempoolTxs) {
            if (!seenTxids.has(tx.txid)) {
                seenTxids.add(tx.txid);
                allActivityTransactions.push(tx);
            }
        }
        for (const tx of chainTxs) {
            if (!seenTxids.has(tx.txid)) {
                seenTxids.add(tx.txid);
                allActivityTransactions.push(tx);
            }
        }
        
        console.log(`✅ Got ${allActivityTransactions.length} transactions`);
        
        if (allActivityTransactions.length === 0) {
            activityList.innerHTML = `
                <div style="display: flex; flex-direction: column; align-items: center; padding: 60px 20px; gap: 16px;">
                    <img src="../assets/logo.png" alt="No transactions" style="width: 48px; height: 48px; opacity: 0.3;">
                    <h3 style="margin: 0; color: #fff; font-size: 16px;">No transactions yet</h3>
                    <p style="margin: 0; color: #888; font-size: 13px;">Your transaction history will appear here</p>
                </div>
            `;
            isLoadingActivity = false;
            return;
        }
        
        // Reset visible count
        activityVisibleCount = ACTIVITY_PAGE_SIZE;
        activityEnrichments.clear();
        activityEnrichedTxIds.clear();
        
        // Render initial transactions
        renderActivityTransactions(address);
        
        // Enrich first 5 only
        enrichActivityBatch(allActivityTransactions.slice(0, 5), address);
        
    } catch (error) {
        console.error('❌ Error:', error);
        activityList.innerHTML = `
            <div style="display: flex; flex-direction: column; align-items: center; padding: 60px 20px; gap: 16px;">
                <img src="../assets/logo.png" alt="Error" style="width: 48px; height: 48px; opacity: 0.3;">
                <h3 style="margin: 0; color: #fff; font-size: 16px;">Error loading transactions</h3>
                <p style="margin: 0; color: #888; font-size: 13px;">${error.message}</p>
            </div>
        `;
    } finally {
        isLoadingActivity = false;
    }
}

// Render visible transactions + View More button
function renderActivityTransactions(address) {
    const activityList = document.getElementById('activity-list');
    if (!activityList) return;
    
    activityList.innerHTML = '';
    
    const visibleTxs = allActivityTransactions.slice(0, activityVisibleCount);
    
    for (const tx of visibleTxs) {
        const enrichment = activityEnrichments.get(tx.txid);
        const item = createActivityItem(tx, address, enrichment);
        activityList.appendChild(item);
    }
    
    // View More button
    if (activityVisibleCount < allActivityTransactions.length) {
        const remaining = allActivityTransactions.length - activityVisibleCount;
        const btn = document.createElement('button');
        btn.style.cssText = 'width: 100%; padding: 14px; background: rgba(255,255,255,0.05); border: 1px solid rgba(255,255,255,0.1); border-radius: 12px; color: #888; font-size: 13px; cursor: pointer; margin-top: 8px; display: flex; align-items: center; justify-content: center; gap: 8px;';
        btn.innerHTML = `View More (${remaining} remaining) <span style="font-size: 10px;">▼</span>`;
        btn.onclick = () => {
            activityVisibleCount += ACTIVITY_PAGE_SIZE;
            renderActivityTransactions(address);
            // Enrich new batch
            const newTxs = allActivityTransactions.slice(activityVisibleCount - ACTIVITY_PAGE_SIZE, activityVisibleCount);
            enrichActivityBatch(newTxs.filter(tx => !activityEnrichedTxIds.has(tx.txid)), address);
        };
        activityList.appendChild(btn);
    }
}

// 🎨 Create activity item with full enrichment (like mobile)
function createActivityItem(tx, myAddress, enrichment) {
    const item = document.createElement('div');
    item.className = 'activity-item';
    item.setAttribute('data-txid', tx.txid);
    item.style.cssText = 'display: flex; align-items: center; padding: 14px; background: rgba(255,255,255,0.03); border-radius: 12px; margin-bottom: 8px; border: 1px solid rgba(255,255,255,0.08); cursor: pointer; transition: all 0.2s;';
    
    // ✂️ Check if this is a Split transaction
    let isSplitTx = false;
    let splitOutputs = 0;
    try {
        const splitTxids = JSON.parse(localStorage.getItem('splitTransactions') || '[]');
        const splitInfo = splitTxids.find(s => s.txid === tx.txid);
        if (splitInfo) {
            isSplitTx = true;
            splitOutputs = splitInfo.outputs || 0;
        }
    } catch (e) { /* ignore */ }
    
    // 🔥 Check if this is a Burn transaction
    let isBurnTx = false;
    let burnInfo = null;
    try {
        const burnTxids = JSON.parse(localStorage.getItem('burnTransactions') || '[]');
        burnInfo = burnTxids.find(b => b.txid === tx.txid);
        if (burnInfo) {
            isBurnTx = true;
        }
    } catch (e) { /* ignore */ }
    
    // Also check if enrichment indicates Cenotaph (burn from API)
    if (enrichment?.data?.isCenotaph || enrichment?.isCenotaph) {
        isBurnTx = true;
    }
    
    // 🌉 Check if this is a Bridge transaction (L2 deposits/withdrawals)
    // Method 1: Check saved txids from storage
    let isBridgeTx = false;
    let bridgeInfo = null;
    try {
        const bridgeTxids = JSON.parse(localStorage.getItem('bridgeTransactions') || '[]');
        bridgeInfo = bridgeTxids.find(b => b.txid === tx.txid);
        if (bridgeInfo) {
            isBridgeTx = true;
        }
    } catch (e) { /* ignore */ }
    
    // Method 2: Address-based heuristic - check if transaction involves bridge address
    const KRAY_BRIDGE_ADDRESS = 'bc1pxtt3tzrcp4zxy5z43vzhwac47dc6tl4s6l0gfdyuzvx66ljr3x7srwetnd';
    const bridgeAddressInvolved = tx.vin?.some(v => 
        v.prevout?.scriptpubkey_address === KRAY_BRIDGE_ADDRESS
    ) || tx.vout?.some(v => 
        v.scriptpubkey_address === KRAY_BRIDGE_ADDRESS
    );
    
    // Calculate market info (exactly like mobile)
    const marketInfo = calculateMarketInfo(tx, myAddress);
    const { isReceive, isSale, isBuy, isMarket, btcAmount, sentFromWallet, isMultiParty } = marketInfo;
    
    const isPending = !tx.status.confirmed;
    const isRuneTx = enrichment?.type === 'rune';
    const isOrdinalTx = enrichment?.type === 'ordinal';
    
    // Combined bridge detection: storage OR address heuristic
    const isBridgeHeuristic = isRuneTx && bridgeAddressInvolved;
    isBridgeTx = isBridgeTx || isBridgeHeuristic;
    
    // 🌉 Detect bridge direction: Withdraw (bridge in inputs) vs Deposit (bridge in outputs)
    const bridgeInInputs = tx.vin?.some(v => 
        v.prevout?.scriptpubkey_address === KRAY_BRIDGE_ADDRESS
    );
    const bridgeInOutputs = tx.vout?.some(v => 
        v.scriptpubkey_address === KRAY_BRIDGE_ADDRESS
    );
    const isBridgeWithdraw = isBridgeTx && bridgeInInputs;  // L2->L1: user receives (green)
    const isBridgeDeposit = isBridgeTx && bridgeInOutputs;  // L1->L2: user sends (red)
    
    // 🔄 SWAP DETECTION: User sends BTC but receives Rune (or vice versa)
    const hasRunestone = tx.vout.some(v => v.scriptpubkey?.startsWith('6a5d'));
    const hasDustOutputToUser = tx.vout.some(v => 
        v.scriptpubkey_address === myAddress && v.value <= 600
    );
    const isSwapTx = isRuneTx && sentFromWallet && hasDustOutputToUser && hasRunestone && !isBridgeTx;
    
    // Check saved swap info for direction
    let swapInfo = null;
    try {
        const swapTxids = JSON.parse(localStorage.getItem('swapTransactions') || '[]');
        swapInfo = swapTxids.find(s => s.txid === tx.txid);
    } catch (e) { /* ignore */ }
    
    // 🔄 SWAP: User ALWAYS receives something (the asset they wanted)
    const swapIsReceive = true;  // Swaps always show as received (green)
    const swapIsSell = swapInfo?.direction === 'SELL';  // Track SELL for amount display
    
    // 🎨 Determine color logic:
    // - SWAP: always green (user received what they wanted)
    // - BRIDGE: withdraw = green (received), deposit = red (sent)
    // - Regular: based on sentFromWallet
    const runeIsReceive = isSwapTx 
        ? swapIsReceive  // Swap: always true (green)
        : (isBridgeTx
            ? isBridgeWithdraw  // Bridge: withdraw = green, deposit = red
            : (isRuneTx ? !sentFromWallet : isReceive));
    
    // Determine type label (simplified - badges/emojis already show asset type)
    let typeLabel;
    if (isBurnTx) {
        typeLabel = '🔥 Burned';
    } else if (isSplitTx) {
        typeLabel = '✂️ Split';
    } else if (isSwapTx && !isBridgeTx) {
        typeLabel = 'Swapped';
    } else if (isMarket && isSale) {
        typeLabel = 'Sold';
    } else if (isMarket && isBuy) {
        typeLabel = 'Bought';
    } else if (isRuneTx ? runeIsReceive : isReceive) {
        typeLabel = 'Received';
    } else {
        typeLabel = 'Sent';
    }
    
    // Colors (like mobile)
    // Swap SELL: green (received BTC)
    // Runes: received = green, sent = red
    // BTC: always orange
    let amountColor;
    if (isSwapTx && swapIsSell) {
        // SELL swap: green (received BTC - user got what they wanted)
        amountColor = '#10b981';
    } else if (isRuneTx) {
        amountColor = runeIsReceive ? '#10b981' : '#ef4444';
    } else {
        amountColor = '#f7931a'; // Bitcoin orange
    }
    
    // Sign: + for received, - for sent
    // Swap SELL: always + (received BTC)
    const sign = (isSwapTx && swapIsSell)
        ? '+'  // SELL swap: always + (received BTC)
        : (isRuneTx && enrichment?.runeAmount) 
            ? (runeIsReceive ? '+' : '-')
            : (isReceive ? '+' : '-');
    
    // Build HTML
    let iconHtml, infoHtml, amountHtml;
    
    // 🔥 BURN transaction display (same layout as regular Rune transactions)
    if (isBurnTx) {
        const thumb = enrichment?.thumbnail || burnInfo?.thumbnail;
        const symbol = enrichment?.runeSymbol || burnInfo?.runeSymbol || '⧈';
        const name = enrichment?.runeName || burnInfo?.runeName || 'Rune';
        const amount = enrichment?.runeAmount || burnInfo?.amount || '';
        
        // Same icon style as regular runes, but with red theme
        iconHtml = `
            <div style="width: 44px; height: 44px; border-radius: 10px; background: rgba(247,147,26,0.15); overflow: hidden; margin-right: 12px; flex-shrink: 0; border: 1px solid rgba(247,147,26,0.3);">
                ${thumb 
                    ? `<img src="${thumb}" style="width:100%;height:100%;object-fit:cover;" onerror="this.outerHTML='<span style=font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#f7931a;>${symbol}</span>'">`
                    : `<span style="font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#f7931a;">${symbol}</span>`}
            </div>`;
        
        // Same info layout as regular runes
        infoHtml = `
            <div style="flex: 1; min-width: 0;">
                <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 2px; flex-wrap: wrap;">
                    <span style="font-weight: 600; font-size: 14px; color: #fff;">${typeLabel}</span>
                    ${isPending ? '<span style="font-size: 9px; background: rgba(245,158,11,0.2); color: #f59e0b; padding: 2px 6px; border-radius: 4px;">⏳</span>' : ''}
                </div>
                <div style="font-size: 12px; color: #f7931a; margin-bottom: 2px;">${name} ${symbol}</div>
                <div style="font-size: 11px; color: #888;">${isPending ? 'In mempool' : tx.status.block_time ? formatTimeAgo(tx.status.block_time) : 'Confirmed'}</div>
            </div>`;
        
        // Amount in red with minus sign
        amountHtml = amount 
            ? `<div style="text-align: right;"><div style="font-weight: 600; font-size: 14px; color: #ef4444;">-${amount}</div></div>`
            : `<div style="text-align: right;"><span style="font-size: 10px; background: rgba(247,147,26,0.2); color: #f7931a; padding: 3px 8px; border-radius: 4px;">⧈ RUNE</span></div>`;
    
    } else if (isRuneTx && enrichment) {
        const thumb = enrichment.thumbnail;
        const symbol = enrichment.runeSymbol || '⧈';
        const name = enrichment.runeName || 'Rune';
        const amount = enrichment.runeAmount || '';
        
        iconHtml = `
            <div style="width: 44px; height: 44px; border-radius: 10px; background: rgba(247,147,26,0.15); overflow: hidden; margin-right: 12px; flex-shrink: 0; border: 1px solid rgba(247,147,26,0.3);">
                ${thumb 
                    ? `<img src="${thumb}" style="width:100%;height:100%;object-fit:cover;" onerror="this.outerHTML='<span style=font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#f7931a;>${symbol}</span>'">`
                    : `<span style="font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#f7931a;">${symbol}</span>`}
            </div>`;
        
        // 🔄 SWAP / 🌉 BRIDGE / 🏪 Market badge for transactions
        const txBadge = isBridgeTx 
            ? '<span style="font-size: 9px; background: rgba(139,92,246,0.2); color: #8b5cf6; padding: 2px 6px; border-radius: 4px;">🌉 Bridge</span>'
            : isSwapBuy 
                ? '<span style="font-size: 9px; background: rgba(59,130,246,0.2); color: #3b82f6; padding: 2px 6px; border-radius: 4px;">🔄 Swap</span>' 
                : (isMarket ? '<span style="font-size: 9px; background: rgba(16,185,129,0.2); color: #10b981; padding: 2px 6px; border-radius: 4px;">🏪 Market</span>' : '');
        
        infoHtml = `
            <div style="flex: 1; min-width: 0;">
                <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 2px; flex-wrap: wrap;">
                    <span style="font-weight: 600; font-size: 14px; color: #fff;">${typeLabel}</span>
                    ${txBadge}
                    ${isPending ? '<span style="font-size: 9px; background: rgba(245,158,11,0.2); color: #f59e0b; padding: 2px 6px; border-radius: 4px;">⏳</span>' : ''}
                </div>
                <div style="font-size: 12px; color: #f7931a; margin-bottom: 2px;">${name} ${symbol}</div>
                <div style="font-size: 11px; color: #888;">${isPending ? 'In mempool' : tx.status.block_time ? formatTimeAgo(tx.status.block_time) : 'Confirmed'}</div>
            </div>`;
        
        amountHtml = amount 
            ? `<div style="text-align: right;"><div style="font-weight: 600; font-size: 14px; color: ${amountColor};">${sign}${amount}</div></div>`
            : `<div style="text-align: right;"><span style="font-size: 10px; background: rgba(247,147,26,0.2); color: #f7931a; padding: 3px 8px; border-radius: 4px;">⧈ RUNE</span></div>`;
    
    } else if (isOrdinalTx && enrichment) {
        const thumb = enrichment.thumbnail || enrichment.contentUrl;
        const inscNum = enrichment.inscriptionNumber;
        
        iconHtml = `
            <div style="width: 50px; height: 50px; border-radius: 10px; background: #111; overflow: hidden; margin-right: 12px; flex-shrink: 0; border: 1px solid rgba(255,255,255,0.2);">
                ${thumb 
                    ? `<img src="${thumb}" style="width:100%;height:100%;object-fit:cover;" onerror="this.outerHTML='<span style=font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#fff;>◉</span>'">`
                    : `<span style="font-size:20px;display:flex;align-items:center;justify-content:center;height:100%;color:#fff;">◉</span>`}
            </div>`;
        
        infoHtml = `
            <div style="flex: 1; min-width: 0;">
                <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 2px; flex-wrap: wrap;">
                    <span style="font-weight: 600; font-size: 14px; color: #fff;">${typeLabel}</span>
                    ${isMarket ? '<span style="font-size: 9px; background: rgba(16,185,129,0.2); color: #10b981; padding: 2px 6px; border-radius: 4px;">🏪 Market</span>' : ''}
                    ${isPending ? '<span style="font-size: 9px; background: rgba(245,158,11,0.2); color: #f59e0b; padding: 2px 6px; border-radius: 4px;">⏳</span>' : ''}
                </div>
                <div style="font-size: 12px; color: #fff; margin-bottom: 2px;">Inscription #${inscNum ? inscNum.toLocaleString() : '...'}</div>
                <div style="font-size: 11px; color: #888;">${isPending ? 'In mempool' : tx.status.block_time ? formatTimeAgo(tx.status.block_time) : 'Confirmed'}</div>
            </div>`;
        
        amountHtml = `<div style="text-align: right;"><span style="font-size: 10px; background: rgba(255,255,255,0.1); color: #fff; padding: 3px 8px; border-radius: 4px;">◉ NFT</span></div>`;
    
    } else if (isSplitTx) {
        // ✂️ Split UTXOs transaction
        iconHtml = `
            <div style="width: 40px; height: 40px; border-radius: 10px; background: rgba(16,185,129,0.15); display: flex; align-items: center; justify-content: center; margin-right: 12px; flex-shrink: 0; border: 1px solid rgba(16,185,129,0.3);">
                <span style="font-size: 18px; color: #10b981;">✂️</span>
            </div>`;
        
        infoHtml = `
            <div style="flex: 1; min-width: 0;">
                <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 4px;">
                    <span style="font-weight: 600; font-size: 14px; color: #fff;">${typeLabel}</span>
                    <span style="font-size: 9px; background: rgba(16,185,129,0.2); color: #10b981; padding: 2px 6px; border-radius: 4px;">SPLIT</span>
                    ${isPending ? '<span style="font-size: 9px; background: rgba(245,158,11,0.2); color: #f59e0b; padding: 2px 6px; border-radius: 4px;">⏳</span>' : ''}
                </div>
                <div style="font-size: 12px; color: #10b981; margin-bottom: 2px;">${splitOutputs} new UTXOs created</div>
                <div style="font-size: 11px; color: #888;">${isPending ? 'In mempool' : tx.status.block_time ? formatTimeAgo(tx.status.block_time) : 'Confirmed'}</div>
            </div>`;
        
        // Calculate total output value (fee subtracted)
        const totalOutput = tx.vout.reduce((sum, v) => sum + (v.value || 0), 0);
        amountHtml = `<div style="text-align: right;"><div style="font-weight: 600; font-size: 14px; color: #10b981;">${totalOutput.toLocaleString()} sats</div></div>`;
    
    } else {
        // Basic BTC transaction (NO HEURISTICS - only rely on enrichment API)
        // If enrichment says it's a rune or ordinal, it would have been caught above
        // Without enrichment confirmation, treat as pure Bitcoin transaction
        
        // Only check for Runestone (OP_RETURN OP_13 = 6a5d) - this is RELIABLE
        const hasRunestonePending = tx.vout.some(v => v.scriptpubkey?.startsWith('6a5d'));
        
        // 🔄 SWAP DETECTION for pending transactions
        const hasDustOutputToUserPending = tx.vout.some(v => 
            v.scriptpubkey_address === myAddress && v.value <= 600
        );
        const isPendingSwapBuy = hasRunestonePending && sentFromWallet && hasDustOutputToUserPending;
        
        // 🔥 FIX: For SWAP BUY, user receives runes (show green +)
        const pendingRuneIsReceive = isPendingSwapBuy ? true : (hasRunestonePending ? !sentFromWallet : isReceive);
        const pendingRuneLabel = hasRunestonePending 
            ? (isPendingSwapBuy ? '🔄 Swapped Rune' : (pendingRuneIsReceive ? 'Received Rune' : 'Sent Rune'))
            : (isReceive ? 'Received' : 'Sent');
        
        // 🔥 FIX: Colors for pending rune transactions
        const pendingAmountColor = hasRunestonePending 
            ? (pendingRuneIsReceive ? '#10b981' : '#ef4444')  // green for receive, red for send
            : '#f7931a';  // Bitcoin orange
        const pendingSign = hasRunestonePending 
            ? (pendingRuneIsReceive ? '+' : '-')
            : (isReceive ? '+' : '-');
        
        // Default to Bitcoin - let the enrichment API tell us if it's something else
        const txType = hasRunestonePending ? 'rune_pending' : 'btc';
        
        // Icon theme
        const iconBg = hasRunestonePending 
            ? (pendingRuneIsReceive ? 'rgba(16,185,129,0.15)' : 'rgba(239,68,68,0.15)')
            : 'rgba(247,147,26,0.15)';
        const iconSymbol = hasRunestonePending ? '⧈' : '₿';
        const iconColor = hasRunestonePending 
            ? (pendingRuneIsReceive ? '#10b981' : '#ef4444')
            : '#f7931a';
        
        iconHtml = `
            <div style="width: 40px; height: 40px; border-radius: 10px; background: ${iconBg}; display: flex; align-items: center; justify-content: center; margin-right: 12px; flex-shrink: 0;">
                <span style="font-size: 18px; color: ${iconColor};">${iconSymbol}</span>
            </div>`;
        
        // Only show RUNE badge if we detected Runestone (reliable), no ORDINAL badge without enrichment
        // 🔄 Add SWAP badge for swap transactions
        const badge = hasRunestonePending 
            ? (isPendingSwapBuy 
                ? '<span style="font-size: 10px; background: rgba(16,185,129,0.2); color: #10b981; padding: 2px 6px; border-radius: 4px;">🔄 SWAP</span>'
                : '<span style="font-size: 10px; background: rgba(247,147,26,0.2); color: #f7931a; padding: 2px 6px; border-radius: 4px;">⧈ RUNE</span>')
            : '';
        
        infoHtml = `
            <div style="flex: 1; min-width: 0;">
                <div style="display: flex; align-items: center; gap: 6px; margin-bottom: 4px;">
                    <span style="font-weight: 600; font-size: 14px; color: #fff;">${pendingRuneLabel}</span>
                    ${badge}
                    ${isPending ? '<span style="font-size: 10px; background: rgba(245,158,11,0.2); color: #f59e0b; padding: 2px 6px; border-radius: 4px;">⏳</span>' : ''}
                </div>
                <div style="font-size: 12px; color: ${hasRunestonePending ? pendingAmountColor : '#f7931a'}; margin-bottom: 2px;">${hasRunestonePending ? 'Rune ⧈' : 'Bitcoin ₿'}</div>
                <div style="font-size: 11px; color: #888;">${isPending ? 'In mempool' : tx.status.block_time ? formatTimeAgo(tx.status.block_time) : 'Confirmed'}</div>
            </div>`;
        
        amountHtml = `<div style="text-align: right;"><div style="font-weight: 600; font-size: 14px; color: ${pendingAmountColor};">${pendingSign}${btcAmount.toLocaleString()} sats</div></div>`;
    }
    
    item.innerHTML = iconHtml + infoHtml + amountHtml;
    
    if (isPending) {
        item.style.borderColor = 'rgba(245,158,11,0.3)';
        item.style.background = 'rgba(245,158,11,0.05)';
    }
    
    item.addEventListener('click', () => {
        chrome.tabs.create({ url: `https://kray.space/krayscan.html?txid=${tx.txid}` });
    });
    
    return item;
}

// 🎯 Calculate market info (EXACTLY like mobile ActivityTab.tsx)
function calculateMarketInfo(tx, myAddress) {
    let myInputs = 0;
    let myOutputs = 0;
    const ourInputs = [];
    const ourOutputs = [];
    const INSCRIPTION_THRESHOLD = 600;
    
    for (const vin of tx.vin) {
        if (vin.prevout?.scriptpubkey_address === myAddress) {
            myInputs += vin.prevout.value || 0;
            ourInputs.push(vin.prevout.value || 0);
        }
    }
    for (const vout of tx.vout) {
        if (vout.scriptpubkey_address === myAddress) {
            myOutputs += vout.value || 0;
            ourOutputs.push(vout.value || 0);
        }
    }
    
    // Count unique addresses (3+ = market transaction)
    const allAddresses = new Set();
    tx.vin.forEach(vin => {
        if (vin.prevout?.scriptpubkey_address) allAddresses.add(vin.prevout.scriptpubkey_address);
    });
    tx.vout.forEach(vout => {
        if (vout.scriptpubkey_address) allAddresses.add(vout.scriptpubkey_address);
    });
    const isMultiParty = allAddresses.size >= 3;
    
    const sentFromWallet = myInputs > 0;
    const btcAmount = sentFromWallet ? Math.abs(myInputs - myOutputs) : myOutputs;
    
    if (sentFromWallet) {
        const sentSmallUtxo = ourInputs.some(v => v <= INSCRIPTION_THRESHOLD);
        const receivedSmallUtxo = ourOutputs.some(v => v <= INSCRIPTION_THRESHOLD);
        
        if (myOutputs > myInputs) {
            // SALE: received more than sent
            return { isReceive: true, isSale: sentSmallUtxo, isBuy: false, isMarket: sentSmallUtxo && isMultiParty, btcAmount: myOutputs, sentFromWallet, isMultiParty };
        } else {
            // Spent more
            if (receivedSmallUtxo && (myInputs - myOutputs) > 0) {
                // BUY: received small UTXO
                return { isReceive: false, isSale: false, isBuy: true, isMarket: isMultiParty, btcAmount, sentFromWallet, isMultiParty };
            }
            return { isReceive: false, isSale: false, isBuy: false, isMarket: false, btcAmount, sentFromWallet, isMultiParty };
        }
    } else {
        // Pure receive
        const receivedSmallUtxo = ourOutputs.some(v => v <= INSCRIPTION_THRESHOLD);
        return { isReceive: true, isSale: false, isBuy: receivedSmallUtxo, isMarket: false, btcAmount, sentFromWallet, isMultiParty };
    }
}

// 🚀 Enrich a batch of transactions
async function enrichActivityBatch(transactions, address) {
    for (const tx of transactions) {
        if (activityEnrichedTxIds.has(tx.txid)) continue;
        
        try {
            const response = await fetch(`https://kraywallet-backend.onrender.com/api/tx-enrich/${tx.txid}`, {
                signal: AbortSignal.timeout(5000)
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    activityEnrichments.set(tx.txid, data);
                    activityEnrichedTxIds.add(tx.txid);
                    
                    // Update UI immediately
                    const item = document.querySelector(`.activity-item[data-txid="${tx.txid}"]`);
                    if (item) {
                        const newItem = createActivityItem(tx, address, data);
                        item.replaceWith(newItem);
                    }
                }
            }
        } catch (e) {
            // Silent fail
        }
        
        // Small delay between requests to avoid rate limiting
        await new Promise(r => setTimeout(r, 200));
    }
}

// 🕐 Format time ago
function formatTimeAgo(timestamp) {
    const now = Date.now();
    const diff = now - (timestamp * 1000);
    
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);
    
    if (minutes < 1) return 'just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 7) return `${days}d ago`;
    return new Date(timestamp * 1000).toLocaleDateString();
}

// Reattach Click Handlers para Activity Items (após carregar do cache)
function attachActivityClickHandlers() {
    console.log('🔗 Attaching click handlers to activity items...');
    const activityItems = document.querySelectorAll('.activity-item');
    
    activityItems.forEach(item => {
        // Extrair TXID do data attribute (vamos adicionar isso)
        const txid = item.getAttribute('data-txid');
        
        if (txid) {
            item.style.cursor = 'pointer';
            item.addEventListener('click', () => {
                const url = `https://kraywallet-backend.onrender.com/krayscan.html?txid=${txid}`;
                console.log(`🔗 Opening TX in KrayScan: ${url}`);
                chrome.tabs.create({ url });
            });
        }
    });
    
    console.log(`✅ Attached click handlers to ${activityItems.length} items`);
}

// 🚀 UNIFIED VERSION: Create Transaction Item - Smart detection + backend enrichment
async function createTransactionItemFast(tx, myAddress, enrichedUtxosMap = new Map(), runesThumbnailsMap = new Map(), runesSymbolsMap = new Map(), runesIdToNameMap = new Map(), inscriptionsMap = new Map()) {
    const item = document.createElement('div');
    item.className = 'activity-item';
    item.setAttribute('data-txid', tx.txid);
    
    // ═══════════════════════════════════════════════════════════════════
    // STEP 1: Determine transaction type (sent/received) and get amounts
    // ═══════════════════════════════════════════════════════════════════
    let type = 'unknown';
    let btcAmount = 0;
    let otherAddress = '';
    let myInputs = 0;
    let myOutputs = 0;
    
    for (const vin of tx.vin) {
        if (vin.prevout && vin.prevout.scriptpubkey_address === myAddress) {
            myInputs += vin.prevout.value;
        }
    }
    
    for (const vout of tx.vout) {
        if (vout.scriptpubkey_address === myAddress) {
            myOutputs += vout.value;
        }
    }
    
    if (myInputs > 0 && myOutputs === 0) {
        type = 'sent';
        for (const vout of tx.vout) {
            if (vout.scriptpubkey_address && vout.scriptpubkey_address !== myAddress) {
                otherAddress = vout.scriptpubkey_address;
                btcAmount = vout.value;
                break;
            }
        }
    } else if (myInputs === 0 && myOutputs > 0) {
        type = 'received';
        btcAmount = myOutputs;
        if (tx.vin.length > 0 && tx.vin[0].prevout) {
            otherAddress = tx.vin[0].prevout.scriptpubkey_address || '';
        }
    } else if (myInputs > 0 && myOutputs > 0) {
        type = 'sent';
        for (const vout of tx.vout) {
            if (vout.scriptpubkey_address && vout.scriptpubkey_address !== myAddress) {
                otherAddress = vout.scriptpubkey_address;
                btcAmount = vout.value;
                break;
            }
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════
    // STEP 2: SMART DETECTION - Check if TX likely has runes/inscriptions
    // ═══════════════════════════════════════════════════════════════════
    let txType = 'bitcoin';
    let runeInfo = null;
    let inscriptionInfo = null;
    
    // Check for indicators of runes/inscriptions:
    // 1. OP_RETURN with 6a5d (Runestone marker)
    // 2. Output with exactly 546 sats (dust limit = inscription/rune output)
    // 3. Pre-cached data from enrichedUtxosMap
    let hasRunestoneMarker = false;
    let hasDustOutput = false;
    
    for (const vout of tx.vout) {
        const script = vout.scriptpubkey || '';
        if (script.startsWith('6a5d')) {
            hasRunestoneMarker = true;
        }
        // Inscription outputs are typically 330-1000 sats
        if (vout.value >= 330 && vout.value <= 1000) {
            hasDustOutput = true;
        }
    }
    
    // Check pre-cached UTXOs first (instant, no API call)
    for (const vout of tx.vout) {
        const voutIndex = vout.n !== undefined ? vout.n : tx.vout.indexOf(vout);
        const key = `${tx.txid}:${voutIndex}`;
        const cached = enrichedUtxosMap.get(key);
        
        if (cached?.hasRunes && cached.runes && Array.isArray(cached.runes) && cached.runes.length > 0) {
            // runes is an ARRAY of objects: [{name, amount: {amount, divisibility, symbol}, symbol, runeId}]
            const runeData = cached.runes[0];
            const runeName = runeData.name || 'Rune';
            
            // amount can be: number OR object {amount, divisibility, symbol}
            let displayAmount = '';
            if (runeData.amount) {
                if (typeof runeData.amount === 'object') {
                    const rawAmount = runeData.amount.amount || 0;
                    const divisibility = runeData.amount.divisibility || 0;
                    if (divisibility > 0) {
                        displayAmount = (rawAmount / Math.pow(10, divisibility)).toLocaleString();
                    } else {
                        displayAmount = rawAmount.toLocaleString();
                    }
                } else {
                    displayAmount = runeData.amount.toLocaleString();
                }
            }
            
            // symbol from runeData.amount.symbol (actual) or runeData.symbol (fallback) or map
            const actualSymbol = runeData.amount?.symbol || runeData.symbol || runesSymbolsMap.get(runeName) || '⧈';
            
            // Try multiple keys to find thumbnail
            const thumbnail = runesThumbnailsMap.get(runeName) || 
                              runesThumbnailsMap.get(runeData.runeId) ||
                              runesThumbnailsMap.get(runeData.name) || null;
            
            if (ACTIVITY_DEBUG) console.log(`   🔍 Thumbnail lookup: ${runeName} -> ${thumbnail ? 'FOUND' : 'NOT FOUND'}`);
            
                        runeInfo = {
                name: runeName,
                symbol: actualSymbol,
                amount: displayAmount,
                thumbnail: thumbnail
            };
            txType = 'rune';
                        break;
                    }
                
        if (cached?.hasInscription && cached.inscription) {
            inscriptionInfo = {
                id: cached.inscription.inscriptionId,
                number: cached.inscription.inscriptionNumber,
                preview: `https://ordinals.com/preview/${cached.inscription.inscriptionId}`,
                value: vout.value || 546
            };
            txType = 'inscription';
                    break;
        }
    }
    
    // Check inscriptionsMap fallback
    if (txType === 'bitcoin') {
        for (const vout of tx.vout) {
            const key = `${tx.txid}:${vout.n !== undefined ? vout.n : tx.vout.indexOf(vout)}`;
            const insc = inscriptionsMap.get(key);
            if (insc) {
                inscriptionInfo = {
                    id: insc.inscriptionId,
                    number: insc.inscriptionNumber,
                    preview: insc.preview || `https://ordinals.com/preview/${insc.inscriptionId}`,
                    value: vout.value || 546
                };
                txType = 'inscription';
                break;
            }
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════
    // STEP 3: Fetch from backend if:
    // 1. TX has rune/inscription indicators AND we don't have cached data
    // 2. We have cached data but NO thumbnail (need to fetch thumbnail)
    // ═══════════════════════════════════════════════════════════════════
    // Also fetch for inscriptions if we don't have the number
    const needsThumbnail = (txType === 'rune' && runeInfo && !runeInfo.thumbnail) || 
                           (txType === 'inscription' && inscriptionInfo && (!inscriptionInfo.number || inscriptionInfo.number === '...'));
    const shouldFetchEnrichment = ((hasRunestoneMarker || hasDustOutput) && txType === 'bitcoin') || needsThumbnail;
    
    if (ACTIVITY_DEBUG && shouldFetchEnrichment) {
        console.log(`   🌐 Fetching enrichment for ${tx.txid.substring(0,12)}... (runestone:${hasRunestoneMarker}, dust:${hasDustOutput}, needsThumb:${needsThumbnail})`);
    }
    
    if (shouldFetchEnrichment) {
        try {
            const response = await fetch(`https://kraywallet-backend.onrender.com/api/explorer/tx/${tx.txid}`, {
                signal: AbortSignal.timeout(8000)
            });
            
            if (response.ok) {
                const data = await response.json();
                
                if (data.success && data.tx) {
                    const vouts = data.tx.vout || [];
                    const vins = data.tx.vin || [];
                    
                    // Check outputs for enrichment
                    for (const vout of vouts) {
                        const enrichment = vout.enrichment;
                            const voutAddress = vout.scriptpubkey_address || vout.scriptPubKey?.address;
                        const isRelevantOutput = (type === 'received' && voutAddress === myAddress) || 
                                                 (type === 'sent' && voutAddress && voutAddress !== myAddress);
                        
                        if (enrichment?.type === 'rune' && enrichment.data) {
                            if (isRelevantOutput || !runeInfo) {
                                const d = enrichment.data;
                                const div = d.divisibility || 0;
                                let displayAmount = d.amount;
                                
                                if (div > 0 && d.amount) {
                                    const numAmount = d.amount / Math.pow(10, div);
                                    displayAmount = numAmount >= 1000 
                                        ? Math.floor(numAmount).toLocaleString() 
                                        : numAmount.toLocaleString(undefined, { maximumFractionDigits: div });
                                } else if (d.amount) {
                                    displayAmount = parseInt(d.amount).toLocaleString();
                                }
                                
                                let thumb = d.thumbnail;
                                if (thumb && !thumb.startsWith('http')) {
                                    thumb = `https://kraywallet-backend.onrender.com${thumb}`;
                                }
                                
                                runeInfo = {
                                    name: d.spacedName || d.name || 'Rune',
                                    symbol: d.symbol || '⧈',
                                    amount: displayAmount || '',
                                    thumbnail: thumb
                                };
                                txType = 'rune';
                                if (isRelevantOutput) break;
                            }
                        } else if (enrichment?.type === 'inscription' && enrichment.data) {
                            if (isRelevantOutput || !inscriptionInfo) {
                                const d = enrichment.data;
                                inscriptionInfo = {
                                    id: d.inscriptionId,
                                    number: d.inscriptionNumber,
                                    preview: d.preview || d.contentUrl,
                                    value: vout.value || 546
                                };
                                txType = 'inscription';
                                if (isRelevantOutput) break;
                            }
                        }
                    }
                    
                    // Check inputs for sent transactions
                    if (txType === 'bitcoin' && type === 'sent') {
                        for (const vin of vins) {
                            const enrichment = vin.enrichment;
                            
                            if (enrichment?.type === 'rune' && enrichment.data) {
                                const d = enrichment.data;
                                const div = d.divisibility || 0;
                                let displayAmount = d.amount;
                                
                                if (div > 0 && d.amount) {
                                    const numAmount = d.amount / Math.pow(10, div);
                                    displayAmount = numAmount >= 1000 
                                        ? Math.floor(numAmount).toLocaleString() 
                                        : numAmount.toLocaleString(undefined, { maximumFractionDigits: div });
                                } else if (d.amount) {
                                    displayAmount = parseInt(d.amount).toLocaleString();
                                }
                                
                                let thumb = d.thumbnail;
                                if (thumb && !thumb.startsWith('http')) {
                                    thumb = `https://kraywallet-backend.onrender.com${thumb}`;
                                }
                                
                                runeInfo = {
                                    name: d.spacedName || d.name || 'Rune',
                                    symbol: d.symbol || '⧈',
                                    amount: displayAmount || '',
                                    thumbnail: thumb
                                };
                                txType = 'rune';
                                break;
                            } else if (enrichment?.type === 'inscription' && enrichment.data) {
                                const d = enrichment.data;
                                inscriptionInfo = {
                                    id: d.inscriptionId,
                                    number: d.inscriptionNumber,
                                    preview: d.preview || d.contentUrl,
                                    value: vin.prevout?.value || 546
                                };
                                txType = 'inscription';
                break;
            }
        }
                    }
                }
            }
        } catch (e) {
            // Silent fail - will show as BTC transaction
        }
    }
    
    // ═══════════════════════════════════════════════════════════════════
    // STEP 3: Render the appropriate UI based on transaction type
    // ═══════════════════════════════════════════════════════════════════
    
    const titleColor = type === 'received' ? '#00ff88' : '#ff6b6b';
    const titleText = type === 'received' ? 'Received' : 'Sent';
    const sign = type === 'received' ? '+' : '-';
    const statusText = tx.status.confirmed ? '✓ Confirmed' : '⏳ Pending';
    const addressText = otherAddress ? (type === 'received' ? 'From: ' : 'To: ') + otherAddress.slice(0, 10) + '...' : 'Internal';
    
    if (txType === 'inscription' && inscriptionInfo) {
        // ═══ INSCRIPTION TX ═══
        item.classList.add('inscription-tx');
        item.style.minHeight = '80px';
        
        // Thumbnail
        const thumbDiv = document.createElement('div');
        thumbDiv.style.cssText = 'width:60px;height:60px;border-radius:8px;background:#1a1a1a;overflow:hidden;flex-shrink:0;margin-right:12px;border:1.5px solid #9333ea;';
        
        const img = document.createElement('img');
        img.style.cssText = 'width:100%;height:100%;object-fit:cover;';
        const backendUrl = `https://kraywallet-backend.onrender.com/api/rune-thumbnail/${inscriptionInfo.id}`;
        const ordinalsUrl = `https://ordinals.com/content/${inscriptionInfo.id}`;
        img.src = backendUrl;
        img.addEventListener('error', function() { this.src = ordinalsUrl; });
        thumbDiv.appendChild(img);
        item.appendChild(thumbDiv);
        
        // Info
        const infoDiv = document.createElement('div');
        infoDiv.style.flex = '1';
        infoDiv.innerHTML = `
            <div style="font-weight:600;font-size:13px;"><span style="color:${titleColor};">${titleText}</span> <span style="color:#9333ea;">Inscription</span></div>
            <div style="font-size:11px;color:#f59e0b;margin-top:4px;font-weight:600;">#${inscriptionInfo.number || '...'}</div>
            <div style="font-size:12px;color:#666;margin-top:4px;">${addressText}</div>
            <div style="font-size:11px;color:#888;margin-top:4px;">${statusText}</div>
        `;
        item.appendChild(infoDiv);
        
        // Amount
        const amountDiv = document.createElement('div');
        amountDiv.innerHTML = `<div style="color:${titleColor};font-weight:600;font-size:14px;text-align:right;">${sign}${inscriptionInfo.value} sats</div>`;
        item.appendChild(amountDiv);
        
    } else if (txType === 'rune' && runeInfo) {
        // ═══ RUNE TX ═══
        
        // Icon/Thumbnail
        const iconDiv = document.createElement('div');
        iconDiv.style.cssText = 'width:50px;height:50px;border-radius:8px;background:#1a1a1a;overflow:hidden;flex-shrink:0;margin-right:12px;border:1.5px solid #f59e0b;';
        
        if (runeInfo.thumbnail) {
            const img = document.createElement('img');
            img.style.cssText = 'width:100%;height:100%;object-fit:cover;';
            img.src = runeInfo.thumbnail;
            img.addEventListener('error', function() {
                this.parentElement.innerHTML = `<div style="width:100%;height:100%;display:flex;align-items:center;justify-content:center;font-size:20px;color:#f59e0b;background:linear-gradient(135deg,rgba(245,158,11,0.2),rgba(251,191,36,0.1));">⧈</div>`;
            });
            iconDiv.appendChild(img);
        } else {
            iconDiv.innerHTML = `<div style="width:100%;height:100%;display:flex;align-items:center;justify-content:center;font-size:20px;color:#f59e0b;background:linear-gradient(135deg,rgba(245,158,11,0.2),rgba(251,191,36,0.1));">${runeInfo.symbol}</div>`;
        }
        item.appendChild(iconDiv);
        
        // Info
        const infoDiv = document.createElement('div');
        infoDiv.style.flex = '1';
        infoDiv.innerHTML = `
            <div style="font-weight:600;font-size:13px;"><span style="color:${titleColor};">${titleText}</span> <span style="color:#f59e0b;">Runes</span></div>
            <div style="font-size:11px;color:#f59e0b;margin-top:4px;">${runeInfo.name} ${runeInfo.symbol}</div>
            <div style="font-size:12px;color:#666;margin-top:4px;">${addressText}</div>
            <div style="font-size:11px;color:#888;margin-top:4px;">${statusText}</div>
        `;
        item.appendChild(infoDiv);
        
        // Amount
        const amountDiv = document.createElement('div');
        amountDiv.style.textAlign = 'right';
        if (runeInfo.amount) {
            amountDiv.innerHTML = `<div style="color:#f59e0b;font-weight:600;font-size:14px;">${runeInfo.amount}</div>`;
            }
        item.appendChild(amountDiv);
        
    } else {
        // ═══ NORMAL BTC TX ═══
        
        // Icon
        const iconDiv = document.createElement('div');
        iconDiv.style.cssText = 'width:40px;height:40px;border-radius:50%;display:flex;align-items:center;justify-content:center;font-size:20px;flex-shrink:0;margin-right:12px;';
        
        if (type === 'received') {
            iconDiv.innerHTML = '<span style="color:#00ff88;">↓</span>';
            iconDiv.style.background = 'rgba(0, 255, 136, 0.1)';
        } else {
            iconDiv.innerHTML = '<span style="color:#ff6b6b;">↑</span>';
            iconDiv.style.background = 'rgba(255, 107, 107, 0.1)';
        }
        item.appendChild(iconDiv);
        
        // Info
        const infoDiv = document.createElement('div');
        infoDiv.style.flex = '1';
        infoDiv.innerHTML = `
            <div style="font-weight:600;font-size:14px;color:${titleColor};">${titleText}</div>
            <div style="font-size:12px;color:#666;margin-top:4px;">${addressText}</div>
            <div style="font-size:11px;color:#888;margin-top:4px;">${statusText}</div>
        `;
        item.appendChild(infoDiv);
        
        // Amount
        const amountDiv = document.createElement('div');
        amountDiv.innerHTML = `<div style="color:${titleColor};font-weight:600;font-size:14px;text-align:right;">${sign}${btcAmount.toLocaleString()} sats</div>`;
        item.appendChild(amountDiv);
    }
    
    // Pending style
        if (!tx.status.confirmed) {
            item.style.opacity = '0.8';
            item.style.border = '1px solid rgba(255, 193, 7, 0.4)';
    }
    
    // Click handler - Always open transaction page in KrayScan
    item.style.cursor = 'pointer';
    item.addEventListener('click', () => {
        chrome.tabs.create({ url: `https://kray.space/krayscan.html?txid=${tx.txid}` });
    });
    
    return item;
}

// Create Transaction Item com Enriched UTXOs e Runes Thumbnails (ORIGINAL - kept for compatibility)
async function createTransactionItem(tx, myAddress, enrichedUtxosMap = new Map(), runesThumbnailsMap = new Map(), runesSymbolsMap = new Map(), runesIdToNameMap = new Map(), inscriptionsMap = new Map()) {
    const item = document.createElement('div');
    item.className = 'activity-item';
    
    // ✅ Adicionar TXID como data attribute para reattach listeners depois
    item.setAttribute('data-txid', tx.txid);
    
    // Determinar tipo de transação (Received/Sent)
    let type = 'unknown';
    let amount = 0;
    let otherAddress = '';
    
    // Calcular entrada/saída para este endereço
    let myInputs = 0;
    let myOutputs = 0;
    
    // DEBUG
    console.log(`\n🔍 Processing TX: ${tx.txid.substring(0, 16)}...`);
    console.log(`   My address: ${myAddress.substring(0, 20)}...`);
    
    // Verificar inputs (se este endereço está enviando)
    for (const vin of tx.vin) {
        if (vin.prevout && vin.prevout.scriptpubkey_address === myAddress) {
            myInputs += vin.prevout.value;
            console.log(`   ✅ Found my input: ${vin.prevout.value} sats`);
        }
    }
    
    // Verificar outputs (se este endereço está recebendo)
    for (const vout of tx.vout) {
        if (vout.scriptpubkey_address === myAddress) {
            myOutputs += vout.value;
            console.log(`   ✅ Found my output: ${vout.value} sats`);
        }
    }
    
    console.log(`   📊 myInputs: ${myInputs}, myOutputs: ${myOutputs}`);
    
    // Determinar tipo e pegar endereço relevante
    if (myInputs > 0 && myOutputs === 0) {
        // Enviamos tudo (sem change)
        type = 'sent';
        amount = myInputs;
        
        // Pegar o primeiro output que não é nosso (destinatário)
        for (const vout of tx.vout) {
            if (vout.scriptpubkey_address && vout.scriptpubkey_address !== myAddress) {
                otherAddress = vout.scriptpubkey_address;
                amount = vout.value; // Valor enviado
                break;
            }
        }
    } else if (myInputs === 0 && myOutputs > 0) {
        // Recebemos
        type = 'received';
        amount = myOutputs;
        
        // Pegar endereço de origem (primeiro input)
        if (tx.vin.length > 0 && tx.vin[0].prevout) {
            otherAddress = tx.vin[0].prevout.scriptpubkey_address || 'Unknown';
        }
    } else if (myInputs > 0 && myOutputs > 0) {
        // Enviamos com change
        type = 'sent';
        amount = myInputs - myOutputs; // Diferença = valor enviado + fee
        console.log(`   🔄 Type: SENT with change`);
        
        // Encontrar endereço de destino (output que não é nosso)
        // Nota: Não verificamos dust limit aqui, pois pode haver outputs válidos < 546
        for (const vout of tx.vout) {
            console.log(`      Checking output: ${vout.value} sats to ${vout.scriptpubkey_address?.substring(0, 20)}...`);
            console.log(`      Is mine? ${vout.scriptpubkey_address === myAddress}`);
            
            if (vout.scriptpubkey_address && vout.scriptpubkey_address !== myAddress) {
                otherAddress = vout.scriptpubkey_address;
                amount = vout.value; // Valor real enviado
                console.log(`      ✅ FOUND RECIPIENT: ${otherAddress.substring(0, 20)}... (${amount} sats)`);
                break;
            }
        }
        console.log(`   📍 Final otherAddress: ${otherAddress ? otherAddress.substring(0, 20) + '...' : 'NONE'}`);
    } else if (myInputs === 0 && myOutputs === 0) {
        // Caso especial: TX pending sem dados de prevout (ainda propagando)
        // Tentar determinar pelo que vemos nos outputs
        type = 'unknown';
        
        // Se há um output que não é para meu endereço, provavelmente estamos enviando
        for (const vout of tx.vout) {
            if (vout.scriptpubkey_address && vout.scriptpubkey_address !== myAddress) {
                type = 'sent';
                otherAddress = vout.scriptpubkey_address;
                amount = vout.value;
                break;
            } else if (vout.scriptpubkey_address === myAddress) {
                type = 'received';
                amount = vout.value;
            }
        }
    }
    
    // 🎯 DETECTAR SE É UMA TRANSAÇÃO DE INSCRIPTION (usando enriched data)
    let inscription = null;
    let inscriptionInfo = null;
    let enrichedData = null;
    
    console.log(`   🔎 Checking TX for inscriptions/runes (ENRICHED)...`);
    console.log(`      TX has ${tx.vout.length} outputs, ${tx.vin.length} inputs`);
    console.log(`      enrichedUtxosMap has ${enrichedUtxosMap.size} entries`);
    
    // Verificar nos outputs se há inscription/runes (usando enriched data)
    for (let i = 0; i < tx.vout.length; i++) {
        const key = `${tx.txid}:${i}`;
        console.log(`      🔍 Checking OUTPUT ${i}: ${key}`);
        
        if (enrichedUtxosMap.has(key)) {
            enrichedData = enrichedUtxosMap.get(key);
            console.log(`         ✅ Found enriched data:`, enrichedData);
            
            if (enrichedData.hasInscription && enrichedData.inscription) {
                inscription = enrichedData.inscription;
                inscriptionInfo = { vout: i, isOutput: true };
                console.log(`   ✅ 🖼️  Found inscription in OUTPUT ${i}:`, inscription.id || inscription.inscription_id);
                break;
            }
        }
        
        // Fallback: usar o mapa antigo de inscriptions
        if (!inscription && inscriptionsMap.has(key)) {
            inscription = inscriptionsMap.get(key);
            inscriptionInfo = { vout: i, isOutput: true };
            console.log(`   ✅ 🖼️  Found inscription in OUTPUT ${i} (fallback):`, inscription.inscriptionId?.substring(0, 16));
            break;
        }
    }
    
    // Se não encontrou nos outputs, verificar nos inputs (inscription sendo enviada)
    if (!inscription) {
        console.log(`      No inscription in outputs, checking inputs...`);
        for (let i = 0; i < tx.vin.length; i++) {
            if (tx.vin[i].txid && tx.vin[i].vout !== undefined) {
                const key = `${tx.vin[i].txid}:${tx.vin[i].vout}`;
                console.log(`      🔍 Checking INPUT ${i}: ${key}`);
                console.log(`         Has in map? ${inscriptionsMap.has(key)}`);
                
                if (inscriptionsMap.has(key)) {
                    inscription = inscriptionsMap.get(key);
                    inscriptionInfo = { vin: i, isInput: true };
                    console.log(`   ✅ 🖼️  Found inscription in INPUT ${i}:`, inscription.inscriptionId.substring(0, 16));
                    break;
                }
                
                // 🔥 NOVO: Se não confirmada, buscar input via backend para detectar inscription
                if (!tx.status?.confirmed) {
                    try {
                        const inputCheckResponse = await fetch(`https://kraywallet-backend.onrender.com/api/output/${tx.vin[i].txid}:${tx.vin[i].vout}`, {
                            signal: AbortSignal.timeout(2000)
                        });
                        
                        if (inputCheckResponse.ok) {
                            const inputData = await inputCheckResponse.json();
                            
                            if (inputData.success && inputData.inscriptions?.length > 0) {
                                const inscriptionId = inputData.inscriptions[0];
                                console.log(`   ✅ Found inscription in UNCONFIRMED TX input: ${inscriptionId}`);
                                
                                // Buscar detalhes
                                try {
                                    const detailsResponse = await fetch(`https://kraywallet-backend.onrender.com/api/ordinals/${inscriptionId}`, {
                                        signal: AbortSignal.timeout(2000)
                                    });
                                    
                                    if (detailsResponse.ok) {
                                        const inscData = await detailsResponse.json();
                                        
                                        inscription = {
                                            inscriptionId: inscriptionId,
                                            inscriptionNumber: inscData.number || inscData.inscription_number || 'N/A',
                                            number: inscData.number || inscData.inscription_number,  // ✅ Add for consistency
                                            inscription_number: inscData.number || inscData.inscription_number,  // ✅ Add all variants
                                            preview: `https://ordinals.com/preview/${inscriptionId}`,
                                            outputValue: tx.vin[i].prevout?.value || 555
                                        };
                                        inscriptionInfo = { vin: i, isInput: true };
                                        console.log(`   ✅ Inscription #${inscription.inscriptionNumber} (MEMPOOL TX)`);
                                        break;
                                    }
                                } catch (e) {
                                    console.warn(`   ⚠️  Could not fetch inscription details: ${e.message}`);
                                }
                            }
                        }
                    } catch (e) {
                        // Continuar
                    }
                }
            }
        }
    }
    
        // 🔍 Se não encontrou no mapa local, tentar detectar por valor típico (600 sats)
        // e buscar na API do Unisat
        if (!inscription) {
            console.log(`   🔍 No inscription in map, checking ALL outputs in ord server...`);
            
            for (let i = 0; i < tx.vout.length; i++) {
                const vout = tx.vout[i];
                
                // ⏸️ DELAY para evitar Chrome throttling
                if (i > 0) {
                    await new Promise(r => setTimeout(r, 100));  // 100ms entre cada output
                }
                
                // 🔍 Buscar QUALQUER output no ord server (não filtrar por valor)
                try {
                    console.log(`   📡 Checking ord server: ${tx.txid}:${i} (${vout.value} sats)`);
                
                // RETRY LOGIC: Tentar até 2x se falhar
                let ordResponse = null;
                let attempts = 0;
                const maxAttempts = 2;
                
                while (attempts < maxAttempts && !ordResponse) {
                    attempts++;
                    
                    try {
                        ordResponse = await fetch(`https://kraywallet-backend.onrender.com/api/output/${tx.txid}:${i}`, {
                            method: 'GET',
                            headers: { 'Accept': 'application/json' },
                            signal: AbortSignal.timeout(15000)  // 15 seg (aumentado!)
                        });
                        
                        if (!ordResponse.ok) {
                            console.warn(`      ⚠️  Attempt ${attempts}: API returned ${ordResponse.status}`);
                            ordResponse = null;  // Retry
                            
                            // Delay antes de retry (150ms)
                            if (attempts < maxAttempts) {
                                await new Promise(r => setTimeout(r, 150));
                            }
                        }
                    } catch (fetchError) {
                        console.warn(`      ⚠️  Attempt ${attempts} failed:`, fetchError.message);
                        ordResponse = null;
                        
                        // Delay antes de retry
                        if (attempts < maxAttempts) {
                            await new Promise(r => setTimeout(r, 150));
                        }
                    }
                }
                
                if (ordResponse.ok) {
                    const outputData = await ordResponse.json();
                    
                    // Verificar se tem inscription (QuickNode retorna array de IDs)
                    if (outputData.success && outputData.inscriptions && outputData.inscriptions.length > 0) {
                        const inscriptionId = outputData.inscriptions[0]; // Primeira inscription
                        console.log(`   ✅ Found inscription in output ${i}: ${inscriptionId}`);
                        
                        // Buscar DETAILS COMPLETOS via QuickNode (sempre!)
                        let inscriptionDetails = null;
                        
                        // RETRY para buscar details (crítico!)
                        let detailsAttempts = 0;
                        const maxDetailsAttempts = 3;  // 3 tentativas
                        
                        while (detailsAttempts < maxDetailsAttempts && !inscriptionDetails) {
                            detailsAttempts++;
                            
                            try {
                                const detailsResponse = await fetch(`https://kraywallet-backend.onrender.com/api/ordinals/${inscriptionId}`, {
                                    signal: AbortSignal.timeout(15000)  // 15 seg
                                });
                            
                                if (detailsResponse.ok) {
                                    const responseData = await detailsResponse.json();
                                    
                                    // /api/explorer/inscription retorna { success: true, inscription: {...} }
                                    inscriptionDetails = responseData.inscription || responseData;
                                    
                                    console.log(`   ✅ Got inscription details (attempt ${detailsAttempts}):`, inscriptionDetails);
                                    break;  // Success!
                                } else {
                                    console.warn(`      ⚠️  Attempt ${detailsAttempts}: Details API ${detailsResponse.status}`);
                                    
                                    // Delay antes de retry
                                    if (detailsAttempts < maxDetailsAttempts) {
                                        await new Promise(r => setTimeout(r, 200));
                                    }
                                }
                            } catch (e) {
                                console.error(`      ❌ Attempt ${detailsAttempts} failed:`, e.message);
                                
                                // Delay antes de retry
                                if (detailsAttempts < maxDetailsAttempts) {
                                    await new Promise(r => setTimeout(r, 200));
                                }
                            }
                        }
                        
                        // DEBUG: Ver o que veio do API
                        console.log(`   🔍 Details response:`, inscriptionDetails);
                        console.log(`   🔍 inscriptionDetails?.number:`, inscriptionDetails?.number);
                        console.log(`   🔍 inscriptionDetails?.inscription_number:`, inscriptionDetails?.inscription_number);
                        
                        // Extrair número (tentar TODOS os campos possíveis)
                        const extractedNumber = inscriptionDetails?.number || 
                                              inscriptionDetails?.inscription_number || 
                                              inscriptionDetails?.inscriptionNumber ||
                                              inscriptionDetails?.data?.number ||
                                              inscriptionDetails?.inscription?.number ||
                                              null;
                        
                        console.log(`   🔍 Extracted number:`, extractedNumber);
                        
                        // Criar inscription object com TODOS os campos
                        inscription = {
                            inscriptionId: inscriptionId,
                            id: inscriptionId,
                            inscriptionNumber: extractedNumber,
                            number: extractedNumber,
                            inscription_number: extractedNumber,
                            preview: `https://ordinals.com/preview/${inscriptionId}`,  // SEMPRE setar!
                            contentUrl: `https://ordinals.com/content/${inscriptionId}`,  // SEMPRE setar!
                            thumbnail: `/api/rune-thumbnail/${inscriptionId}`,  // Para backend
                            outputValue: vout.value,
                            value: vout.value,
                            utxo: { value: vout.value, txid: tx.txid, vout: i }
                        };
                        inscriptionInfo = { vout: i, isOutput: true };
                        console.log(`   ✅ 🖼️  Inscription created: #${extractedNumber || '?'}, preview: ${inscription.preview}`);
                        break;
                    }
                }
            } catch (error) {
                if (error.name === 'TimeoutError') {
                    console.warn(`   ⏱️  Timeout checking output ${i}`);
                } else {
                    console.error(`   ⚠️ Error checking output ${i}:`, error.message);
                }
            }
        }
    }
    
    const isInscriptionTx = !!inscription;
    console.log(`   📋 Is inscription TX? ${isInscriptionTx}`);
    
    // 🎯 DETECTAR SE É TRANSAÇÃO DE RUNES (usando enriched data)
    let isRunesTx = false;
    let runeInfo = null;
    let runesData = [];
    
    console.log(`   🪙 Checking for RUNES in outputs...`);
    
    // Verificar se algum output tem runes (usando enriched data)
    for (let i = 0; i < tx.vout.length; i++) {
        const key = `${tx.txid}:${i}`;
        
        if (enrichedUtxosMap.has(key)) {
            const data = enrichedUtxosMap.get(key);
            
            if (data.hasRunes && data.runes && data.runes.length > 0) {
                isRunesTx = true;
                runesData = data.runes;
                
                // 🔄 CONVERTER RUNE IDs PARA NOMES (se necessário)
                runesData = runesData.map(r => {
                    // Se 'name' parece ser um ID (formato block:tx), converter para nome
                    if (r.name && r.name.includes(':')) {
                        const runeId = r.name;
                        const actualName = runesIdToNameMap.get(runeId) || r.name;
                        const symbol = runesSymbolsMap.get(actualName) || r.symbol || '⧈';
                        
                        console.log(`      🔄 Converting ID ${runeId} → ${actualName} ${symbol}`);
                        
                        return {
                            ...r,
                            name: actualName,
                            symbol: symbol,
                            runeId: runeId
                        };
                    }
                    return r;
                });
                
                // Montar runeInfo com todas as runes
                const runeNames = runesData.map(r => r.name).join(', ');
                const totalAmount = runesData.reduce((sum, r) => sum + parseInt(r.amount || 0), 0);
                
                runeInfo = {
                    name: runesData.length === 1 ? runesData[0].name : `${runesData.length} Runes`,
                    amount: runesData.length === 1 ? runesData[0].amount : totalAmount.toString(),
                    runes: runesData
                };
                
                console.log(`   ✅ 🪙 Found ${runesData.length} rune(s) in OUTPUT ${i}:`, runeNames);
                break;
            }
        }
    }
    
    // 🎯 FALLBACK: Detectar OP_RETURN com Runestone (para PENDING e CONFIRMED)
    // Runestone sempre começa com OP_RETURN (6a) + OP_PUSHNUM_13 (5d)
    if (!isRunesTx) {
        const txStatus = tx.status.confirmed ? 'CONFIRMED' : 'PENDING';
        console.log(`   🔍 TX is ${txStatus} - checking for Runestone in OP_RETURN...`);
        
        for (const vout of tx.vout) {
            const script = vout.scriptpubkey;
            
            // Verificar se é OP_RETURN com Runestone (6a5d = OP_RETURN + OP_PUSHNUM_13)
            if (script && script.startsWith('6a5d')) {
                console.log(`   ✅ Found Runestone OP_RETURN: ${script.substring(0, 20)}...`);
                
                // Marcar como transação de Runes
                isRunesTx = true;
                
                // 🎯 BUSCAR INFO DA RUNE NO CACHE LOCAL ou do runesThumbnailsMap
                let runeName = 'Rune Transfer';
                let runeAmount = '';
                let runeThumbnail = null;
                let runeSymbol = '⧈';
                
                // 1. Tentar pegar do cache de lastSentRune (se disponível)
                if (window.lastSentRune && window.lastSentRune.txid === tx.txid) {
                    runeName = window.lastSentRune.name;
                    runeAmount = window.lastSentRune.amount;
                    runeSymbol = window.lastSentRune.symbol || '⧈';
                    runeThumbnail = window.lastSentRune.thumbnail;
                    console.log(`   📌 Using cached lastSentRune: ${runeName} • ${runeAmount}`);
                } else {
                    // 2. Decode Runestone from OP_RETURN to get Rune ID, then fetch details
                    console.log(`   🔍 Trying to decode Runestone from OP_RETURN...`);
                    try {
                        // Decodificar o Runestone do OP_RETURN
                        const opReturnHex = script;
                        const payload = opReturnHex.substring(6); // Pular 6a5d e length byte
                        
                        // Decodificar LEB128 para extrair Rune ID (block:tx)
                        const buffer = [];
                        for (let i = 0; i < payload.length; i += 2) {
                            buffer.push(parseInt(payload.substring(i, i + 2), 16));
                        }
                        
                        // Função para decodificar LEB128
                        function decodeLEB128(bytes, offset = 0) {
                            let result = 0;
                            let shift = 0;
                            let i = offset;
                            
                            while (i < bytes.length) {
                                const byte = bytes[i++];
                                result |= (byte & 0x7F) << shift;
                                shift += 7;
                                
                                if (!(byte & 0x80)) {
                                    break;
                                }
                            }
                            
                            return { value: result, nextOffset: i };
                        }
                        
                        // Decodificar valores: Tag, Block, TxIndex, Amount, Output
                        let offset = 0;
                        const values = [];
                        while (offset < buffer.length && values.length < 6) {
                            const decoded = decodeLEB128(buffer, offset);
                            values.push(decoded.value);
                            offset = decoded.nextOffset;
                        }
                        
                        const tag = values[0];
                        console.log(`   🏷️  Tag: ${tag} (${tag === 0 ? 'Edicts ✅' : tag === 10 ? 'INVALID (Cenotaph) ⚠️' : 'Unknown'})`);
                        
                        // Determine rune ID based on tag type
                        let runeId = null;
                        
                        if (tag !== 0) {
                            // Cenotaph (burn) - values are shifted
                            console.log(`   ⚠️  CENOTAPH DETECTED! Invalid Tag ${tag} - Runes were BURNED!`);
                            if (values.length >= 4) {
                                runeId = `${values[2]}:${values[3]}`;
                            }
                            runeName = '🔥 Burned Rune';
                            runeSymbol = '🔥';
                        } else if (values.length >= 3) {
                            // Normal transfer - tag 0
                            runeId = `${values[1]}:${values[2]}`;
                        }
                        
                        if (runeId) {
                            console.log(`   🆔 Decoded Rune ID: ${runeId}`);
                            
                            // Try to get rune name from cache first
                            if (runesIdToNameMap.has(runeId)) {
                                const cachedName = runesIdToNameMap.get(runeId);
                                runeName = tag !== 0 ? `🔥 ${cachedName}` : cachedName;
                                runeThumbnail = runesThumbnailsMap.get(cachedName);
                                runeSymbol = tag !== 0 ? '🔥' : (runesSymbolsMap.get(cachedName) || '⧈');
                                console.log(`   ✅ Found in cache: ${runeName} ${runeSymbol}`);
                            } else {
                                // Fetch rune details from backend
                                console.log(`   🔍 Rune ID ${runeId} not in cache, fetching...`);
                                try {
                                    const runeResponse = await fetch(`https://kraywallet-backend.onrender.com/api/rune/${runeId}`, {
                                        method: 'GET',
                                        headers: { 'Accept': 'application/json' },
                                        signal: AbortSignal.timeout(5000)
                                    });
                                    
                                    if (runeResponse.ok) {
                                        const runeData = await runeResponse.json();
                                        if (runeData.success) {
                                            const fetchedName = runeData.name;
                                            runeName = tag !== 0 ? `🔥 ${fetchedName}` : fetchedName;
                                            runeSymbol = tag !== 0 ? '🔥' : (runeData.symbol || '⧈');
                                            runeThumbnail = runeData.thumbnail;
                                            
                                            // Cache it
                                            runesIdToNameMap.set(runeId, fetchedName);
                                            runesSymbolsMap.set(fetchedName, runeData.symbol || '⧈');
                                            if (runeThumbnail) runesThumbnailsMap.set(fetchedName, runeThumbnail);
                                            if (runeData.divisibility !== undefined) {
                                                window.runesDivisibilityMap = window.runesDivisibilityMap || new Map();
                                                window.runesDivisibilityMap.set(fetchedName, runeData.divisibility);
                                            }
                                            console.log(`   ✅ Fetched: ${runeName} ${runeSymbol}`);
                                        }
                                    }
                                } catch (fetchError) {
                                    console.error(`   ❌ Error fetching rune:`, fetchError.message);
                                }
                            }
                            
                            // 🔥 NOW fetch the AMOUNT from explorer (QuickNode has accurate amounts)
                            try {
                                console.log(`   💰 Fetching amount from explorer...`);
                                const txResponse = await fetch(`https://kraywallet-backend.onrender.com/api/explorer/tx/${tx.txid}`, {
                                    method: 'GET',
                                    signal: AbortSignal.timeout(5000)
                                });
                                
                                if (txResponse.ok) {
                                    const txData = await txResponse.json();
                                    if (txData.success && txData.tx && txData.tx.vout) {
                                        for (const vout of txData.tx.vout) {
                                            const enrichment = vout.enrichment;
                                            if (enrichment?.type === 'rune' && enrichment.data?.amount) {
                                                const voutAddress = vout.scriptpubkey_address || vout.scriptPubKey?.address;
                                                const isOurAddress = voutAddress === myAddress;
                                                
                                                // For SENT: use recipient's output (not ours)
                                                // For RECEIVED: use our output
                                                if ((type === 'received' && isOurAddress) || 
                                                    (type === 'sent' && !isOurAddress)) {
                                                    runeAmount = enrichment.data.amount.toString();
                                                    console.log(`   ✅ Amount: ${runeAmount} (${type} to ${voutAddress?.slice(0,12)}...)`);
                                                    break;
                                                }
                                            }
                                        }
                                    }
                                }
                            } catch (amountError) {
                                console.warn(`   ⚠️ Could not fetch amount:`, amountError.message);
                            }
                        }
                    } catch (decodeError) {
                        console.error(`   ❌ Error decoding Runestone:`, decodeError);
                        
                        // Fallback: Se tivermos runes no runesThumbnailsMap, usar a primeira
                        if (runesThumbnailsMap.size > 0) {
                            const firstRune = Array.from(runesThumbnailsMap.keys())[0];
                            runeName = firstRune;
                            runeThumbnail = runesThumbnailsMap.get(firstRune);
                            runeSymbol = runesSymbolsMap.get(firstRune) || '⧈';
                            console.log(`   🎯 Fallback to first rune from thumbnailsMap: ${runeName} ${runeSymbol}`);
                        }
                    }
                }
                
                // Adicionar thumbnail ao map se tiver
                if (runeThumbnail) {
                    runesThumbnailsMap.set(runeName, runeThumbnail);
                }
                
                // Criar dados da rune para array
                runesData = [{
                    name: runeName,
                    amount: runeAmount,
                    symbol: runeSymbol
                }];
                
                runeInfo = {
                    name: runeName,
                    amount: runeAmount,
                    runes: runesData
                };
                
                console.log(`   🪙 Detected ${txStatus} Runes transaction: ${runeName}`);
                break;
            }
        }
    }
    
    // ==========================================
    // 🎨 RENDERIZAR UI
    // ==========================================
    
    if (isInscriptionTx) {
        // ✅ TRANSAÇÃO DE INSCRIPTION (com thumbnail e detalhes)
        item.classList.add('inscription-tx');
        item.style.minHeight = '80px';
        
        // Thumbnail da inscription
        const thumbnail = document.createElement('div');
        thumbnail.className = 'activity-thumbnail';
        thumbnail.style.cssText = `
            width: 60px;
            height: 60px;
            border-radius: 8px;
            background: #1a1a1a;
            overflow: hidden;
            flex-shrink: 0;
            margin-right: 12px;
        `;
        
        const contentUrl = sanitizeURL(`https://kraywallet-backend.onrender.com/api/rune-thumbnail/${sanitizeHTML(inscription.inscriptionId)}`);
        thumbnail.innerHTML = `<img src="${contentUrl}" style="width: 100%; height: 100%; object-fit: cover;" onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><text y=%22.9em%22 font-size=%2290%22>◉</text></svg>'">`;
        
        // Info
        const info = document.createElement('div');
        info.className = 'activity-info';
        info.style.flex = '1';
        
        // Título
        const title = document.createElement('div');
        title.className = 'activity-title';
        title.style.fontWeight = '600';
        title.style.fontSize = '13px'; // Reduzir 1px para caber melhor
        title.style.lineHeight = '1.2';
        
        if (type === 'received') {
            title.innerHTML = '<span style="white-space: nowrap;">📥 <span style="color: #00ff88;">Received</span> <span style="color: #9333ea;">Inscription</span></span>';
        } else if (type === 'sent') {
            title.innerHTML = '<span style="white-space: nowrap;">📤 <span style="color: #ff6b6b;">Sent</span> <span style="color: #9333ea;">Inscription</span></span>';
        } else {
            title.innerHTML = '<span style="white-space: nowrap;">◉ <span style="color: #9333ea;">Inscription</span> Transaction</span>';
        }
        
        // TAG: Número da Inscription
        const inscIdEl = document.createElement('div');
        inscIdEl.className = 'activity-inscription-id';
        inscIdEl.style.cssText = `
            font-size: 11px;
            color: #f59e0b;
            margin-top: 4px;
            font-weight: 600;
        `;
        // ✅ SEMPRE mostrar número (buscar se necessário)
        console.log(`🔍 Inscription data:`, inscription);
        
        const inscriptionNum = inscription.inscriptionNumber || 
                              inscription.inscription_number || 
                              inscription.number;
        
        console.log(`🔍 Inscription number found:`, inscriptionNum);
        
        if (inscriptionNum && inscriptionNum !== 'Pending') {
            inscIdEl.textContent = `#${inscriptionNum}`;
            console.log(`   ✅ Showing number: #${inscriptionNum}`);
        } else {
            // Não tem número - mostrar loading e buscar
            inscIdEl.textContent = `#...`;
            inscIdEl.style.opacity = '0.7';
            
            console.warn(`   ⚠️  No inscription number! Fetching now...`);
            
            // Buscar AGORA (não async) - garantir que aparece
            const inscId = inscription.id || inscription.inscriptionId;
            
            if (inscId) {
                console.log(`   📡 Fetching number for ${inscId.substring(0, 16)}...`);
                
                (async () => {
                    try {
                        const response = await fetch(`https://kraywallet-backend.onrender.com/api/explorer/inscription/${inscId}`);
                        
                        if (response.ok) {
                            const data = await response.json();
                            
                            // /api/explorer/inscription retorna { inscription: {...} }
                            const inscData = data.inscription || data;
                            const number = inscData.number || inscData.inscriptionNumber || inscData.inscription_number;
                            
                            if (number) {
                                inscIdEl.textContent = `#${number}`;
                                inscIdEl.style.opacity = '1';
                                console.log(`   ✅ Got number: #${number}`);
                                
                                // Atualizar objeto também
                                inscription.number = number;
                                inscription.inscriptionNumber = number;
                            } else {
                                console.error(`   ❌ API returned no number`);
                                inscIdEl.textContent = `#?`;
                            }
                        } else {
                            console.error(`   ❌ API error ${response.status}`);
                            inscIdEl.textContent = `#?`;
                        }
                    } catch (error) {
                        console.error(`   ❌ Fetch failed:`, error);
                        inscIdEl.textContent = `#?`;
                    }
                })();
            } else {
                console.error(`   ❌ No inscription ID to fetch!`);
                inscIdEl.textContent = `#?`;
            }
        }
        
        // Address
        const addressEl = document.createElement('div');
        addressEl.className = 'activity-address';
        addressEl.style.fontSize = '12px';
        addressEl.style.color = '#666';
        addressEl.style.marginTop = '4px';
        
        if (otherAddress) {
            const prefix = type === 'received' ? 'From: ' : 'To: ';
            addressEl.textContent = prefix + (otherAddress.slice(0, 10) + '...' + otherAddress.slice(-6));
        } else {
            addressEl.textContent = 'Internal Transfer';
        }
        
        // Meta (data e status)
        const meta = document.createElement('div');
        meta.className = 'activity-meta';
        meta.style.fontSize = '11px';
        meta.style.color = '#888';
        meta.style.marginTop = '4px';
        
        let timeAgo = 'Just now';
        if (tx.status.block_time) {
            const date = new Date(tx.status.block_time * 1000);
            timeAgo = getTimeAgo(date);
        }
        
        const confirmations = tx.status.confirmed ? '✓ Confirmed' : '⏳ Pending';
        meta.textContent = `${timeAgo} • ${confirmations}`;
        
        info.appendChild(title);
        info.appendChild(inscIdEl);
        info.appendChild(addressEl);
        info.appendChild(meta);
        
        // Amount (UTXO value + possível inscription price)
        const amountEl = document.createElement('div');
        amountEl.className = 'activity-amount';
        amountEl.style.textAlign = 'right';
        amountEl.style.marginLeft = '8px';
        
        const sign = type === 'received' ? '+' : type === 'sent' ? '-' : '';
        const color = type === 'received' ? '#00ff88' : type === 'sent' ? '#ff6b6b' : '#888';
        
        // UTXO value (inscription sat value)
        const inscriptionValue = inscription.outputValue || inscription.utxo?.value || 600;
        
        amountEl.innerHTML = `
            <div style="color: ${color}; font-weight: 600; font-size: 14px;">${sign}${inscriptionValue} <span style="font-size: 11px; font-weight: 400;">sats</span></div>
            <div style="font-size: 11px; color: #666; margin-top: 2px;">UTXO Value</div>
        `;
        
        // TODO: Se tivéssemos o preço de venda do marketplace, mostraríamos aqui
        // Por exemplo: "Sold for: 10,000 sats"
        
        // Montar item
        item.appendChild(thumbnail);
        item.appendChild(info);
        item.appendChild(amountEl);
        
        // Adicionar borda especial se pending
        if (!tx.status.confirmed) {
            item.style.opacity = '0.8';
            item.style.border = '1px solid rgba(255, 193, 7, 0.4)';
            item.style.background = 'rgba(255, 193, 7, 0.02)';
        }
        
    } else {
        // ✅ TRANSAÇÃO NORMAL DE BITCOIN OU RUNES (sem inscription)
        
        // Ícone ou Thumbnail (para Runes)
        const icon = document.createElement('div');
        icon.className = 'activity-icon';
        
        if (isRunesTx && runesData && runesData.length > 0) {
            // Para RUNES: Mostrar thumbnail do parent (igual às inscriptions)
            
            // Verificar se é Cenotaph (runes queimadas)
            const isCenotaph = runeInfo && runeInfo.name && runeInfo.name.startsWith('🔥');
            
            icon.style.cssText = `
                width: 50px;
                height: 50px;
                border-radius: 8px;
                background: #1a1a1a;
                overflow: hidden;
                flex-shrink: 0;
                margin-right: 12px;
                border: 1.5px solid ${isCenotaph ? '#ff4444' : '#f59e0b'};
            `;
            
            // 🔥 Se for BURNED (Cenotaph), mostrar caixão
            if (isCenotaph) {
                icon.innerHTML = `<div style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; font-size: 24px; background: linear-gradient(135deg, rgba(255, 68, 68, 0.3), rgba(139, 0, 0, 0.2));">⚰️</div>`;
            } else {
                // Buscar thumbnail da primeira rune
                const firstRune = runesData[0];
                const runeName = firstRune.name;
                const runeSymbol = firstRune.symbol || '⧈';
                const thumbnailUrl = runesThumbnailsMap.get(runeName);
                
                if (thumbnailUrl) {
                    // ✅ Mostrar SOMENTE thumbnail real do parent (sem fallback emoji)
                    icon.innerHTML = `<img src="${thumbnailUrl}" style="width: 100%; height: 100%; object-fit: cover; background: #1a1a1a;" onerror="this.style.background='linear-gradient(135deg, rgba(245, 158, 11, 0.2), rgba(251, 191, 36, 0.1))';">`;
                } else {
                    // Fallback: Se não tem thumbnail, mostrar placeholder neutro (não emoji)
                    icon.innerHTML = `<div style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; font-size: 16px; color: #f59e0b; background: linear-gradient(135deg, rgba(245, 158, 11, 0.2), rgba(251, 191, 36, 0.1));">🪙</div>`;
                }
            }
            
        } else {
            // Para BTC normal: Ícone circular
            icon.style.cssText = `
                width: 40px;
                height: 40px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-size: 20px;
                flex-shrink: 0;
                margin-right: 12px;
            `;
            
            if (type === 'received') {
                icon.innerHTML = '<span style="color: #00ff88;">↓</span>';
                icon.style.background = 'rgba(0, 255, 136, 0.1)';
            } else if (type === 'sent') {
                icon.innerHTML = '<span style="color: #ff6b6b;">↑</span>';
                icon.style.background = 'rgba(255, 107, 107, 0.1)';
            } else {
                icon.innerHTML = '<span style="color: #888;">↔</span>';
                icon.style.background = 'rgba(136, 136, 136, 0.1)';
            }
        }
        
        // Info
        const info = document.createElement('div');
        info.className = 'activity-info';
        info.style.flex = '1';
        
        // Título
        const title = document.createElement('div');
        title.className = 'activity-title';
        title.style.fontWeight = '600';
        
        if (isRunesTx) {
            // Transação de RUNES - verificar se é Cenotaph (queimada)
            const isCenotaph = runeInfo && runeInfo.name && runeInfo.name.startsWith('🔥');
            
            if (isCenotaph) {
                // Cenotaph: Runes foram queimadas acidentalmente
                title.innerHTML = '🔥 <span style="color: #ff6b6b;">Runes BURNED (Cenotaph)</span>';
            } else if (type === 'received') {
                title.innerHTML = '📥 <span style="color: #00ff88;">Received Runes</span>';
            } else if (type === 'sent') {
                title.innerHTML = '📤 <span style="color: #ff6b6b;">Sent Runes</span>';
            } else {
                title.textContent = 'Runes Transaction';
            }
        } else {
            // Transação normal de Bitcoin - SEM TAG
            if (type === 'received') {
                title.innerHTML = '📥 <span style="color: #00ff88;">Received</span>';
            } else if (type === 'sent') {
                title.innerHTML = '📤 <span style="color: #ff6b6b;">Sent</span>';
            } else {
                title.textContent = 'Transaction';
            }
        }
        
        info.appendChild(title);
        
        // Se for Runes, adicionar info da Rune logo após o título
        if (isRunesTx && runeInfo) {
            const runeInfoEl = document.createElement('div');
            runeInfoEl.style.cssText = `
                font-size: 12px;
                margin-top: 4px;
                font-weight: 600;
            `;
            
            // Formatar display: Nome (branco) + Amount (verde) + Symbol (grande)
            let displayHTML = `<span style="color: #fff;">${sanitizeHTML(runeInfo.name)}</span>`;
            
            if (runeInfo.amount && runeInfo.amount !== '') {
                // Buscar divisibility do map
                const divisibility = (window.runesDivisibilityMap && window.runesDivisibilityMap.get(runeInfo.name)) || 0;
                const formattedAmount = formatRuneAmount(runeInfo.amount, divisibility);
                
                // Symbol ao lado do amount
                const symbol = (runesData && runesData.length > 0 && runesData[0].symbol) || '⧈';
                
                displayHTML += ` <span style="color: #888;">•</span> <span style="color: #10b981;">${sanitizeHTML(formattedAmount)} ${sanitizeHTML(symbol)}</span>`;
            }
            
            runeInfoEl.innerHTML = displayHTML;
            info.appendChild(runeInfoEl);
        }
        
        const addressEl = document.createElement('div');
        addressEl.className = 'activity-address';
        addressEl.style.fontSize = '12px';
        addressEl.style.color = '#666';
        addressEl.style.marginTop = '4px';
        
        if (otherAddress) {
            const prefix = type === 'received' ? 'From: ' : 'To: ';
            addressEl.textContent = prefix + (otherAddress.slice(0, 12) + '...' + otherAddress.slice(-8));
        } else {
            addressEl.textContent = 'Internal';
        }
        
        // Data e confirmações
        const meta = document.createElement('div');
        meta.className = 'activity-meta';
        meta.style.fontSize = '11px';
        meta.style.color = '#888';
        meta.style.marginTop = '4px';
        
        let timeAgo = 'Just now';
        if (tx.status.block_time) {
            const date = new Date(tx.status.block_time * 1000);
            timeAgo = getTimeAgo(date);
        }
        
        const confirmations = tx.status.confirmed ? '✓ Confirmed' : '⏳ Pending';
        meta.textContent = `${timeAgo} • ${confirmations}`;
        
        // Adicionar classe pending se não confirmado
        if (!tx.status.confirmed) {
            item.classList.add('pending-tx');
            item.style.opacity = '0.7';
            item.style.border = '1px solid rgba(255, 193, 7, 0.3)';
        }
        
        info.appendChild(addressEl);
        info.appendChild(meta);
        
        // Amount
        const amountEl = document.createElement('div');
        amountEl.className = 'activity-amount';
        amountEl.style.textAlign = 'right';
        amountEl.style.marginLeft = '8px';
        
        const sign = type === 'received' ? '+' : type === 'sent' ? '-' : '';
        const color = type === 'received' ? '#00ff88' : type === 'sent' ? '#ff6b6b' : '#888';
        
        amountEl.innerHTML = `
            <span style="color: ${color}; font-weight: 600;">${sign}${amount.toLocaleString()}</span>
            <span style="font-size: 12px; color: #666;"> sats</span>
        `;
        
        // Montar item
        item.appendChild(icon);
        item.appendChild(info);
        item.appendChild(amountEl);
    }
    
    // Click handler: abrir no explorador apropriado
    item.style.cursor = 'pointer';
    item.addEventListener('click', () => {
        // 🚀 KRAYSCAN - Nosso explorador próprio!
        const url = `https://kraywallet-backend.onrender.com/krayscan.html?txid=${tx.txid}`;
        console.log(`🔗 Opening TX in KrayScan: ${url}`);
        chrome.tabs.create({ url });
    });
    
    return item;
}

// Helper: Time ago
function getTimeAgo(date) {
    const seconds = Math.floor((new Date() - date) / 1000);
    
    if (seconds < 60) return 'Just now';
    if (seconds < 3600) return Math.floor(seconds / 60) + 'm ago';
    if (seconds < 86400) return Math.floor(seconds / 3600) + 'h ago';
    if (seconds < 604800) return Math.floor(seconds / 86400) + 'd ago';
    
    return date.toLocaleDateString();
}

// ============================================================================
// ANALYTICS TRACKING
// ============================================================================

async function trackUserAction(address, actionType, actionData = {}) {
    try {
        const API_URL = 'https://kraywallet-backend.onrender.com/api';
        
        const response = await fetch(`${API_URL}/analytics/track`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                address: address,
                action_type: actionType,
                action_data: actionData,
                offer_id: actionData.offer_id || null,
                inscription_id: actionData.inscription_id || null,
                rune_id: actionData.rune_id || null,
                amount: actionData.amount || null
            })
        });
        
        const data = await response.json();
        
        if (data.success) {
            console.log(`✅ Tracked action: ${actionType}`, data);
        }
        
        return data;
    } catch (error) {
        console.warn('⚠️  Failed to track action:', error.message);
        // Não bloquear a UI se o tracking falhar
        return null;
    }
}

// Load Wallet Data
async function loadWalletData() {
    try {
        console.log('📊 Loading wallet data...');
        
        // 🔥 VERIFICAR REDE PRIMEIRO - antes de carregar qualquer coisa
        const activeNetworkResult = await chrome.storage.local.get(['activeNetwork']);
        const isKrayL2 = activeNetworkResult.activeNetwork === 'kray-l2';
        
        console.log(`📡 Current network: ${activeNetworkResult.activeNetwork || 'mainnet'}`);
        
        const response = await chrome.runtime.sendMessage({
            action: 'getWalletInfo'
        });
        
        console.log('📨 Wallet info response:', response);
        
        if (response && response.success) {
            const { address, balance } = response.data;
            
            // 📊 Track wallet open
            trackUserAction(address, 'wallet_open', {
                balance: balance
            });
            
            console.log('✅ Address:', address);
            console.log('✅ Balance:', balance);
            
            // Atualizar UI com verificação
            const walletAddressEl = document.getElementById('wallet-address');
            if (walletAddressEl) {
                walletAddressEl.textContent = address.slice(0, 12) + '...' + address.slice(-8);
                walletAddressEl.setAttribute('data-full-address', address); // Guardar endereço completo
            }
            
            const receiveAddressEl = document.getElementById('receive-address');
            if (receiveAddressEl) {
                receiveAddressEl.textContent = address;
            }
            
            // 🔥 SÓ MOSTRAR BALANCE DO MAINNET SE NÃO ESTIVER EM L2!
            if (!isKrayL2) {
                const balanceEl = document.getElementById('wallet-balance');
                if (balanceEl) {
                    balanceEl.textContent = balance.total.toLocaleString() + ' sats';
                }
                
                const balanceBtcEl = document.getElementById('wallet-balance-btc');
                if (balanceBtcEl) {
                    balanceBtcEl.textContent = (balance.total / 100000000).toFixed(8) + ' BTC';
                }
            }
            
            // Gerar QR Code para o endereço
            generateQRCode(address);
            
            // 🔥 SE ESTIVER EM L2, CARREGAR L2 DATA E APLICAR SWITCH
            if (isKrayL2) {
                console.log('⚡ Starting in KRAY L2 mode - loading L2 data...');
                
                // Register showScreen first
                if (window.krayL2 && typeof window.krayL2.setShowScreen === 'function') {
                    window.krayL2.setShowScreen(showScreen);
                }
                
                // Apply L2 network switch (hide mainnet UI, show L2 UI)
                await switchNetwork('kray-l2');
                
            } else {
                // Carregar Ordinals, Runes e Activity (só no mainnet)
                await loadOrdinals(address);
                await loadRunes(address);
                await loadActivity(address);
                
                // ⚡ Pre-load L2 data in background (without updating UI!)
                if (window.krayL2) {
                    // Register showScreen first
                    if (typeof window.krayL2.setShowScreen === 'function') {
                        window.krayL2.setShowScreen(showScreen);
                    }
                    // Load L2 data in background (updateUI = false to prevent overwriting mainnet balance!)
                    if (typeof window.krayL2.loadL2Data === 'function') {
                        window.krayL2.loadL2Data(false).catch(e => console.warn('L2 preload error:', e));
                    }
                }
            }
            
            console.log('✅ Wallet data loaded successfully');
            
            // Garantir que event listeners estão vinculados
            console.log('🔗 Re-binding event listeners after data load...');
            rebindWalletButtons();
        } else {
            console.error('❌ Failed to get wallet info:', response?.error);
            showNotification(response?.error || 'Failed to load wallet', 'error');
        }
    } catch (error) {
        console.error('❌ Error loading wallet data:', error);
        showNotification('Error loading wallet: ' + error.message, 'error');
    }
}

// Gerar QR Code
function generateQRCode(address) {
    try {
        console.log('🔲 Generating QR Code for address:', address);
        
        const qrContainer = document.getElementById('qr-code');
        
        if (!qrContainer) {
            console.warn('⚠️  QR Code container not found');
            return;
        }
        
        // Limpar conteúdo anterior
        qrContainer.innerHTML = '<div class="qr-placeholder">⏳ Loading...</div>';
        
        // Criar imagem do QR Code usando API gratuita
        const qrImg = document.createElement('img');
        
        // API 1: QRServer.com (mais rápida)
        // Formato: bitcoin:ADDRESS para wallets reconhecerem automaticamente
        const qrUrl1 = `https://api.qrserver.com/v1/create-qr-code/?size=200x200&data=bitcoin:${address}`;
        
        // API 2: Quickchart.io (fallback)
        const qrUrl2 = `https://quickchart.io/qr?text=bitcoin:${address}&size=200`;
        
        qrImg.src = qrUrl1;
        qrImg.alt = 'QR Code';
        qrImg.style.width = '200px';
        qrImg.style.height = '200px';
        qrImg.style.borderRadius = '8px';
        qrImg.style.display = 'block';
        
        // Sucesso ao carregar
        qrImg.onload = () => {
            console.log('✅ QR Code loaded successfully');
            qrContainer.innerHTML = '';
            qrContainer.appendChild(qrImg);
        };
        
        // Se falhar, tentar API alternativa
        qrImg.onerror = () => {
            console.log('⚠️  Primary QR API failed, trying alternative...');
            
            const qrImgFallback = document.createElement('img');
            qrImgFallback.src = qrUrl2;
            qrImgFallback.alt = 'QR Code';
            qrImgFallback.style.width = '200px';
            qrImgFallback.style.height = '200px';
            qrImgFallback.style.borderRadius = '8px';
            qrImgFallback.style.display = 'block';
            
            qrImgFallback.onload = () => {
                console.log('✅ QR Code loaded from fallback API');
                qrContainer.innerHTML = '';
                qrContainer.appendChild(qrImgFallback);
            };
            
            qrImgFallback.onerror = () => {
                console.error('❌ All QR APIs failed');
                qrContainer.innerHTML = '<div class="qr-placeholder">❌ Failed to load</div>';
            };
        };
        
    } catch (error) {
        console.error('❌ Error generating QR Code:', error);
        const qrContainer = document.getElementById('qr-code');
        qrContainer.innerHTML = '<div class="qr-placeholder">❌ Error</div>';
    }
}

// Load Ordinals (Inscriptions)
async function loadOrdinals(address) {
    const container = document.getElementById('ordinals-list');
    
    if (!container) {
        console.error('❌ ordinals-list container not found!');
        return;
    }
    
    // 🔄 SEMPRE mostrar loading primeiro
    container.innerHTML = `
        <div class="loading-container">
            <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
            <div class="loading-spinner"></div>
            <p class="loading-text">Loading inscriptions...</p>
        </div>
    `;
    
    // 💾 VERIFICAR CACHE PRIMEIRO
    if (isCacheValid('ordinals')) {
        console.log('⚡ Using cached ordinals data (saved API call & energy)');
        const cachedData = dataCache.ordinals.data;
        
        if (cachedData && cachedData.length > 0) {
            container.innerHTML = '';
            cachedData.forEach(inscription => {
                const item = createOrdinalItem(inscription);
                container.appendChild(item);
            });
        } else {
            container.innerHTML = '<div class="empty-state"><span class="empty-icon">◉</span><p>No inscriptions yet</p></div>';
        }
        return;
    }
    
    try {
        console.log('═══════════════════════════════════════════════════');
        console.log('🔧 POPUP.JS VERSION: 2024-FIXED-MULTIPLE-CALLS ✅');
        console.log('═══════════════════════════════════════════════════');
        console.log('🖼️  Loading ordinals for address:', address);
        
        // Buscar inscriptions via API
        const response = await chrome.runtime.sendMessage({
            action: 'getInscriptions',
            data: { address }
        });
        
        console.log('📦 Inscriptions response:', response);
        console.log('   response.success:', response?.success);
        console.log('   response.inscriptions:', response?.inscriptions);
        console.log('   inscriptions.length:', response?.inscriptions?.length);
        
        if (response && response.success && response.inscriptions && response.inscriptions.length > 0) {
            console.log(`✅ Found ${response.inscriptions.length} inscriptions`);
            
            // 💾 SALVAR NO CACHE
            dataCache.ordinals.data = response.inscriptions;
            dataCache.ordinals.timestamp = Date.now();
            dataCache.ordinals.loaded = true;
            console.log('💾 Ordinals cached for 5 minutes');
            
            container.innerHTML = '';
            
            response.inscriptions.forEach(inscription => {
                console.log('   Creating container for:', inscription.id);
                const item = createOrdinalItem(inscription);
                container.appendChild(item);
            });
            
            console.log('✅ All containers created!');
        } else {
            console.log('ℹ️  No inscriptions found - Reason:');
            console.log('   response exists?', !!response);
            console.log('   response.success?', response?.success);
            console.log('   response.inscriptions exists?', !!response?.inscriptions);
            console.log('   inscriptions.length > 0?', (response?.inscriptions?.length || 0) > 0);
            
            // 💾 SALVAR CACHE VAZIO
            dataCache.ordinals.data = [];
            dataCache.ordinals.timestamp = Date.now();
            dataCache.ordinals.loaded = true;
            
            container.innerHTML = '<div class="empty-state"><span class="empty-icon">◉</span><p>No inscriptions yet</p></div>';
        }
    } catch (error) {
        console.error('❌ Error loading ordinals:', error);
        container.innerHTML = '<div class="empty-state"><span class="empty-icon">⚠️</span><p>Failed to load inscriptions</p></div>';
    }
}

/**
 * Load Inscription Buttons State (verificar se está listado e mostrar botões corretos)
 */
async function loadInscriptionButtonsState(inscription, buttonsContainer) {
    try {
        // Buscar wallet address
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const address = result?.data?.address || walletState.address;
        
        if (!address) {
            // Sem wallet, mostrar botões padrão desabilitados
            buttonsContainer.innerHTML = '<p style="color: #888; font-size: 11px; text-align: center;">Connect wallet</p>';
            return;
        }
        
        // Marketplace disabled in production - show default buttons directly
        // const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/?seller_address=${address}`);
        // const data = await response.json();
        const data = { listings: [] }; // No listings in production
        
        // Verificar se esta inscrição está listada
        const activeListing = data.listings?.find(listing => 
            listing.inscription_id === inscription.id &&
            (listing.status === 'PENDING_SIGNATURES' || listing.status === 'OPEN')
        );
        
        if (activeListing) {
            // ✅ LISTADO - Mostrar botões "Change Price" e "Cancel"
            buttonsContainer.innerHTML = '';
            
            // Badge "Listed"
            const badge = document.createElement('div');
            badge.style.cssText = `
                width: 100%;
                padding: 6px 8px;
                background: rgba(76, 175, 80, 0.1);
                border: 1px solid #4CAF50;
                border-radius: 6px;
                margin-bottom: 8px;
                text-align: center;
            `;
            badge.innerHTML = `<span style="color: #4CAF50; font-size: 10px; font-weight: 600;">✅ LISTED: ${activeListing.price_sats.toLocaleString()} sats</span>`;
            buttonsContainer.appendChild(badge);
            
            // Botão: Change Price (se PENDING_SIGNATURES)
            if (activeListing.status === 'PENDING_SIGNATURES') {
                const changePriceBtn = document.createElement('button');
                changePriceBtn.innerHTML = '💰 Change Price';
                changePriceBtn.style.cssText = `
                    flex: 1;
                    padding: 8px 12px;
                    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
                    color: white;
                    border: none;
                    border-radius: 6px;
                    font-size: 11px;
                    font-weight: 600;
                    cursor: pointer;
                `;
                changePriceBtn.addEventListener('click', (e) => {
                    e.stopPropagation();
                    showUpdatePriceModal(activeListing);
                });
                buttonsContainer.appendChild(changePriceBtn);
            }
            
            // Botão: Cancel Listing
            const cancelBtn = document.createElement('button');
            cancelBtn.innerHTML = '❌ Cancel';
            cancelBtn.style.cssText = `
                flex: 1;
                padding: 8px 12px;
                background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
                color: white;
                border: none;
                border-radius: 6px;
                font-size: 11px;
                font-weight: 600;
                cursor: pointer;
            `;
            cancelBtn.addEventListener('click', async (e) => {
                e.stopPropagation();
                // Vai direto para assinatura (sem confirm)
                await cancelAtomicListing(activeListing.order_id);
                // Recarregar ordinals após cancelar
                await loadOrdinals();
            });
            buttonsContainer.appendChild(cancelBtn);
            
        } else {
            // ❌ NÃO LISTADO - Mostrar botões "List" e "Send" normais
            buttonsContainer.innerHTML = '';
            
            // Botão: List
            const listBtn = document.createElement('button');
            listBtn.innerHTML = '🛒 List';
            listBtn.style.cssText = `
                flex: 1;
                padding: 10px 16px;
                background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 12px;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.2s ease;
                box-shadow: 0 2px 8px rgba(245, 158, 11, 0.4);
            `;
            listBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                showListMarketModal(inscription);
            });
            buttonsContainer.appendChild(listBtn);
            
            // Botão: Send
            const sendBtn = document.createElement('button');
            sendBtn.innerHTML = '📤 Send';
            sendBtn.style.cssText = `
                flex: 1;
                padding: 10px 16px;
                background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 12px;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.2s ease;
                box-shadow: 0 2px 8px rgba(59, 130, 246, 0.4);
            `;
            sendBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                showSendInscriptionScreen(inscription);
            });
            buttonsContainer.appendChild(sendBtn);
        }
        
    } catch (error) {
        console.error('❌ Error loading buttons state:', error);
        // Fallback: mostrar botões padrão
        buttonsContainer.innerHTML = '<p style="color: #ef4444; font-size: 10px; text-align: center;">Error loading</p>';
    }
}

// Create Ordinal Item
function createOrdinalItem(inscription) {
    const item = document.createElement('div');
    item.className = 'ordinal-item';
    
    const preview = document.createElement('div');
    preview.className = 'ordinal-preview';
    
    // Determinar URL do conteúdo - Usar ORD local se disponível, senão ordinals.com
    const contentUrl = sanitizeURL(inscription.preview || `https://ordinals.com/content/${sanitizeHTML(inscription.id)}`);
    const contentType = (inscription.content_type || '').toLowerCase();
    
    console.log('🎨 Creating ordinal item:', {
        id: inscription.id,
        number: inscription.number,
        type: contentType,
        url: contentUrl
    });
    
    // ✅ SUPORTE COMPLETO PARA TODOS OS TIPOS DE CONTEÚDO
    
    // ⚠️ FALLBACK UNIVERSAL: Se content_type for 'unknown' ou vazio, SEMPRE tentar carregar como imagem primeiro!
    const isUnknown = !contentType || contentType === 'unknown' || contentType === '';
    
    // 1. IMAGENS (png, jpeg, jpg, webp, avif, gif, svg, bmp) OU UNKNOWN
    if (isUnknown || 
        contentType.includes('image/') || 
        contentType.includes('png') || 
        contentType.includes('jpeg') || 
        contentType.includes('jpg') || 
        contentType.includes('webp') || 
        contentType.includes('avif') || 
        contentType.includes('gif') || 
        contentType.includes('svg') ||
        contentType.includes('bmp')) {
        
        const img = document.createElement('img');
        img.src = contentUrl;
        img.alt = `Inscription #${inscription.number}`;
        img.style.width = '100%';
        img.style.height = '100%';
        img.style.objectFit = 'cover';
        img.style.borderRadius = '8px';
        
        img.onload = () => {
            preview.style.background = 'transparent';
            console.log('✅ Image loaded:', contentUrl);
        };
        
        img.onerror = () => {
            console.error('❌ Image failed to load, trying iframe fallback:', contentUrl);
            
            // FALLBACK: Tentar carregar em iframe (funciona para HTML/inscriptions complexas)
            preview.innerHTML = '';
            const iframe = document.createElement('iframe');
            iframe.src = contentUrl;
            iframe.style.width = '100%';
            iframe.style.height = '100%';
            iframe.style.border = 'none';
            iframe.style.borderRadius = '8px';
            iframe.style.background = '#1a1a1a';
            
            // Se iframe também falhar, mostrar link direto
            iframe.onerror = () => {
                console.error('❌ Iframe also failed, showing link');
                preview.innerHTML = `
                    <a href="${contentUrl}" target="_blank" style="
                        display: flex;
                        flex-direction: column;
                        align-items: center;
                        justify-content: center;
                        width: 100%;
                        height: 100%;
                        text-decoration: none;
                        color: #888;
                    ">
                        <span style="font-size: 32px;">◉</span>
                        <span style="font-size: 10px; margin-top: 8px;">Click to view</span>
                    </a>
                `;
            };
            
            preview.appendChild(iframe);
        };
        
        preview.appendChild(img);
    }
    // 2. VÍDEOS (mp4, webm, ogg)
    else if (contentType.includes('video/') || 
             contentType.includes('mp4') || 
             contentType.includes('webm') || 
             contentType.includes('ogg')) {
        
        const video = document.createElement('video');
        video.src = contentUrl;
        video.style.width = '100%';
        video.style.height = '100%';
        video.style.objectFit = 'cover';
        video.style.borderRadius = '8px';
        video.controls = true;
        video.muted = true;
        video.autoplay = false;
        
        preview.appendChild(video);
    }
    // 3. HTML/TEXT
    else if (contentType.includes('html') || contentType.includes('text/html')) {
        
        const iframe = document.createElement('iframe');
        iframe.src = contentUrl;
        iframe.style.width = '100%';
        iframe.style.height = '100%';
        iframe.style.border = 'none';
        iframe.style.borderRadius = '8px';
        iframe.sandbox = 'allow-scripts allow-same-origin';
        
        preview.appendChild(iframe);
    }
    // 4. TEXTO (text/plain, json, etc)
    else if (contentType.includes('text/')) {
        
        const textPreview = document.createElement('div');
        textPreview.style.padding = '12px';
        textPreview.style.fontSize = '10px';
        textPreview.style.fontFamily = 'monospace';
        textPreview.style.overflow = 'hidden';
        textPreview.style.textOverflow = 'ellipsis';
        textPreview.style.whiteSpace = 'pre-wrap';
        textPreview.style.maxHeight = '100%';
        textPreview.textContent = '📄 Text content';
        
        preview.appendChild(textPreview);
    }
    // 5. ÁUDIO (mp3, wav, ogg)
    else if (contentType.includes('audio/') || 
             contentType.includes('mp3') || 
             contentType.includes('wav')) {
        
        preview.innerHTML = '<span style="font-size: 32px;">🎵</span>';
        preview.style.display = 'flex';
        preview.style.alignItems = 'center';
        preview.style.justifyContent = 'center';
        
        const audio = document.createElement('audio');
        audio.src = contentUrl;
        audio.controls = true;
        audio.style.width = '90%';
        audio.style.marginTop = '8px';
        preview.appendChild(audio);
    }
    // 6. FALLBACK - Mostrar ícone genérico
    else {
        console.log('⚠️  Unknown content type, using fallback icon');
        preview.innerHTML = '<span style="font-size: 32px;">📦</span>';
        preview.style.display = 'flex';
        preview.style.alignItems = 'center';
        preview.style.justifyContent = 'center';
    }
    
    const info = document.createElement('div');
    info.className = 'ordinal-info';
    info.style.cssText = `
        padding: 8px 12px;
        display: flex;
        flex-direction: column;
        gap: 4px;
    `;
    
    // ✨ LINHA 1: Inscription Number (destaque)
    const number = document.createElement('div');
    number.className = 'ordinal-number';
    number.style.cssText = `
        font-size: 13px;
        font-weight: 600;
        color: var(--color-text-primary);
        margin-bottom: 2px;
    `;
    number.textContent = `Inscription #${inscription.number || '?'}`;
    
    // ⏰ BADGE "PENDING" se a inscription ainda não foi indexada pela API
    if (inscription.pending) {
        const pendingBadge = document.createElement('span');
        pendingBadge.style.cssText = `
            display: inline-block;
            margin-left: 8px;
            padding: 2px 8px;
            background: #ff9800;
            color: white;
            border-radius: 10px;
            font-size: 9px;
            font-weight: 600;
            vertical-align: middle;
        `;
        pendingBadge.textContent = '⏰ Pending';
        pendingBadge.title = 'Awaiting blockchain confirmation';
        number.appendChild(pendingBadge);
    }
    
    // 🌟 SAT RARITY BADGE - Detect and show rare sat indicator
    if (inscription.sat && typeof SatRarity !== 'undefined') {
        const rarity = SatRarity.detect(inscription.sat);
        if (rarity && rarity.isRare) {
            const rarityBadge = document.createElement('span');
            rarityBadge.style.cssText = `
                display: inline-block;
                margin-left: 6px;
                padding: 2px 6px;
                background: ${rarity.primaryColor}20;
                color: ${rarity.primaryColor};
                border: 1px solid ${rarity.primaryColor}40;
                border-radius: 8px;
                font-size: 9px;
                font-weight: 600;
                vertical-align: middle;
                cursor: help;
            `;
            rarityBadge.textContent = rarity.rarities.slice(0, 2).map(r => r.emoji).join('');
            rarityBadge.title = rarity.description;
            number.appendChild(rarityBadge);
        }
    }
    
    // ✨ LINHA 2: Value (sats) - ao invés de content_type
    const valueDiv = document.createElement('div');
    valueDiv.className = 'ordinal-value';
    valueDiv.style.cssText = `
        font-size: 11px;
        color: #4CAF50;
        font-weight: 500;
    `;
    const satsValue = inscription.outputValue || inscription.value || 600; // ✅ Use real UTXO value from backend!
    console.log(`🔍 Creating ordinal item #${inscription.number}: outputValue=${inscription.outputValue}, value=${inscription.value}, using=${satsValue}`);
    valueDiv.textContent = `UTXO: ${satsValue.toLocaleString()} sats`;
    
    // ✨ LINHA 3: ID truncado (small)
    const id = document.createElement('div');
    id.className = 'ordinal-id';
    id.style.cssText = `
        font-size: 9px;
        color: var(--color-text-secondary);
        font-family: monospace;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    `;
    const fullId = inscription.id || 'Unknown ID';
    id.textContent = fullId.length > 24 ? fullId.substring(0, 12) + '...' + fullId.substring(fullId.length - 8) : fullId;
    id.title = fullId; // Tooltip com ID completo
    
    info.appendChild(number);
    info.appendChild(valueDiv);
    info.appendChild(id);
    
    item.appendChild(preview);
    item.appendChild(info);
    
    // ✅ CONTAINER DE BOTÕES (embaixo da info - dois botões lado a lado)
    const buttonsContainer = document.createElement('div');
    buttonsContainer.style.cssText = `
        padding: 8px 12px;
        padding-top: 4px;
        display: flex;
        gap: 8px;
        border-top: 1px solid var(--color-border);
    `;
    
    // 🔍 VERIFICAR SE ESTÁ LISTADO (carregar dinamicamente - async)
    (async () => {
        await loadInscriptionButtonsState(inscription, buttonsContainer);
    })();
    
    item.appendChild(buttonsContainer);
    
    // Click handler do card (mostrar detalhes internos)
    item.addEventListener('click', () => {
        showInscriptionDetails(inscription);
    });
    
    // Tornar o item posicionável para o botão absoluto
    item.style.position = 'relative';
    
    return item;
}

// Show Send Inscription Screen
async function showSendInscriptionScreen(inscription) {
    console.log('📤 Opening send inscription screen for:', inscription.id);
    
    // Esconder todas as screens
    document.querySelectorAll('.screen').forEach(s => s.classList.add('hidden'));
    
    // Mostrar tela de envio
    const sendInscriptionScreen = document.getElementById('send-inscription-screen');
    if (sendInscriptionScreen) {
        sendInscriptionScreen.classList.remove('hidden');
        
        // Preencher detalhes da inscription (agora é async)
        await populateSendInscriptionForm(inscription);
        
        // Verificar se o botão existe
        const confirmBtn = document.getElementById('send-inscription-confirm-btn');
        console.log('   🔍 Confirm button found:', !!confirmBtn);
        console.log('   🔍 Button has click listener:', confirmBtn?._clickListener ? 'YES' : 'NO');
    } else {
        console.error('❌ Send inscription screen not found!');
    }
}

// Populate Send Inscription Form
async function populateSendInscriptionForm(inscription) {
    console.log('📋 Populating form with inscription:', inscription);
    
    // ✅ CONVERTER OUTPUT PARA UTXO (backend retorna "output": "txid:vout")
    if (!inscription.utxo || !inscription.utxo.txid) {
        console.log('   🔄 Converting output to UTXO format...');
        
        if (inscription.output) {
            const parts = inscription.output.split(':');
            if (parts.length >= 2) {
                inscription.utxo = {
                    txid: parts[0],
                    vout: parseInt(parts[1]),
                    value: inscription.outputValue || inscription.value || 600
                };
                console.log('   ✅ UTXO created from output:', inscription.utxo);
            }
        } else {
            console.error('   ❌ Inscription has no output/utxo!', inscription);
        }
    }
    
    // Preview da inscription
    const previewEl = document.getElementById('send-inscription-preview');
    if (previewEl) {
        const contentUrl = sanitizeURL(inscription.preview || `https://ordinals.com/content/${sanitizeHTML(inscription.id)}`);
        previewEl.innerHTML = `<img src="${contentUrl}" style="width: 100%; height: 100%; object-fit: cover; border-radius: 8px;" onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><text y=%22.9em%22 font-size=%2290%22>◉</text></svg>'">`;
    }
    
    // Inscription ID
    const inscIdEl = document.getElementById('send-inscription-id');
    if (inscIdEl) {
        inscIdEl.textContent = `#${inscription.number || '?'}`;
    }
    
    // Inscription details
    const detailsEl = document.getElementById('send-inscription-details');
    if (detailsEl) {
        detailsEl.innerHTML = `
            <div style="font-size: 10px; color: #888; word-break: break-all;">
                ${sanitizeHTML(inscription.id)}
            </div>
            <div style="font-size: 11px; color: #aaa; margin-top: 4px;">
                Type: ${sanitizeHTML(inscription.content_type || 'unknown')}
            </div>
            ${inscription.utxo ? `
                <div style="font-size: 10px; color: #666; margin-top: 4px;">
                    UTXO: ${inscription.utxo.txid.substring(0, 16)}...
                </div>
            ` : ''}
        `;
    }
    
    // Mostrar valor da inscription
    const inscValueEl = document.getElementById('send-inscription-value');
    if (inscValueEl && inscription.utxo) {
        inscValueEl.textContent = `${inscription.utxo.value} sats`;
    }
    
    // Limpar campos
    const recipientInput = document.getElementById('send-inscription-recipient');
    if (recipientInput) {
        recipientInput.value = '';
    }
    
    // Armazenar inscription globalmente para o envio
    window.currentSendingInscription = inscription;
    
    console.log('   ✅ Form populated, stored inscription:', window.currentSendingInscription);
}

// Load Runes
async function loadRunes(address) {
    console.log('🪙 loadRunes called with address:', address);
    const container = document.getElementById('runes-list');
    
    if (!container) {
        console.error('❌ runes-list container not found!');
        return;
    }
    
    console.log('   Container found:', container);
    
    // 🔄 SEMPRE mostrar loading primeiro
    container.innerHTML = `
        <div class="loading-container">
            <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
            <div class="loading-spinner"></div>
            <p class="loading-text">Loading runes...</p>
        </div>
    `;
    
    // 💾 VERIFICAR CACHE PRIMEIRO
    if (isCacheValid('runes')) {
        console.log('⚡ Using cached runes data (saved API call & energy)');
        const cachedData = dataCache.runes.data;
        
        if (cachedData && cachedData.length > 0) {
            container.innerHTML = '';
            cachedData.forEach(rune => {
                const item = createRuneItem(rune);
                container.appendChild(item);
            });
        } else {
            container.innerHTML = `
                <div class="empty-state">
                    <span class="empty-icon">⧈</span>
                    <p>No runes yet</p>
                    <small>Send runes to this address to see them here</small>
                </div>
            `;
        }
        return;
    }
    
    try {
        console.log('   📡 Sending message to background script...');
        
        // Buscar runes via API
        const response = await chrome.runtime.sendMessage({
            action: 'getRunes',
            data: { address }
        });
        
        console.log('   📦 Response received:', response);
        
        if (response && response.success && response.runes && response.runes.length > 0) {
            console.log(`   ✅ Found ${response.runes.length} runes`);
            
            // 💾 SALVAR NO CACHE
            dataCache.runes.data = response.runes;
            dataCache.runes.timestamp = Date.now();
            dataCache.runes.loaded = true;
            console.log('💾 Runes cached for 5 minutes');
            
            // 🌐 CACHE SYMBOLS GLOBALLY for Market tab
            window.runesSymbolsMap = window.runesSymbolsMap || new Map();
            response.runes.forEach(rune => {
                if (rune.symbol) {
                    window.runesSymbolsMap.set(rune.name, rune.symbol);
                    if (rune.displayName) window.runesSymbolsMap.set(rune.displayName, rune.symbol);
                    if (rune.runeId) window.runesSymbolsMap.set(rune.runeId, rune.symbol);
                    console.log(`   📋 Cached symbol: ${rune.name} → ${rune.symbol}`);
                }
            });
            
            container.innerHTML = '';
            
            response.runes.forEach(rune => {
                console.log('   🪙 Creating rune item:', rune.displayName || rune.name);
                const item = createRuneItem(rune);
                container.appendChild(item);
            });
        } else {
            console.log('   ℹ️  No runes found');
            
            // 💾 SALVAR CACHE VAZIO
            dataCache.runes.data = [];
            dataCache.runes.timestamp = Date.now();
            dataCache.runes.loaded = true;
            
            container.innerHTML = `
                <div class="empty-state">
                    <span class="empty-icon">⧈</span>
                    <p>No runes yet</p>
                    <small>Send runes to this address to see them here</small>
                </div>
            `;
        }
    } catch (error) {
        console.error('Error loading runes:', error);
        container.innerHTML = '<div class="empty-state">Failed to load runes</div>';
    }
}

// 🔐 RUNES VERIFICADAS - MyWallet Proprietary Feature
// 
// Esta é uma FEATURE PROPRIETÁRIA da MyWallet para dar credibilidade e segurança.
// Mesmo sendo uma wallet DESCENTRALIZADA, oferecemos features CENTRALIZADAS OPCIONAIS
// que agregam VALOR e CONFIANÇA ao produto.
//
// 🎯 Propósito:
// - Proteger usuários contra scams e runes falsas
// - Diferencial competitivo vs outras wallets
// - Posicionar MyWallet como autoridade no ecossistema Runes
// - Criar oportunidade de monetização futura (verificação paga)
//
// 📋 Critérios para Verificação:
// ✅ Projeto legítimo com comunidade ativa
// ✅ Parent inscription válido
// ✅ Rune ID oficial confirmado
// ✅ Sem histórico de scam
//
// ⚠️ SOMENTE O ADMIN PODE ADICIONAR RUNES AQUI
// Para adicionar, insira o nome EXATO da rune:
//
const VERIFIED_RUNES = [
    'DOG•GO•TO•THE•MOON',
    'LOBO•THE•WOLF•PUP'
];

// Função auxiliar para verificar se uma rune é verificada
function isRuneVerified(runeName) {
    return VERIFIED_RUNES.includes(runeName);
}

// Create Rune Item
function createRuneItem(rune) {
    const item = document.createElement('div');
    item.className = 'rune-item';
    item.dataset.runeName = rune.name;
    
    // ✨ GRID LAYOUT: 4 colunas alinhadas
    // Coluna 1: Thumbnail → Coluna 2: Nome → Coluna 3: Amount → Coluna 4: Emoji
    
    // 1️⃣ COLUNA 1: Thumbnail (52px fixo)
    const thumbnail = document.createElement('div');
    thumbnail.className = 'rune-thumbnail';
    
    const thumbnailUrl = rune.thumbnail || rune.parentPreview;
    
    if (thumbnailUrl) {
        const img = document.createElement('img');
        img.src = thumbnailUrl;
        img.alt = rune.displayName || rune.name;
        // ✅ CSP-safe error handler
        img.addEventListener('error', function() {
            const fallback = document.createElement('div');
            fallback.className = 'rune-thumbnail-fallback';
            fallback.textContent = rune.symbol || '⧈';
            thumbnail.innerHTML = '';
            thumbnail.appendChild(fallback);
        });
        thumbnail.appendChild(img);
    } else {
        const fallback = document.createElement('div');
        fallback.className = 'rune-thumbnail-fallback';
        fallback.textContent = rune.symbol || '⧈';
        thumbnail.appendChild(fallback);
    }
    
    // ✅ Adicionar badge de verificado se a rune estiver na lista
    if (isRuneVerified(rune.name)) {
        const verifiedBadge = document.createElement('div');
        verifiedBadge.className = 'rune-verified-badge';
        verifiedBadge.title = 'Verified by MyWallet';
        thumbnail.appendChild(verifiedBadge);
    }
    
    // 2️⃣ COLUNA 2: Nome (flex-grow)
    const nameContainer = document.createElement('div');
    nameContainer.className = 'rune-info';
    
    const nameSpan = document.createElement('div');
    nameSpan.className = 'rune-name';
    // Limpar o nome (remover emoji do final)
    const cleanName = (rune.displayName || rune.name).replace(/\s*[🐕🪙🔥💎⚡🌟🚀🌙🐂🐻🦁🐶🦴🎯🎨🎭🎪🎡🎢🎠🎰⧈]+\s*$/g, '');
    nameSpan.textContent = cleanName;
    
    nameContainer.appendChild(nameSpan);
    
    // 3️⃣ COLUNA 3: Amount (alinhado à direita)
    const amountDiv = document.createElement('div');
    amountDiv.className = 'rune-amount';
    amountDiv.textContent = formatRuneAmount(rune.amount);
    
    // 4️⃣ COLUNA 4: Emoji GRANDE (40px fixo)
    const emojiDiv = document.createElement('div');
    emojiDiv.className = 'rune-arrow';
    emojiDiv.textContent = rune.symbol || '⧈';
    
    // ✅ Montar Grid (ordem: thumb → nome → amount → emoji)
    item.appendChild(thumbnail);
    item.appendChild(nameContainer);
    item.appendChild(amountDiv);
    item.appendChild(emojiDiv);
    
    // Click handler
    item.addEventListener('click', () => {
        showRuneDetails(rune);
    });
    
    return item;
}

// Get Rune Emoji baseado no nome
function getRuneEmoji(name) {
    const nameLower = (name || '').toLowerCase();
    
    // Mapear nomes comuns para emojis
    if (nameLower.includes('dog')) return '🐕';
    if (nameLower.includes('cat')) return '🐱';
    if (nameLower.includes('uncommon')) return '💎';
    if (nameLower.includes('goods')) return '📦';
    if (nameLower.includes('fehu')) return '⚡';
    if (nameLower.includes('moon')) return '🌙';
    if (nameLower.includes('rocket')) return '🚀';
    if (nameLower.includes('fire')) return '🔥';
    if (nameLower.includes('water')) return '💧';
    if (nameLower.includes('gold')) return '🏆';
    if (nameLower.includes('silver')) return '🥈';
    if (nameLower.includes('bronze')) return '🥉';
    if (nameLower.includes('king')) return '👑';
    if (nameLower.includes('queen')) return '👸';
    if (nameLower.includes('dragon')) return '🐉';
    if (nameLower.includes('bear')) return '🐻';
    if (nameLower.includes('bull')) return '🐂';
    if (nameLower.includes('lion')) return '🦁';
    if (nameLower.includes('tiger')) return '🐯';
    if (nameLower.includes('wolf')) return '🐺';
    if (nameLower.includes('fox')) return '🦊';
    if (nameLower.includes('eagle')) return '🦅';
    if (nameLower.includes('pepe')) return '🐸';
    if (nameLower.includes('satoshi')) return '₿';
    if (nameLower.includes('bitcoin')) return '₿';
    if (nameLower.includes('ordinal')) return '⚙️';
    if (nameLower.includes('rune')) return '🔮';
    
    // Default: primeiro emoji ou símbolo genérico
    return '⭐';
}

// Format Rune Amount
function formatRuneAmount(rawAmount, divisibility = 0) {
    // Converter raw amount para display amount usando divisibility
    if (divisibility === 0) {
        // Sem divisibility, mostra inteiro
        const num = parseFloat(rawAmount);
        if (num >= 1000000) {
            return (num / 1000000).toFixed(2) + 'M';
        } else if (num >= 1000) {
            return (num / 1000).toFixed(2) + 'K';
        }
        return num.toLocaleString();
    }
    
    const amount = parseFloat(rawAmount);
    const displayAmount = amount / Math.pow(10, divisibility);
    
    // Se o valor for inteiro (sem decimais reais), mostra sem casas decimais
    if (displayAmount % 1 === 0) {
        const num = Math.floor(displayAmount);
        if (num >= 1000000) {
            return (num / 1000000).toFixed(2) + 'M';
        } else if (num >= 1000) {
            return (num / 1000).toFixed(2) + 'K';
        }
        return num.toLocaleString('en-US', {
            minimumFractionDigits: 0,
            maximumFractionDigits: 0
        });
    }
    
    // Se tiver decimais, mostra apenas decimais significativos (remove zeros à direita)
    return displayAmount.toLocaleString('en-US', {
        minimumFractionDigits: 0,
        maximumFractionDigits: divisibility
    });
}

// Copy Address
async function handleCopyAddress() {
    const response = await chrome.runtime.sendMessage({
        action: 'getWalletInfo'
    });
    
    if (response.success) {
        const address = response.data.address;
        await navigator.clipboard.writeText(address);
        showNotification('Address copied!', 'success');
    }
}

// Send - Show Password Modal First
async function handleSend() {
    console.log('💸 ========== HANDLE SEND CALLED ==========');
    
    const toAddress = document.getElementById('send-address').value.trim();
    const amount = parseInt(document.getElementById('send-amount').value);
    const feeSelect = document.getElementById('send-fee').value;
    
    // Determinar fee rate (custom ou preset)
    let feeRate;
    if (feeSelect === 'custom') {
        const customFee = document.getElementById('send-fee-custom').value;
        if (!customFee || customFee < 1) {
            showNotification('Please enter a valid custom fee rate (minimum 1 sat/vB)', 'error');
            return;
        }
        feeRate = parseInt(customFee);
    } else {
        feeRate = parseInt(feeSelect);
    }
    
    console.log('   To Address:', toAddress);
    console.log('   Amount:', amount);
    console.log('   Fee Rate:', feeRate);
    
    if (!toAddress || !amount || !feeRate) {
        console.error('   ❌ Missing fields!');
        showNotification('Please fill all fields', 'error');
        return;
    }
    
    console.log('   ✅ All fields valid');
    
    // Mostrar modal de confirmação com resumo da transação
    showPasswordModal(toAddress, amount, feeRate);
}

// Show Password Confirmation Modal
function showPasswordModal(toAddress, amount, feeRate) {
    console.log('🔐 Showing password confirmation modal...');
    
    // Preencher resumo da transação
    document.getElementById('confirm-to-address').textContent = 
        toAddress.slice(0, 20) + '...' + toAddress.slice(-10);
    document.getElementById('confirm-amount').textContent = 
        amount.toLocaleString() + ' sats';
    document.getElementById('confirm-fee-rate').textContent = 
        feeRate + ' sat/vB';
    
    // Limpar senha anterior
    document.getElementById('confirm-password-input').value = '';
    
    // Mostrar modal
    const modal = document.getElementById('password-modal');
    modal.classList.remove('hidden');
    
    // Focar no campo de senha
    setTimeout(() => {
        document.getElementById('confirm-password-input').focus();
    }, 100);
    
    // Armazenar dados da transação para uso posterior
    window.pendingTransaction = { toAddress, amount, feeRate };
}

// Handle Send Inscription
async function handleSendInscription() {
    console.log('📤 ========== HANDLE SEND INSCRIPTION CALLED ==========');
    console.log('   🔍 Function entered successfully!');
    
    const recipientAddress = document.getElementById('send-inscription-recipient').value.trim();
    console.log('   📝 Recipient from input:', recipientAddress);
    const feeRate = parseInt(document.getElementById('send-inscription-fee').value);
    const inscription = window.currentSendingInscription;
    
    console.log('   Inscription:', inscription);
    console.log('   Recipient:', recipientAddress);
    console.log('   Fee Rate:', feeRate);
    
    // Validações
    if (!inscription) {
        console.error('   ❌ No inscription selected!');
        showNotification('No inscription selected', 'error');
        return;
    }
    
    if (!recipientAddress) {
        console.error('   ❌ No recipient address!');
        showNotification('Please enter recipient address', 'error');
        return;
    }
    
    if (!feeRate || feeRate < 1) {
        console.error('   ❌ Invalid fee rate!');
        showNotification('Please enter a valid fee rate (min: 1 sat/vB)', 'error');
        return;
    }
    
    // Validar formato do endereço (Taproot)
    if (!recipientAddress.startsWith('bc1p') && !recipientAddress.startsWith('tb1p')) {
        console.error('   ❌ Invalid Taproot address!');
        showNotification('Please enter a valid Taproot address (bc1p...)', 'error');
        return;
    }
    
    console.log('   ✅ All fields valid');
    console.log('=========================================');
    
    // 🔐 SOLICITAR SENHA (modal de confirmação)
    // Armazenar dados da transação para uso posterior
    window.pendingInscriptionTransaction = { inscription, recipientAddress, feeRate };
    
    // Mostrar modal de senha
    document.getElementById('password-modal').classList.remove('hidden');
    document.getElementById('confirm-password-input').value = '';
    
    // Preencher resumo
    document.getElementById('confirm-to-address').textContent = 
        recipientAddress.slice(0, 20) + '...' + recipientAddress.slice(-10);
    document.getElementById('confirm-amount').textContent = 
        `Inscription #${inscription.number}`;
    document.getElementById('confirm-fee-rate').textContent = 
        feeRate + ' sat/vB';
    
    // Focar no campo de senha
    setTimeout(() => {
        document.getElementById('confirm-password-input').focus();
    }, 100);
}

// Confirm and Send Inscription (chamado quando o usuário confirma a senha)
async function confirmAndSendInscription(password) {
    console.log('🔐 ========== CONFIRMING INSCRIPTION SEND ==========');
    
    // 🔒 MUTEX: Block concurrent sends (shared with rune sends)
    if (isSendingTransaction) {
        showNotification('⏳ Transaction in progress. Please wait for it to complete.', 'warning');
        return;
    }
    isSendingTransaction = true;
    
    const pending = window.pendingInscriptionTransaction;
    
    if (!pending) {
        console.error('❌ No pending inscription transaction!');
        showNotification('No pending transaction', 'error');
        isSendingTransaction = false;
        return;
    }
    
    const { inscription, recipientAddress, feeRate } = pending;
    
    console.log('   Inscription:', inscription.id);
    console.log('   Recipient:', recipientAddress);
    console.log('   Fee Rate:', feeRate);
    console.log('   Password provided:', password ? 'YES ✅' : 'NO ❌');
    
    try {
        // Fechar modal
        closePasswordModal();
        
        // Mostrar notificação de processamento
        showNotification('🔧 Creating transaction...', 'info');
        
        // Verificar se a wallet está desbloqueada
        const walletResponse = await chrome.runtime.sendMessage({
            action: 'getWalletInfo'
        });
        
        if (!walletResponse || !walletResponse.success || !walletResponse.data || !walletResponse.data.address) {
            console.error('   ❌ Wallet locked or not found!');
            showNotification('Please unlock your wallet first', 'error');
            return;
        }
        
        console.log('   ✅ Wallet unlocked:', walletResponse.data.address);
        
        // 🔓 DESBLOQUEAR WALLET COM A SENHA
        console.log('   🔓 Unlocking wallet with password...');
        
        const unlockResult = await chrome.runtime.sendMessage({
            action: 'unlockWallet',
            data: { password }
        });
        
        if (!unlockResult || !unlockResult.success) {
            throw new Error('Failed to unlock wallet with provided password');
        }
        
        console.log('   ✅ Wallet unlocked successfully');
        
        // Solicitar ao backend para criar e assinar a transação
        const sendResult = await chrome.runtime.sendMessage({
            action: 'sendInscription',
            data: {
                inscription: inscription,
                recipientAddress: recipientAddress,
                feeRate: feeRate,
                password: password
            }
        });
        
        console.log('   📡 Send result:', sendResult);
        
        if (sendResult && sendResult.success) {
            console.log('   ✅ Transaction broadcasted!');
            console.log('      TXID:', sendResult.txid);
            
            // 📤 ADICIONAR PENDING INSCRIPTION PARA O DESTINATÁRIO
            console.log('📤 Adding pending inscription to recipient cache...');
            
            // Enviar mensagem para adicionar ao cache do destinatário
            const pendingInscription = {
                id: inscription.id,
                number: inscription.number,
                address: recipientAddress, // Endereço do destinatário
                content_type: inscription.content_type || 'unknown',
                timestamp: Date.now(),
                genesis_transaction: sendResult.txid,
                pending: true
            };
            
            // ⚠️ Nota: Isso só funciona se o destinatário também usar MyWallet
            // Se for outra carteira, a inscription aparecerá após confirmação no blockchain
            window.postMessage({
                type: 'MYWALLET_ADD_PENDING_INSCRIPTION',
                data: pendingInscription
            }, '*');
            
            console.log('   ✅ Pending inscription sent to cache');
            
            // Limpar pending transaction
            window.pendingInscriptionTransaction = null;
            
            // 🔄 Clear caches to prevent stale UTXO data on next send
            dataCache.ordinals = { data: null, timestamp: null, loaded: false };
            dataCache.runes = { data: null, timestamp: null, loaded: false };
            dataCache.activity = { data: null, timestamp: null, loaded: false };
            clearActivityCache();
            console.log('🔔 ALL caches cleared after inscription send');
            
            // 🎉 Show success modal with mempool.space and KrayScan links
            showSuccessModal(
                sendResult.txid,
                'Inscription Sent!',
                'Your inscription has been transferred successfully.'
            );
            
        } else {
            console.error('   ❌ Send failed:', sendResult?.error || 'Unknown error');
            showNotification(`❌ Failed to send: ${sendResult?.error || 'Unknown error'}`, 'error');
        }
        
    } catch (error) {
        console.error('   ❌ Error sending inscription:', error);
        showNotification(`❌ Error: ${error.message}`, 'error');
        
        // 🔓 On failure, tell backend to unlock UTXOs for this address
        try {
            const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
            if (walletInfo?.data?.address) {
                fetch('https://kraywallet-backend.onrender.com/api/runes/utxo-unlock', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ address: walletInfo.data.address })
                }).catch(e => console.warn('⚠️ Could not unlock UTXOs:', e));
            }
        } catch (unlockErr) {
            console.warn('⚠️ Could not unlock UTXOs:', unlockErr);
        }
    } finally {
        // 🔒 Always release the mutex
        isSendingTransaction = false;
        console.log('🔓 Inscription send mutex released');
    }
    
    console.log('=========================================');
}

// Close Password Modal
function closePasswordModal() {
    console.log('🔐 Closing password modal...');
    const modal = document.getElementById('password-modal');
    modal.classList.add('hidden');
    document.getElementById('confirm-password-input').value = '';
    delete window.pendingTransaction;
    
    // If there's a pending password promise, resolve with null (cancelled)
    if (window._passwordResolve) {
        window._passwordResolve(null);
        window._passwordResolve = null;
    }
}

// 🔒 SECURITY: Restore functionality removed from unlock/welcome screens
// Users must reinstall extension to reset wallet (standard security practice)

// Confirm and Send Transaction
async function confirmAndSend() {
    console.log('✅ Confirming and sending transaction...');
    
    const password = document.getElementById('confirm-password-input').value.trim();
    
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    // 🔥 FIX: Save transaction data BEFORE closing modal (closePasswordModal deletes it!)
    const savedTransaction = window.pendingTransaction ? { ...window.pendingTransaction } : null;
    const savedInscription = window.pendingInscriptionTransaction ? { ...window.pendingInscriptionTransaction } : null;
    const savedPasswordResolve = window._passwordResolve;
    
    // Close the modal (this will delete window.pendingTransaction!)
    closePasswordModal();
    
    // 🔐 If there's a password promise waiting (from requestPassword), resolve it
    if (savedPasswordResolve) {
        console.log('🔐 Resolving password promise...');
        savedPasswordResolve(password);
        return;
    }
    
    // 🎯 DETECTAR TIPO DE TRANSAÇÃO (legacy flow)
    if (savedInscription) {
        // É uma transação de INSCRIPTION
        console.log('   📤 Detected INSCRIPTION transaction');
        window.pendingInscriptionTransaction = savedInscription; // Restore for the function
        await confirmAndSendInscription(password);
        return;
    }
    
    if (!savedTransaction) {
        showNotification('Transaction data not found', 'error');
        return;
    }
    
    // É uma transação NORMAL de Bitcoin
    console.log('   💸 Detected BITCOIN transaction');
    const { toAddress, amount, feeRate } = savedTransaction;
    
    // Mostrar loading
    showLoading('Creating transaction...');
    
    try {
        const response = await chrome.runtime.sendMessage({
            action: 'sendBitcoin',
            data: { toAddress, amount, feeRate, password }
        });
        
        hideLoading();
        
        if (response.success) {
            // 🔒 SECURITY: Clear password field immediately
            clearAllPasswordFields();
            
            // 🎉 Show success modal with mempool.space and KrayScan links
            showSuccessModal(
                response.txid,
                'Bitcoin Sent!',
                'Your Bitcoin transaction has been broadcast successfully.'
            );
        } else {
            showNotification(response.error || 'Failed to send transaction', 'error');
        }
    } catch (error) {
        hideLoading();
        console.error('Error sending:', error);
        showNotification('Failed to send transaction: ' + error.message, 'error');
    }
}

// Loading Overlay
function showLoading(message = 'Loading...') {
    const loadingMessage = document.getElementById('loading-message') || document.getElementById('loading-text');
    const loadingOverlay = document.getElementById('loading-overlay');
    
    if (loadingMessage && loadingOverlay) {
        loadingMessage.textContent = message;
        loadingOverlay.classList.remove('hidden');
    } else {
        console.warn('⚠️ Loading overlay elements not found');
    }
}

function hideLoading() {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.classList.add('hidden');
    }
}

/**
 * Load Inscription Actions (dynamic buttons based on listing status)
 */
async function loadInscriptionActions(fullDetails, inscription, detailsScreen) {
    const actionsContainer = document.getElementById('inscription-actions');
    if (!actionsContainer) return;
    
    try {
        // Verificar se a inscrição está listada
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const address = result?.data?.address || walletState.address;
        
        if (!address) {
            actionsContainer.innerHTML = '<p style="color: #888; text-align: center;">Wallet not connected</p>';
            return;
        }
        
        // Buscar listings do usuário
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/?seller_address=${address}`);
        const data = await response.json();
        
        // Verificar se esta inscrição específica está listada
        const activeListing = data.listings?.find(listing => 
            listing.inscription_id === fullDetails.id &&
            (listing.status === 'PENDING_SIGNATURES' || listing.status === 'OPEN')
        );
        
        // Renderizar botões
        if (activeListing) {
            // ✅ INSCRIÇÃO ESTÁ LISTADA - BOTÕES DINÂMICOS
            actionsContainer.innerHTML = `
                <!-- Info Box: Listada -->
                <div style="padding: 12px; background: rgba(76, 175, 80, 0.1); border: 1px solid #4CAF50; border-radius: 8px; margin-bottom: 12px;">
                    <div style="font-size: 12px; color: #4CAF50; font-weight: 600; margin-bottom: 4px;">✅ Listed on Market</div>
                    <div style="font-size: 16px; color: white; font-weight: 700;">${activeListing.price_sats.toLocaleString()} sats</div>
                    <div style="font-size: 11px; color: #888; margin-top: 4px;">Status: ${activeListing.status}</div>
                </div>
                
                <!-- Botão: Change Price (substitui "List") -->
                ${activeListing.status === 'PENDING_SIGNATURES' ? `
                <button class="rune-action-btn" id="update-price-btn" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border: none; color: white;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M12 2v20M17 12l-5 5-5-5"/>
                    </svg>
                    💰 Change Price
                </button>
                ` : ''}
                
                <!-- Botão: Cancel Listing (substitui "Send") -->
                <button class="rune-action-btn primary" id="cancel-listing-btn" style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); border: none;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <circle cx="12" cy="12" r="10"/>
                        <path d="M15 9l-6 6M9 9l6 6"/>
                    </svg>
                    ❌ Cancel Listing
                </button>
                
                <!-- Botão: View on Ordinals.com -->
                <button class="rune-action-btn" id="view-ordinals-btn" style="background: rgba(255, 136, 0, 0.1); border-color: #ff8800;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M18 13v6a2 2 0 01-2 2H5a2 2 0 01-2-2V8a2 2 0 012-2h6"/>
                        <path d="M15 3h6v6M10 14L21 3"/>
                    </svg>
                    View on Ordinals.com
                </button>
            `;
            
            // Event listeners
            if (activeListing.status === 'PENDING_SIGNATURES') {
                document.getElementById('update-price-btn')?.addEventListener('click', () => {
                    showUpdatePriceModal(activeListing);
                });
            }
            
            document.getElementById('cancel-listing-btn')?.addEventListener('click', async () => {
                // Vai direto para assinatura (sem confirm)
                await cancelAtomicListing(activeListing.order_id);
                detailsScreen.remove(); // Fechar detalhes após cancelar
                showNotification('✅ Listing cancelled!', 'success');
            });
            
        } else {
            // ❌ INSCRIÇÃO NÃO ESTÁ LISTADA - BOTÕES PADRÃO
            actionsContainer.innerHTML = `
                <!-- Botão: Send -->
                <button class="rune-action-btn primary" id="send-inscription-btn">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/>
                    </svg>
                    🚀 Send
                </button>
                
                <!-- Botão: List on Market -->
                <button class="rune-action-btn" id="list-inscription-btn" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%); border: none; color: white;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M3 3h18v18H3z"/>
                        <path d="M3 9h18M9 21V9"/>
                    </svg>
                    📜 List on Market
                </button>
                
                <!-- Botão: View on Ordinals.com -->
                <button class="rune-action-btn" id="view-ordinals-btn" style="background: rgba(255, 136, 0, 0.1); border-color: #ff8800;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M18 13v6a2 2 0 01-2 2H5a2 2 0 01-2-2V8a2 2 0 012-2h6"/>
                        <path d="M15 3h6v6M10 14L21 3"/>
                    </svg>
                    View on Ordinals.com
                </button>
            `;
            
            // Event listeners
            document.getElementById('send-inscription-btn')?.addEventListener('click', () => {
                detailsScreen.remove();
                showSendInscriptionScreen(inscription);
            });
            
            document.getElementById('list-inscription-btn')?.addEventListener('click', () => {
                alert('🏪 List on Market feature coming soon!\n\nThis will allow you to list your inscription on the KrayWallet marketplace!');
            });
        }
        
        // Event listener comum (View on Ordinals.com)
        document.getElementById('view-ordinals-btn')?.addEventListener('click', () => {
            const url = `https://ordinals.com/inscription/${fullDetails.id}`;
            chrome.tabs.create({ url });
        });
        
    } catch (error) {
        console.error('❌ Error loading inscription actions:', error);
        actionsContainer.innerHTML = `
            <p style="color: #ef4444; text-align: center; font-size: 12px;">
                Failed to load actions
            </p>
        `;
    }
}

// Show Inscription Details (tela completa após clicar na inscription)
async function showInscriptionDetails(inscription) {
    console.log('🖼️  Showing inscription details:', inscription.id);
    
    // Criar tela de detalhes (loading state)
    const detailsScreen = document.createElement('div');
    detailsScreen.className = 'rune-details-screen'; // Reusar o mesmo estilo das runes
    detailsScreen.id = 'inscription-details-screen';
    
    detailsScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="inscription-details-back-temp">←</button>
            <div class="rune-details-title">Loading...</div>
        </div>
        <div class="rune-details-content" style="display: flex; align-items: center; justify-content: center; padding: 40px;">
            <div class="spinner"></div>
        </div>
    `;
    
    document.body.appendChild(detailsScreen);
    
    // Event listener temporário para voltar
    document.getElementById('inscription-details-back-temp').addEventListener('click', () => {
        detailsScreen.remove();
    });
    
    try {
        // Use inscription data we already have (backend details endpoint may not be available)
        console.log('   📡 Using cached inscription data...');
        
        const fullDetails = {
            id: inscription.id || inscription.inscriptionId,
            number: inscription.number || inscription.inscription_number,
            content_type: inscription.content_type || inscription.type || 'unknown',
            address: inscription.address || 'unknown',
            output: inscription.output || inscription.utxo,
            preview: `https://ordinals.com/preview/${inscription.id || inscription.inscriptionId}`,
            inscriptionUrl: `https://ordinals.com/inscription/${inscription.id || inscription.inscriptionId}`
        };
        console.log('   ✅ Details ready:', fullDetails);
        
        // Atualizar UI com detalhes completos
        const contentUrl = fullDetails.preview || inscription.preview || `https://ordinals.com/content/${inscription.id}`;
        const contentType = (fullDetails.content_type || inscription.content_type || '').toLowerCase();
        
        detailsScreen.innerHTML = `
            <div class="rune-details-header">
                <button class="rune-details-back" id="inscription-details-back">←</button>
                <div class="rune-details-title">Inscription #${fullDetails.number || inscription.number || 'N/A'}</div>
            </div>
            
            <div class="rune-details-content">
                <!-- Content Preview (Thumbnail GRANDE) -->
                <div class="rune-parent-preview" style="position: relative;">
                    <img src="${contentUrl}" 
                         alt="Inscription #${fullDetails.number || inscription.number}" 
                         style="width: 100%; height: 100%; object-fit: contain; border-radius: 12px;" 
                         onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                    <div class="rune-thumbnail-fallback" style="display: none; width: 100%; height: 100%; align-items: center; justify-content: center; font-size: 64px;">◉</div>
                </div>
                
                <!-- Inscription Info Grid -->
                <div class="rune-info-grid">
                    <div class="rune-info-item">
                        <div class="rune-info-label">Inscription ID</div>
                        <div class="rune-info-value" style="font-size: 12px; font-family: var(--font-mono); word-break: break-all;">${fullDetails.id}</div>
                    </div>
                    
                    <div class="rune-info-item">
                        <div class="rune-info-label">Inscription Number</div>
                        <div class="rune-info-value" style="color: #8b5cf6;">#${(fullDetails.number || inscription.number || 'N/A').toLocaleString()}</div>
                    </div>
                    
                    <div class="rune-info-item">
                        <div class="rune-info-label">Content Type</div>
                        <div class="rune-info-value">${fullDetails.content_type || 'unknown'}</div>
                    </div>
                    
                    ${fullDetails.content_length ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Content Length</div>
                        <div class="rune-info-value">${fullDetails.content_length.toLocaleString()} bytes</div>
                    </div>
                    ` : ''}
                    
                    ${fullDetails.output ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Output (Location)</div>
                        <div class="rune-info-value" style="font-size: 12px; font-family: var(--font-mono); word-break: break-all;">${fullDetails.output}</div>
                    </div>
                    ` : ''}
                    
                    ${fullDetails.genesis_height ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Genesis Height</div>
                        <div class="rune-info-value">${fullDetails.genesis_height.toLocaleString()}</div>
                    </div>
                    ` : ''}
                    
                    ${fullDetails.genesis_fee ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Genesis Fee</div>
                        <div class="rune-info-value">${fullDetails.genesis_fee.toLocaleString()} sats</div>
                    </div>
                    ` : ''}
                    
                    ${fullDetails.timestamp ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Timestamp</div>
                        <div class="rune-info-value">${fullDetails.timestamp}</div>
                    </div>
                    ` : ''}
                    
                    ${fullDetails.sat ? `
                    <div class="rune-info-item">
                        <div class="rune-info-label">Sat Number</div>
                        <div class="rune-info-value">${fullDetails.sat.toLocaleString()}</div>
                    </div>
                    ` : ''}
                </div>
                
                <!-- Actions (será populado dinamicamente) -->
                <div class="rune-actions" id="inscription-actions">
                    <!-- Carregando botões... -->
                </div>
            </div>
        `;
        
        // Event listeners
        document.getElementById('inscription-details-back').addEventListener('click', () => {
            detailsScreen.remove();
        });
        
        // Carregar botões dinâmicos (verificar se está listado)
        await loadInscriptionActions(fullDetails, inscription, detailsScreen);
        
    } catch (error) {
        console.error('❌ Error loading inscription details:', error);
        
        // Mostrar erro na UI
        detailsScreen.innerHTML = `
            <div class="rune-details-header">
                <button class="rune-details-back" id="inscription-details-back-error">←</button>
                <div class="rune-details-title">Error</div>
            </div>
            <div class="rune-details-content" style="padding: 40px; text-align: center;">
                <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
                <div style="color: #ef4444; font-weight: 600; margin-bottom: 8px;">Failed to load inscription details</div>
                <div style="color: #888; font-size: 14px;">${error.message}</div>
            </div>
        `;
        
        document.getElementById('inscription-details-back-error').addEventListener('click', () => {
            detailsScreen.remove();
        });
    }
}

// Show Rune Details (tela completa após clicar na rune)
function showRuneDetails(rune) {
    console.log('🪙 Showing rune details:', rune.name);
    
    // Criar tela de detalhes
    const detailsScreen = document.createElement('div');
    detailsScreen.className = 'rune-details-screen';
    detailsScreen.id = 'rune-details-screen';
    
    detailsScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="rune-details-back">←</button>
            <div class="rune-details-title">${rune.displayName || rune.name}</div>
        </div>
        
        <div class="rune-details-content">
            <!-- Parent Preview (Thumbnail) com Badge de Verificado -->
            <div class="rune-parent-preview" style="position: relative;">
                ${(rune.thumbnail || rune.parentPreview)
                    ? `<img src="${rune.thumbnail || rune.parentPreview}" alt="${rune.displayName || rune.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                       <div class="rune-parent-preview-fallback" style="display: none;">${rune.symbol || '⧈'}</div>`
                    : `<div class="rune-parent-preview-fallback">${rune.symbol || '⧈'}</div>`
                }
                ${isRuneVerified(rune.name) 
                    ? `<div class="rune-verified-badge-large" style="position: absolute; top: 12px; right: 12px; width: 48px; height: 48px; background: linear-gradient(135deg, #3b82f6, #2563eb); border-radius: 50%; display: flex; align-items: center; justify-content: center; border: 3px solid var(--color-bg-primary); box-shadow: 0 4px 12px rgba(59, 130, 246, 0.6); z-index: 10;">
                        <span style="font-size: 28px; color: white; font-weight: bold;">✓</span>
                       </div>`
                    : ''
                }
            </div>
            
            <!-- Rune Info Grid -->
            <div class="rune-info-grid">
                <div class="rune-info-item">
                    <div class="rune-info-label">Rune Name</div>
                    <div class="rune-info-value">${rune.displayName || rune.name}</div>
                </div>
                
                <div class="rune-info-item">
                    <div class="rune-info-label">Your Balance</div>
                    <div class="rune-info-value">${formatRuneAmount(rune.amount)} ${rune.symbol || ''}</div>
                </div>
                
                ${rune.supply ? `
                <div class="rune-info-item">
                    <div class="rune-info-label">Total Supply</div>
                    <div class="rune-info-value">${rune.supply}</div>
                </div>
                ` : ''}
                
                ${rune.etching ? `
                <div class="rune-info-item">
                    <div class="rune-info-label">Etching</div>
                    <div class="rune-info-value" style="font-size: 12px; font-family: var(--font-mono);">${rune.etching}</div>
                </div>
                ` : ''}
                
                ${rune.runeId ? `
                <div class="rune-info-item">
                    <div class="rune-info-label">Rune ID</div>
                    <div class="rune-info-value" style="font-size: 14px; font-family: var(--font-mono); color: #f59e0b;">${rune.runeId}</div>
                </div>
                ` : ''}
                
                ${rune.parent ? `
                <div class="rune-info-item">
                    <div class="rune-info-label">Parent Inscription</div>
                    <div class="rune-info-value" style="font-size: 12px; font-family: var(--font-mono);">${rune.parent}</div>
                </div>
                ` : ''}
                
                <div class="rune-info-item">
                    <div class="rune-info-label">UTXOs</div>
                    <div class="rune-info-value">${rune.utxos ? rune.utxos.length : 0} UTXO(s)</div>
                </div>
            </div>
            
            <!-- Actions -->
            <div class="rune-actions">
                <button class="rune-action-btn primary" id="send-rune-btn">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/>
                    </svg>
                    Send
                </button>
                
                <button class="rune-action-btn" id="receive-rune-btn">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M22 12h-6l-2 3h-4l-2-3H2"/>
                        <path d="M5.45 5.11L2 12v6a2 2 0 0 0 2 2h16a2 2 0 0 0 2-2v-6l-3.45-6.89A2 2 0 0 0 16.76 4H7.24a2 2 0 0 0-1.79 1.11z"/>
                    </svg>
                    Receive
                </button>
                
                <button class="rune-action-btn" id="list-rune-btn" style="background: linear-gradient(135deg, #ff6b35 0%, #f59e0b 100%); color: white;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M9 5H7a2 2 0 0 0-2 2v12a2 2 0 0 0 2 2h10a2 2 0 0 0 2-2V7a2 2 0 0 0-2-2h-2"/>
                        <path d="M9 5a2 2 0 0 1 2-2h2a2 2 0 0 1 2 2v0a2 2 0 0 1-2 2h-2a2 2 0 0 1-2-2z"/>
                        <path d="M12 11v6M9 14h6"/>
                    </svg>
                    ⧈ List for Sale
                </button>
                
                <button class="rune-action-btn burn" id="burn-rune-btn" style="background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%); color: white;">
                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                        <path d="M12 2c-4 4-6 8-6 12a6 6 0 0 0 12 0c0-4-2-8-6-12z"/>
                        <path d="M12 11l-2 3h4l-2 3"/>
                    </svg>
                    🔥 Burn
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(detailsScreen);
    
    // Event listeners
    document.getElementById('rune-details-back').addEventListener('click', () => {
        detailsScreen.remove();
    });
    
    document.getElementById('send-rune-btn').addEventListener('click', () => {
        detailsScreen.remove();
        showSendRuneScreen(rune);
    });
    
    document.getElementById('receive-rune-btn').addEventListener('click', () => {
        detailsScreen.remove();
        showScreen('receive');
    });
    
    document.getElementById('list-rune-btn').addEventListener('click', () => {
        detailsScreen.remove();
        showRunePackages(rune); // Show UTXO packages for THIS rune only
    });
    
    document.getElementById('burn-rune-btn').addEventListener('click', () => {
        detailsScreen.remove();
        showBurnRuneScreen(rune);
    });
}

// Show Send Rune Screen
function showSendRuneScreen(rune) {
    console.log('🪙 Preparing to send rune:', rune.name);
    
    // Criar tela de envio
    const sendScreen = document.createElement('div');
    sendScreen.className = 'rune-details-screen';
    sendScreen.id = 'send-rune-screen';
    
    sendScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="send-rune-back">←</button>
            <div class="rune-details-title">Send ${rune.symbol || '⧈'} ${rune.displayName || rune.name}</div>
        </div>
        
        <div class="rune-details-content" style="padding: 20px;">
            <!-- Rune Info Summary com Thumbnail e Badge -->
            <div class="send-rune-summary">
                <div class="send-rune-icon" style="position: relative;">
                    ${(rune.thumbnail || rune.parentPreview)
                        ? `<img src="${rune.thumbnail || rune.parentPreview}" alt="${rune.displayName || rune.name}" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                           <div class="rune-thumbnail-fallback" style="font-size: 48px; display: none;">${rune.symbol || '⧈'}</div>`
                        : `<div class="rune-thumbnail-fallback" style="font-size: 48px;">${rune.symbol || '⧈'}</div>`
                    }
                    ${isRuneVerified(rune.name) 
                        ? `<div class="rune-verified-badge" style="position: absolute; top: -4px; right: -4px;"></div>`
                        : ''
                    }
                </div>
                <div class="send-rune-balance">
                    <div class="label">Available Balance</div>
                    <div class="value">${formatRuneAmount(rune.amount)} ${rune.symbol || ''}</div>
                </div>
            </div>
            
            <!-- Send Form -->
            <form id="send-rune-form" class="send-form">
                <div class="form-group">
                    <label for="send-rune-address">Recipient Address</label>
                    <input 
                        type="text" 
                        id="send-rune-address" 
                        class="form-input" 
                        placeholder="bc1p..." 
                        required
                    />
                    <small class="form-hint">Enter a valid Bitcoin address</small>
                </div>
                
                <div class="form-group">
                    <label for="send-rune-amount">Amount</label>
                    <div class="input-with-max">
                        <input 
                            type="number" 
                            id="send-rune-amount" 
                            class="form-input" 
                            placeholder="0" 
                            min="0"
                            step="any"
                            max="${rune.amount}"
                            required
                        />
                        <button type="button" class="max-btn" id="send-rune-max-btn">MAX</button>
                    </div>
                    <small class="form-hint">Max: ${formatRuneAmount(rune.amount)}</small>
                </div>
                
                <div class="form-group">
                    <label for="send-rune-fee">Fee Rate (sat/vB)</label>
                    <select id="send-rune-fee" class="form-select">
                        <option value="loading" disabled selected>Loading fees...</option>
                    </select>
                    <div class="fee-custom-input" id="fee-custom-container" style="display: none;">
                        <input 
                            type="number" 
                            id="send-rune-fee-custom" 
                            class="form-input" 
                            placeholder="Enter custom fee rate" 
                            min="1"
                        />
                        <small class="form-hint">Minimum: 1 sat/vB</small>
                    </div>
                </div>
                
                <div class="fee-estimate" id="send-rune-fee-estimate">
                    <div class="fee-label">Estimated Fee:</div>
                    <div class="fee-value">Calculating...</div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="btn-secondary" id="send-rune-cancel">Cancel</button>
                    <button type="submit" class="btn-primary" id="send-rune-submit">
                        <span class="btn-text">Send Rune</span>
                        <span class="btn-loading" style="display: none;">
                            <span class="spinner"></span>
                            Sending...
                        </span>
                    </button>
                </div>
            </form>
        </div>
    `;
    
    document.body.appendChild(sendScreen);
    
    // Load dynamic fees from mempool.space
    loadMempoolFees();
    
    // Event Listeners
    document.getElementById('send-rune-back').addEventListener('click', () => {
        sendScreen.remove();
        showRuneDetails(rune);
    });
    
    document.getElementById('send-rune-cancel').addEventListener('click', () => {
        sendScreen.remove();
        showRuneDetails(rune);
    });
    
    // MAX button
    document.getElementById('send-rune-max-btn').addEventListener('click', () => {
        // ✅ NÃO DIVIDIR! Preencher com o valor RAW
        document.getElementById('send-rune-amount').value = rune.amount;
    });
    
    // Fee selector change (mostrar/ocultar custom input + calcular fee)
    document.getElementById('send-rune-fee').addEventListener('change', (e) => {
        const customContainer = document.getElementById('fee-custom-container');
        if (e.target.value === 'custom') {
            customContainer.style.display = 'block';
        } else {
            customContainer.style.display = 'none';
        }
        // Atualizar estimativa de fee
        updateRuneFeeEstimate();
    });
    
    // Custom fee input change
    document.getElementById('send-rune-fee-custom')?.addEventListener('input', () => {
        updateRuneFeeEstimate();
    });
    
    // Form submission
    document.getElementById('send-rune-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const address = document.getElementById('send-rune-address').value.trim();
        const amount = document.getElementById('send-rune-amount').value;
        const feeSelect = document.getElementById('send-rune-fee').value;
        
        // Determinar fee rate (custom ou preset)
        let feeRate;
        if (feeSelect === 'custom') {
            const customFee = document.getElementById('send-rune-fee-custom').value;
            if (!customFee || customFee < 1) {
                showNotification('Please enter a valid custom fee rate (minimum 1 sat/vB)', 'error');
                return;
            }
            feeRate = parseInt(customFee);
        } else {
            feeRate = parseInt(feeSelect);
        }
        
        // Validações
        if (!address) {
            showNotification('Please enter a recipient address', 'error');
            return;
        }
        
        if (!amount || amount <= 0) {
            showNotification('Please enter a valid amount', 'error');
            return;
        }
        
        if (parseInt(amount) > parseInt(rune.amount)) {
            showNotification('Insufficient balance', 'error');
            return;
        }
        
        // Mostrar loading
        const submitBtn = document.getElementById('send-rune-submit');
        const btnText = submitBtn.querySelector('.btn-text');
        const btnLoading = submitBtn.querySelector('.btn-loading');
        
        submitBtn.disabled = true;
        btnText.style.display = 'none';
        btnLoading.style.display = 'flex';
        
        try {
            console.log('📤 Sending rune:', {
                rune: rune.name,
                to: address,
                amount: amount,
                feeRate: feeRate
            });
            
            // Chamar função de envio (shows success modal internally)
            await sendRuneTransaction(rune, address, amount, feeRate);
            
            // Remove send screen - modal handles wallet navigation
            sendScreen.remove();
            
        } catch (error) {
            console.error('❌ Error sending rune:', error);
            showNotification(`Failed to send rune: ${error.message}`, 'error');
            
            // Restaurar botão
            submitBtn.disabled = false;
            btnText.style.display = 'block';
            btnLoading.style.display = 'none';
        }
    });
}

// Update Rune Fee Estimate
function updateRuneFeeEstimate() {
    try {
        const feeSelect = document.getElementById('send-rune-fee');
        const feeValue = document.getElementById('send-rune-fee-estimate')?.querySelector('.fee-value');
        
        if (!feeSelect || !feeValue) return;
        
        let feeRate;
        if (feeSelect.value === 'custom') {
            const customFee = document.getElementById('send-rune-fee-custom')?.value;
            feeRate = customFee ? parseInt(customFee) : 0;
        } else if (feeSelect.value !== 'loading') {
            feeRate = parseInt(feeSelect.value);
        } else {
            return; // Still loading
        }
        
        if (!feeRate || feeRate < 1) {
            feeValue.textContent = 'Enter fee rate';
            return;
        }
        
        // Estimativa de tamanho de TX para Rune send
        // Típico: 1-2 inputs + 3 outputs (receiver, change, OP_RETURN) = ~250-350 vBytes
        const estimatedSize = 300; // vBytes (média)
        const estimatedFeeSats = feeRate * estimatedSize;
        const estimatedFeeBTC = (estimatedFeeSats / 100000000).toFixed(8);
        
        feeValue.textContent = `~${estimatedFeeSats.toLocaleString()} sats (~${estimatedFeeBTC} BTC)`;
        
    } catch (error) {
        console.error('Error updating fee estimate:', error);
    }
}

// Load fees for Bitcoin Send screen (via backend API - uses QuickNode first!)
async function loadBitcoinSendFees() {
    try {
        console.log('📊 Loading fees via backend API...');
        
        // Use backend API which uses QuickNode first, then Mempool.space as fallback
        let fees;
        try {
            const backendResponse = await fetch('https://kraywallet-backend.onrender.com/api/wallet/fees');
            const backendData = await backendResponse.json();
            if (backendData.success) {
                fees = {
                    minimumFee: backendData.fees.minimum,
                    hourFee: backendData.fees.low,
                    halfHourFee: backendData.fees.medium,
                    fastestFee: backendData.fees.high
                };
                console.log(`✅ Fees from backend (${backendData.source}):`, fees);
            }
        } catch (backendError) {
            console.log('⚠️ Backend failed, trying mempool.space directly...');
            const response = await fetch('https://mempool.space/api/v1/fees/recommended');
            fees = await response.json();
            console.log('✅ Fees from mempool.space fallback:', fees);
        }
        
        console.log('✅ Fees loaded:', fees);
        
        const feeSelect = document.getElementById('send-fee');
        
        if (!feeSelect) {
            console.error('❌ send-fee select not found');
            return;
        }
        
        // Limpar options anteriores
        feeSelect.innerHTML = '';
        
        // Adicionar opções com fees dinâmicas
        const options = [
            { 
                value: fees.minimumFee || 1, 
                label: `🐢 Economy (${fees.minimumFee || 1} sat/vB)`
            },
            { 
                value: fees.hourFee || 5, 
                label: `⏱️  Normal (${fees.hourFee || 5} sat/vB)`
            },
            { 
                value: fees.halfHourFee || 10, 
                label: `⚡ Fast (${fees.halfHourFee || 10} sat/vB)`,
                selected: true
            },
            { 
                value: fees.fastestFee || 20, 
                label: `🚀 Priority (${fees.fastestFee || 20} sat/vB)`
            },
            { 
                value: 'custom', 
                label: '⚙️  Custom'
            }
        ];
        
        options.forEach(opt => {
            const option = document.createElement('option');
            option.value = opt.value;
            option.textContent = opt.label;
            if (opt.selected) {
                option.selected = true;
            }
            feeSelect.appendChild(option);
        });
        
        // Add change listener
        feeSelect.addEventListener('change', (e) => {
            const customContainer = document.getElementById('send-fee-custom-container');
            if (e.target.value === 'custom') {
                customContainer.style.display = 'block';
            } else {
                customContainer.style.display = 'none';
            }
        });
        
    } catch (error) {
        console.error('❌ Error loading mempool fees:', error);
        
        // Fallback para fees estáticas se API falhar
        const feeSelect = document.getElementById('send-fee');
        if (feeSelect) {
            feeSelect.innerHTML = `
                <option value="1">🐢 Economy (1 sat/vB)</option>
                <option value="5">⏱️  Normal (5 sat/vB)</option>
                <option value="10" selected>⚡ Fast (10 sat/vB)</option>
                <option value="20">🚀 Priority (20 sat/vB)</option>
                <option value="custom">⚙️  Custom</option>
            `;
        }
    }
}

// Load recommended fees for Runes (via backend API - uses QuickNode first!)
async function loadMempoolFees() {
    try {
        console.log('📊 Loading fees via backend API...');
        
        // Use backend API which uses QuickNode first, then Mempool.space as fallback
        let fees;
        try {
            const backendResponse = await fetch('https://kraywallet-backend.onrender.com/api/wallet/fees');
            const backendData = await backendResponse.json();
            if (backendData.success) {
                fees = {
                    minimumFee: backendData.fees.minimum,
                    hourFee: backendData.fees.low,
                    halfHourFee: backendData.fees.medium,
                    fastestFee: backendData.fees.high
                };
                console.log(`✅ Fees from backend (${backendData.source}):`, fees);
            }
        } catch (backendError) {
            console.log('⚠️ Backend failed, trying mempool.space directly...');
            const response = await fetch('https://mempool.space/api/v1/fees/recommended');
            fees = await response.json();
            console.log('✅ Fees from mempool.space fallback:', fees);
        }
        
        const feeSelect = document.getElementById('send-rune-fee');
        
        // Limpar options anteriores
        feeSelect.innerHTML = '';
        
        // Adicionar opções com fees dinâmicas
        const options = [
            { 
                value: fees.minimumFee || 1, 
                label: `🐢 Economy (${fees.minimumFee || 1} sat/vB)`
            },
            { 
                value: fees.hourFee || 5, 
                label: `⏱️  Normal (${fees.hourFee || 5} sat/vB)`
            },
            { 
                value: fees.halfHourFee || 10, 
                label: `⚡ Fast (${fees.halfHourFee || 10} sat/vB)`,
                selected: true
            },
            { 
                value: fees.fastestFee || 20, 
                label: `🚀 Priority (${fees.fastestFee || 20} sat/vB)`
            },
            { 
                value: 'custom', 
                label: '⚙️  Custom'
            }
        ];
        
        options.forEach(opt => {
            const option = document.createElement('option');
            option.value = opt.value;
            option.textContent = opt.label;
            if (opt.selected) {
                option.selected = true;
            }
            feeSelect.appendChild(option);
        });
        
        // Calcular estimativa inicial com fee selecionada (Fast)
        setTimeout(() => updateRuneFeeEstimate(), 100);
        
    } catch (error) {
        console.error('❌ Error loading mempool fees:', error);
        
        // Fallback para fees estáticas se API falhar
        const feeSelect = document.getElementById('send-rune-fee');
        feeSelect.innerHTML = `
            <option value="1">🐢 Economy (1 sat/vB)</option>
            <option value="5">⏱️  Normal (5 sat/vB)</option>
            <option value="10" selected>⚡ Fast (10 sat/vB)</option>
            <option value="20">🚀 Priority (20 sat/vB)</option>
            <option value="custom">⚙️  Custom</option>
        `;
        
        showNotification('Using default fee rates (mempool.space unavailable)', 'warning');
        
        // Calcular estimativa inicial com fee padrão
        setTimeout(() => updateRuneFeeEstimate(), 100);
    }
}

// 🔒 Global send mutex - prevents concurrent Rune sends that cause UTXO conflicts
let isSendingTransaction = false;

// Send Rune Transaction (chamada ao backend + sign + broadcast)
async function sendRuneTransaction(rune, toAddress, amount, feeRate) {
    // 🔒 MUTEX: Block concurrent sends
    if (isSendingTransaction) {
        showNotification('⏳ Transaction in progress. Please wait for it to complete.', 'warning');
        throw new Error('Transaction already in progress');
    }
    isSendingTransaction = true;
    
    try {
        console.log('\n🚀 ========== SEND RUNE TRANSACTION ==========');
        
        // 1. Get wallet info
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        
        if (!walletInfo.success || !walletInfo.data.address) {
            throw new Error('Wallet not unlocked');
        }
        
        const fromAddress = walletInfo.data.address;
        console.log('From:', fromAddress);
        console.log('To:', toAddress);
        console.log('Rune:', rune.name);
        console.log('Amount:', amount);
        
        // ✅ VALIDAÇÃO: Verificar decimais permitidos
        const divisibility = rune.divisibility || 0;
        const amountNum = parseFloat(amount);
        
        // Calcular raw amount
        const rawAmount = amountNum * Math.pow(10, divisibility);
        
        console.log('   Divisibility:', divisibility);
        console.log('   Raw amount:', rawAmount);
        
        // Validar que raw amount é inteiro (respeitando divisibility)
        if (!Number.isInteger(rawAmount)) {
            throw new Error(
                `Invalid amount precision. ${rune.name} has ${divisibility} decimals. ` +
                `Maximum precision: ${divisibility} decimal places.`
            );
        }
        
        // Validar que amount é positivo
        if (rawAmount <= 0) {
            throw new Error('Amount must be greater than 0');
        }
        
        console.log('   ✅ Amount validation passed');
        
        // 🛡️ CRITICAL: Filter rune UTXOs to exclude those with inscriptions
        const allRuneUtxos = rune.utxos || [];
        let safeRuneUtxos = allRuneUtxos.filter(u => !u.hasInscription);
        
        console.log('   🛡️ UTXO Filter (Inscriptions):');
        console.log('      Total UTXOs:', allRuneUtxos.length);
        console.log('      Safe UTXOs (no inscription):', safeRuneUtxos.length);
        console.log('      Filtered out:', allRuneUtxos.length - safeRuneUtxos.length);
        
        // 🛡️ MARKETPLACE PROTECTION: Filter out UTXOs that are listed for sale
        try {
            console.log('   🛡️ Checking for marketplace-listed UTXOs...');
            const listedResp = await fetch(`https://kraywallet-backend.onrender.com/api/runes-atomic-swap/my-listed-utxos?address=${fromAddress}`);
            const listedData = await listedResp.json();
            
            if (listedData.success && listedData.listedUtxos?.length > 0) {
                const listedSet = new Set(listedData.listedUtxos.map(u => `${u.txid}:${u.vout}`));
                const beforeCount = safeRuneUtxos.length;
                
                safeRuneUtxos = safeRuneUtxos.filter(u => !listedSet.has(`${u.txid}:${u.vout}`));
                
                const marketFiltered = beforeCount - safeRuneUtxos.length;
                console.log('   🛡️ UTXO Filter (Marketplace):');
                console.log('      Listed UTXOs found:', listedData.listedUtxos.length);
                console.log('      Filtered out (listed for sale):', marketFiltered);
                console.log('      Available after filter:', safeRuneUtxos.length);
                
                if (marketFiltered > 0 && safeRuneUtxos.length === 0) {
                    throw new Error(
                        'All your rune UTXOs are listed on the marketplace! ' +
                        'Cancel your listings first to unlock them for sending.'
                    );
                }
            } else {
                console.log('   ✅ No marketplace listings found');
            }
        } catch (marketError) {
            // If it's our custom error about all UTXOs being listed, re-throw it
            if (marketError.message.includes('listed on the marketplace')) {
                throw marketError;
            }
            // Otherwise just log and continue (don't block on marketplace check failures)
            console.warn('   ⚠️ Could not check marketplace listings:', marketError.message);
        }
        
        if (safeRuneUtxos.length === 0 && allRuneUtxos.length > 0) {
            throw new Error(
                'All rune UTXOs contain inscriptions! ' +
                'Use "Separate Assets" in Settings → Advanced Tools to split them first.'
            );
        }
        
        if (safeRuneUtxos.length === 0) {
            throw new Error('No rune UTXOs available. Please refresh your wallet.');
        }
        
        // 2. Build PSBT via backend
        console.log('\n📦 Step 1: Building PSBT...');
        console.log('   ✅ Sending rune data to backend (from QuickNode):');
        console.log('      runeId:', rune.runeId);
        console.log('      divisibility:', divisibility);
        console.log('      runeUtxos:', safeRuneUtxos.length);
        
        const buildResponse = await fetch('https://kraywallet-backend.onrender.com/api/runes/build-send-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                fromAddress: fromAddress,
                toAddress: toAddress,
                runeName: rune.name,
                amount: amountNum,
                feeRate: feeRate,
                runeId: rune.runeId,
                divisibility: divisibility,
                runeUtxos: safeRuneUtxos
            })
        });
        
        const buildData = await buildResponse.json();
        
        if (!buildData.success) {
            // Show user-friendly notification immediately for marketplace errors
            if (buildData.code === 'UTXO_LISTED_ON_MARKETPLACE') {
                showNotification(
                    '🚫 Cannot send: This UTXO is listed on marketplace. Cancel listing first!', 
                    'error'
                );
            }
            throw new Error(buildData.error || 'Failed to build PSBT');
        }
        
        console.log('✅ PSBT built:', buildData.psbt);
        console.log('   Fee:', buildData.fee, 'sats');
        
        // 3. Sign PSBT (mostrar modal de confirmação de senha)
        console.log('\n✍️  Step 2: Requesting password for signing...');
        
        // Criar modal de confirmação inline
        const confirmModal = document.createElement('div');
        confirmModal.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.8);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        `;
        
        confirmModal.innerHTML = `
            <div style="background: #1a1a1a; border-radius: 16px; padding: 24px; width: 90%; max-width: 400px; border: 1px solid #333;">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3 style="margin: 0; color: #fff; font-size: 18px;">🔏 Confirm Transaction</h3>
                    <button id="confirm-close-btn" style="background: none; border: none; color: #888; font-size: 24px; cursor: pointer; padding: 0; width: 30px; height: 30px;">×</button>
                </div>
                
                <div style="background: #2a2a2a; border-radius: 8px; padding: 16px; margin-bottom: 20px;">
                    <div style="color: #ff9500; font-weight: 600; margin-bottom: 12px;">⧈ Rune Transfer</div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Rune:</span>
                        <span style="color: #fff;">${rune.name}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Amount:</span>
                        <span style="color: #fff;">${amount}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">To:</span>
                        <span style="color: #fff; font-size: 11px;">${toAddress.substring(0,20)}...</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #888;">Fee:</span>
                        <span style="color: #fff;">${buildData.fee} sats</span>
                    </div>
                </div>
                
                <div style="margin-bottom: 20px;">
                    <label style="display: block; color: #fff; margin-bottom: 8px; font-weight: 500;">Password</label>
                    <input type="password" id="rune-confirm-password-input" placeholder="Enter your password" 
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px; box-sizing: border-box;" />
                </div>
                
                <div style="display: flex; gap: 12px;">
                    <button id="confirm-cancel-btn" style="flex: 1; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-weight: 600; cursor: pointer;">Cancel</button>
                    <button id="confirm-sign-btn" style="flex: 1; padding: 12px; background: #ff9500; border: none; border-radius: 8px; color: #000; font-weight: 600; cursor: pointer;">Sign & Send</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(confirmModal);
        
        // Aguardar assinatura do usuário
        const signResult = await new Promise((resolve, reject) => {
            const signBtn = document.getElementById('confirm-sign-btn');
            const cancelBtn = document.getElementById('confirm-cancel-btn');
            const closeBtn = document.getElementById('confirm-close-btn');
            const passwordInput = document.getElementById('rune-confirm-password-input');
            
            // Focus no campo de senha após modal estar no DOM
            setTimeout(() => {
                if (passwordInput) {
                    passwordInput.focus();
                    console.log('✅ Password input focused');
                }
            }, 100);
            
            // Permitir Enter para submeter
            if (passwordInput) {
                passwordInput.addEventListener('keypress', (e) => {
                    if (e.key === 'Enter') {
                        signBtn.click();
                    }
                });
            }
            
            const cleanup = () => {
                confirmModal.remove();
            };
            
            signBtn.onclick = async () => {
                const passwordInput = document.getElementById('rune-confirm-password-input');
                const password = passwordInput ? passwordInput.value : '';
                
                console.log('🔐 Password input element:', passwordInput);
                console.log('🔐 Password value:', password ? '***' : '(empty)');
                console.log('🔐 Password length:', password.length);
                
                if (!password || password.length === 0) {
                    showNotification('❌ Please enter your password', 'error');
                    return;
                }
                
                try {
                    showLoading('Signing transaction...');
                    
                    // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
                    console.log('🔐 Signing PSBT locally...');
                    // ✅ Sign ALL inputs (rune UTXOs + fee UTXOs)
                    const inputCount = buildData.inputCount || 3; // Default to 3 if not provided
                    const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
                    console.log('   Inputs to sign:', inputsToSign);
                    
                    const signData = await chrome.runtime.sendMessage({
                        action: 'signPsbtWithPassword',
                        data: {
                            psbt: buildData.psbt,
                            password: password,
                            sighashType: 0x01, // SIGHASH_ALL for rune sends
                            inputsToSign: inputsToSign // ✅ Sign ALL inputs
                        }
                    });
                    
                    if (!signData.success) {
                        hideLoading();
                        showNotification('❌ ' + (signData.error || 'Failed to sign PSBT'), 'error');
                        return;
                    }
                    
                    console.log('✅ PSBT signed locally');
                    
                    cleanup();
                    resolve({ success: true, signedPsbt: signData.signedPsbt });
                    
                } catch (error) {
                    hideLoading();
                    showNotification('❌ ' + error.message, 'error');
                }
            };
            
            cancelBtn.onclick = () => {
                cleanup();
                reject(new Error('User cancelled'));
            };
            
            closeBtn.onclick = () => {
                cleanup();
                reject(new Error('User cancelled'));
            };
        });
        
        if (!signResult.success) {
            throw new Error('Failed to sign PSBT');
        }
        
        console.log('✅ PSBT signed:', signResult.signedPsbt ? 'Yes' : 'No');
        
        // Mostrar loading novamente para finalização
        showLoading('Finalizing transaction...');
        
        // 3.5. Finalizar PSBT e extrair hex (chamar backend)
        console.log('\n🔨 Step 2.5: Finalizing PSBT...');
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                psbt: signResult.signedPsbt
            })
        });
        
        const finalizeData = await finalizeResponse.json();
        
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize PSBT');
        }
        
        console.log('✅ PSBT finalized');
        console.log('   Hex length:', finalizeData.hex?.length || 0);
        
        // 4. Broadcast transaction
        console.log('\n📡 Step 3: Broadcasting transaction...');
        const broadcastResult = await chrome.runtime.sendMessage({
            action: 'broadcastTransaction',
            hex: finalizeData.hex
        });
        
        if (!broadcastResult.success) {
            throw new Error(broadcastResult.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Transaction broadcast!');
        console.log('   TXID:', broadcastResult.txid);
        console.log('========== SEND COMPLETE ==========\n');
        
        // 💾 SALVAR INFO DA RUNE PARA ENRICHMENT NO ACTIVITY
        window.lastSentRune = {
            name: rune.displayName || rune.name,
            amount: amount.toString(),
            thumbnail: rune.thumbnail || rune.parentPreview,
            symbol: rune.symbol,
            txid: broadcastResult.txid,
            timestamp: Date.now()
        };
        console.log('💾 Saved rune info for activity enrichment:', window.lastSentRune);
        
        // 🔄 AGGRESSIVE CACHE INVALIDATION after broadcast
        // This ensures the next send uses fresh UTXO data, preventing conflicts
        console.log('🔄 Force clearing ALL caches after broadcast...');
        
        // Clear activity cache
        clearActivityCache();
        
        // 🔒 CRITICAL: Nuke ALL data caches to force fresh data on next view
        // This prevents stale UTXO data from being used in a rapid second send
        dataCache.runes = { data: null, timestamp: null, loaded: false };
        dataCache.ordinals = { data: null, timestamp: null, loaded: false };
        dataCache.activity = { data: null, timestamp: null, loaded: false };
        console.log('🔔 ALL caches cleared (runes, ordinals, activity) - will force refresh on next view');
        
        // 🔄 OPTIMISTIC UPDATE: Update rune balance immediately for UI feedback
        // (The cache is cleared so next full load will get real data)
        console.log('🔄 Optimistic UI update for immediate feedback...');
        const runesList = document.getElementById('runes-list');
        if (runesList) {
            const runeItems = runesList.querySelectorAll('.rune-item');
            runeItems.forEach(item => {
                const runeName = item.dataset?.runeName;
                if (runeName === rune.name) {
                    const amountDiv = item.querySelector('.rune-amount');
                    if (amountDiv) {
                        const currentAmount = parseInt(rune.amount) || 0;
                        const sentAmount = parseInt(amount) || 0;
                        const newAmount = currentAmount - sentAmount;
                        if (newAmount > 0) {
                            amountDiv.textContent = formatRuneAmount(newAmount);
                        } else {
                            item.style.opacity = '0.3';
                        }
                    }
                }
            });
        }
        
        // Esconder loading
        hideLoading();
        
        // 🎉 Show success modal with mempool.space and KrayScan links
        showSuccessModal(
            broadcastResult.txid,
            'Runes Sent!',
            'Your runes have been transferred successfully.'
        );
        
        return {
            success: true,
            txid: broadcastResult.txid,
            fee: buildData.fee
        };
        
    } catch (error) {
        console.error('❌ Error in sendRuneTransaction:', error);
        
        // 🔓 On failure, tell backend to unlock UTXOs for this address
        try {
            const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
            if (walletInfo?.data?.address) {
                fetch('https://kraywallet-backend.onrender.com/api/runes/utxo-unlock', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ address: walletInfo.data.address })
                }).catch(e => console.warn('⚠️ Could not unlock UTXOs:', e));
            }
        } catch (unlockErr) {
            console.warn('⚠️ Could not unlock UTXOs:', unlockErr);
        }
        
        throw error;
    } finally {
        // 🔒 Always release the mutex
        isSendingTransaction = false;
        console.log('🔓 Send mutex released');
    }
}

// ========================================
// 🔥 BURN RUNES - Cenotaph (Tag 126)
// ========================================

let burnRunesState = {
    selectedUtxo: null,
    utxosList: [],
    runeName: null,
    runeSymbol: null,
    thumbnail: null
};

// 🔥 Show Burn Rune Screen with UTXO Selection
function showBurnRuneScreen(rune) {
    console.log('🔥 Preparing to burn rune:', rune.name);
    
    // Store rune info (same pattern as Send Runes)
    burnRunesState.runeName = rune.displayName || rune.name;
    burnRunesState.runeSymbol = rune.symbol || '⧈';
    burnRunesState.thumbnail = rune.thumbnail || rune.parentPreview || null;
    burnRunesState.selectedUtxo = null;
    burnRunesState.utxosList = [];
    
    console.log('🔥 Burn state:', { name: burnRunesState.runeName, thumbnail: burnRunesState.thumbnail });
    
    const burnScreen = document.createElement('div');
    burnScreen.className = 'rune-details-screen';
    burnScreen.id = 'burn-rune-screen';
    
    burnScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="burn-rune-back">←</button>
            <div class="rune-details-title">🔥 Burn ${rune.symbol || '⧈'} ${rune.displayName || rune.name}</div>
        </div>
        
        <div class="rune-details-content" style="padding: 24px;">
            <!-- Warning Banner -->
            <div class="burn-warning" style="background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%); padding: 16px; border-radius: 12px; margin-bottom: 20px;">
                <div style="display: flex; align-items: center; gap: 12px;">
                    <div style="font-size: 32px;">⚠️</div>
                    <div>
                        <div style="color: white; font-weight: 600; margin-bottom: 4px;">PERMANENT ACTION</div>
                        <div style="color: rgba(255,255,255,0.85); font-size: 13px;">
                            ALL runes in the selected UTXO will be permanently destroyed via Cenotaph. This cannot be undone!
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Info Box -->
            <div style="background: rgba(255,255,255,0.03); border: 1px solid #333; border-radius: 8px; padding: 12px; margin-bottom: 20px;">
                <div style="color: #888; font-size: 12px; margin-bottom: 8px;">💡 How Burn Works:</div>
                <div style="color: #aaa; font-size: 13px; line-height: 1.5;">
                    • Select a UTXO containing runes<br>
                    • <strong>ALL</strong> runes in that UTXO will be burned<br>
                    • The BTC sats return to you as change<br>
                    • Want to burn specific amount? <strong>Split first</strong>, then burn
                </div>
            </div>
            
            <!-- UTXO Selection -->
            <div style="margin-bottom: 16px;">
                <label style="color: #888; font-size: 13px; margin-bottom: 8px; display: block;">
                    Select UTXO to Burn (${rune.displayName || rune.name})
                </label>
                <div id="burn-rune-utxo-list" style="max-height: 240px; overflow-y: auto; border: 1px solid #333; border-radius: 8px; background: rgba(0,0,0,0.2);">
                    <div style="color: #888; text-align: center; padding: 20px;">
                        <div class="loading-spinner" style="margin: 0 auto 8px;"></div>
                        Loading UTXOs...
                    </div>
                </div>
            </div>
            
            <!-- Fee Rate -->
            <div style="margin-bottom: 20px;">
                <label style="color: #888; font-size: 13px; margin-bottom: 8px; display: block;">
                    Fee Rate (sat/vB)
                </label>
                <input type="number" id="burn-fee-rate" min="1" value="2" 
                       style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 16px;">
            </div>
            
            <!-- Selected UTXO Summary -->
            <div id="burn-rune-summary" style="display: none; background: rgba(255, 68, 68, 0.1); border: 1px solid rgba(255, 68, 68, 0.3); border-radius: 8px; padding: 12px; margin-bottom: 20px;">
                <div style="color: #ff4444; font-weight: 600; margin-bottom: 8px;">🔥 Will Burn:</div>
                <div style="display: flex; justify-content: space-between; color: #fff;">
                    <span>Amount:</span>
                    <span id="burn-rune-amount-display">-</span>
                </div>
                <div style="display: flex; justify-content: space-between; color: #888; font-size: 13px; margin-top: 4px;">
                    <span>UTXO Value:</span>
                    <span id="burn-rune-sats-display">-</span>
                </div>
            </div>
            
            <!-- Burn Button -->
            <button id="execute-burn-runes-btn" class="btn btn-primary" disabled
                    style="width: 100%; background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%); padding: 16px; font-weight: 600; opacity: 0.5;">
                🔥 Select UTXO to Burn
            </button>
        </div>
    `;
    
    document.body.appendChild(burnScreen);
    
    // Event listeners
    document.getElementById('burn-rune-back').addEventListener('click', () => {
        burnRunesState = { selectedUtxo: null, utxosList: [], runeName: null, runeSymbol: null, thumbnail: null };
        burnScreen.remove();
    });
    
    document.getElementById('execute-burn-runes-btn').addEventListener('click', () => {
        if (burnRunesState.selectedUtxo) {
            showBurnConfirmModal();
        }
    });
    
    // Load UTXOs for this rune
    loadBurnRuneUtxos(rune.name);
}

// Load UTXOs containing the specific rune (same structure as Split Runes)
async function loadBurnRuneUtxos(runeName) {
    const listContainer = document.getElementById('burn-rune-utxo-list');
    if (!listContainer) return;
    
    console.log('🔥 Loading burn UTXOs for:', runeName);
    
    try {
        const address = await getWalletAddress();
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await response.json();
        
        console.log('🔥 UTXOs response:', data.success, 'total:', data.utxos?.length);
        
        if (!data.success || !data.utxos) {
            throw new Error('Failed to load UTXOs');
        }
        
        // Normalize runeName for comparison (remove bullets)
        const normalizedRuneName = runeName.replace(/•/g, '').toUpperCase();
        
        // Filter for rune UTXOs without inscriptions (same as Split Runes)
        const runeUtxos = data.utxos
            .filter(u => u.hasRunes && !u.hasInscription)
            .map(utxo => {
                // Process runes array to find matching rune and extract rawAmount/divisibility
                const processedRunes = (utxo.runes || []).map(r => {
                    let rawAmount = 0;
                    let divisibility = 0;
                    let displayAmount = 0;
                    
                    if (r.amount && typeof r.amount === 'object') {
                        // Structured amount: {amount, divisibility, symbol}
                        rawAmount = r.amount.amount || 0;
                        divisibility = r.amount.divisibility || 0;
                    } else {
                        // Simple amount (already raw or display)
                        rawAmount = r.amount || 0;
                    }
                    
                    // Calculate display amount
                    displayAmount = divisibility > 0 
                        ? rawAmount / Math.pow(10, divisibility) 
                        : rawAmount;
                    
                    return {
                        ...r,
                        rawAmount: rawAmount,
                        divisibility: divisibility,
                        amount: displayAmount,
                        name: r.name || 'Unknown',
                        runeId: r.runeId || r.id
                    };
                });
                
                // Find the specific rune we're looking for
                const matchingRune = processedRunes.find(r => {
                    const rName = (r.name || '').replace(/•/g, '').toUpperCase();
                    return rName === normalizedRuneName;
                });
                
                if (!matchingRune) return null;
                
                return {
                    ...utxo,
                    runes: processedRunes,
                    runeName: matchingRune.name,
                    rawRuneAmount: matchingRune.rawAmount,
                    displayRuneAmount: matchingRune.amount,
                    divisibility: matchingRune.divisibility,
                    runeId: matchingRune.runeId
                };
            })
            .filter(Boolean);
        
        console.log('🔥 Filtered burn UTXOs:', runeUtxos.length);
        
        if (runeUtxos.length === 0) {
            listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;">No burnable UTXOs found for this rune</div>';
            return;
        }
        
        burnRunesState.utxosList = runeUtxos;
        renderBurnRuneUtxos();
        
    } catch (error) {
        console.error('❌ Error loading burn UTXOs:', error);
        listContainer.innerHTML = '<div style="color: #ef4444; text-align: center; padding: 20px;">Failed to load UTXOs</div>';
    }
}

// Render UTXO list for burn selection
function renderBurnRuneUtxos() {
    const listContainer = document.getElementById('burn-rune-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '';
    
    burnRunesState.utxosList.forEach((utxo) => {
        const isSelected = burnRunesState.selectedUtxo?.txid === utxo.txid && 
                          burnRunesState.selectedUtxo?.vout === utxo.vout;
        
        const item = document.createElement('div');
        item.style.cssText = `
            display: flex; align-items: center; padding: 12px; margin: 4px;
            background: ${isSelected ? 'rgba(255, 68, 68, 0.15)' : 'rgba(255,255,255,0.03)'};
            border: 1px solid ${isSelected ? 'rgba(255, 68, 68, 0.5)' : 'transparent'};
            border-radius: 8px; cursor: pointer;
            transition: all 0.2s ease;
        `;
        
        item.innerHTML = `
            <div style="flex: 1;">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                    <span style="color: #ff4444; font-size: 16px;">🔥</span>
                    <span style="color: #fff; font-weight: 600;">${utxo.displayRuneAmount?.toLocaleString() || 0}</span>
                    <span style="color: #888;">${burnRunesState.runeSymbol}</span>
                </div>
                <div style="font-size: 11px; color: #666;">
                    ${utxo.txid.slice(0, 8)}...${utxo.txid.slice(-8)}:${utxo.vout} • ${utxo.value} sats
                </div>
            </div>
            <div style="color: ${isSelected ? '#ff4444' : '#444'}; font-size: 20px;">
                ${isSelected ? '🔥' : '○'}
            </div>
        `;
        
        item.addEventListener('click', () => {
            burnRunesState.selectedUtxo = isSelected ? null : utxo;
            renderBurnRuneUtxos();
            updateBurnRuneSummary();
        });
        
        listContainer.appendChild(item);
    });
}

// Update summary when UTXO is selected
function updateBurnRuneSummary() {
    const summary = document.getElementById('burn-rune-summary');
    const btn = document.getElementById('execute-burn-runes-btn');
    
    if (!summary || !btn) return;
    
    if (burnRunesState.selectedUtxo) {
        summary.style.display = 'block';
        document.getElementById('burn-rune-amount-display').textContent = 
            `${burnRunesState.selectedUtxo.displayRuneAmount?.toLocaleString() || 0} ${burnRunesState.runeSymbol}`;
        document.getElementById('burn-rune-sats-display').textContent = 
            `${burnRunesState.selectedUtxo.value?.toLocaleString() || 0} sats (returns as change)`;
        
        btn.disabled = false;
        btn.style.opacity = '1';
        btn.textContent = '🔥 Burn Runes Permanently';
    } else {
        summary.style.display = 'none';
        btn.disabled = true;
        btn.style.opacity = '0.5';
        btn.textContent = '🔥 Select UTXO to Burn';
    }
}

// Show confirmation modal before burn
function showBurnConfirmModal() {
    if (!burnRunesState.selectedUtxo) return;
    
    const modal = document.createElement('div');
    modal.id = 'burn-runes-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(255, 68, 68, 0.3); background: rgba(255, 68, 68, 0.1);">
                <h3 style="color: #ff4444;">🔥 Confirm Burn</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 48px; margin-bottom: 12px;">⚠️</div>
                
                <p style="color: #ff4444; font-weight: 600; margin-bottom: 8px;">
                    This action is PERMANENT!
                </p>
                
                <p style="color: #ccc; margin-bottom: 16px;">
                    You are about to burn <strong style="color: #ff4444;">${burnRunesState.selectedUtxo.displayRuneAmount?.toLocaleString() || 0} ${burnRunesState.runeSymbol}</strong>
                </p>
                
                <div style="background: rgba(0,0,0,0.3); border-radius: 8px; padding: 12px; margin-bottom: 16px; text-align: left;">
                    <div style="color: #888; font-size: 12px;">UTXO:</div>
                    <div style="color: #fff; font-size: 11px; word-break: break-all;">
                        ${burnRunesState.selectedUtxo.txid}:${burnRunesState.selectedUtxo.vout}
                    </div>
                </div>
                
                <div style="display: flex; align-items: center; gap: 12px; background: rgba(255,255,255,0.03); padding: 12px; border-radius: 8px; margin-bottom: 16px;">
                    <input type="checkbox" id="burn-confirm-checkbox" style="width: 20px; height: 20px; accent-color: #ff4444;">
                    <label for="burn-confirm-checkbox" style="color: #aaa; font-size: 13px; cursor: pointer;">
                        I understand this is permanent and cannot be undone
                    </label>
                </div>
                
                <input type="password" id="burn-runes-password" class="input-field" placeholder="Enter password to confirm" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-burn-runes-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-burn-runes-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #ff4444 0%, #cc0000 100%);">
                    🔥 Burn Forever
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.getElementById('burn-runes-password').focus();
    
    document.getElementById('cancel-burn-runes-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-burn-runes-btn').addEventListener('click', confirmBurnRunes);
}

// Execute the burn transaction
async function confirmBurnRunes() {
    const checkbox = document.getElementById('burn-confirm-checkbox');
    if (!checkbox?.checked) {
        showNotification('⚠️ Please confirm you understand this is permanent', 'error');
        return;
    }
    
    const password = document.getElementById('burn-runes-password').value;
    if (!password) {
        showNotification('⚠️ Enter password', 'error');
        return;
    }
    
    const confirmBtn = document.getElementById('confirm-burn-runes-btn');
    confirmBtn.disabled = true;
    confirmBtn.textContent = '⏳ Burning...';
    
    try {
        const address = await getWalletAddress();
        const feeRate = parseInt(document.getElementById('burn-fee-rate')?.value) || 2;
        
        console.log('🔥 Burn Runes - Building PSBT...');
        console.log('   UTXO:', burnRunesState.selectedUtxo.txid + ':' + burnRunesState.selectedUtxo.vout);
        console.log('   Amount:', burnRunesState.selectedUtxo.displayRuneAmount);
        
        // Get pure UTXOs for fees
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const utxosData = await utxosResponse.json();
        const pureUtxos = utxosData.utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        
        if (pureUtxos.length === 0) {
            throw new Error('No pure UTXOs available for fees');
        }
        
        // Build Burn PSBT via backend
        const response = await fetch('https://kraywallet-backend.onrender.com/api/runes/burn', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                runeUtxo: {
                    txid: burnRunesState.selectedUtxo.txid,
                    vout: burnRunesState.selectedUtxo.vout,
                    value: burnRunesState.selectedUtxo.value
                },
                paymentUtxos: pureUtxos.slice(0, 5).map(u => ({ txid: u.txid, vout: u.vout, value: u.value })),
                feeRate
            })
        });
        
        const result = await response.json();
        if (!result.success) throw new Error(result.error || 'Failed to build burn PSBT');
        
        console.log('🔥 Burn PSBT built, signing locally...');
        
        // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
        // ✅ Sign ALL inputs (rune UTXO + fee UTXOs)
        const inputCount = result.inputs || 2; // Default to 2 if not provided
        const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
        console.log('   Inputs to sign:', inputsToSign);
        
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01, // SIGHASH_ALL
                inputsToSign: inputsToSign // ✅ Sign ALL inputs
            }
        });
        
        if (!signResult.success) throw new Error(signResult.error || 'Sign failed');
        
        // Finalize and broadcast
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeResult = await finalizeResponse.json();
        if (!finalizeResult.success) throw new Error(finalizeResult.error || 'Finalize failed');
        
        console.log('🔥 Runes burned! TXID:', finalizeResult.txid);
        
        showNotification(`🔥 Runes burned! TX: ${finalizeResult.txid.slice(0, 8)}...`, 'success');
        
        // 🔥 Save burn transaction info for Activity display
        try {
            const burnTxids = JSON.parse(localStorage.getItem('burnTransactions') || '[]');
            burnTxids.push({
                txid: finalizeResult.txid,
                runeName: burnRunesState.runeName,
                runeSymbol: burnRunesState.runeSymbol,
                thumbnail: burnRunesState.thumbnail,
                amount: burnRunesState.selectedUtxo?.displayRuneAmount,
                timestamp: Date.now()
            });
            // Keep only last 100 burn transactions
            localStorage.setItem('burnTransactions', JSON.stringify(burnTxids.slice(-100)));
            console.log('✅ Burn transaction saved to localStorage');
        } catch (e) {
            console.error('Failed to save burn tx:', e);
        }
        
        // Clean up
        document.getElementById('burn-runes-confirm-modal')?.remove();
        document.getElementById('burn-rune-screen')?.remove();
        burnRunesState = { selectedUtxo: null, utxosList: [], runeName: null, runeSymbol: null, thumbnail: null };
        
        // Refresh wallet
        if (typeof loadWalletData === 'function') {
            setTimeout(() => loadWalletData(), 2000);
        }
        
    } catch (error) {
        console.error('❌ Burn error:', error);
        showNotification('❌ Burn failed: ' + error.message, 'error');
        confirmBtn.disabled = false;
        confirmBtn.textContent = '🔥 Burn Forever';
    }
}

// 🎨 Show Create New Rune Screen (Etching)
function showCreateRuneScreen() {
    console.log('🎨 Opening Create New Rune screen');
    
    const createScreen = document.createElement('div');
    createScreen.className = 'rune-details-screen';
    createScreen.id = 'create-rune-screen';
    
    createScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="create-rune-back">←</button>
            <div class="rune-details-title">🎨 Create New Rune</div>
        </div>
        
        <div class="rune-details-content" style="padding: 24px;">
            <div style="background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%); padding: 20px; border-radius: 12px; margin-bottom: 24px; text-align: center;">
                <div style="font-size: 48px; margin-bottom: 12px;">✨</div>
                <div style="color: white; font-weight: 600; margin-bottom: 8px;">Launch Your Own Rune</div>
                <div style="color: rgba(255,255,255,0.9); font-size: 14px;">
                    Create a fungible token on Bitcoin using the Runes protocol
                </div>
            </div>
            
            <form id="create-rune-form">
                <div class="form-group">
                    <label>Rune Name *</label>
                    <input type="text" id="rune-name" placeholder="MY•AWESOME•RUNE" required
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #6366f1; border-radius: 8px; color: #fff; font-size: 16px; text-transform: uppercase;">
                    <div style="color: #888; font-size: 12px; margin-top: 8px;">
                        Use bullets (•) to separate words. Only A-Z allowed.
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Symbol/Emoji *</label>
                    <input type="text" id="rune-symbol" placeholder="🚀" maxlength="2" required
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 24px; text-align: center;">
                    <div style="color: #888; font-size: 12px; margin-top: 8px;">
                        Single character or emoji
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Decimals</label>
                    <input type="number" id="rune-decimals" min="0" max="38" value="0"
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 16px;">
                    <div style="color: #888; font-size: 12px; margin-top: 8px;">
                        Number of decimal places (0-38)
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Total Supply</label>
                    <input type="number" id="rune-supply" min="0" placeholder="0 = unlimited" value="1000000"
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 16px;">
                    <div style="color: #888; font-size: 12px; margin-top: 8px;">
                        0 = unlimited (open mint)
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Premine</label>
                    <input type="number" id="rune-premine" min="0" value="0"
                           style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 16px;">
                    <div style="color: #888; font-size: 12px; margin-top: 8px;">
                        Amount minted to your address immediately
                    </div>
                </div>
                
                <div class="form-group">
                    <label style="display: flex; align-items: center; gap: 12px; cursor: pointer;">
                        <input type="checkbox" id="rune-turbo" style="width: 20px; height: 20px;">
                        <span>⚡ Turbo Mode (fast minting)</span>
                    </label>
                </div>
                
                <div style="background: #2a2a2a; padding: 16px; border-radius: 8px; margin-top: 24px;">
                    <div style="color: #ff9500; font-weight: 600; margin-bottom: 8px;">💰 Estimated Cost</div>
                    <div style="color: #888; font-size: 14px;">
                        • Etching Fee: ~0.001 BTC<br>
                        • Network Fee: Variable (based on congestion)
                    </div>
                </div>
                
                <button type="submit" class="btn btn-primary" style="width: 100%; background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%); margin-top: 24px; padding: 16px; font-weight: 600;">
                    ✨ Create Rune
                </button>
            </form>
        </div>
    `;
    
    document.body.appendChild(createScreen);
    
    // Event listeners
    document.getElementById('create-rune-back').addEventListener('click', () => {
        createScreen.remove();
    });
    
    document.getElementById('create-rune-form').addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const name = document.getElementById('rune-name').value;
        const symbol = document.getElementById('rune-symbol').value;
        const decimals = parseInt(document.getElementById('rune-decimals').value);
        const supply = parseInt(document.getElementById('rune-supply').value);
        const premine = parseInt(document.getElementById('rune-premine').value);
        const turbo = document.getElementById('rune-turbo').checked;
        
        try {
            showLoading('🎨 Creating your rune...');
            
            // TODO: Implementar createRuneTransaction
            showNotification('🚧 Create Rune feature coming soon! Backend ready, testing in progress.', 'info');
            hideLoading();
            
        } catch (error) {
            hideLoading();
            showNotification('❌ ' + error.message, 'error');
        }
    });
}

// Notifications
function showNotification(message, type = 'info') {
    const notification = document.getElementById('notification');
    
    if (!notification) {
        console.error('❌ Notification element not found!');
        console.error('   Message was:', message);
        // Fallback: usar console
        if (type === 'error') {
            console.error('🔴', message);
        } else if (type === 'success') {
            console.log('✅', message);
        } else {
            console.log('ℹ️', message);
        }
        return;
    }
    
    notification.textContent = message;
    notification.className = `notification ${type}`;
    notification.classList.remove('hidden');
    
    setTimeout(() => {
        notification.classList.add('hidden');
    }, 3000);
}

/**
 * Tela de sucesso para listagem no marketplace (dentro do popup)
 */
function showListingSuccessScreen(inscriptionId, price, orderId) {
    // Atualizar preço na tela
    const priceEl = document.getElementById('listing-success-price');
    if (priceEl) {
        priceEl.textContent = `${price.toLocaleString()} sats`;
    }
    
    // Configurar botão View on KrayScan
    const viewBtn = document.getElementById('listing-success-view-btn');
    if (viewBtn) {
        viewBtn.onclick = () => {
            const krayscanUrl = `https://kraywallet-backend.onrender.com/krayscan.html?inscription=${inscriptionId}`;
            window.open(krayscanUrl, '_blank');
        };
    }
    
    // Configurar botão Done
    const doneBtn = document.getElementById('listing-success-done-btn');
    if (doneBtn) {
        doneBtn.onclick = () => {
            showScreen('wallet');
            loadWalletData();
        };
    }
    
    // Mostrar tela de sucesso
    showScreen('listing-success');
}

/**
 * 🎉 Show Listing Success Modal
 * Beautiful success modal for completed listings (same style as Buy Now)
 */
function showListingSuccessModal(inscriptionId, price) {
    console.log('🎉 Showing Listing success modal');
    console.log('   Inscription:', inscriptionId);
    console.log('   Price:', price);
    
    // Remove any existing modal
    const existingModal = document.getElementById('listing-success-modal');
    if (existingModal) existingModal.remove();
    
    // Create modal overlay
    const modal = document.createElement('div');
    modal.id = 'listing-success-modal';
    modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
        animation: fadeIn 0.3s ease;
    `;
    
    modal.innerHTML = `
        <div style="
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 20px;
            padding: 30px;
            max-width: 360px;
            width: 90%;
            text-align: center;
            border: 1px solid #3d5a80;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            animation: slideUp 0.4s ease;
        ">
            <div style="
                width: 80px;
                height: 80px;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0 auto 20px;
                animation: pulse 2s infinite;
            ">
                <span style="font-size: 40px;">✅</span>
            </div>
            
            <h2 style="
                color: #fff;
                font-size: 24px;
                margin: 0 0 10px 0;
                font-weight: 700;
            ">Listing is LIVE!</h2>
            
            <p style="
                color: #94a3b8;
                font-size: 14px;
                margin: 0 0 20px 0;
            ">Your inscription is now on the marketplace.</p>
            
            <div style="
                background: rgba(34, 197, 94, 0.1);
                border: 1px solid rgba(34, 197, 94, 0.3);
                border-radius: 12px;
                padding: 15px;
                margin-bottom: 20px;
            ">
                <div style="color: #94a3b8; font-size: 12px; margin-bottom: 5px;">Price</div>
                <div style="
                    color: #22c55e;
                    font-size: 28px;
                    font-weight: 700;
                ">${price.toLocaleString()} sats</div>
            </div>
            
            <a href="https://krayspace.com/krayscan.html?inscription=${inscriptionId}" target="_blank" style="
                display: block;
                width: 100%;
                background: transparent;
                border: 1px solid #3d5a80;
                color: #fff;
                padding: 12px;
                border-radius: 10px;
                font-size: 14px;
                cursor: pointer;
                transition: all 0.2s;
                margin-bottom: 10px;
                text-decoration: none;
                text-align: center;
                box-sizing: border-box;
            ">
                🔍 View on KrayScan
            </a>
            
            <button id="listing-modal-done-btn" style="
                width: 100%;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border: none;
                color: #fff;
                padding: 14px;
                border-radius: 10px;
                font-size: 16px;
                font-weight: 600;
                cursor: pointer;
                transition: all 0.2s;
            ">
                ✅ Done
            </button>
        </div>
        
        <style>
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
            @keyframes slideUp {
                from { transform: translateY(20px); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
            @keyframes pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.05); }
            }
        </style>
    `;
    
    document.body.appendChild(modal);
    
    // Add click handler for Done button
    document.getElementById('listing-modal-done-btn').onclick = () => {
        modal.remove();
        showScreen('wallet');
        loadWalletData();
    };
}

/**
 * 🎉 Show Buy Now Success Modal
 * Beautiful success modal for completed Buy Now sales
 */
function showBuyNowSuccessModal(txid, orderId) {
    console.log('🎉 Showing Buy Now success modal');
    console.log('   TXID:', txid);
    
    // Create modal overlay
    const modal = document.createElement('div');
    modal.id = 'buynow-success-modal';
    modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
        animation: fadeIn 0.3s ease;
    `;
    
    modal.innerHTML = `
        <div style="
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 20px;
            padding: 30px;
            max-width: 360px;
            width: 90%;
            text-align: center;
            border: 1px solid #3d5a80;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            animation: slideUp 0.4s ease;
        ">
            <div style="
                width: 80px;
                height: 80px;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0 auto 20px;
                animation: pulse 2s infinite;
            ">
                <span style="font-size: 40px;">✅</span>
            </div>
            
            <h2 style="
                color: #fff;
                font-size: 24px;
                margin: 0 0 10px 0;
                font-weight: 700;
            ">Sale Complete!</h2>
            
            <p style="
                color: #94a3b8;
                font-size: 14px;
                margin: 0 0 20px 0;
            ">The inscription has been transferred successfully.</p>
            
            <div style="
                background: rgba(34, 197, 94, 0.1);
                border: 1px solid rgba(34, 197, 94, 0.3);
                border-radius: 12px;
                padding: 15px;
                margin-bottom: 20px;
            ">
                <div style="color: #94a3b8; font-size: 12px; margin-bottom: 5px;">Transaction ID</div>
                <div style="
                    color: #22c55e;
                    font-size: 13px;
                    font-family: monospace;
                    word-break: break-all;
                ">${txid}</div>
            </div>
            
            <div style="display: flex; gap: 10px;">
                <button id="buynow-mempool-btn" style="
                    flex: 1;
                    background: transparent;
                    border: 1px solid #3d5a80;
                    color: #fff;
                    padding: 12px;
                    border-radius: 10px;
                    font-size: 14px;
                    cursor: pointer;
                    transition: all 0.2s;
                ">
                    🔗 Mempool
                </button>
                <button id="buynow-krayscan-btn" style="
                    flex: 1;
                    background: transparent;
                    border: 1px solid #3d5a80;
                    color: #fff;
                    padding: 12px;
                    border-radius: 10px;
                    font-size: 14px;
                    cursor: pointer;
                    transition: all 0.2s;
                ">
                    🔍 KrayScan
                </button>
            </div>
            
            <button id="buynow-success-done-btn" style="
                width: 100%;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border: none;
                color: #fff;
                padding: 14px;
                border-radius: 10px;
                font-size: 16px;
                font-weight: 600;
                cursor: pointer;
                margin-top: 15px;
                transition: all 0.2s;
            ">
                ✅ Done
            </button>
        </div>
        
        <style>
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
            @keyframes slideUp {
                from { transform: translateY(20px); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
            @keyframes pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.05); }
            }
        </style>
    `;
    
    document.body.appendChild(modal);
    
    // Mempool button - open in new tab
    document.getElementById('buynow-mempool-btn').onclick = () => {
        chrome.tabs.create({ url: `https://mempool.space/tx/${txid}` });
    };
    
    // KrayScan button - open in new tab
    document.getElementById('buynow-krayscan-btn').onclick = () => {
        chrome.tabs.create({ url: `https://kray.space/krayscan.html?txid=${txid}` });
    };
    
    // Done button closes modal and returns to wallet
    document.getElementById('buynow-success-done-btn').onclick = () => {
        modal.remove();
        showScreen('wallet');
        loadWalletData();
    };
}

// Notificação especial para transações (clicável)
function showTransactionNotification(txid, txidShort) {
    const notification = document.getElementById('notification');
    
    if (!notification) {
        console.error('❌ Notification element not found!');
        return;
    }
    
    // Limpar conteúdo anterior
    notification.innerHTML = '';
    notification.className = 'notification success';
    notification.classList.remove('hidden');
    
    // Criar conteúdo clicável
    const content = document.createElement('div');
    content.style.cursor = 'pointer';
    content.style.display = 'flex';
    content.style.flexDirection = 'column';
    content.style.gap = '4px';
    
    const title = document.createElement('div');
    title.textContent = '✅ Transaction Broadcasted!';
    title.style.fontWeight = '600';
    
    const txidEl = document.createElement('div');
    txidEl.textContent = `TXID: ${txidShort}`;
    txidEl.style.fontSize = '12px';
    txidEl.style.opacity = '0.9';
    
    const hint = document.createElement('div');
    hint.textContent = '👆 Click to view on mempool.space';
    hint.style.fontSize = '11px';
    hint.style.opacity = '0.7';
    hint.style.marginTop = '2px';
    
    content.appendChild(title);
    content.appendChild(txidEl);
    content.appendChild(hint);
    
    // Adicionar click handler - abre no mempool.space
    content.addEventListener('click', () => {
        const url = `https://mempool.space/tx/${txid}`;
        chrome.tabs.create({ url });
        notification.classList.add('hidden');
    });
    
    notification.appendChild(content);
    
    // Auto-hide após 5 segundos
    setTimeout(() => {
        notification.classList.add('hidden');
    }, 5000);
}

// ==========================================
// 🎉 SUCCESS MODAL (Reusable for all broadcasts)
// ==========================================
function showSuccessModal(txid, title = 'Transaction Broadcasted!', subtitle = 'Your transaction has been sent successfully.') {
    // Remove existing modal if any
    const existingModal = document.getElementById('success-modal');
    if (existingModal) existingModal.remove();
    
    const modal = document.createElement('div');
    modal.id = 'success-modal';
    modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
        animation: fadeIn 0.3s ease;
    `;
    
    const txidShort = txid.length > 20 ? txid.slice(0, 10) + '...' + txid.slice(-10) : txid;
    
    modal.innerHTML = `
        <div style="
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 20px;
            padding: 30px;
            max-width: 360px;
            width: 90%;
            text-align: center;
            border: 1px solid #3d5a80;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            animation: slideUp 0.4s ease;
        ">
            <div style="
                width: 80px;
                height: 80px;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0 auto 20px;
                animation: pulse 2s infinite;
            ">
                <span style="font-size: 40px;">✅</span>
            </div>
            
            <h2 style="
                color: #fff;
                font-size: 24px;
                margin: 0 0 10px 0;
                font-weight: 700;
            ">${title}</h2>
            
            <p style="
                color: #94a3b8;
                font-size: 14px;
                margin: 0 0 20px 0;
            ">${subtitle}</p>
            
            <div style="
                background: rgba(34, 197, 94, 0.1);
                border: 1px solid rgba(34, 197, 94, 0.3);
                border-radius: 12px;
                padding: 15px;
                margin-bottom: 20px;
            ">
                <div style="color: #94a3b8; font-size: 12px; margin-bottom: 5px;">Transaction ID</div>
                <div id="success-modal-txid" style="
                    color: #22c55e;
                    font-size: 13px;
                    font-family: monospace;
                    word-break: break-all;
                    cursor: pointer;
                " title="Click to copy">${txidShort}</div>
            </div>
            
            <div style="display: flex; gap: 10px;">
                <button id="success-mempool-btn" style="
                    flex: 1;
                    background: transparent;
                    border: 1px solid #3d5a80;
                    color: #fff;
                    padding: 12px;
                    border-radius: 10px;
                    font-size: 14px;
                    cursor: pointer;
                    transition: all 0.2s;
                ">
                    🔗 Mempool
                </button>
                <button id="success-krayscan-btn" style="
                    flex: 1;
                    background: transparent;
                    border: 1px solid #3d5a80;
                    color: #fff;
                    padding: 12px;
                    border-radius: 10px;
                    font-size: 14px;
                    cursor: pointer;
                    transition: all 0.2s;
                ">
                    🔍 KrayScan
                </button>
            </div>
            
            <button id="success-done-btn" style="
                width: 100%;
                background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
                border: none;
                color: #fff;
                padding: 14px;
                border-radius: 10px;
                font-size: 16px;
                font-weight: 600;
                cursor: pointer;
                margin-top: 15px;
                transition: all 0.2s;
            ">
                ✅ Done
            </button>
        </div>
        
        <style>
            @keyframes fadeIn {
                from { opacity: 0; }
                to { opacity: 1; }
            }
            @keyframes slideUp {
                from { transform: translateY(20px); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
            @keyframes pulse {
                0%, 100% { transform: scale(1); }
                50% { transform: scale(1.05); }
            }
        </style>
    `;
    
    document.body.appendChild(modal);
    
    // Copy TXID on click
    document.getElementById('success-modal-txid').onclick = async () => {
        try {
            await navigator.clipboard.writeText(txid);
            document.getElementById('success-modal-txid').textContent = '✅ Copied!';
            setTimeout(() => {
                document.getElementById('success-modal-txid').textContent = txidShort;
            }, 1500);
        } catch (e) {
            console.error('Failed to copy:', e);
        }
    };
    
    // Mempool button - open in new tab
    document.getElementById('success-mempool-btn').onclick = () => {
        chrome.tabs.create({ url: `https://mempool.space/tx/${txid}` });
    };
    
    // KrayScan button - open in new tab
    document.getElementById('success-krayscan-btn').onclick = () => {
        chrome.tabs.create({ url: `https://kray.space/krayscan.html?txid=${txid}` });
    };
    
    // Done button closes modal and returns to wallet
    document.getElementById('success-done-btn').onclick = () => {
        modal.remove();
        showScreen('wallet');
        loadWalletData();
    };
}

// ==========================================
// 🔧 SETTINGS HANDLERS
// ==========================================

// Reset Wallet
// 🔒 Show reset confirmation modal
function showResetConfirmModal() {
    console.log('⚠️ Showing reset confirmation modal...');
    const modal = document.getElementById('reset-confirm-modal');
    const input = document.getElementById('reset-confirm-input');
    const confirmBtn = document.getElementById('confirm-reset-btn');
    
    // Reset state
    if (input) input.value = '';
    if (confirmBtn) {
        confirmBtn.disabled = true;
        confirmBtn.style.opacity = '0.5';
    }
    
    // Show modal
    modal.classList.remove('hidden');
    
    // Focus input
    setTimeout(() => input?.focus(), 100);
}

function closeResetConfirmModal() {
    console.log('❌ Closing reset confirmation modal...');
    const modal = document.getElementById('reset-confirm-modal');
    const input = document.getElementById('reset-confirm-input');
    
    modal.classList.add('hidden');
    if (input) input.value = '';
}

function handleResetConfirmInput() {
    const input = document.getElementById('reset-confirm-input');
    const confirmBtn = document.getElementById('confirm-reset-btn');
    
    if (!input || !confirmBtn) return;
    
    const isValid = input.value.toUpperCase() === 'DELETE';
    confirmBtn.disabled = !isValid;
    confirmBtn.style.opacity = isValid ? '1' : '0.5';
}

async function handleResetWallet() {
    // Show confirmation modal instead of confirm()
    showResetConfirmModal();
}

async function executeResetWallet() {
    const input = document.getElementById('reset-confirm-input');
    
    if (input?.value.toUpperCase() !== 'DELETE') {
        showNotification('❌ Type DELETE to confirm', 'error');
        return;
    }
    
    closeResetConfirmModal();
    showLoading('Resetting wallet...');
    
    try {
        // Limpar storage
        await chrome.storage.local.clear();
        
        // Limpar estado
        wallet = null;
        
        showNotification('✅ Wallet reset successfully!', 'success');
        
        // Voltar para tela inicial
        setTimeout(() => {
            showScreen('no-wallet');
            hideLoading();
        }, 1000);
        
    } catch (error) {
        console.error('Error resetting wallet:', error);
        showNotification('❌ Failed to reset wallet', 'error');
        hideLoading();
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🔐 MNEMONIC REVEAL - Secure with Hold-to-Reveal + Internet Check
// ═══════════════════════════════════════════════════════════════════════════

let currentMnemonicWords = []; // Store words temporarily

// 🛡️ Check Internet Connection Status
async function checkInternetConnection() {
    const statusDiv = document.getElementById('connection-status');
    if (!statusDiv) return true;
    
    // Method 1: navigator.onLine
    const isOnline = navigator.onLine;
    
    // Method 2: Try to fetch a tiny resource (more reliable)
    let canReachInternet = false;
    try {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 2000);
        
        const response = await fetch('https://www.google.com/favicon.ico', {
            method: 'HEAD',
            mode: 'no-cors',
            signal: controller.signal
        });
        clearTimeout(timeoutId);
        canReachInternet = true;
    } catch (e) {
        canReachInternet = false;
    }
    
    const isConnected = isOnline && canReachInternet;
    
    if (isConnected) {
        statusDiv.style.background = 'rgba(239, 68, 68, 0.2)';
        statusDiv.style.color = '#fca5a5';
        statusDiv.innerHTML = `
            <span style="font-size: 16px;">🔴</span>
            <span>CONNECTED - We recommend disconnecting before viewing</span>
        `;
    } else {
        statusDiv.style.background = 'rgba(16, 185, 129, 0.2)';
        statusDiv.style.color = '#6ee7b7';
        statusDiv.innerHTML = `
            <span style="font-size: 16px;">🟢</span>
            <span>OFFLINE - Safe to view recovery phrase</span>
        `;
    }
    
    return isConnected;
}

// Update connection status when screen is shown
function initMnemonicSecurityScreen() {
    checkInternetConnection();
    
    // Also listen for online/offline events
    window.addEventListener('online', checkInternetConnection);
    window.addEventListener('offline', checkInternetConnection);
}

// Reveal Mnemonic (Step 1: Validate Password)
async function handleRevealMnemonic() {
    const password = document.getElementById('view-mnemonic-password').value;
    
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    // 🛡️ SECURITY CHECK: Warn if connected to internet
    const isConnected = await checkInternetConnection();
    
    if (isConnected) {
        const proceed = confirm(
            '⚠️ SECURITY WARNING ⚠️\n\n' +
            'Your computer is connected to the internet!\n\n' +
            'For maximum security, we strongly recommend:\n' +
            '1. Disconnect from Wi-Fi/Ethernet\n' +
            '2. Then view your recovery phrase\n\n' +
            'Malware and screen-capture viruses can steal your keys while online.\n\n' +
            'Do you want to continue anyway? (NOT RECOMMENDED)'
        );
        
        if (!proceed) {
            showNotification('👍 Good choice! Disconnect from internet first', 'info');
            return;
        }
    }
    
    showLoading('Decrypting...');
    
    try {
        const response = await chrome.runtime.sendMessage({
            action: 'getMnemonic',
            data: { password }
        });
        
        if (response.success && response.mnemonic) {
            // 🔒 SECURITY: Clear password field immediately
            clearAllPasswordFields();
            
            // Store words
            currentMnemonicWords = response.mnemonic.trim().split(/\s+/);
            
            // Hide password step and warning, show revealed step
            document.getElementById('mnemonic-password-step').classList.add('hidden');
            document.getElementById('internet-security-warning').classList.add('hidden');
            document.getElementById('mnemonic-revealed-step').classList.remove('hidden');
            
            // Render grid with asterisks (hidden)
            renderMnemonicGrid(false);
            
            // Setup hold-to-reveal button
            setupHoldToReveal();
            
            showNotification('✅ Password verified! Hold button to reveal words', 'success');
        } else {
            showNotification('❌ ' + (response.error || 'Invalid password'), 'error');
        }
    } catch (error) {
        console.error('Error revealing mnemonic:', error);
        showNotification('❌ Failed to reveal mnemonic', 'error');
    } finally {
        hideLoading();
    }
}

// Render mnemonic grid (numbered format like wallet creation)
function renderMnemonicGrid(showWords) {
    const grid = document.getElementById('mnemonic-grid-security');
    if (!grid) return;
    
    grid.innerHTML = '';
    
    currentMnemonicWords.forEach((word, index) => {
        const wordBox = document.createElement('div');
        wordBox.style.cssText = `
            background: rgba(255,255,255,0.05);
            border: 1px solid rgba(255,255,255,0.1);
            border-radius: 8px;
            padding: 10px 8px;
            text-align: center;
            font-size: 12px;
        `;
        
        const numberSpan = document.createElement('span');
        numberSpan.style.cssText = 'color: #888; font-size: 10px; display: block; margin-bottom: 4px;';
        numberSpan.textContent = `${index + 1}`;
        
        const wordSpan = document.createElement('span');
        wordSpan.style.cssText = `
            color: ${showWords ? '#fff' : '#666'};
            font-weight: 600;
            font-family: monospace;
            letter-spacing: ${showWords ? '0' : '2px'};
        `;
        wordSpan.textContent = showWords ? word : '••••••';
        
        wordBox.appendChild(numberSpan);
        wordBox.appendChild(wordSpan);
        grid.appendChild(wordBox);
    });
}

// Setup hold-to-reveal functionality
function setupHoldToReveal() {
    const btn = document.getElementById('hold-to-reveal-btn');
    if (!btn) return;
    
    let isHolding = false;
    
    const showWords = () => {
        isHolding = true;
        btn.textContent = '👁️ Revealing...';
        btn.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
        renderMnemonicGrid(true);
    };
    
    const hideWords = () => {
        isHolding = false;
        btn.textContent = '👁️ Hold to Reveal';
        btn.style.background = 'linear-gradient(135deg, #333 0%, #222 100%)';
        renderMnemonicGrid(false);
    };
    
    // Mouse events
    btn.addEventListener('mousedown', showWords);
    btn.addEventListener('mouseup', hideWords);
    btn.addEventListener('mouseleave', hideWords);
    
    // Touch events (mobile)
    btn.addEventListener('touchstart', (e) => {
        e.preventDefault();
        showWords();
    });
    btn.addEventListener('touchend', hideWords);
    btn.addEventListener('touchcancel', hideWords);
}

// Copy Mnemonic
async function handleCopyMnemonic() {
    if (currentMnemonicWords.length === 0) {
        showNotification('❌ No mnemonic to copy', 'error');
        return;
    }
    
    const mnemonic = currentMnemonicWords.join(' ');
    
    try {
        await navigator.clipboard.writeText(mnemonic);
        showNotification('✅ Recovery phrase copied!', 'success');
    } catch (error) {
        console.error('Error copying:', error);
        showNotification('❌ Failed to copy', 'error');
    }
}

// Reveal Private Key
async function handleRevealPrivateKey() {
    const password = document.getElementById('view-key-password').value;
    
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    showLoading('Decrypting...');
    
    try {
        const response = await chrome.runtime.sendMessage({
            action: 'getPrivateKey',
            data: { password }
        });
        
        if (response.success) {
            // 🔒 SECURITY: Clear password field immediately
            clearAllPasswordFields();
            document.getElementById('private-key-display').textContent = response.privateKey;
            document.getElementById('private-key-display').classList.remove('hidden');
            document.getElementById('copy-key-btn').classList.remove('hidden');
            document.getElementById('reveal-key-btn').classList.add('hidden');
            showNotification('✅ Private key revealed', 'success');
        } else {
            showNotification('❌ ' + (response.error || 'Invalid password'), 'error');
        }
    } catch (error) {
        console.error('Error revealing private key:', error);
        showNotification('❌ Failed to reveal private key', 'error');
    } finally {
        hideLoading();
    }
}

// Copy Private Key
async function handleCopyPrivateKey() {
    const privateKey = document.getElementById('private-key-display').textContent;
    
    try {
        await navigator.clipboard.writeText(privateKey);
        showNotification('✅ Private key copied to clipboard!', 'success');
    } catch (error) {
        console.error('Error copying:', error);
        showNotification('❌ Failed to copy', 'error');
    }
}

// ==========================================
// 🔏 PSBT CONFIRMATION SCREEN
// ==========================================

/**
 * Setup fee selector for atomic swap
 */
function setupFeeSelector(pendingPsbt) {
    console.log('💰 Setting up fee selector...');
    
    // Buscar fees disponíveis
    fetch('https://kraywallet-backend.onrender.com/api/wallet/fees')
        .then(r => r.json())
        .then(data => {
            const fees = data.fees;
            let selectedFee = pendingPsbt.feeRate || fees.high;
            
            // Event listeners para botões
            const feeButtons = document.querySelectorAll('.fee-btn');
            feeButtons.forEach(btn => {
                btn.addEventListener('click', () => {
                    const feeType = btn.dataset.feeType;
                    
                    // Remover active de todos
                    feeButtons.forEach(b => {
                        b.classList.remove('fee-btn-active');
                        b.style.border = '1px solid #444';
                        b.style.background = '#2a2a2a';
                        b.style.color = '#fff';
                        b.textContent = b.dataset.feeType.charAt(0).toUpperCase() + b.dataset.feeType.slice(1);
                    });
                    
                    // Ativar o clicado
                    btn.classList.add('fee-btn-active');
                    btn.style.border = '1px solid #FFC107';
                    btn.style.background = 'rgba(255,193,7,0.2)';
                    btn.style.color = '#FFC107';
                    btn.textContent = feeType.charAt(0).toUpperCase() + feeType.slice(1) + ' ✓';
                    
                    // Atualizar fee selecionada
                    selectedFee = fees[feeType];
                    pendingPsbt.feeRate = selectedFee;
                    
                    console.log(`💰 Fee changed to ${feeType}: ${selectedFee} sat/vB`);
                    
                    document.querySelector('.fee-info div:first-child').textContent = 
                        `⚡ Network Fee: ${selectedFee} sat/vB`;
                });
            });
            
            // Custom fee input
            const customInput = document.getElementById('custom-fee-input');
            if (customInput) {
                customInput.addEventListener('input', (e) => {
                    const customFee = parseInt(e.target.value);
                    if (customFee && customFee > 0) {
                        selectedFee = customFee;
                        pendingPsbt.feeRate = selectedFee;
                        
                        console.log(`💰 Custom fee: ${selectedFee} sat/vB`);
                        
                        // Desativar botões
                        feeButtons.forEach(b => {
                            b.classList.remove('fee-btn-active');
                            b.style.border = '1px solid #444';
                            b.style.background = '#2a2a2a';
                            b.style.color = '#fff';
                            b.textContent = b.dataset.feeType.charAt(0).toUpperCase() + b.dataset.feeType.slice(1);
                        });
                        
                        document.querySelector('.fee-info div:first-child').textContent = 
                            `⚡ Network Fee: ${selectedFee} sat/vB (Custom)`;
                    }
                });
            }
        })
        .catch(err => console.warn('⚠️  Failed to fetch fees for selector:', err));
}

async function showPsbtConfirmation() {
    try {
        console.log('🔏 Loading PSBT confirmation DATA (screen already shown)...');
        
        // NÃO chama showScreen aqui - a tela JÁ foi trocada antes!
        // showScreen('confirm-psbt'); // ❌ REMOVIDO - causa conflito!
        
        // Buscar pending PSBT do background
        const response = await sendMessage({ action: 'getPendingPsbt' });
        
        if (!response.success || !response.pending) {
            console.warn('⚠️ No pending PSBT found in showPsbtConfirmation');
            // NÃO FECHAR O POPUP! Apenas voltar para a tela da wallet
            showScreen('wallet');
            await loadWalletData();
            return;
        }
        
        const pendingPsbt = response.pending;
        console.log('📋 Pending PSBT:', pendingPsbt);
        
        // Decodificar PSBT para extrair informações
        let psbtDetails = null;
        try {
            const decodeResponse = await fetch('https://kraywallet-backend.onrender.com/api/psbt/decode', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ psbt: pendingPsbt.psbt })
            });
            psbtDetails = await decodeResponse.json();
            console.log('📊 PSBT decoded:', psbtDetails);
        } catch (decodeError) {
            console.warn('⚠️ Could not decode PSBT:', decodeError);
        }
        
        // Construir visualização detalhada do PSBT (design profissional)
        const detailsContainer = document.getElementById('psbt-details-container');
        
        // Verificar tipo de transação
        const isRuneTransfer = pendingPsbt.isRuneTransfer || false;
        const isAtomicSwap = pendingPsbt.type === 'createOffer' || pendingPsbt.type === 'buyAtomicSwap' || pendingPsbt.type === 'buyNow';
        const isSeller = pendingPsbt.type === 'createOffer';
        const isBuyer = pendingPsbt.type === 'buyAtomicSwap' || pendingPsbt.type === 'buyNow';
        
        // 🛡️ DESIGN PROFISSIONAL PARA ATOMIC SWAP (VENDEDOR E COMPRADOR)
        if (isAtomicSwap) {
            // Detect Buy Now model (uses SIGHASH_ALL, not Guardians)
            const isBuyNowModel = pendingPsbt.type === 'buyNow' || pendingPsbt.orderId?.startsWith('buynow_');
            const sighashDisplay = isBuyNowModel ? 'ALL (0x01)' : (pendingPsbt.sighashType || (isSeller ? 'NONE|ANYONECANPAY' : 'ALL'));
            const securityText = isBuyNowModel ? '🔐 Atomic Swap' : '🛡️ Guardian Protected';
            const roleEmoji = isSeller ? '🏷️' : '🛒';
            const roleText = isSeller ? 'List Inscription' : 'Buy Inscription';
            const roleDescription = isSeller 
                ? 'You are listing your inscription for sale on the marketplace.'
                : 'You are purchasing an inscription from the marketplace.';
            
            let html = `
                <div style="background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%); border-radius: 16px; padding: 20px; margin-bottom: 16px;">
                    <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 16px;">
                        <div style="font-size: 32px;">${roleEmoji}</div>
                        <div>
                            <div style="font-size: 18px; font-weight: 700; color: #fff;">${roleText}</div>
                            <div style="font-size: 12px; color: #888; margin-top: 4px;">${roleDescription}</div>
                        </div>
                    </div>
                    
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px; margin-bottom: 16px;">
                        <div style="background: rgba(255,255,255,0.05); border-radius: 12px; padding: 12px;">
                            <div style="font-size: 10px; color: #888; text-transform: uppercase; margin-bottom: 4px;">Security</div>
                            <div style="font-size: 14px; color: #10B981; font-weight: 600;">${securityText}</div>
                        </div>
                        <div style="background: rgba(255,255,255,0.05); border-radius: 12px; padding: 12px;">
                            <div style="font-size: 10px; color: #888; text-transform: uppercase; margin-bottom: 4px;">SIGHASH</div>
                            <div style="font-size: 12px; color: #FFC107; font-weight: 600; font-family: monospace;">${sighashDisplay}</div>
                        </div>
                    </div>
            `;
            
            // Info específica para vendedor
            if (isSeller && pendingPsbt.seller_value) {
                html += `
                    <div style="background: rgba(139,92,246,0.1); border: 1px solid rgba(139,92,246,0.3); border-radius: 12px; padding: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="font-size: 12px; color: #8B5CF6;">Inscription Value</span>
                            <span style="font-size: 14px; color: #fff; font-weight: 600;">${pendingPsbt.seller_value} sats</span>
                        </div>
                        ${pendingPsbt.price ? `
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 8px; padding-top: 8px; border-top: 1px solid rgba(139,92,246,0.2);">
                            <span style="font-size: 12px; color: #8B5CF6;">Listing Price</span>
                            <span style="font-size: 16px; color: #10B981; font-weight: 700;">${pendingPsbt.price} sats</span>
                        </div>
                        ` : ''}
                    </div>
                `;
            }
            
            // Info específica para comprador - DESIGN COM UTXO DETAILS
            if (isBuyer) {
                const b = pendingPsbt.breakdown || {};
                const inscriptionId = pendingPsbt.inscriptionId || pendingPsbt.inscription_id || '';
                const inscriptionNumber = pendingPsbt.inscriptionNumber || pendingPsbt.inscription_number || '?';
                // 🎯 DYNAMIC: Get real UTXO value from backend (never hardcode!)
                const sellerValue = b.sellerValue || b.inscriptionPostage || pendingPsbt.seller_value || 546;
                const price = b.price || pendingPsbt.price || 0;
                const marketFee = b.marketFee || Math.max(Math.floor(price * 0.02), 546);
                const totalRequired = b.totalRequired || (price + marketFee);
                
                html += `
                    <!-- UTXO Details Section -->
                    <div style="background: rgba(255,255,255,0.03); border: 1px solid rgba(255,255,255,0.1); border-radius: 12px; padding: 14px; margin-bottom: 12px;">
                        <div style="font-size: 10px; color: #888; text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 10px;">
                            📦 Transaction Details
                        </div>
                        
                        <!-- Inscription you're buying (OUTPUT to you) -->
                        <div style="background: rgba(34, 197, 94, 0.1); border: 1px solid rgba(34, 197, 94, 0.3); border-radius: 8px; padding: 10px; margin-bottom: 8px;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                                <span style="font-size: 10px; color: #22c55e; font-weight: 600; text-transform: uppercase;">📥 YOU RECEIVE</span>
                                <span style="font-size: 11px; color: #22c55e; font-weight: 700;">${sellerValue} sats</span>
                            </div>
                            <div style="display: flex; gap: 10px; align-items: center;">
                                ${inscriptionId ? `
                                <img src="https://ordinals.com/content/${inscriptionId}" 
                                     style="width: 44px; height: 44px; border-radius: 6px; object-fit: cover; border: 2px solid rgba(34, 197, 94, 0.5);"
                                     onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/></svg>'">
                                ` : ''}
                                <div style="flex: 1; min-width: 0;">
                                    <div style="font-size: 13px; font-weight: 600; color: #22c55e;">Inscription #${inscriptionNumber}</div>
                                    <div style="font-size: 10px; color: #888;">Transfers to your wallet</div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Arrow -->
                        <div style="text-align: center; color: #555; font-size: 12px; margin: 4px 0;">↑</div>
                        
                        <!-- Payment (YOUR INPUT) -->
                        <div style="background: rgba(239, 68, 68, 0.1); border: 1px solid rgba(239, 68, 68, 0.3); border-radius: 8px; padding: 10px;">
                            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 6px;">
                                <span style="font-size: 10px; color: #ef4444; font-weight: 600; text-transform: uppercase;">📤 YOU PAY</span>
                                <span style="font-size: 11px; color: #ef4444; font-weight: 700;">${totalRequired} sats</span>
                            </div>
                            <div style="font-size: 10px; color: #888;">From your wallet UTXOs</div>
                        </div>
                    </div>
                    
                    <!-- Price Breakdown -->
                    <div style="background: rgba(16,185,129,0.1); border: 1px solid rgba(16,185,129,0.3); border-radius: 12px; padding: 12px;">
                        <div style="font-size: 10px; color: #888; text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 8px;">
                            💰 Price Breakdown
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 6px;">
                            <span style="font-size: 12px; color: #10B981;">Inscription Price</span>
                            <span style="font-size: 13px; color: #fff;">${price.toLocaleString()} sats</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 6px;">
                            <span style="font-size: 12px; color: #10B981;">Market Fee (2%)</span>
                            <span style="font-size: 13px; color: #fff;">${marketFee.toLocaleString()} sats</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 6px;">
                            <span style="font-size: 12px; color: #10B981;">Network Fee</span>
                            <span style="font-size: 13px; color: #fff;">~${(b.minerFee || 0).toLocaleString()} sats</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; align-items: center; padding-top: 8px; margin-top: 8px; border-top: 1px solid rgba(16,185,129,0.2);">
                            <span style="font-size: 12px; color: #10B981; font-weight: 600;">Total Payment</span>
                            <span style="font-size: 16px; color: #10B981; font-weight: 700;">${totalRequired.toLocaleString()} sats</span>
                        </div>
                    </div>
                `;
            }
            
            html += `</div>`;
            
            detailsContainer.innerHTML = html;
        } else if (isRuneTransfer) {
            // Transação de Runes - mostrar info simplificada
            detailsContainer.innerHTML = `
                <div class="alert alert-info" style="margin-bottom: 16px;">
                    <strong>⧈ Rune Transfer</strong>
                    <p>You are about to send a Rune token.</p>
                </div>
                <div class="detail-row">
                    <span class="label">Transaction Type:</span>
                    <span class="value">Rune Transfer</span>
                </div>
                <div class="detail-row">
                    <span class="label">SIGHASH Type:</span>
                    <span class="value">${pendingPsbt.sighashType || 'ALL'}</span>
                </div>
                <div class="detail-row">
                    <span class="label">Inputs to sign:</span>
                    <span class="value">All</span>
                </div>
                <div class="detail-row">
                    <span class="label">Network:</span>
                    <span class="value">Bitcoin Mainnet</span>
                </div>
            `;
        } else if (psbtDetails && psbtDetails.inputs && psbtDetails.outputs) {
            // Calcular totais
            const totalInput = psbtDetails.inputs.reduce((sum, inp) => sum + (inp.value || 0), 0);
            const totalOutput = psbtDetails.outputs.reduce((sum, out) => sum + out.value, 0);
            const fee = totalInput - totalOutput;
            
            // Construir HTML detalhado
            let html = `
                <div class="psbt-section">
                    <h3>📥 Inputs (${psbtDetails.inputs.length})</h3>
            `;
            
            // Listar cada input
            psbtDetails.inputs.forEach((input, index) => {
                const addressShort = input.address 
                    ? input.address.slice(0, 8) + '...' + input.address.slice(-8)
                    : 'Unknown';
                const isSigning = pendingPsbt.inputsToSign?.find(i => i.index === index);
                
                html += `
                    <div class="psbt-item ${isSigning ? 'highlight' : ''}">
                        <div class="psbt-item-label">
                            Input #${index} ${isSigning ? '✍️ (You sign)' : '✅ (Signed)'}
                        </div>
                        <div class="psbt-item-value">
                            ${input.value || 0} sats
                        </div>
                        <div class="psbt-item-address">
                            ${addressShort}
                        </div>
                    </div>
                `;
            });
            
            html += `</div><div class="psbt-section">
                    <h3>📤 Outputs (${psbtDetails.outputs.length})</h3>
            `;
            
            // Listar cada output
            psbtDetails.outputs.forEach((output, index) => {
                const addressShort = output.address
                    ? output.address.slice(0, 8) + '...' + output.address.slice(-8)
                    : 'Script';
                    
                html += `
                    <div class="psbt-item">
                        <div class="psbt-item-label">
                            Output #${index}
                        </div>
                        <div class="psbt-item-value">
                            ${output.value} sats
                        </div>
                        <div class="psbt-item-address">
                            → ${addressShort}
                        </div>
                    </div>
                `;
            });
            
            html += `</div><div class="psbt-summary">
                    <div class="summary-row">
                        <span>Total Input:</span>
                        <span>${totalInput} sats</span>
                    </div>
                    <div class="summary-row">
                        <span>Total Output:</span>
                        <span>${totalOutput} sats</span>
                    </div>
                    <div class="summary-row fee-row">
                        <span>Network Fee:</span>
                        <span>${fee} sats${pendingPsbt.feeRate ? ` (${pendingPsbt.feeRate} sat/vB)` : ''}</span>
                    </div>
                </div>
                
                ${pendingPsbt.type === 'buyAtomicSwap' && pendingPsbt.feeRate ? `
                <div class="fee-info" style="margin-top: 12px; padding: 8px; background: rgba(255,193,7,0.1); border-radius: 8px; font-size: 11px;">
                    <div style="margin-bottom: 4px; font-weight: 600; color: #FFC107;">⚡ Network Fee: ${pendingPsbt.feeRate} sat/vB</div>
                    <div style="color: #888;">Using HIGH priority for atomic swap (fast confirmation)</div>
                </div>
                
                <div class="fee-selector" style="margin-top: 12px;">
                    <div style="font-size: 11px; color: #888; margin-bottom: 6px;">Adjust fee (optional):</div>
                    <div style="display: flex; gap: 6px; margin-bottom: 8px;">
                        <button class="fee-btn" data-fee-type="low" style="flex: 1; padding: 6px; font-size: 11px; border: 1px solid #444; background: #2a2a2a; color: #fff; border-radius: 6px; cursor: pointer;">Low</button>
                        <button class="fee-btn" data-fee-type="medium" style="flex: 1; padding: 6px; font-size: 11px; border: 1px solid #444; background: #2a2a2a; color: #fff; border-radius: 6px; cursor: pointer;">Medium</button>
                        <button class="fee-btn fee-btn-active" data-fee-type="high" style="flex: 1; padding: 6px; font-size: 11px; border: 1px solid #FFC107; background: rgba(255,193,7,0.2); color: #FFC107; border-radius: 6px; cursor: pointer; font-weight: 600;">High ✓</button>
                    </div>
                    <input type="number" id="custom-fee-input" placeholder="Custom fee (sat/vB)" min="1" max="1000" style="width: 100%; padding: 8px; background: #2a2a2a; border: 1px solid #444; border-radius: 6px; color: #fff; font-size: 12px;" />
                </div>
                ` : ''}
            `;
            
            detailsContainer.innerHTML = html;
            
            // 🎯 Adicionar event listeners para fee buttons (se for atomic swap)
            if (pendingPsbt.type === 'buyAtomicSwap') {
                setupFeeSelector(pendingPsbt);
            }
            
        } else {
            // Fallback se não conseguir decodificar
            detailsContainer.innerHTML = `
                <div class="detail-row">
                    <span class="label">Type:</span>
                    <span class="value">Atomic Swap</span>
                </div>
                <div class="detail-row">
                    <span class="label">Inputs to sign:</span>
                    <span class="value">${pendingPsbt.inputsToSign?.length || 'All'}</span>
                </div>
                <div class="detail-row">
                    <span class="label">Network:</span>
                    <span class="value">Bitcoin Mainnet</span>
                </div>
            `;
        }
        
        // Focus no campo de senha
        // (Event listeners são registrados em DOMContentLoaded)
        document.getElementById('psbt-confirm-password').focus();
        
    } catch (error) {
        console.error('❌ Error showing PSBT confirmation:', error);
        showNotification('❌ Error loading PSBT request', 'error');
        window.close();
    }
}

// ==========================================
// ⚡ LIGHTNING PAYMENT CONFIRMATION
// ==========================================

async function showLightningPaymentConfirmation(paymentRequest) {
    try {
        console.log('⚡ Loading Lightning payment confirmation...');
        
        // Se não foi passado paymentRequest, buscar do storage
        if (!paymentRequest) {
            const lightningCheck = await chrome.storage.local.get(['pendingPaymentRequest']);
            paymentRequest = lightningCheck.pendingPaymentRequest;
        }
        
        if (!paymentRequest || !paymentRequest.invoice) {
            console.warn('⚠️  No pending Lightning payment found');
            showScreen('wallet');
            await loadWalletData();
            return;
        }
        
        const decoded = paymentRequest.decoded;
        console.log('⚡ Payment details:', decoded);
        
        // Preencher detalhes do pagamento
        document.getElementById('lightning-amount').textContent = 
            `${decoded.amount?.toLocaleString() || '?'} sats`;
        
        document.getElementById('lightning-description').textContent = 
            decoded.description || 'No description';
        
        document.getElementById('lightning-destination').textContent = 
            decoded.destination ? 
            decoded.destination.substring(0, 20) + '...' + decoded.destination.substring(decoded.destination.length - 20) :
            'Unknown';
        
        document.getElementById('lightning-payment-hash').textContent = 
            decoded.paymentHash ? 
            decoded.paymentHash.substring(0, 20) + '...' + decoded.paymentHash.substring(decoded.paymentHash.length - 20) :
            'Unknown';
        
        // Formatar expiry
        if (decoded.expiry) {
            const expiryDate = new Date(decoded.expiry * 1000);
            const now = new Date();
            const diff = expiryDate - now;
            const minutes = Math.floor(diff / 60000);
            const seconds = Math.floor((diff % 60000) / 1000);
            
            if (diff > 0) {
                document.getElementById('lightning-expiry').textContent = 
                    `${minutes}m ${seconds}s`;
            } else {
                document.getElementById('lightning-expiry').textContent = 
                    '⚠️  Expired';
                document.getElementById('lightning-expiry').style.color = '#f44336';
            }
        } else {
            document.getElementById('lightning-expiry').textContent = 'No expiry';
        }
        
        // Focus no campo de senha
        document.getElementById('lightning-payment-password').focus();
        
    } catch (error) {
        console.error('❌ Error showing Lightning payment confirmation:', error);
        showNotification('❌ Error loading Lightning payment request', 'error');
        showScreen('wallet');
        await loadWalletData();
    }
}

async function handleLightningPaymentConfirm() {
    const password = document.getElementById('lightning-payment-password').value;
    
    if (!password) {
        showNotification('❌ Please enter your password', 'error');
        return;
    }
    
    try {
        // Mostrar status de loading
        const statusDiv = document.getElementById('lightning-payment-status');
        const statusText = document.getElementById('lightning-payment-status-text');
        statusDiv.classList.remove('hidden');
        statusDiv.className = 'alert alert-info';
        statusText.textContent = '⏳ Processing Lightning payment...';
        
        // Desabilitar botão
        const confirmBtn = document.getElementById('lightning-payment-confirm-btn');
        confirmBtn.disabled = true;
        confirmBtn.textContent = '⏳ Processing...';
        
        console.log('⚡ Confirming Lightning payment...');
        
        // Buscar pending payment
        const lightningCheck = await chrome.storage.local.get(['pendingPaymentRequest']);
        const paymentRequest = lightningCheck.pendingPaymentRequest;
        
        if (!paymentRequest || !paymentRequest.invoice) {
            throw new Error('No pending payment found');
        }
        
        // Chamar backend para pagar
        console.log('📡 Calling /api/lightning/pay...');
        const response = await fetch('https://kraywallet-backend.onrender.com/api/lightning/pay', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                invoice: paymentRequest.invoice
            })
        });
        
        const result = await response.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Payment failed');
        }
        
        console.log('✅ Payment successful!');
        console.log('   Preimage:', result.preimage);
        console.log('   Payment Hash:', result.paymentHash);
        
        // 🔒 SECURITY: Clear password field immediately
        clearAllPasswordFields();
        
        // Salvar resultado no storage
        await chrome.storage.local.set({
            paymentResult: {
                success: true,
                preimage: result.preimage,
                paymentHash: result.paymentHash,
                amountSats: result.amountSats,
                timestamp: result.timestamp
            }
        });
        
        // Mostrar sucesso
        statusDiv.className = 'alert alert-success';
        statusText.textContent = '✅ Payment successful!';
        
        // Aguardar 1 segundo e fechar
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // Limpar pending payment
        await chrome.storage.local.remove('pendingPaymentRequest');
        
        // Voltar para wallet
        showScreen('wallet');
        await loadWalletData();
        showNotification('✅ Lightning payment sent successfully!', 'success');
        
    } catch (error) {
        console.error('❌ Error processing Lightning payment:', error);
        
        // Mostrar erro
        const statusDiv = document.getElementById('lightning-payment-status');
        const statusText = document.getElementById('lightning-payment-status-text');
        statusDiv.classList.remove('hidden');
        statusDiv.className = 'alert alert-error';
        statusText.textContent = `❌ ${error.message}`;
        
        // Salvar erro no storage
        await chrome.storage.local.set({
            paymentResult: {
                success: false,
                error: error.message
            }
        });
        
        // Reabilitar botão
        const confirmBtn = document.getElementById('lightning-payment-confirm-btn');
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = '⚡ Pay Invoice';
    }
}

async function handleLightningPaymentCancel() {
    console.log('❌ Lightning payment cancelled by user');
    
    // 🔒 SECURITY: Clear password field immediately
    clearAllPasswordFields();
    
    // Salvar cancelamento no storage
    await chrome.storage.local.set({
        paymentResult: {
            success: false,
            error: 'User cancelled'
        }
    });
    
    // Limpar pending payment
    await chrome.storage.local.remove('pendingPaymentRequest');
    
    // Voltar para wallet
    showScreen('wallet');
    await loadWalletData();
}

async function handlePsbtSign() {
    const password = document.getElementById('psbt-confirm-password').value;
    
    if (!password) {
        showNotification('❌ Please enter your password', 'error');
        return;
    }
    
    try {
        showLoading('Signing transaction...');
        
        console.log('✍️  Confirming PSBT sign...');
        
        // 🔐 CRITICAL FIX: Salvar pendingPsbt ANTES de assinar!
        // Depois de confirmPsbtSign, o pendingPsbtRequest é limpo do storage
        const pendingResponse = await sendMessage({ action: 'getPendingPsbt' });
        const pendingPsbt = pendingResponse?.pending;
        
        console.log('📋 Pending PSBT type (BEFORE sign):', pendingPsbt?.type);
        console.log('📋 Pending PSBT orderId:', pendingPsbt?.orderId);
        
        let response;
        
        // 🔐 LOCAL SIGNING for ALL PSBTs (LEI PRIMORDIAL!)
        // Always use local signing when TaprootSignerPure is available
        if (typeof TaprootSignerPure !== 'undefined' && TaprootSignerPure.signPsbtTaprootLocal) {
            console.log('🔐 ===== LOCAL SIGNING IN POPUP (LEI PRIMORDIAL!) =====');
            console.log('   Type:', pendingPsbt?.type || 'generic');
            console.log('   LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
            
            // Step 1: Decrypt mnemonic locally
            const decryptResult = await sendMessage({
                action: 'decryptWallet',
                data: { password: password }
            });
            
            if (!decryptResult.success || !decryptResult.mnemonic) {
                throw new Error(decryptResult.error || 'Failed to decrypt wallet');
            }
            
            let mnemonic = decryptResult.mnemonic;
            console.log('   ✅ Wallet decrypted');
            
            // Step 2: Parse signing parameters from pendingPsbt
            let inputsToSign = [0]; // default
            let sighashType = 0x01; // default SIGHASH_ALL
            
            if (pendingPsbt?.inputsToSign) {
                inputsToSign = pendingPsbt.inputsToSign.map(i => typeof i === 'number' ? i : i.index);
                // Get sighash from first input
                const firstInput = pendingPsbt.inputsToSign[0];
                if (firstInput && firstInput.sighashTypes && firstInput.sighashTypes[0]) {
                    sighashType = firstInput.sighashTypes[0];
                }
            }
            // 🔧 FIX: Only use pendingPsbt.sighashType if it's a NUMBER (not string like 'ALL')
            if (pendingPsbt?.sighashType && typeof pendingPsbt.sighashType === 'number') {
                sighashType = pendingPsbt.sighashType;
            }
            
            console.log('   🔐 Signing PSBT 100% LOCALLY...');
            console.log('   inputsToSign:', inputsToSign);
            console.log('   sighashType: 0x' + sighashType.toString(16));
            
            try {
                const signResult = await TaprootSignerPure.signPsbtTaprootLocal(
                    mnemonic,
                    pendingPsbt.psbt,
                    {
                        inputsToSign: inputsToSign,
                        sighashType: sighashType,
                        autoFinalize: false
                    }
                );
                
                // 🗑️ Clear mnemonic immediately!
                mnemonic = null;
                console.log('   🗑️ Mnemonic cleared from memory');
                
                if (signResult && signResult.signedPsbt) {
                    response = { success: true, signedPsbt: signResult.signedPsbt };
                    console.log('   ✅ PSBT signed 100% LOCALLY! (Pure JS)');
                } else {
                    throw new Error('No signed PSBT returned');
                }
            } catch (signErr) {
                mnemonic = null;
                console.error('   ❌ Local signing failed:', signErr.message);
                throw new Error('Local signing failed: ' + signErr.message);
            }
        } else {
            // Fallback to background script (should not happen if TaprootSignerPure is loaded)
            console.warn('⚠️ TaprootSignerPure not available, using background script');
            response = await sendMessage({
                action: 'confirmPsbtSign',
                data: { password }
            });
        }
        
        if (!response.success) {
            throw new Error(response.error || 'Failed to sign PSBT');
        }
        
        // 🔒 SECURITY: Clear password field immediately
        clearAllPasswordFields();
        
        console.log('✅ PSBT signed successfully!');
        console.log('📋 Pending PSBT type (using saved):', pendingPsbt?.type);
        console.log('📋 Pending PSBT orderId:', pendingPsbt?.orderId);
        console.log('📋 Signed PSBT length:', response.signedPsbt?.length || 0);
        console.log('📋 Full pendingPsbt object:', JSON.stringify(pendingPsbt, null, 2));
        
        // 🎯 SE FOR createOffer, enviar para /api/atomic-swap/:id/sign automaticamente!
        if (pendingPsbt?.type === 'createOffer') {
            console.log('🛒 ===== CREATE OFFER FLOW (ATOMIC SWAP) =====');
            console.log('   Sending signed PSBT to /api/atomic-swap/:id/sign...');
            console.log('   Order ID:', pendingPsbt.order_id);
            console.log('   SIGHASH: NONE|ANYONECANPAY (0x82) - ARA MODEL');
            
            showLoading('Publishing atomic swap listing...');
            
            // Verificar se temos order_id
            if (!pendingPsbt.order_id) {
                throw new Error('Missing order_id from listing creation');
            }
            
            // Enviar para /api/atomic-swap/:id/sign (endpoint correto!)
            const offerResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${pendingPsbt.order_id}/sign`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    signed_psbt_base64: response.signedPsbt
                })
            });
            
            if (!offerResponse.ok) {
                const errorData = await offerResponse.json();
                throw new Error(`Failed to publish listing: ${errorData.error || 'Unknown error'}`);
            }
            
            const offerResult = await offerResponse.json();
            console.log('✅ Atomic swap listing published:', offerResult);
            console.log('   Status:', offerResult.status);
            console.log('   Order ID:', offerResult.order_id);
            
            // Limpar pending PSBT
            await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
            
            hideLoading();
            
            // Mostrar tela de sucesso (dentro do popup, padrão KrayWallet)
            const inscriptionId = pendingPsbt.inscriptionId;
            const price = pendingPsbt.price;
            showListingSuccessScreen(inscriptionId, price, offerResult.order_id);
            
        } else if (pendingPsbt?.type === 'listRuneAtomic') {
            // 🪙 LIST RUNE ATOMIC SWAP - Submit signed PSBT to runes market
            console.log('🪙 ===== LIST RUNE ATOMIC SWAP =====');
            console.log('   Sending signed PSBT to /api/runes-atomic-swap/:id/sign...');
            console.log('   Order ID:', pendingPsbt.order_id);
            console.log('   SIGHASH: SINGLE|ANYONECANPAY (0x83)');
            
            showLoading('Publishing rune listing...');
            
            // Verificar se temos order_id
            if (!pendingPsbt.order_id) {
                throw new Error('Missing order_id from listing creation');
            }
            
            // Enviar para /api/runes-atomic-swap/:id/sign
            const runeOfferResponse = await fetch(`https://kraywallet-backend.onrender.com/api/runes-atomic-swap/${pendingPsbt.order_id}/sign`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    signed_psbt_base64: response.signedPsbt
                })
            });
            
            if (!runeOfferResponse.ok) {
                const errorData = await runeOfferResponse.json();
                throw new Error(`Failed to publish rune listing: ${errorData.error || 'Unknown error'}`);
            }
            
            const runeOfferResult = await runeOfferResponse.json();
            console.log('✅ Rune atomic swap listing published:', runeOfferResult);
            console.log('   Status:', runeOfferResult.status);
            console.log('   Order ID:', runeOfferResult.order_id);
            
            // Limpar pending PSBT
            await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
            
            // Limpar storage do resultado
            await chrome.storage.local.set({ 
                psbtSignResult: { success: true, signedPsbt: response.signedPsbt } 
            });
            
            hideLoading();
            
            // Esconder explicitamente a tela de confirmação PSBT
            const confirmScreen = document.getElementById('confirm-psbt-screen');
            if (confirmScreen) {
                confirmScreen.classList.add('hidden');
            }
            
            // Remover o listScreen que foi criado por showListRuneScreen
            const listScreen = document.getElementById('list-rune-screen');
            if (listScreen) {
                listScreen.remove();
                console.log('✅ List rune screen removed');
            }
            
            // Mostrar sucesso e voltar para wallet
            showScreen('wallet');
            
            // Mostrar notificação de sucesso mais visível
            const runeName = pendingPsbt.runeName || 'Rune';
            showNotification(`🎉 ${runeName} listed for sale! Check Market tab.`, 'success');
            
            // Recarregar dados
            await loadWalletData();
            
        } else if (pendingPsbt?.type === 'buyAtomicSwap') {
            console.log('🛒 ===== BUY ATOMIC SWAP FLOW =====');
            console.log('   Sending signed PSBT to /api/atomic-swap/:id/broadcast...');
            console.log('   Order ID:', pendingPsbt.orderId);
            console.log('   Buyer Address:', pendingPsbt.buyerAddress);
            
            showLoading('Broadcasting purchase (with consensus validation)...');
            
            // 🔍 DEBUG: Log what we're sending
            console.log('📤 Sending to GUARDIAN BUILD broadcast:');
            console.log(`   signed_psbt_base64: ${response.signedPsbt?.length || 0} chars`);
            console.log(`   buyer_address: ${pendingPsbt.buyerAddress}`);
            console.log(`   seller_signature_hex: ${pendingPsbt.sellerSignatureHex ? `${pendingPsbt.sellerSignatureHex.length} chars` : 'NOT FOUND!'}`);
            console.log(`   seller_tx_context: ${pendingPsbt.sellerTxContext ? 'PROVIDED' : 'NOT FOUND!'}`);
            console.log(`   model: ${pendingPsbt.model || 'GUARDIAN_BUILD'}`);
            
            // Enviar para /api/atomic-swap/:orderId/broadcast (Guardians will BUILD final TX!)
            const broadcastResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${pendingPsbt.orderId}/broadcast`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    signed_psbt_base64: response.signedPsbt,
                    buyer_address: pendingPsbt.buyerAddress,
                    seller_signature_hex: pendingPsbt.sellerSignatureHex,
                    seller_tx_context: pendingPsbt.sellerTxContext // 🔐 Full seller TX context
                })
            });
            
            const broadcastResult = await broadcastResponse.json();
            
            if (!broadcastResponse.ok || !broadcastResult.success) {
                // Check if it was a consensus rejection
                if (broadcastResult.consensus && !broadcastResult.consensus.approved) {
                    console.error('❌ Consensus rejected:', broadcastResult.consensus.errors);
                    throw new Error(`Consensus validation failed: ${broadcastResult.consensus.errors.join(', ')}`);
                }
                throw new Error(broadcastResult.error || 'Failed to broadcast purchase');
            }
            
            console.log('✅ Purchase broadcast!', broadcastResult);
            console.log('   TXID:', broadcastResult.txid);
            console.log('   Consensus:', broadcastResult.consensus);
            
            // Limpar pending PSBT
            await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
            
            hideLoading();
            
            // 🔄 IMEDIATAMENTE voltar para wallet (antes da notificação)
            showScreen('wallet');
            
            // Mostrar sucesso com detalhes do consenso
            const consensusInfo = broadcastResult.consensus ? 
                `\n🗳️ Validated by ${broadcastResult.consensus.votes.approvals}/${broadcastResult.consensus.votes.totalVotes} validators` : '';
            
            showNotification(`✅ Purchase successful!${consensusInfo}\n\n📋 TXID: ${broadcastResult.txid.slice(0, 16)}...\n\n🎨 The inscription is now yours!`, 'success');
            
            // Recarregar dados da wallet
            await loadWalletData();
            
        } else if (pendingPsbt?.type === 'createBuyNowListing') {
            // 🏷️ CREATE BUY NOW LISTING - Seller signs to activate listing (KrayWallet Model)
            console.log('🏷️ ===== CREATE BUY NOW LISTING (KRAYWALLET MODEL) =====');
            console.log('   Confirming listing with signed PSBT...');
            console.log('   Order ID:', pendingPsbt.orderId);
            console.log('   Inscription ID:', pendingPsbt.inscriptionId);
            console.log('   Price:', pendingPsbt.priceSats, 'sats');
            console.log('   Signed PSBT length:', response.signedPsbt?.length || 0);
            console.log('   SIGHASH: SINGLE|ANYONECANPAY (0x83)');
            
            showLoading('Activating listing...');
            
            try {
                // Send signed PSBT to confirm the listing
                const requestBody = {
                    inscription_id: pendingPsbt.inscriptionId,
                    price_sats: pendingPsbt.priceSats,
                    order_id: pendingPsbt.orderId,
                    seller_signed_psbt: response.signedPsbt
                };
                console.log('📤 Sending to backend:', JSON.stringify(requestBody, null, 2));
                
                const confirmResponse = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/list', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(requestBody)
                });
                
                const confirmResult = await confirmResponse.json();
                console.log('📥 Backend response:', confirmResult);
                
                if (!confirmResponse.ok || !confirmResult.success) {
                    console.error('❌ Backend error:', confirmResult);
                    throw new Error(confirmResult.error || 'Failed to activate listing');
                }
                
                console.log('✅ Listing is LIVE!', confirmResult);
                
                // Clear pending PSBT
                await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
                
                hideLoading();
                
                // 🎉 Show beautiful success modal (same style as Buy Now success)
                const inscriptionId = pendingPsbt.inscriptionId;
                const price = pendingPsbt.priceSats;
                
                showListingSuccessModal(inscriptionId, price);
                
            } catch (confirmError) {
                console.error('❌ Error confirming listing:', confirmError);
                hideLoading();
                showNotification('❌ ' + confirmError.message, 'error');
                showScreen('wallet');
            }
            
        } else if (pendingPsbt?.type === 'buyNow') {
            // 🛒 BUY NOW FLOW (KRAYWALLET MODEL) - Buyer signs → INSTANT BROADCAST!
            console.log('🛒 ===== BUY NOW FLOW (KRAYWALLET MODEL) =====');
            console.log('   Broadcasting purchase IMMEDIATELY...');
            console.log('   Order ID:', pendingPsbt.orderId);
            console.log('   Purchase ID:', pendingPsbt.purchaseId);
            console.log('   Seller signature: PRE-SIGNED ✓');
            
            showLoading('Broadcasting purchase...');
            
            // Send signed PSBT to confirm endpoint → INSTANT BROADCAST!
            const confirmResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${pendingPsbt.orderId}/confirm`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    buyer_signed_psbt: response.signedPsbt,
                    purchase_id: pendingPsbt.purchaseId
                })
            });
            
            const confirmResult = await confirmResponse.json();
            
            if (!confirmResponse.ok || !confirmResult.success) {
                throw new Error(confirmResult.error || 'Failed to broadcast purchase');
            }
            
            console.log('✅ Purchase BROADCAST!', confirmResult);
            console.log('   TXID:', confirmResult.txid);
            
            // Clear pending PSBT
            await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
            
            hideLoading();
            
            // Show success modal with TXID!
            showBuyNowSuccessModal(confirmResult.txid, pendingPsbt.orderId);
            
            await loadWalletData();
            
        } else if (pendingPsbt?.type === 'acceptBuyNow') {
            // 🎉 ACCEPT BUY NOW FLOW - Seller signs and broadcasts
            console.log('🎉 ===== ACCEPT BUY NOW FLOW =====');
            console.log('   Broadcasting sale...');
            console.log('   Order ID:', pendingPsbt.orderId);
            console.log('   Purchase ID:', pendingPsbt.purchaseId);
            
            showLoading('Broadcasting sale...');
            
            // Send signed PSBT to accept endpoint (broadcasts automatically!)
            const acceptResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${pendingPsbt.orderId}/accept`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    seller_signed_psbt: response.signedPsbt,
                    purchase_id: pendingPsbt.purchaseId
                })
            });
            
            const acceptResult = await acceptResponse.json();
            
            if (!acceptResponse.ok || !acceptResult.success) {
                throw new Error(acceptResult.error || 'Failed to broadcast sale');
            }
            
            console.log('✅ Sale broadcast!', acceptResult);
            console.log('   TXID:', acceptResult.txid);
            
            // Clear pending PSBT
            await sendMessage({ action: 'cancelPsbtSign', data: { cancelled: false } });
            
            hideLoading();
            
            // Show success modal
            showBuyNowSuccessModal(acceptResult.txid, pendingPsbt.orderId);
            
        } else {
            // 🎯 FLUXO NORMAL (outros): Salvar resultado no storage
            // Este é o fluxo para requisições EXTERNAS (KrayScan, sites, etc)
            console.log('🛍️ Default flow (external request): saving result to storage');
            await chrome.storage.local.set({
                psbtSignResult: response
            });
            
            // Esconder tela de confirmação
            const confirmScreen = document.getElementById('confirm-psbt-screen');
            if (confirmScreen) {
                confirmScreen.classList.add('hidden');
            }
            
            // Mostrar sucesso brevemente
            showNotification('✅ Transaction signed!', 'success');
            
            // Fechar popup após 500ms (para mostrar a notificação)
            console.log('🪟 External request complete, closing popup in 500ms...');
            setTimeout(() => {
                window.close();
            }, 500);
        }
        
    } catch (error) {
        console.error('❌ Error signing PSBT:', error);
        showNotification('❌ ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

async function handlePsbtCancel() {
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    console.log('❌ USER CANCELLED - CLEANING EVERYTHING');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    
    // 🔒 SECURITY: Clear password field immediately
    clearAllPasswordFields();
    
    // 0. 🗑️ If this was a Rune listing, delete from backend to free UTXO
    try {
        const pendingResponse = await sendMessage({ action: 'getPendingPsbt' });
        const pendingPsbt = pendingResponse?.pending;
        
        if (pendingPsbt?.type === 'listRuneAtomic' && pendingPsbt?.order_id) {
            console.log('🗑️ Cancelling PENDING rune listing from backend...');
            console.log('   Order ID:', pendingPsbt.order_id);
            
            const walletInfo = await sendMessage({ action: 'getWalletInfo' });
            const sellerAddress = walletInfo?.data?.address || walletInfo?.address;
            
            if (sellerAddress) {
                const deleteRes = await fetch(`https://kraywallet-backend.onrender.com/api/runes-atomic-swap/${pendingPsbt.order_id}`, {
                    method: 'DELETE',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ seller_address: sellerAddress })
                });
                
                if (deleteRes.ok) {
                    console.log('✅ PENDING listing deleted from backend - UTXO freed');
                } else {
                    console.warn('⚠️  Could not delete PENDING listing (may not exist yet)');
                }
            }
        }
    } catch (error) {
        console.warn('⚠️  Error cleaning up pending listing:', error.message);
    }
    
    // 1. Notificar background que foi cancelado (limpa memória)
    try {
        await chrome.runtime.sendMessage({
            action: 'cancelPsbtSign',
            data: { cancelled: true }
        });
        console.log('✅ Background notified');
    } catch (error) {
        console.warn('⚠️  Could not notify background:', error);
    }
    
    // 2. 🔥 LIMPAR TUDO DO STORAGE (agressivo)
    await chrome.storage.local.remove([
        'psbtSignResult',
        'pendingPsbtRequest',
        'pendingMarketListing'
    ]);
    console.log('✅ Storage cleaned');
    
    // 3. Resetar flag de criação de listing
    if (typeof isCreatingListing !== 'undefined') {
        isCreatingListing = false;
        console.log('✅ isCreatingListing reset');
    }
    
    showNotification('❌ Transaction cancelled', 'info');
    console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
    
    // Verificar se popup é uma janela separada ou popup da extensão
    const isStandaloneWindow = window.opener === null && window.location.href.includes('popup.html');
    
    if (isStandaloneWindow) {
        // Se for popup da extensão, voltar para tela principal
        console.log('📱 Popup window, returning to main screen...');
        showScreen('wallet');
    } else {
        // Se for janela separada, fechar
        console.log('🪟 Separate window, closing...');
        setTimeout(() => {
            window.close();
        }, 100);
    }
}

/**
 * Wait for PSBT Sign Result (via chrome.storage)
 */
function waitForPsbtSignResult() {
    return new Promise((resolve, reject) => {
        console.log('⏳ Waiting for PSBT sign result...');
        
        const timeout = setTimeout(() => {
            console.error('⏱️  Timeout waiting for signature');
            reject(new Error('Signature timeout'));
        }, 120000); // 2 minutos
        
        // Listener para mudanças no storage
        const listener = (changes, namespace) => {
            if (namespace === 'local' && changes.psbtSignResult) {
                clearTimeout(timeout);
                chrome.storage.onChanged.removeListener(listener);
                
                const result = changes.psbtSignResult.newValue;
                console.log('✅ PSBT sign result received:', result);
                
                // Limpar resultado do storage
                chrome.storage.local.remove('psbtSignResult');
                
                resolve(result);
            }
        };
        
        chrome.storage.onChanged.addListener(listener);
    });
}

// Export Wallet
async function handleExportWallet() {
    showNotification('Export wallet feature coming soon!', 'info');
}

// ========================================
// DEX - LIQUIDITY POOLS
// ========================================

/**
 * Carregar e exibir liquidity pools
 */
async function loadLiquidityPools() {
    console.log('🏊 Loading liquidity pools...');
    
    const container = document.getElementById('pools-list');
    if (!container) {
        console.error('❌ Pools container not found');
        return;
    }

    try {
        // Buscar pools do backend
        const response = await fetch('https://kraywallet-backend.onrender.com/api/dex/pools?sortBy=tvl');
        const data = await response.json();

        if (!data.success) {
            throw new Error(data.error || 'Failed to load pools');
        }

        const pools = data.pools || [];
        console.log(`✅ Loaded ${pools.length} pools`);

        if (pools.length === 0) {
            container.innerHTML = `
                <div class="empty-state" style="padding: 40px; text-align: center;">
                    <span style="font-size: 48px;">🏊</span>
                    <p style="margin-top: 16px; color: #888;">No liquidity pools yet</p>
                    <p style="margin-top: 8px; color: #666; font-size: 14px;">Create the first pool and start earning fees!</p>
                </div>
            `;
            return;
        }

        // Renderizar pools
        container.innerHTML = pools.map(pool => {
            // Determinar imagem da pool
            let poolImageHTML;
            if (pool.use_inscription && pool.pool_image) {
                // Usar inscription
                poolImageHTML = `
                    <div style="position: relative;">
                        <img src="${pool.pool_image}" 
                             style="width: 48px; height: 48px; border-radius: 8px; object-fit: cover; border: 2px solid #f59e0b;" 
                             onerror="this.style.display='none'; this.nextElementSibling.style.display='block';" />
                        <div style="display: none; width: 48px; height: 48px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 8px; display: flex; align-items: center; justify-content: center; font-size: 24px;">💎</div>
                        <div style="position: absolute; bottom: -4px; right: -4px; background: #f59e0b; border-radius: 50%; width: 20px; height: 20px; display: flex; align-items: center; justify-content: center; font-size: 12px; border: 2px solid #2a2a2a;">💎</div>
                    </div>
                `;
            } else if (pool.pool_image) {
                // Usar URL normal
                poolImageHTML = `<img src="${pool.pool_image}" style="width: 48px; height: 48px; border-radius: 50%; object-fit: cover;" onerror="this.style.display='none'; this.nextElementSibling.style.display='inline';" /><span style="display: none; font-size: 32px;">🏊</span>`;
            } else {
                // Emoji padrão
                poolImageHTML = '<span style="font-size: 32px;">🏊</span>';
            }

            return `
            <div class="pool-card" data-pool-id="${pool.id}" style="background: #2a2a2a; border-radius: 12px; padding: 16px; margin-bottom: 12px; cursor: pointer; transition: all 0.2s;">
                <div style="display: flex; align-items: center; justify-content: space-between; margin-bottom: 12px;">
                    <div style="display: flex; align-items: center; gap: 12px;">
                        ${poolImageHTML}
                        <div>
                            <div style="display: flex; align-items: center; gap: 6px;">
                                <span style="font-weight: 600; font-size: 16px;">${pool.pool_name}</span>
                                ${pool.use_inscription ? '<span style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; font-size: 10px; padding: 2px 6px; border-radius: 4px; font-weight: 600;">ORDINAL</span>' : ''}
                            </div>
                            <div style="font-size: 12px; color: #888;">${pool.rune_a_name} / ${pool.rune_b_name || 'BTC'}</div>
                            ${pool.pool_inscription_number ? `<div style="font-size: 10px; color: #f59e0b; margin-top: 2px;">Inscription #${pool.pool_inscription_number}</div>` : ''}
                        </div>
                    </div>
                    <div style="text-align: right;">
                        <div style="font-size: 18px; font-weight: 700; color: #10b981;">${pool.apr}%</div>
                        <div style="font-size: 12px; color: #888;">APR</div>
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 12px; padding-top: 12px; border-top: 1px solid #333;">
                    <div>
                        <div style="font-size: 12px; color: #888; margin-bottom: 4px;">TVL</div>
                        <div style="font-size: 14px; font-weight: 600;">${(pool.tvl / 100000000).toFixed(4)} BTC</div>
                    </div>
                    <div>
                        <div style="font-size: 12px; color: #888; margin-bottom: 4px;">Volume 24h</div>
                        <div style="font-size: 14px; font-weight: 600;">${(pool.volume_24h / 100000000).toFixed(4)} BTC</div>
                    </div>
                    <div>
                        <div style="font-size: 12px; color: #888; margin-bottom: 4px;">Fee</div>
                        <div style="font-size: 14px; font-weight: 600;">${pool.fee_percentage}</div>
                    </div>
                    <div>
                        <div style="font-size: 12px; color: #888; margin-bottom: 4px;">Swaps</div>
                        <div style="font-size: 14px; font-weight: 600;">${pool.swap_count || 0}</div>
                    </div>
                </div>

                <button class="swap-in-pool-btn" data-pool-id="${pool.id}" style="width: 100%; margin-top: 12px; padding: 10px; background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); border: none; border-radius: 8px; color: white; font-weight: 600; cursor: pointer;">
                    Swap
                </button>
            </div>
        `;
        }).join('');

        // Adicionar event listeners para cada pool
        container.querySelectorAll('.pool-card').forEach(card => {
            card.addEventListener('click', (e) => {
                if (e.target.classList.contains('swap-in-pool-btn')) {
                    e.stopPropagation();
                    const poolId = e.target.dataset.poolId;
                    showSwapScreen(poolId);
                } else {
                    const poolId = card.dataset.poolId;
                    showPoolDetails(poolId);
                }
            });
        });

    } catch (error) {
        console.error('❌ Error loading pools:', error);
        container.innerHTML = `
            <div class="empty-state" style="padding: 40px; text-align: center;">
                <span style="font-size: 48px;">❌</span>
                <p style="margin-top: 16px; color: #ff4444;">Failed to load pools</p>
                <p style="margin-top: 8px; color: #666; font-size: 14px;">${error.message}</p>
            </div>
        `;
    }
}

/**
 * Mostrar tela de criação de pool
 */
async function showCreatePoolScreen() {
    console.log('🏊 ===== CREATE POOL SCREEN OPENING =====');
    console.log('   📍 User wants to create a new liquidity pool');
    console.log('   🔄 Loading user runes dynamically...');
    
    const existingScreen = document.querySelector('.create-pool-screen');
    if (existingScreen) {
        console.log('   🗑️ Removing existing screen');
        existingScreen.remove();
    }

    const screen = document.createElement('div');
    screen.className = 'create-pool-screen';
    screen.style.cssText = 'position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: #1a1a1a; z-index: 10000; overflow-y: auto;';

    screen.innerHTML = `
        <div style="padding: 20px;">
            <div style="display: flex; align-items: center; gap: 16px; margin-bottom: 24px;">
                <button class="btn-back" id="create-pool-back-btn">←</button>
                <h2 style="margin: 0; font-size: 20px;">🏊 Create Liquidity Pool</h2>
            </div>

            <div style="background: linear-gradient(135deg, #10b981 0%, #059669 100%); border-radius: 12px; padding: 20px; margin-bottom: 24px;">
                <h3 style="margin: 0 0 8px 0; font-size: 16px;">💰 Earn Trading Fees</h3>
                <p style="margin: 0; font-size: 14px; opacity: 0.9;">Provide liquidity and earn a share of all trading fees in your pool</p>
            </div>

            <form id="create-pool-form" style="display: flex; flex-direction: column; gap: 16px;">
                <div>
                    <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #888;">Pool Name</label>
                    <input type="text" id="pool-name" placeholder="e.g., DOG/BTC Official Pool" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px;">
                </div>

                <div style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); border-radius: 12px; padding: 16px; margin-bottom: 8px;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 12px;">
                        <span style="font-size: 24px;">💎</span>
                        <div>
                            <h4 style="margin: 0; font-size: 14px; font-weight: 600;">Use Your Ordinal Inscription!</h4>
                            <p style="margin: 4px 0 0 0; font-size: 12px; opacity: 0.9;">Give value to your NFT by making it represent your pool</p>
                        </div>
                    </div>
                    
                    <label style="display: flex; align-items: center; gap: 8px; cursor: pointer; background: rgba(0,0,0,0.2); padding: 10px; border-radius: 8px;">
                        <input type="checkbox" id="use-inscription" style="width: 18px; height: 18px; cursor: pointer;">
                        <span style="font-size: 14px; font-weight: 500;">🖼️ Use My Inscription as Pool Image</span>
                    </label>
                </div>

                <div id="inscription-inputs" style="display: none; background: #2a2a2a; border-radius: 12px; padding: 16px; border: 2px solid #f59e0b;">
                    <div>
                        <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #f59e0b; font-weight: 600;">🖼️ Select Your Inscription</label>
                        
                        <!-- Campo de Busca -->
                        <div style="position: relative; margin-bottom: 12px;">
                            <input type="text" id="inscription-search" placeholder="🔍 Search by inscription number or ID..." style="width: 100%; padding: 12px 12px 12px 36px; background: #1a1a1a; border: 1px solid #f59e0b; border-radius: 8px; color: #fff; font-size: 14px;">
                            <span style="position: absolute; left: 12px; top: 50%; transform: translateY(-50%); font-size: 16px;">🔍</span>
                        </div>
                        
                        <!-- Info sobre quantidade -->
                        <div id="inscription-count-info" style="margin-bottom: 8px; font-size: 12px; color: #888; display: none;">
                            Showing <span id="inscription-shown-count">0</span> of <span id="inscription-total-count">0</span> inscriptions
                        </div>
                        
                        <!-- Dropdown Dinâmico de Inscriptions -->
                        <select id="inscription-select" size="6" style="width: 100%; padding: 8px; background: #1a1a1a; border: 1px solid #f59e0b; border-radius: 8px; color: #fff; font-size: 14px; cursor: pointer; max-height: 240px; overflow-y: auto;">
                            <option value="">Loading your inscriptions...</option>
                        </select>
                        
                        <!-- Preview da Inscription Selecionada -->
                        <div id="inscription-preview" style="display: none; margin-top: 12px; padding: 12px; background: rgba(245, 158, 11, 0.1); border: 1px solid #f59e0b; border-radius: 8px;">
                            <div style="display: flex; gap: 12px; align-items: center;">
                                <div id="inscription-preview-image" style="width: 80px; height: 80px; border-radius: 8px; overflow: hidden; border: 2px solid #f59e0b; background: #1a1a1a; display: flex; align-items: center; justify-content: center;">
                                    <div class="spinner" style="width: 20px; height: 20px;"></div>
                                </div>
                                <div style="flex: 1;">
                                    <div id="inscription-preview-number" style="font-weight: 600; color: #f59e0b; font-size: 14px;">#0</div>
                                    <div id="inscription-preview-id" style="font-size: 11px; color: #888; margin-top: 4px; word-break: break-all;">ID: ...</div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Hidden inputs para o form -->
                        <input type="hidden" id="pool-inscription-id" />
                        <input type="hidden" id="pool-inscription-number" />
                    </div>
                </div>

                <div id="image-url-input">
                    <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #888;">
                        <span>Pool Image URL</span>
                        <span style="font-size: 12px; opacity: 0.7;"> (or use inscription above)</span>
                    </label>
                    <input type="url" id="pool-image" placeholder="https://..." style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px;">
                </div>

                <div>
                    <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #888;">First Token</label>
                    
                    <!-- Dropdown de Runes -->
                    <select id="rune-a-select" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px; cursor: pointer;">
                        <option value="">Loading your runes...</option>
                    </select>
                    
                    <!-- Info da Rune Selecionada -->
                    <div id="rune-a-info" style="display: none; margin-top: 8px; padding: 12px; background: rgba(16, 185, 129, 0.1); border: 1px solid #10b981; border-radius: 8px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <div id="rune-a-name-display" style="font-weight: 600; color: #10b981; font-size: 14px;"></div>
                                <div id="rune-a-id-display" style="font-size: 12px; color: #888; margin-top: 2px;"></div>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-size: 12px; color: #888;">Your Balance</div>
                                <div id="rune-a-balance-display" style="font-weight: 700; color: #10b981; font-size: 16px;">0</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Hidden inputs para o form -->
                    <input type="hidden" id="rune-a" />
                    <input type="hidden" id="rune-a-name" />
                </div>

                <div>
                    <label style="display: flex; align-items: center; gap: 8px; margin-bottom: 8px; font-size: 14px;">
                        <input type="checkbox" id="is-btc-pair" checked>
                        <span style="color: #888;">Pair with BTC (leave unchecked for Rune/Rune pair)</span>
                    </label>
                </div>

                <div id="rune-b-container" style="display: none;">
                    <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #888;">Second Token</label>
                    
                    <!-- Dropdown de Runes -->
                    <select id="rune-b-select" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px; cursor: pointer;">
                        <option value="">Loading your runes...</option>
                    </select>
                    
                    <!-- Info da Rune Selecionada -->
                    <div id="rune-b-info" style="display: none; margin-top: 8px; padding: 12px; background: rgba(16, 185, 129, 0.1); border: 1px solid #10b981; border-radius: 8px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <div id="rune-b-name-display" style="font-weight: 600; color: #10b981; font-size: 14px;"></div>
                                <div id="rune-b-id-display" style="font-size: 12px; color: #888; margin-top: 2px;"></div>
                            </div>
                            <div style="text-align: right;">
                                <div style="font-size: 12px; color: #888;">Your Balance</div>
                                <div id="rune-b-balance-display" style="font-weight: 700; color: #10b981; font-size: 16px;">0</div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Hidden inputs -->
                    <input type="hidden" id="rune-b" />
                    <input type="hidden" id="rune-b-name" />
                </div>

                <div>
                    <label style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                        <span style="font-size: 14px; color: #888;">Initial Amount (Token A)</span>
                        <button type="button" id="max-amount-a-btn" style="padding: 4px 8px; background: #3b82f6; border: none; border-radius: 4px; color: white; font-size: 12px; cursor: pointer; display: none;">MAX</button>
                    </label>
                    <input type="number" id="initial-amount-a" placeholder="0" min="1" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px;">
                    <div id="amount-a-warning" style="display: none; margin-top: 6px; padding: 8px; background: rgba(239, 68, 68, 0.1); border: 1px solid #ef4444; border-radius: 6px; font-size: 12px; color: #ef4444;">
                        ⚠️ Amount exceeds your balance!
                    </div>
                </div>

                <div>
                    <label style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                        <span style="font-size: 14px; color: #888;">Initial Amount (Token B / BTC sats)</span>
                        <button type="button" id="max-amount-b-btn" style="padding: 4px 8px; background: #3b82f6; border: none; border-radius: 4px; color: white; font-size: 12px; cursor: pointer; display: none;">MAX</button>
                    </label>
                    <input type="number" id="initial-amount-b" placeholder="0" min="1" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px;">
                    <div id="amount-b-warning" style="display: none; margin-top: 6px; padding: 8px; background: rgba(239, 68, 68, 0.1); border: 1px solid #ef4444; border-radius: 6px; font-size: 12px; color: #ef4444;">
                        ⚠️ Amount exceeds your balance!
                    </div>
                </div>

                <div>
                    <label style="display: block; margin-bottom: 8px; font-size: 14px; color: #888;">Fee Rate (%)</label>
                    <select id="fee-rate" style="width: 100%; padding: 12px; background: #2a2a2a; border: 1px solid #444; border-radius: 8px; color: #fff; font-size: 14px;">
                        <option value="5">0.05%</option>
                        <option value="10">0.10%</option>
                        <option value="30" selected>0.30%</option>
                        <option value="100">1.00%</option>
                    </select>
                </div>

                <button type="submit" style="padding: 16px; background: linear-gradient(135deg, #10b981 0%, #059669 100%); border: none; border-radius: 8px; color: white; font-weight: 600; font-size: 16px; cursor: pointer;">
                    🏊 Create Pool
                </button>
            </form>
        </div>
    `;

    document.body.appendChild(screen);

    // ✅ Event listener para botão de voltar (CSP-safe)
    const backBtn = screen.querySelector('#create-pool-back-btn');
    if (backBtn) {
        backBtn.addEventListener('click', () => {
            screen.remove();
        });
    }

    // 🔥 CARREGAR RUNES DO USUÁRIO
    await loadUserRunesForPool(screen);

    // Toggle inscription inputs
    const useInscription = screen.querySelector('#use-inscription');
    const inscriptionInputs = screen.querySelector('#inscription-inputs');
    const imageUrlInput = screen.querySelector('#image-url-input');
    
    useInscription.addEventListener('change', () => {
        if (useInscription.checked) {
            inscriptionInputs.style.display = 'block';
            imageUrlInput.style.opacity = '0.5';
            imageUrlInput.style.pointerEvents = 'none';
        } else {
            inscriptionInputs.style.display = 'none';
            imageUrlInput.style.opacity = '1';
            imageUrlInput.style.pointerEvents = 'auto';
        }
    });

    // Toggle rune-b inputs based on checkbox
    const isBtcPair = screen.querySelector('#is-btc-pair');
    const runeBContainer = screen.querySelector('#rune-b-container');
    
    isBtcPair.addEventListener('change', () => {
        runeBContainer.style.display = isBtcPair.checked ? 'none' : 'block';
    });

    // Handle form submission
    screen.querySelector('#create-pool-form').addEventListener('submit', handleCreatePool);
}

/**
 * Carregar runes do usuário para o form de criar pool
 */
async function loadUserRunesForPool(screen) {
    console.log('🔥 ===== LOADING USER RUNES FOR POOL =====');
    console.log('   📊 Fetching wallet info...');
    
    try {
        // Buscar endereço do usuário
        const walletInfo = await sendMessage({ action: 'getWalletInfo' });
        if (!walletInfo.success || !walletInfo.data || !walletInfo.data.address) {
            console.error('❌ Wallet not found or not unlocked');
            throw new Error('Wallet not found');
        }

        const userAddress = walletInfo.data.address;
        console.log(`   ✅ User address: ${userAddress}`);

        // 🖼️ CARREGAR INSCRIPTIONS DO USUÁRIO (em paralelo com runes)
        loadUserInscriptionsForPool(screen, userAddress);

        // Buscar runes do usuário
        const response = await sendMessage({ 
            action: 'getRunes',
            address: userAddress 
        });

        if (!response.success || !response.runes) {
            throw new Error('Failed to load runes');
        }

        const runes = response.runes;
        console.log(`   ✅ Loaded ${runes.length} runes for pool`);
        console.log(`   📋 Runes:`, runes.map(r => `${r.name} (${r.amount})`).join(', '));

        // Preencher dropdown de Rune A
        const runeASelect = screen.querySelector('#rune-a-select');
        runeASelect.innerHTML = '<option value="">Select a rune...</option>';
        
        runes.forEach(rune => {
            const option = document.createElement('option');
            option.value = JSON.stringify({
                id: rune.runeId,
                name: rune.name,
                symbol: rune.symbol,
                balance: rune.amount
            });
            option.textContent = `${rune.name} ${rune.symbol} (${rune.amount.toLocaleString()})`;
            runeASelect.appendChild(option);
        });

        // Preencher dropdown de Rune B (mesmas runes)
        const runeBSelect = screen.querySelector('#rune-b-select');
        runeBSelect.innerHTML = '<option value="">Select a rune...</option>';
        
        runes.forEach(rune => {
            const option = document.createElement('option');
            option.value = JSON.stringify({
                id: rune.runeId,
                name: rune.name,
                symbol: rune.symbol,
                balance: rune.amount
            });
            option.textContent = `${rune.name} ${rune.symbol} (${rune.amount.toLocaleString()})`;
            runeBSelect.appendChild(option);
        });

        // Event listener para Rune A
        runeASelect.addEventListener('change', () => {
            if (runeASelect.value) {
                const runeData = JSON.parse(runeASelect.value);
                
                // Preencher hidden inputs
                screen.querySelector('#rune-a').value = runeData.id;
                screen.querySelector('#rune-a-name').value = runeData.name;
                
                // Mostrar info
                const infoDiv = screen.querySelector('#rune-a-info');
                infoDiv.style.display = 'block';
                screen.querySelector('#rune-a-name-display').textContent = `${runeData.name} ${runeData.symbol}`;
                screen.querySelector('#rune-a-id-display').textContent = `ID: ${runeData.id}`;
                screen.querySelector('#rune-a-balance-display').textContent = runeData.balance.toLocaleString();
                
                // Mostrar botão MAX
                const maxBtn = screen.querySelector('#max-amount-a-btn');
                maxBtn.style.display = 'block';
                maxBtn.onclick = () => {
                    screen.querySelector('#initial-amount-a').value = runeData.balance;
                };
                
                // Validação em tempo real
                const amountInput = screen.querySelector('#initial-amount-a');
                const warningDiv = screen.querySelector('#amount-a-warning');
                
                amountInput.addEventListener('input', () => {
                    const amount = parseInt(amountInput.value);
                    if (amount > runeData.balance) {
                        warningDiv.style.display = 'block';
                        amountInput.style.borderColor = '#ef4444';
                    } else {
                        warningDiv.style.display = 'none';
                        amountInput.style.borderColor = '#444';
                    }
                });
            } else {
                screen.querySelector('#rune-a-info').style.display = 'none';
                screen.querySelector('#max-amount-a-btn').style.display = 'none';
            }
        });

        // Event listener para Rune B
        runeBSelect.addEventListener('change', () => {
            if (runeBSelect.value) {
                const runeData = JSON.parse(runeBSelect.value);
                
                // Preencher hidden inputs
                screen.querySelector('#rune-b').value = runeData.id;
                screen.querySelector('#rune-b-name').value = runeData.name;
                
                // Mostrar info
                const infoDiv = screen.querySelector('#rune-b-info');
                infoDiv.style.display = 'block';
                screen.querySelector('#rune-b-name-display').textContent = `${runeData.name} ${runeData.symbol}`;
                screen.querySelector('#rune-b-id-display').textContent = `ID: ${runeData.id}`;
                screen.querySelector('#rune-b-balance-display').textContent = runeData.balance.toLocaleString();
                
                // Mostrar botão MAX
                const maxBtn = screen.querySelector('#max-amount-b-btn');
                maxBtn.style.display = 'block';
                maxBtn.onclick = () => {
                    screen.querySelector('#initial-amount-b').value = runeData.balance;
                };
                
                // Validação em tempo real
                const amountInput = screen.querySelector('#initial-amount-b');
                const warningDiv = screen.querySelector('#amount-b-warning');
                
                amountInput.addEventListener('input', () => {
                    const amount = parseInt(amountInput.value);
                    if (amount > runeData.balance) {
                        warningDiv.style.display = 'block';
                        amountInput.style.borderColor = '#ef4444';
                    } else {
                        warningDiv.style.display = 'none';
                        amountInput.style.borderColor = '#444';
                    }
                });
            } else {
                screen.querySelector('#rune-b-info').style.display = 'none';
                screen.querySelector('#max-amount-b-btn').style.display = 'none';
            }
        });

        console.log('✅ Rune dropdowns configured!');

    } catch (error) {
        console.error('❌ Error loading runes:', error);
        const runeASelect = screen.querySelector('#rune-a-select');
        runeASelect.innerHTML = '<option value="">Error loading runes</option>';
    }
}

/**
 * Carregar inscriptions do usuário para o form de criar pool
 */
async function loadUserInscriptionsForPool(screen, userAddress) {
    console.log('🖼️ Loading user inscriptions for pool creation...');
    
    try {
        // Buscar inscriptions do usuário
        const response = await sendMessage({ 
            action: 'getInscriptions',
            data: { address: userAddress }
        });

        console.log('📦 Inscriptions response:', response);

        if (!response || !response.success) {
            throw new Error('Failed to load inscriptions');
        }

        const allInscriptions = response.inscriptions || [];
        console.log(`✅ Loaded ${allInscriptions.length} inscriptions for pool`);

        const inscriptionSelect = screen.querySelector('#inscription-select');
        const inscriptionSearch = screen.querySelector('#inscription-search');
        const countInfo = screen.querySelector('#inscription-count-info');
        const shownCount = screen.querySelector('#inscription-shown-count');
        const totalCount = screen.querySelector('#inscription-total-count');
        
        if (allInscriptions.length === 0) {
            inscriptionSelect.innerHTML = '<option value="">You have no inscriptions yet</option>';
            return;
        }

        // Mostrar info de quantidade
        totalCount.textContent = allInscriptions.length;
        countInfo.style.display = 'block';

        // Criar um container customizado para o dropdown (com imagens)
        const inscriptionListContainer = document.createElement('div');
        inscriptionListContainer.id = 'inscription-list-container';
        inscriptionListContainer.style.cssText = 'max-height: 280px; overflow-y: auto; border: 1px solid #f59e0b; border-radius: 8px; background: #1a1a1a;';
        
        // Substituir o select por um container custom
        inscriptionSelect.style.display = 'none';
        inscriptionSelect.parentNode.insertBefore(inscriptionListContainer, inscriptionSelect);

        // Função para renderizar inscriptions com imagens (limitado a 12)
        const renderInscriptions = (inscriptionsToShow) => {
            inscriptionListContainer.innerHTML = '';
            
            const limited = inscriptionsToShow.slice(0, 12);
            shownCount.textContent = limited.length;
            
            if (limited.length === 0) {
                inscriptionListContainer.innerHTML = '<div style="padding: 20px; text-align: center; color: #888;">No inscriptions found</div>';
                return;
            }
            
            limited.forEach(inscription => {
                const item = document.createElement('div');
                item.style.cssText = 'display: flex; align-items: center; gap: 12px; padding: 10px; cursor: pointer; border-bottom: 1px solid #333; transition: background 0.2s;';
                item.dataset.inscription = JSON.stringify({
                    id: inscription.id,
                    inscriptionId: inscription.inscriptionId,
                    inscriptionNumber: inscription.inscriptionNumber,
                    contentType: inscription.contentType
                });
                
                // Thumbnail da inscription
                const thumbnail = document.createElement('div');
                thumbnail.style.cssText = 'width: 40px; height: 40px; border-radius: 6px; overflow: hidden; border: 1px solid #f59e0b; background: #2a2a2a; display: flex; align-items: center; justify-content: center; flex-shrink: 0;';
                
                const inscriptionId = inscription.inscriptionId || inscription.id;
                if (inscriptionId) {
                    const contentUrl = `https://ordinals.com/content/${inscriptionId}`;
                    const contentType = inscription.contentType || '';
                    
                    // Detectar tipo de conteúdo
                    if (contentType.includes('image') || contentType.includes('png') || contentType.includes('jpeg') || contentType.includes('jpg') || contentType.includes('webp') || contentType.includes('gif')) {
                        // Imagem estática ou GIF
                        thumbnail.innerHTML = `<img src="${contentUrl}" style="width: 100%; height: 100%; object-fit: cover;" onerror="this.parentElement.innerHTML='🖼️'">`;
                    } else if (contentType.includes('video') || contentType.includes('mp4') || contentType.includes('webm')) {
                        // Vídeo
                        thumbnail.innerHTML = `<video src="${contentUrl}" style="width: 100%; height: 100%; object-fit: cover;" muted loop autoplay playsinline onerror="this.parentElement.innerHTML='🎥'"></video>`;
                    } else if (contentType.includes('text') || contentType.includes('html')) {
                        // Texto/HTML
                        thumbnail.innerHTML = '<div style="font-size: 20px;">📝</div>';
                    } else {
                        // Tentar carregar como imagem (fallback)
                        thumbnail.innerHTML = `<img src="${contentUrl}" style="width: 100%; height: 100%; object-fit: cover;" onerror="this.parentElement.innerHTML='💎'">`;
                    }
                } else {
                    thumbnail.innerHTML = '<div style="font-size: 20px;">💎</div>';
                }
                
                // Info da inscription
                const info = document.createElement('div');
                info.style.cssText = 'flex: 1; min-width: 0;';
                const number = inscription.inscriptionNumber || inscription.number || '?';
                const idShort = (inscriptionId || '').substring(0, 10);
                info.innerHTML = `
                    <div style="font-weight: 600; color: #f59e0b; font-size: 14px;">Inscription #${number}</div>
                    <div style="font-size: 11px; color: #888; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">${idShort}...</div>
                `;
                
                item.appendChild(thumbnail);
                item.appendChild(info);
                
                // Hover effect
                item.addEventListener('mouseenter', () => {
                    item.style.background = 'rgba(245, 158, 11, 0.1)';
                });
                item.addEventListener('mouseleave', () => {
                    item.style.background = 'transparent';
                });
                
                // Click para selecionar
                item.addEventListener('click', () => {
                    // Remover seleção anterior
                    inscriptionListContainer.querySelectorAll('div[data-inscription]').forEach(el => {
                        el.style.background = 'transparent';
                        el.style.borderLeft = 'none';
                    });
                    
                    // Marcar como selecionado
                    item.style.background = 'rgba(245, 158, 11, 0.2)';
                    item.style.borderLeft = '3px solid #f59e0b';
                    
                    // Trigger change event
                    const inscriptionData = JSON.parse(item.dataset.inscription);
                    handleInscriptionSelect(inscriptionData);
                });
                
                inscriptionListContainer.appendChild(item);
            });
        };

        // Renderizar primeiras 12 inscriptions
        renderInscriptions(allInscriptions);

        // Event listener para busca
        inscriptionSearch.addEventListener('input', (e) => {
            const searchTerm = e.target.value.toLowerCase().trim();
            
            if (!searchTerm) {
                // Se busca vazia, mostrar primeiras 12
                renderInscriptions(allInscriptions);
                return;
            }

            // Filtrar inscriptions
            const filtered = allInscriptions.filter(inscription => {
                const number = String(inscription.inscriptionNumber || inscription.number || '');
                const id = (inscription.inscriptionId || inscription.id || '').toLowerCase();
                
                return number.includes(searchTerm) || id.includes(searchTerm);
            });

            console.log(`🔍 Search "${searchTerm}": ${filtered.length} results`);
            renderInscriptions(filtered);
        });

        // Função para lidar com seleção de inscription
        const handleInscriptionSelect = async (inscriptionData) => {
            const previewDiv = screen.querySelector('#inscription-preview');
            
            console.log('🖼️ Inscription selected:', inscriptionData);
            
            // Preencher hidden inputs
            screen.querySelector('#pool-inscription-id').value = inscriptionData.inscriptionId || inscriptionData.id;
            screen.querySelector('#pool-inscription-number').value = inscriptionData.inscriptionNumber || '';
            
            // Mostrar preview
            previewDiv.style.display = 'block';
            screen.querySelector('#inscription-preview-number').textContent = `Inscription #${inscriptionData.inscriptionNumber || '?'}`;
            screen.querySelector('#inscription-preview-id').textContent = `ID: ${(inscriptionData.inscriptionId || inscriptionData.id).substring(0, 20)}...`;
            
            // Carregar imagem/content da inscription com tamanho maior
            const imageContainer = screen.querySelector('#inscription-preview-image');
            imageContainer.innerHTML = '<div class="spinner" style="width: 20px; height: 20px;"></div>';
            
            try {
                const inscriptionId = inscriptionData.inscriptionId || inscriptionData.id;
                const contentUrl = `https://ordinals.com/content/${inscriptionId}`;
                const contentType = inscriptionData.contentType || '';
                
                console.log('🖼️ Loading content:', { inscriptionId, contentType, contentUrl });
                
                // Limpar container
                imageContainer.innerHTML = '';
                
                // Detectar tipo de conteúdo e renderizar apropriadamente
                if (contentType.includes('image') || contentType.includes('png') || contentType.includes('jpeg') || contentType.includes('jpg') || contentType.includes('webp') || contentType.includes('gif')) {
                    // Imagem estática ou GIF
                    const img = document.createElement('img');
                    img.src = contentUrl;
                    img.style.cssText = 'width: 100%; height: 100%; object-fit: cover;';
                    img.onerror = () => {
                        console.error('❌ Failed to load image:', contentUrl);
                        imageContainer.innerHTML = '<div style="font-size: 32px;">🖼️</div>';
                    };
                    img.onload = () => {
                        console.log('✅ Image loaded successfully');
                    };
                    imageContainer.appendChild(img);
                } else if (contentType.includes('video') || contentType.includes('mp4') || contentType.includes('webm')) {
                    // Vídeo
                    const video = document.createElement('video');
                    video.src = contentUrl;
                    video.style.cssText = 'width: 100%; height: 100%; object-fit: cover;';
                    video.muted = true;
                    video.loop = true;
                    video.autoplay = true;
                    video.playsInline = true;
                    video.onerror = () => {
                        console.error('❌ Failed to load video:', contentUrl);
                        imageContainer.innerHTML = '<div style="font-size: 32px;">🎥</div>';
                    };
                    video.onloadeddata = () => {
                        console.log('✅ Video loaded successfully');
                    };
                    imageContainer.appendChild(video);
                } else if (contentType.includes('text') || contentType.includes('html')) {
                    // Texto/HTML
                    imageContainer.innerHTML = '<div style="font-size: 32px;">📝</div>';
                } else {
                    // Tentar carregar como imagem (fallback universal)
                    console.log('⚠️ Unknown content type, trying as image:', contentType);
                    const img = document.createElement('img');
                    img.src = contentUrl;
                    img.style.cssText = 'width: 100%; height: 100%; object-fit: cover;';
                    img.onerror = () => {
                        console.error('❌ Failed to load content:', contentUrl);
                        imageContainer.innerHTML = '<div style="font-size: 32px;">💎</div>';
                    };
                    img.onload = () => {
                        console.log('✅ Content loaded as image');
                    };
                    imageContainer.appendChild(img);
                }
            } catch (err) {
                console.error('❌ Error loading inscription preview:', err);
                imageContainer.innerHTML = '<div style="font-size: 32px;">💎</div>';
            }
        };

        console.log('✅ Inscription dropdown configured with search!');

    } catch (error) {
        console.error('❌ Error loading inscriptions:', error);
        const inscriptionSelect = screen.querySelector('#inscription-select');
        inscriptionSelect.innerHTML = '<option value="">Error loading inscriptions</option>';
    }
}

/**
 * Criar nova pool
 */
async function handleCreatePool(e) {
    e.preventDefault();
    
    const form = e.target;
    const poolName = form.querySelector('#pool-name').value;
    const poolImage = form.querySelector('#pool-image').value;
    const useInscription = form.querySelector('#use-inscription').checked;
    const poolInscriptionId = useInscription ? form.querySelector('#pool-inscription-id').value : null;
    const poolInscriptionNumber = useInscription ? form.querySelector('#pool-inscription-number').value : null;
    const runeA = form.querySelector('#rune-a').value;
    const runeAName = form.querySelector('#rune-a-name').value;
    const isBtcPair = form.querySelector('#is-btc-pair').checked;
    const runeB = isBtcPair ? null : form.querySelector('#rune-b').value;
    const runeBName = isBtcPair ? 'BTC' : form.querySelector('#rune-b-name').value;
    const initialAmountA = parseInt(form.querySelector('#initial-amount-a').value);
    const initialAmountB = parseInt(form.querySelector('#initial-amount-b').value);
    const feeRate = parseInt(form.querySelector('#fee-rate').value);

    // Validações
    if (!poolName || !runeA || !runeAName || !initialAmountA || !initialAmountB) {
        showNotification('❌ Please fill all required fields', 'error');
        return;
    }

    if (useInscription && !poolInscriptionId) {
        showNotification('❌ Please provide an Inscription ID', 'error');
        return;
    }

    try {
        showLoading('Creating pool...');

        const walletInfo = await sendMessage({ action: 'getWalletInfo' });
        if (!walletInfo.success || !walletInfo.address) {
            throw new Error('Wallet not found');
        }

        const response = await fetch('https://kraywallet-backend.onrender.com/api/dex/pools/create', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                poolName,
                poolImage,
                poolInscriptionId,
                poolInscriptionNumber: poolInscriptionNumber ? parseInt(poolInscriptionNumber) : null,
                useInscription,
                creatorAddress: walletInfo.address,
                runeA,
                runeAName,
                runeB,
                runeBName,
                isBtcPair,
                initialAmountA,
                initialAmountB,
                feeRate
            })
        });

        const data = await response.json();

        if (!data.success) {
            throw new Error(data.error || 'Failed to create pool');
        }

        console.log('✅ Pool created:', data.poolId);
        showNotification('✅ Pool created successfully!', 'success');
        
        // Fechar tela e recarregar pools
        document.querySelector('.create-pool-screen').remove();
        loadLiquidityPools();

    } catch (error) {
        console.error('❌ Error creating pool:', error);
        showNotification('❌ ' + error.message, 'error');
    } finally {
        hideLoading();
    }
}

/**
 * Mostrar detalhes da pool
 */
async function showPoolDetails(poolId) {
    console.log('🔍 Loading pool details:', poolId);
    showNotification('Pool details coming soon!', 'info');
}

/**
 * Mostrar tela de swap
 */
async function showSwapScreen(poolId) {
    console.log('💱 Opening swap screen for pool:', poolId);
    showNotification('Swap feature coming soon!', 'info');
}

// ==========================================
// ⚡ NETWORK SWITCHER (Mainnet/Lightning/Testnet)
// ==========================================

/**
 * Switch between networks (mainnet, lightning, testnet)
 * @param {string} network - 'mainnet', 'lightning', or 'testnet'
 */
async function switchNetwork(network) {
    console.log(`⚡ ========== SWITCHING TO ${network.toUpperCase()} ==========`);
    
    try {
        // Update dropdown label
        const currentNetworkLabel = document.getElementById('current-network-label');
        const balanceLabel = document.getElementById('balance-label');
        const lightningInfo = document.getElementById('lightning-info');
        const lightningActions = document.getElementById('lightning-actions');
        const actionButtons = document.querySelector('.action-buttons');
        
        // Update balance icon
        const balanceNetworkIcon = document.getElementById('balance-network-icon');
        const balanceNetworkEmoji = document.getElementById('balance-network-emoji');
        
        if (network === 'mainnet') {
            // Mainnet (Bitcoin Layer 1)
            if (currentNetworkLabel) {
                currentNetworkLabel.innerHTML = '<img src="../images/bitcoin.png" alt="Bitcoin" onerror="this.style.display=\'none\'; this.nextElementSibling.style.display=\'inline\';" style="width: 16px; height: 16px; vertical-align: middle; margin-right: 4px; display: inline;" /><span style="display: none;">🔗</span> Mainnet';
            }
            if (balanceLabel) {
                balanceLabel.textContent = 'Total Balance';
            }
            // Update balance icon to Bitcoin (normal size)
            if (balanceNetworkIcon) {
                balanceNetworkIcon.src = '../images/bitcoin.png';
                balanceNetworkIcon.style.display = 'inline';
                balanceNetworkIcon.style.borderRadius = '0';
                balanceNetworkIcon.style.width = '40px';
                balanceNetworkIcon.style.height = '40px';
                balanceNetworkEmoji.style.display = 'none';
            }
            
            // LIMPAR balance antes de atualizar
            const walletBalance = document.getElementById('wallet-balance');
            const walletBalanceBtc = document.getElementById('wallet-balance-btc');
            if (walletBalance) walletBalance.textContent = 'Loading...';
            if (walletBalanceBtc) walletBalanceBtc.textContent = '';
            
            // Hide Lightning and L2 specific UI
            lightningInfo?.classList.add('hidden');
            lightningActions?.classList.add('hidden');
            document.getElementById('kray-l2-content')?.classList.add('hidden');
            
            // Show normal UI
            actionButtons?.classList.remove('hidden');
            document.querySelector('.tabs-container')?.classList.remove('hidden');
            
            // Show normal tab contents
            document.getElementById('ordinals-tab')?.classList.remove('hidden');
            
            // Update Bitcoin balance
            await updateMainnetBalance();
            
            console.log('✅ Switched to Mainnet (Bitcoin Layer 1)');
            
        } else if (network === 'kray-l2') {
            // KRAY SPACE L2 (Our own Layer 2!)
            if (currentNetworkLabel) {
                currentNetworkLabel.innerHTML = '<img src="../assets/kray-space.webp" alt="KRAY L2" onerror="this.style.display=\'none\'; this.nextElementSibling.style.display=\'inline\';" style="width: 16px; height: 16px; vertical-align: middle; margin-right: 4px; display: inline; border-radius: 3px;" /><span style="display: none;">⚡</span> KRAY L2';
            }
            if (balanceLabel) {
                balanceLabel.textContent = 'L2 Balance';
            }
            // Update balance icon to KRAY L2 (40% bigger than Bitcoin icon)
            if (balanceNetworkIcon && balanceNetworkEmoji) {
                balanceNetworkIcon.src = '../assets/kray-space.webp';
                balanceNetworkIcon.style.display = 'inline';
                balanceNetworkIcon.style.borderRadius = '8px';
                balanceNetworkIcon.style.width = '56px';
                balanceNetworkIcon.style.height = '56px';
                balanceNetworkEmoji.style.display = 'none';
            }
            
            // Hide normal action buttons and tabs
            actionButtons?.classList.add('hidden');
            document.querySelector('.tabs-container')?.classList.add('hidden');
            
            // Hide ALL tab contents (Ordinals, Runes, Swap, Activity)
            document.getElementById('ordinals-tab')?.classList.add('hidden');
            document.getElementById('runes-tab')?.classList.add('hidden');
            document.getElementById('swap-tab')?.classList.add('hidden');
            document.getElementById('activity-tab')?.classList.add('hidden');
            
            // Hide Lightning-specific UI (if any)
            lightningInfo?.classList.add('hidden');
            lightningActions?.classList.add('hidden');
            
            // Show ONLY KRAY L2 content
            const krayL2Content = document.getElementById('kray-l2-content');
            if (krayL2Content) {
                krayL2Content.classList.remove('hidden');
            }
            
            // Clear Bitcoin balance (don't show in L2 mode)
            const walletBalance = document.getElementById('wallet-balance');
            const walletBalanceBtc = document.getElementById('wallet-balance-btc');
            if (walletBalance) walletBalance.textContent = 'Loading...';
            if (walletBalanceBtc) walletBalanceBtc.textContent = '';
            
            // Load L2 balance immediately
            if (window.krayL2 && typeof window.krayL2.refreshL2Data === 'function') {
                console.log('📡 Calling refreshL2Data...');
                await window.krayL2.refreshL2Data();
            } else {
                console.warn('⚠️ krayL2.refreshL2Data not available');
            }
            
            console.log('✅ Switched to KRAY L2 (Layer 2)');
            
        } else if (network === 'lightning') {
            // Lightning Network (commented out - using KRAY L2 instead)
            // Keeping code for reference
            if (currentNetworkLabel) {
                currentNetworkLabel.textContent = '⚡ Lightning';
            }
            if (balanceLabel) {
                balanceLabel.textContent = 'Total Balance (Lightning)';
            }
            // Update balance icon to Lightning
            if (balanceNetworkIcon && balanceNetworkEmoji) {
                balanceNetworkIcon.style.display = 'none';
                balanceNetworkEmoji.textContent = '⚡';
                balanceNetworkEmoji.style.display = 'inline';
            }
            
            // LIMPAR balance antes de atualizar (evita mostrar balance do Mainnet)
            const walletBalance = document.getElementById('wallet-balance');
            const walletBalanceBtc = document.getElementById('wallet-balance-btc');
            if (walletBalance) walletBalance.textContent = 'Loading...';
            if (walletBalanceBtc) walletBalanceBtc.textContent = '';
            
            // Show Lightning-specific UI
            lightningInfo?.classList.remove('hidden');
            lightningActions?.classList.remove('hidden');
            actionButtons?.classList.add('hidden');
            
            // Update Lightning balance
            await updateLightningBalance();
            
            console.log('✅ Switched to Lightning Network (Layer 2)');
            
        } else if (network === 'testnet') {
            // Testnet
            if (currentNetworkLabel) {
                currentNetworkLabel.textContent = '🧪 Testnet';
            }
            if (balanceLabel) {
                balanceLabel.textContent = 'Total Balance (Testnet)';
            }
            // Update balance icon to Testnet
            if (balanceNetworkIcon && balanceNetworkEmoji) {
                balanceNetworkIcon.style.display = 'none';
                balanceNetworkEmoji.textContent = '🧪';
                balanceNetworkEmoji.style.display = 'inline';
            }
            
            // Hide Lightning and L2 specific UI
            lightningInfo?.classList.add('hidden');
            lightningActions?.classList.add('hidden');
            document.getElementById('kray-l2-content')?.classList.add('hidden');
            
            // Show normal UI
            actionButtons?.classList.remove('hidden');
            document.querySelector('.tabs-container')?.classList.remove('hidden');
            
            // Show normal tab contents
            document.getElementById('ordinals-tab')?.classList.remove('hidden');
            
            // Update testnet balance (mock for now)
            const walletBalance = document.getElementById('wallet-balance');
            const walletBalanceBtc = document.getElementById('wallet-balance-btc');
            if (walletBalance) walletBalance.textContent = '0 sats';
            if (walletBalanceBtc) walletBalanceBtc.textContent = '0.00000000 BTC';
            
            showNotification('🧪 Testnet mode (not implemented yet)', 'info');
            
            console.log('✅ Switched to Testnet');
        }
        
        // Mark active option
        document.querySelectorAll('.network-option').forEach(opt => {
            if (opt.dataset.network === network) {
                opt.classList.add('active');
            } else {
                opt.classList.remove('active');
            }
        });
        
        // Save preference
        chrome.storage.local.set({ activeNetwork: network }, () => {
            console.log(`💾 Network preference saved: ${network}`);
        });
        
    } catch (error) {
        console.error('❌ Error switching network:', error);
        showNotification('❌ Failed to switch network', 'error');
    }
}

/**
 * Update Mainnet (Bitcoin) balance
 */
async function updateMainnetBalance() {
    try {
        console.log('💰 Updating Mainnet balance...');
        
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        
        console.log('📊 Wallet info received:', walletInfo);
        
        if (walletInfo.success && walletInfo.data) {
            // Balance vem como { total, confirmed, unconfirmed }
            const balanceData = walletInfo.data.balance;
            
            // ✅ FIX: Usar typeof para verificar se é objeto, então pegar .total
            let balance;
            if (typeof balanceData === 'object' && balanceData !== null) {
                // Balance é um objeto { total, confirmed, unconfirmed }
                balance = balanceData.total ?? 0;
            } else {
                // Balance já é um número
                balance = balanceData || 0;
            }
            
            const balanceBtc = (balance / 100000000).toFixed(8);
            
            console.log('💰 Balance data:', balanceData);
            console.log('💰 Balance total:', balance);
            
            const walletBalance = document.getElementById('wallet-balance');
            const walletBalanceBtc = document.getElementById('wallet-balance-btc');
            
            if (walletBalance) {
                walletBalance.textContent = `${balance.toLocaleString()} sats`;
            }
            if (walletBalanceBtc) {
                walletBalanceBtc.textContent = `${balanceBtc} BTC`;
            }
            
            console.log(`✅ Mainnet balance updated: ${balance} sats`);
        }
    } catch (error) {
        console.error('❌ Error updating Mainnet balance:', error);
    }
}

/**
 * Update Lightning Network balance and channels
 */
async function updateLightningBalance() {
    const walletBalance = document.getElementById('wallet-balance');
    const walletBalanceBtc = document.getElementById('wallet-balance-btc');
    const channelsText = document.getElementById('lightning-channels-text');
    
    try {
        console.log('⚡ Updating Lightning balance...');
        
        // Get wallet address
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        
        if (!walletInfo.success || !walletInfo.data?.address) {
            throw new Error('Wallet not found');
        }
        
        const address = walletInfo.data.address;
        
        console.log(`⚡ Fetching Lightning balance for: ${address.slice(0, 20)}...`);
        
        // Fetch Lightning balance from backend com TIMEOUT
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 3000); // 3 segundos timeout
        
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/lightning/balance/${address}`, {
            signal: controller.signal
        });
        
        clearTimeout(timeoutId);
        
        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }
        
        const data = await response.json();
        
        console.log('⚡ Lightning API response:', data);
        
        const balance = data.balance || 0;
        const balanceBtc = (balance / 100000000).toFixed(8);
        const channelsActive = data.channels?.active || 0;
        const channelsTotal = data.channels?.total || 0;
        const lndStatus = data.lndStatus || 'unknown';
        const walletStatus = data.walletStatus || '';
        
        console.log(`💰 Balance: ${balance} sats`);
        console.log(`📡 Channels: ${channelsActive} active / ${channelsTotal} total`);
        console.log(`🔌 LND Status: ${lndStatus}`);
        console.log(`🔐 Wallet Status: ${walletStatus}`);
        
        // Se wallet está inicializando, mostrar mensagem específica
        if (walletStatus === 'locked_or_initializing') {
            if (walletBalance) {
                walletBalance.innerHTML = `<span style="color: #ff9500;">Initializing...</span>`;
            }
            if (walletBalanceBtc) {
                walletBalanceBtc.textContent = '';
            }
            if (channelsText) {
                channelsText.innerHTML = `<div style="color: #888; font-size: 12px; margin-top: 8px;">⏳ Setting up Lightning Network...<br>This may take 1-2 minutes</div>`;
            }
            return;
        }
        
        // Update UI normal
        if (walletBalance) {
            walletBalance.textContent = `${balance.toLocaleString()} sats`;
        }
        if (walletBalanceBtc) {
            walletBalanceBtc.textContent = `${balanceBtc} BTC`;
        }
        if (channelsText) {
            channelsText.textContent = `${channelsActive} ${channelsActive === 1 ? 'channel' : 'channels'} active`;
        }
        
        console.log(`✅ Lightning balance updated: ${balance} sats, ${channelsActive} channels`);
        
        // Mensagem informativa baseada no status
        if (lndStatus === 'disconnected') {
            console.log('ℹ️  LND not running. Start with: ./start-lnd.sh');
        } else if (lndStatus === 'connected' && channelsActive === 0) {
            console.log('ℹ️  LND connected! No channels yet. Use "Open Channel" to get started!');
        }
        
    } catch (error) {
        console.error('❌ Error updating Lightning balance:', error);
        console.error('   Error type:', error.name);
        console.error('   Error message:', error.message);
        
        // Mensagem específica baseada no erro
        let errorMessage = '0 sats';
        if (error.name === 'AbortError') {
            console.log('⏱️  Request timeout - backend may be slow or LND not running');
            errorMessage = '0 sats';
        } else if (error.message.includes('Failed to fetch')) {
            console.log('🔌 Backend not responding - is it running?');
            errorMessage = '0 sats';
        }
        
        // Show 0 on error
        if (walletBalance) walletBalance.textContent = errorMessage;
        if (walletBalanceBtc) walletBalanceBtc.textContent = '0.00000000 BTC';
        if (channelsText) channelsText.textContent = '0 channels active';
        
        console.log('ℹ️  Lightning showing 0 sats (LND not connected or backend issue)');
    }
}

// ==========================================
// 💰 PURE BITCOIN BALANCE
// ==========================================

/**
 * Calculate pure Bitcoin balance (UTXOs without inscriptions or runes)
 */
async function getPureBitcoinBalance(address, userRunes) {
    try {
        console.log('💰 Calculating pure Bitcoin balance...');
        console.log(`   Address: ${address}`);
        console.log(`   User has ${userRunes.length} runes`);
        
        // 1. Buscar balance total
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (!walletInfo.success) {
            throw new Error('Failed to get wallet info');
        }
        
        const totalBalance = walletInfo.data.balance.total || 0;
        console.log(`   Total balance: ${totalBalance} sats`);
        
        // 2. Buscar inscriptions
        const inscriptionsResponse = await chrome.runtime.sendMessage({
            action: 'getInscriptions',
            data: { address }
        });
        
        const inscriptions = inscriptionsResponse.inscriptions || [];
        console.log(`   Inscriptions: ${inscriptions.length}`);
        
        // 3. Calcular sats em inscriptions (geralmente 546 sats cada)
        const inscriptionsSats = inscriptions.reduce((sum, ins) => sum + (ins.outputValue || ins.value || 600), 0);
        console.log(`   Sats in inscriptions: ${inscriptionsSats} sats`);
        
        // 4. Calcular sats em runes
        // Cada UTXO de rune tem pelo menos 546 sats (dust limit)
        let runesSats = 0;
        for (const rune of userRunes) {
            // Cada rune pode ter múltiplos UTXOs
            const utxoCount = rune.utxos ? rune.utxos.length : 1;
            runesSats += utxoCount * 546; // Estimativa conservadora
        }
        console.log(`   Estimated sats in runes: ${runesSats} sats`);
        
        // 5. Calcular pure balance
        const pureBalance = Math.max(0, totalBalance - inscriptionsSats - runesSats);
        console.log(`   Pure Bitcoin balance: ${pureBalance} sats`);
        
        return pureBalance;
        
    } catch (error) {
        console.error('❌ Error calculating pure Bitcoin balance:', error);
        return 0;
    }
}

// ==========================================
// ⚡ LIGHTNING ACTIONS (Open Channel, Deposit, Withdraw)
// ==========================================

/**
 * Open Lightning Channel
 */
async function handleOpenChannel() {
    console.log('📡 ========== OPEN LIGHTNING CHANNEL ==========');
    
    try {
        // TODO: Implementar UI para abrir channel
        showNotification('🚧 Open Channel coming soon!', 'info');
        
        console.log('📡 Open Channel feature will allow:');
        console.log('   1. Select remote peer');
        console.log('   2. Choose amount to lock');
        console.log('   3. Create funding transaction');
        console.log('   4. Wait for confirmations');
        console.log('   5. Channel active!');
        
    } catch (error) {
        console.error('❌ Error opening channel:', error);
        showNotification('❌ Failed to open channel', 'error');
    }
}

/**
 * Deposit to Lightning (Mainnet → Lightning)
 */
async function handleDepositToLightning() {
    console.log('💰 ========== DEPOSIT TO LIGHTNING ==========');
    
    try {
        // Buscar wallet info
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        
        if (!walletInfo.success || !walletInfo.data?.address) {
            throw new Error('Wallet not found');
        }
        
        const address = walletInfo.data.address;
        
        // Buscar Runes do usuário
        console.log('📊 Fetching user Runes for deposit...');
        console.log(`   Address: ${address}`);
        
        const runesResponse = await chrome.runtime.sendMessage({
            action: 'getRunes',
            data: { address }
        });
        
        console.log('📦 Runes response:', runesResponse);
        
        if (!runesResponse.success) {
            console.error('❌ Failed to fetch runes:', runesResponse.error);
            throw new Error('Failed to fetch Runes');
        }
        
        // ✅ CORRIGIR: O backend retorna `runes` não `data`
        const userRunes = runesResponse.runes || [];
        console.log(`✅ Found ${userRunes.length} Runes`);
        
        // Mostrar tela de seleção
        showDepositToLightningScreen(address, userRunes);
        
    } catch (error) {
        console.error('❌ Error opening deposit screen:', error);
        showNotification('❌ Failed to open deposit screen', 'error');
    }
}

/**
 * Mostrar tela de Deposit para Lightning
 */
async function showDepositToLightningScreen(address, userRunes) {
    console.log('💰 Showing Deposit to Lightning screen...');
    console.log(`   Address: ${address}`);
    console.log(`   Runes available: ${userRunes.length}`);
    
    // Criar overlay
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.95);
        z-index: 10000;
        display: flex;
        flex-direction: column;
        padding: 20px;
        overflow-y: auto;
    `;
    
    // Header
    const header = document.createElement('div');
    header.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 24px;
    `;
    header.innerHTML = `
        <h2 style="color: #fff; font-size: 20px; font-weight: 700; margin: 0;">
            💰 Deposit to Lightning
        </h2>
        <button id="close-deposit-modal" style="
            background: transparent;
            border: none;
            color: #fff;
            font-size: 24px;
            cursor: pointer;
            padding: 0;
            width: 32px;
            height: 32px;
        ">×</button>
    `;
    
    // Info
    const info = document.createElement('div');
    info.style.cssText = `
        background: rgba(255, 159, 0, 0.1);
        border: 1px solid rgba(255, 159, 0, 0.3);
        border-radius: 12px;
        padding: 16px;
        margin-bottom: 24px;
        color: #ff9500;
        font-size: 13px;
        line-height: 1.6;
    `;
    info.innerHTML = `
        <strong>⚡ How it works:</strong><br>
        1. Select which Runes you want to send to Lightning<br>
        2. Or send pure Bitcoin (minimum 1 sat)<br>
        3. Creates a Lightning channel with your assets<br>
        4. Assets locked in channel for instant swaps!
    `;
    
    // Lista de Runes
    const runesList = document.createElement('div');
    runesList.style.cssText = `
        display: flex;
        flex-direction: column;
        gap: 12px;
        margin-bottom: 24px;
    `;
    
    // Opção: Pure Bitcoin
    const bitcoinOption = document.createElement('div');
    bitcoinOption.className = 'deposit-option';
    bitcoinOption.style.cssText = `
        background: #1a1a1a;
        border: 2px solid #333;
        border-radius: 12px;
        padding: 16px;
        cursor: pointer;
        transition: all 0.2s ease;
    `;
    
    // Buscar sats puros (sem inscriptions e sem runes)
    console.log('💰 Fetching pure Bitcoin balance...');
    const pureBitcoinBalance = await getPureBitcoinBalance(address, userRunes);
    console.log(`   Pure sats: ${pureBitcoinBalance} sats`);
    
    // Inscription ID do símbolo Bitcoin na blockchain
    const bitcoinInscriptionId = 'cfab194b924f7785c6e453728e1c264b89b74843633278cda3ad3f57576c1e93i0';
    const bitcoinContentUrl = `https://ordinals.com/content/${bitcoinInscriptionId}`;
    
    bitcoinOption.innerHTML = `
        <div style="display: flex; align-items: center; gap: 12px;">
            <div style="
                width: 48px;
                height: 48px;
                border-radius: 8px;
                overflow: hidden;
                background: #000;
                display: flex;
                align-items: center;
                justify-content: center;
            ">
                <img src="${bitcoinContentUrl}" 
                     style="width: 100%; height: 100%; object-fit: cover;"
                     onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"
                />
                <div style="
                    display: none;
                    font-size: 28px;
                    color: #f7931a;
                ">₿</div>
            </div>
            <div style="flex: 1;">
                <div style="color: #fff; font-weight: 600; font-size: 15px;">Pure Bitcoin</div>
                <div style="color: #888; font-size: 13px;">${pureBitcoinBalance.toLocaleString()} sats available</div>
            </div>
            <div style="color: #fff; font-weight: 600;">›</div>
        </div>
    `;
    bitcoinOption.addEventListener('click', () => {
        showDepositAmountScreen(null, address, overlay, pureBitcoinBalance);
    });
    bitcoinOption.addEventListener('mouseenter', () => {
        bitcoinOption.style.borderColor = '#ff9500';
        bitcoinOption.style.background = '#222';
    });
    bitcoinOption.addEventListener('mouseleave', () => {
        bitcoinOption.style.borderColor = '#333';
        bitcoinOption.style.background = '#1a1a1a';
    });
    
    runesList.appendChild(bitcoinOption);
    
    // Opções de Runes
    if (userRunes.length > 0) {
        userRunes.forEach(rune => {
            const runeOption = document.createElement('div');
            runeOption.className = 'deposit-option';
            runeOption.style.cssText = `
                background: #1a1a1a;
                border: 2px solid #333;
                border-radius: 12px;
                padding: 16px;
                cursor: pointer;
                transition: all 0.2s ease;
            `;
            
            // Determinar display (thumbnail ou símbolo)
            const runeName = rune.displayName || rune.name || 'Unknown Rune';
            const runeSymbol = rune.symbol || 'ᚱ';
            const hasParent = rune.parent && rune.parentPreview;
            
            // HTML com thumbnail se tiver parent
            runeOption.innerHTML = `
                <div style="display: flex; align-items: center; gap: 12px;">
                    ${hasParent ? `
                        <div style="
                            width: 48px;
                            height: 48px;
                            border-radius: 8px;
                            overflow: hidden;
                            background: #000;
                            display: flex;
                            align-items: center;
                            justify-content: center;
                        ">
                            <img src="${rune.parentPreview}" 
                                 style="width: 100%; height: 100%; object-fit: cover;"
                                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"
                            />
                            <div style="display: none; font-size: 24px;">${runeSymbol}</div>
                        </div>
                    ` : `
                        <div style="font-size: 32px;">${runeSymbol}</div>
                    `}
                    <div style="flex: 1;">
                        <div style="color: #fff; font-weight: 600; font-size: 15px;">${runeName}</div>
                        <div style="color: #888; font-size: 13px;">${parseInt(rune.amount).toLocaleString()} available</div>
                    </div>
                    <div style="color: #fff; font-weight: 600;">›</div>
                </div>
            `;
            runeOption.addEventListener('click', () => {
                showDepositAmountScreen(rune, address, overlay);
            });
            runeOption.addEventListener('mouseenter', () => {
                runeOption.style.borderColor = '#ff9500';
                runeOption.style.background = '#222';
            });
            runeOption.addEventListener('mouseleave', () => {
                runeOption.style.borderColor = '#333';
                runeOption.style.background = '#1a1a1a';
            });
            
            runesList.appendChild(runeOption);
        });
    } else {
        const noRunes = document.createElement('div');
        noRunes.style.cssText = `
            text-align: center;
            padding: 32px;
            color: #666;
            font-size: 14px;
        `;
        noRunes.innerHTML = `
            <div style="font-size: 48px; margin-bottom: 16px;">ᚱ</div>
            <div>No Runes detected</div>
            <div style="font-size: 12px; margin-top: 8px; color: #888;">
                Loading from blockchain...<br>
                Or deposit pure Bitcoin below
            </div>
        `;
        runesList.appendChild(noRunes);
    }
    
    // Montar
    overlay.appendChild(header);
    overlay.appendChild(info);
    overlay.appendChild(runesList);
    document.body.appendChild(overlay);
    
    // Close button
    document.getElementById('close-deposit-modal').addEventListener('click', () => {
        document.body.removeChild(overlay);
    });
}

/**
 * Mostrar tela de quantidade para deposit
 */
function showDepositAmountScreen(rune, address, previousOverlay, pureBitcoinBalance = 0) {
    console.log('💰 Showing amount screen...');
    
    // Preparar dados da Rune ou Bitcoin
    let assetName = 'Pure Bitcoin';
    let assetSymbol = '₿';
    let hasParent = false;
    let parentPreview = '';
    let availableAmount = pureBitcoinBalance;
    let minAmount = 10000; // 10k sats minimum for Lightning channel
    
    if (rune) {
        // É uma Rune
        assetName = rune.displayName || rune.name || 'Unknown Rune';
        assetSymbol = rune.symbol || 'ᚱ';
        hasParent = rune.parent && rune.parentPreview;
        parentPreview = rune.parentPreview || '';
        availableAmount = parseInt(rune.amount);
        minAmount = 1; // Runes podem enviar qualquer quantidade
        
        console.log(`   Rune: ${assetName}`);
        console.log(`   Available: ${availableAmount}`);
        console.log(`   Has parent: ${hasParent}`);
    } else {
        // É Pure Bitcoin
        // Buscar inscription do Bitcoin
        const bitcoinInscriptionId = 'cfab194b924f7785c6e453728e1c264b89b74843633278cda3ad3f57576c1e93i0';
        hasParent = true;
        parentPreview = `https://ordinals.com/content/${bitcoinInscriptionId}`;
        
        console.log('   Type: Pure Bitcoin');
        console.log(`   Available: ${availableAmount} sats`);
        console.log(`   Min amount: ${minAmount} sats`);
    }
    
    // Remover overlay anterior
    if (previousOverlay) {
        document.body.removeChild(previousOverlay);
    }
    
    // Criar novo overlay
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.95);
        z-index: 10000;
        display: flex;
        flex-direction: column;
        padding: 20px;
    `;
    
    overlay.innerHTML = `
        <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 24px;">
            <button id="back-deposit-btn" style="
                background: transparent;
                border: none;
                color: #fff;
                font-size: 24px;
                cursor: pointer;
                padding: 0;
            ">←</button>
            ${hasParent ? `
                <div style="
                    width: 40px;
                    height: 40px;
                    border-radius: 8px;
                    overflow: hidden;
                    background: #000;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                ">
                    <img src="${parentPreview}" 
                         style="width: 100%; height: 100%; object-fit: cover;"
                         onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';"
                    />
                    <div style="display: none; font-size: 20px;">${assetSymbol}</div>
                </div>
            ` : ''}
            <h2 style="color: #fff; font-size: 20px; font-weight: 700; margin: 0; flex: 1;">
                ${assetName}
            </h2>
        </div>
        
        <div style="background: #1a1a1a; border-radius: 16px; padding: 24px; margin-bottom: 24px;">
            <div style="color: #888; font-size: 13px; margin-bottom: 8px;">Available</div>
            <div style="color: #fff; font-size: 24px; font-weight: 700; margin-bottom: 24px;">
                ${availableAmount.toLocaleString()} ${rune ? assetSymbol : 'sats'}
            </div>
            
            <div style="color: #888; font-size: 13px; margin-bottom: 8px;">
                ${rune ? 'Amount to deposit' : 'Bitcoin amount (sats)'}
            </div>
            <div style="display: flex; gap: 12px; align-items: center;">
                <input type="number" id="deposit-amount" placeholder="${rune ? 'Enter amount' : `Min ${minAmount.toLocaleString()} sats`}" 
                    style="
                        flex: 1;
                        background: #000;
                        border: 2px solid #333;
                        border-radius: 12px;
                        padding: 16px;
                        color: #fff;
                        font-size: 18px;
                        font-weight: 600;
                    "
                    max="${availableAmount}"
                    min="${minAmount}"
                />
                <button id="max-deposit-btn" style="
                    background: #ff9500;
                    border: none;
                    border-radius: 8px;
                    padding: 16px 24px;
                    color: #000;
                    font-weight: 700;
                    cursor: pointer;
                ">MAX</button>
            </div>
        </div>
        
        <div style="background: rgba(255, 159, 0, 0.1); border: 1px solid rgba(255, 159, 0, 0.3); border-radius: 12px; padding: 16px; margin-bottom: 24px;">
            <div style="color: #ff9500; font-size: 13px; line-height: 1.6;">
                <strong>⚡ What happens:</strong><br>
                1. Creates Lightning channel<br>
                2. ${rune ? 'Runes + BTC' : 'BTC'} locked in channel<br>
                3. Can do instant swaps (<1 second)<br>
                4. Fee: 1 sat per swap!
            </div>
        </div>
        
        <button id="confirm-deposit-btn" style="
            background: linear-gradient(135deg, #ff9500 0%, #ff6b00 100%);
            border: none;
            border-radius: 12px;
            padding: 18px;
            color: #fff;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            width: 100%;
        ">
            💰 Deposit to Lightning
        </button>
    `;
    
    document.body.appendChild(overlay);
    
    // Event listeners
    document.getElementById('back-deposit-btn').addEventListener('click', () => {
        document.body.removeChild(overlay);
        handleDepositToLightning(); // Voltar para lista
    });
    
    // Botão MAX (agora para Runes e Bitcoin)
    document.getElementById('max-deposit-btn').addEventListener('click', () => {
        document.getElementById('deposit-amount').value = availableAmount;
    });
    
    document.getElementById('confirm-deposit-btn').addEventListener('click', async () => {
        const amount = document.getElementById('deposit-amount').value;
        
        if (!amount || parseFloat(amount) <= 0) {
            showNotification('❌ Enter valid amount', 'error');
            return;
        }
        
        if (parseFloat(amount) > availableAmount) {
            showNotification('❌ Insufficient balance', 'error');
            return;
        }
        
        if (parseFloat(amount) < minAmount) {
            showNotification(`❌ Minimum ${minAmount.toLocaleString()} ${rune ? assetSymbol : 'sats'}`, 'error');
            return;
        }
        
        // Processar deposit
        document.body.removeChild(overlay);
        await processDepositToLightning(rune, amount, address);
    });
}

/**
 * Processar deposit para Lightning
 */
async function processDepositToLightning(rune, amount, address) {
    console.log('💰 ========== PROCESSING DEPOSIT ==========');
    console.log(`   Asset: ${rune ? (rune.displayName || rune.name) : 'Pure Bitcoin'}`);
    console.log(`   Amount: ${amount} sats`);
    console.log(`   Address: ${address}`);
    
    try {
        // 1. Verificar se LND está sincronizado
        console.log('🔍 Checking LND status...');
        showNotification('⚡ Checking Lightning Network status...', 'info');
        
        const statusResponse = await fetch('https://kraywallet-backend.onrender.com/api/lightning/status');
        const statusData = await statusResponse.json();
        
        console.log('📊 LND Status:', statusData);
        
        if (!statusData.connected) {
            throw new Error('Lightning Network not connected. LND may be syncing or not running.');
        }
        
        if (!statusData.info?.synced_to_chain) {
            throw new Error(`Lightning Network is syncing. Currently at block ${statusData.info?.block_height || 0}. Please wait ~15-30 minutes.`);
        }
        
        // 2. Preparar dados do channel
        console.log('📡 Preparing channel funding transaction...');
        showNotification('⚡ Opening Lightning channel...', 'info');
        
        const channelData = {
            capacity: amount, // Amount in sats
            fromAddress: address,
            assetType: rune ? 'rune' : 'btc',
            rune: rune ? {
                id: rune.id,
                name: rune.name || rune.displayName,
                amount: rune.amount
            } : null
        };
        
        console.log('📦 Channel data:', channelData);
        
        // 3. Chamar backend para criar o channel
        const response = await fetch('https://kraywallet-backend.onrender.com/api/lightning/open-channel', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(channelData)
        });
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'Failed to open channel');
        }
        
        const result = await response.json();
        console.log('✅ Channel opening result:', result);
        
        // 4. Mostrar sucesso
        showNotification(`✅ Channel opening! TXID: ${result.txid?.slice(0, 10)}...`, 'success');
        
        // 5. Fechar overlay
        const overlay = document.querySelector('.modal-overlay');
        if (overlay) overlay.remove();
        
        // 6. Atualizar balances
        setTimeout(() => {
            updateLightningBalance();
        }, 2000);
        
    } catch (error) {
        console.error('❌ Error processing deposit:', error);
        showNotification(`❌ ${error.message}`, 'error');
    }
}

/**
 * Withdraw from Lightning (Lightning → Mainnet)
 */
async function handleWithdrawFromLightning() {
    console.log('📤 ========== WITHDRAW FROM LIGHTNING ==========');
    
    try {
        // Verificar se tem balance Lightning
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        
        if (!walletInfo.success || !walletInfo.data?.address) {
            throw new Error('Wallet not found');
        }
        
        const address = walletInfo.data.address;
        
        // Buscar balance Lightning
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/lightning/balance/${address}`);
        const data = await response.json();
        
        const balance = data.balance || 0;
        const channelsActive = data.channels?.active || 0;
        
        console.log(`💰 Lightning balance: ${balance} sats`);
        console.log(`📡 Active channels: ${channelsActive}`);
        
        if (balance === 0 || channelsActive === 0) {
            showNotification('⚠️ No balance or channels to withdraw', 'warning');
            return;
        }
        
        // TODO: Implementar UI de withdraw
        showNotification('🚧 Withdraw feature coming soon!', 'info');
        
        console.log('📤 Withdraw will allow:');
        console.log('   1. Choose amount to withdraw');
        console.log('   2. Close Lightning channel (cooperative)');
        console.log('   3. Create closing transaction');
        console.log('   4. Runes + BTC return to Mainnet');
        console.log('   5. Everything documented on-chain!');
        
        // Preview do que vai fazer:
        console.log('\n📋 Withdraw Preview:');
        console.log(`   From: Lightning Channel`);
        console.log(`   To: ${address}`);
        console.log(`   Amount: ${balance} sats`);
        console.log(`   + Runes locked in channel (if any)`);
        console.log(`   Fee: ~200-500 sats (on-chain closing)`);
        console.log(`   Time: ~10 minutes (on-chain confirmation)`);
        
    } catch (error) {
        console.error('❌ Error withdrawing:', error);
        showNotification('❌ Failed to withdraw', 'error');
    }
}

/**
 * Initialize network selector with saved preference
 */
async function initializeNetworkSelector() {
    try {
        console.log('⚡ Initializing Network Selector...');
        
        // Get saved preference
        const result = await chrome.storage.local.get(['activeNetwork']);
        const savedNetwork = result.activeNetwork || 'mainnet'; // Default to Mainnet
        
        console.log(`📊 Saved network preference: ${savedNetwork}`);
        
        // Apply saved network
        await switchNetwork(savedNetwork);
        
        console.log('✅ Network Selector initialized');
        
    } catch (error) {
        console.error('❌ Error initializing Network Selector:', error);
        // Default to Mainnet on error
        await switchNetwork('mainnet');
    }
}

// ==========================================
// 🔒 LOCK/UNLOCK FUNCTIONS
// ==========================================

/**
 * Handle unlock wallet button click
 */
async function handleUnlockWallet() {
    try {
        console.log('🔓 ========== UNLOCKING WALLET ==========');
        
        const passwordInput = document.getElementById('unlock-password');
        const password = passwordInput.value.trim();
        
        if (!password) {
            showNotification('❌ Please enter your password', 'error');
            return;
        }
        
        if (password.length < 6) {
            showNotification('❌ Password must be at least 6 characters', 'error');
            return;
        }
        
        // Show loading
        showLoading('Unlocking wallet...');
        
        // Call background to unlock
        const response = await sendMessage({
            action: 'unlockWallet',
            data: { password }
        });
        
        hideLoading();
        
        if (response.success) {
            // 🔒 SECURITY: Clear password field immediately
            clearAllPasswordFields();
            console.log('✅ Wallet unlocked successfully');
            showNotification('✅ Welcome back!', 'success');
            
            // 🔒 SEGURANÇA: Limpar password imediatamente após uso
            passwordInput.value = '';
            console.log('✅ Password cleared from input field');
            
            // 💾 LIMPAR CACHE AO UNLOCK (dados podem ter mudado)
            clearAllCache();
            
            // 🔔 NOTIFICAR TODAS AS PÁGINAS QUE A WALLET FOI DESBLOQUEADA
            try {
                // Buscar todas as tabs
                const tabs = await chrome.tabs.query({});
                for (const tab of tabs) {
                    try {
                        // Enviar mensagem para content script de cada tab
                        await chrome.tabs.sendMessage(tab.id, {
                            type: 'MYWALLET_UNLOCKED',
                            address: response.address
                        });
                        console.log(`📡 Notified tab ${tab.id} about unlock`);
                    } catch (e) {
                        // Tab pode não ter content script, ignorar
                    }
                }
            } catch (error) {
                console.error('⚠️ Error notifying tabs:', error);
            }
            
            // Show wallet screen and load data
            showScreen('wallet');
            await loadWalletData();
            
            // ⚡ Initialize KRAY L2 after unlock
            if (window.krayL2) {
                if (typeof window.krayL2.setShowScreen === 'function') {
                    window.krayL2.setShowScreen(showScreen);
                }
                if (typeof window.krayL2.initL2 === 'function') {
                    console.log('⚡ Initializing L2 after unlock...');
                    setTimeout(() => window.krayL2.initL2(), 500);
                }
            }
        } else {
            console.error('❌ Failed to unlock:', response.error);
            showNotification('❌ ' + response.error, 'error');
            
            // Clear password input
            passwordInput.value = '';
            passwordInput.focus();
        }
        
    } catch (error) {
        hideLoading();
        console.error('❌ Error unlocking wallet:', error);
        showNotification('❌ Failed to unlock wallet', 'error');
    }
}

/**
 * Handle lock wallet button click (from Settings)
 */
async function handleLockWallet() {
    try {
        console.log('🔒 Locking wallet...');
        
        // Call background to lock
        const response = await sendMessage({ action: 'lockWallet' });
        
        if (response.success) {
            console.log('✅ Wallet locked successfully');
            showNotification('🔒 Wallet locked', 'success');
            
            // 🔒 SEGURANÇA: Limpar campo de password completamente
            const unlockPasswordInput = document.getElementById('unlock-password');
            if (unlockPasswordInput) {
                unlockPasswordInput.value = '';
                console.log('✅ Password field cleared for security');
            }
            
            // Show unlock screen
            showScreen('unlock');
        } else {
            showNotification('❌ Failed to lock wallet', 'error');
        }
        
    } catch (error) {
        console.error('❌ Error locking wallet:', error);
        showNotification('❌ Failed to lock wallet', 'error');
    }
}

/**
 * Handle auto-lock timeout change (from Settings)
 */
async function handleAutolockTimeoutChange(event) {
    try {
        const timeout = parseInt(event.target.value, 10);
        console.log(`🔒 Setting auto-lock timeout to: ${timeout} minutes`);
        
        // Call background to set timeout
        const response = await sendMessage({
            action: 'setAutolockTimeout',
            data: { timeout }
        });
        
        if (response.success) {
            console.log('✅ Auto-lock timeout set successfully');
            
            let message = '';
            if (timeout === 0) {
                message = '⏰ Auto-lock disabled';
            } else {
                message = `⏰ Auto-lock set to ${timeout} minutes`;
            }
            
            showNotification(message, 'success');
        } else {
            showNotification('❌ Failed to set auto-lock timeout', 'error');
        }
        
    } catch (error) {
        console.error('❌ Error setting auto-lock timeout:', error);
        showNotification('❌ Failed to set auto-lock timeout', 'error');
    }
}

/**
 * Load auto-lock setting on Settings screen
 */
async function loadAutolockSetting() {
    try {
        const result = await chrome.storage.local.get(['autolockTimeout']);
        const timeout = result.autolockTimeout !== undefined ? result.autolockTimeout : 15; // Default 15 minutes
        
        const select = document.getElementById('autolock-timeout');
        if (select) {
            select.value = timeout.toString();
        }
    } catch (error) {
        console.error('❌ Error loading autolock setting:', error);
    }
}

// ═══════════════════════════════════════════════════════════════
// 🏪 L1 MARKET - INSCRIPTIONS & RUNES MARKETPLACE
// ═══════════════════════════════════════════════════════════════

const INSCRIPTIONS_MARKET_API = 'https://kraywallet-backend.onrender.com/api/atomic-swap';
const RUNES_MARKET_API = 'https://kraywallet-backend.onrender.com/api/runes-atomic-swap';

/**
 * Load Market Data - Called when Market tab is clicked
 */
async function loadMarketData() {
    console.log('🏪 Loading market data...');
    
    // Load the active market tab FIRST (don't wait for stats)
    const activeSubtab = document.querySelector('.market-subtab.active');
    if (activeSubtab?.dataset.marketTab === 'runes-market') {
        loadRunesMarket();
    } else {
        loadInscriptionsMarket();
    }
    
    // Load stats in background (don't block listings)
    try {
        const [inscStatsRes, runesStatsRes] = await Promise.all([
            fetch(`${INSCRIPTIONS_MARKET_API}/stats`).catch(() => null),
            fetch(`${RUNES_MARKET_API}/stats`).catch(() => null)
        ]);
        
        let totalListings = 0;
        let totalVolume = 0;
        let totalSales = 0;
        
        // Inscriptions stats - uses "marketplace" key
        if (inscStatsRes && inscStatsRes.ok) {
            const inscStats = await inscStatsRes.json();
            console.log('📊 Inscriptions stats:', inscStats);
            // atomicSwap.js returns { success, marketplace: { active_listings, total_sales, total_volume } }
            const stats = inscStats.marketplace || inscStats.stats || {};
            if (inscStats.success) {
                totalListings += stats.active_listings || 0;
                totalVolume += stats.total_volume || 0;
                totalSales += stats.total_sales || 0;
            }
        }
        
        // Runes stats - uses "stats" key
        if (runesStatsRes && runesStatsRes.ok) {
            const runesStats = await runesStatsRes.json();
            console.log('📊 Runes stats:', runesStats);
            // runesAtomicSwap.js returns { success, stats: { active_listings, total_sales, total_volume } }
            const stats = runesStats.stats || runesStats.marketplace || {};
            if (runesStats.success) {
                totalListings += stats.active_listings || 0;
                totalVolume += stats.total_volume || 0;
                totalSales += stats.total_sales || 0;
            }
        }
        
        console.log('📊 Total stats:', { totalListings, totalVolume, totalSales });
        
        // Update UI - with debug
        const listingsEl = document.getElementById('market-stat-listings');
        const volumeEl = document.getElementById('market-stat-volume');
        const salesEl = document.getElementById('market-stat-sales');
        
        console.log('📊 Stats elements found:', { 
            listings: !!listingsEl, 
            volume: !!volumeEl, 
            sales: !!salesEl 
        });
        
        if (listingsEl) listingsEl.textContent = totalListings;
        if (volumeEl) volumeEl.textContent = formatSatsShort(totalVolume);
        if (salesEl) salesEl.textContent = totalSales;
        
        console.log('✅ Stats UI updated!');
        
    } catch (err) {
        console.error('Error loading market stats:', err);
    }
}

/**
 * Format sats to short format (1K, 1M, etc)
 */
function formatSatsShort(sats) {
    if (sats >= 100000000) return (sats / 100000000).toFixed(2) + ' BTC';
    if (sats >= 1000000) return (sats / 1000000).toFixed(1) + 'M';
    if (sats >= 1000) return (sats / 1000).toFixed(1) + 'K';
    return sats.toLocaleString();
}

/**
 * Load Inscriptions Market Listings
 */
async function loadInscriptionsMarket() {
    console.log('◉ Loading inscriptions market...');
    
    const listContainer = document.getElementById('inscriptions-market-list');
    listContainer.innerHTML = `
        <div class="loading-state">
            <div class="spinner"></div>
            <p>Loading listings...</p>
        </div>
    `;
    
    try {
        // Get connected wallet address
        let connectedAddress = '';
        try {
            const walletInfo = await sendMessage({ action: 'getWalletInfo' });
            connectedAddress = walletInfo?.data?.address || walletInfo?.address || '';
            console.log('📍 Connected address for market:', connectedAddress);
        } catch (e) {
            console.log('Could not get wallet address');
        }
        
        const res = await fetch(`${INSCRIPTIONS_MARKET_API}/?sort=price_asc&limit=50`);
        const data = await res.json();
        
        if (!data.success || !data.listings || data.listings.length === 0) {
            listContainer.innerHTML = `
                <div style="text-align: center; padding: 40px;">
                    <div style="font-size: 48px; margin-bottom: 16px;">◉</div>
                    <div style="color: #888; font-size: 14px;">No inscriptions for sale</div>
                    <div style="color: #666; font-size: 12px; margin-top: 8px;">Be the first to list!</div>
                </div>
            `;
            return;
        }
        
        listContainer.innerHTML = '';
        
        data.listings.forEach(listing => {
            const card = document.createElement('div');
            card.className = 'market-listing-card';
            
            // Always try to load thumbnail from ordinals.com
            const thumbnailUrl = `https://ordinals.com/content/${listing.inscription_id}`;
            
            // Check if this listing belongs to connected user
            const isOwner = connectedAddress && listing.seller_payout_address === connectedAddress;
            
            // Display inscription number or first 8 chars of ID
            const displayNumber = listing.inscription_number 
                ? `#${listing.inscription_number}` 
                : `#${listing.inscription_id.slice(0, 8)}`;
            
            // 🌟 Get rarity badge (from listing if available, or default to Common)
            const satRarity = listing.satRarity;
            let rarityBadge = '<span style="background: #88888830; color: #888; padding: 2px 6px; border-radius: 4px; font-size: 9px; font-weight: 600; text-transform: uppercase;">Common</span>';
            if (satRarity && satRarity.attributes && satRarity.attributes.length > 0) {
                const primary = satRarity.attributes[0];
                rarityBadge = `<span style="background: ${primary.color}30; color: ${primary.color}; padding: 2px 6px; border-radius: 4px; font-size: 9px; font-weight: 600; text-transform: uppercase; border: 1px solid ${primary.color}50;">${primary.emoji} ${primary.name}</span>`;
            }
            
            card.innerHTML = `
                <div class="market-listing-thumb">
                    <img src="${thumbnailUrl}" alt="${listing.inscription_id}" onerror="this.parentElement.innerHTML='◉'">
                </div>
                <div class="market-listing-info">
                    <div class="market-listing-name">${displayNumber}</div>
                    <div class="market-listing-rarity" style="margin-top: 2px;">${rarityBadge}</div>
                </div>
                <div class="market-listing-price">
                    <div class="market-listing-price-value">${formatSatsShort(listing.price_sats)}</div>
                    ${isOwner 
                        ? `<button class="market-listing-cancel-btn" style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); margin-top: 4px;">Cancel</button>`
                        : `<button class="market-listing-buy-btn">Buy</button>`
                    }
                </div>
            `;
            
            if (isOwner) {
                card.querySelector('.market-listing-cancel-btn').addEventListener('click', (e) => {
                    e.stopPropagation();
                    cancelInscriptionListing(listing);
                });
            } else {
                card.querySelector('.market-listing-buy-btn').addEventListener('click', (e) => {
                    e.stopPropagation();
                    buyInscriptionListing(listing);
                });
            }
            
            listContainer.appendChild(card);
        });
        
    } catch (err) {
        console.error('Error loading inscriptions market:', err);
        listContainer.innerHTML = `
            <div style="text-align: center; padding: 40px; color: #ff6b35;">
                Error loading listings
            </div>
        `;
    }
}

/**
 * Load Runes Market Listings
 */
async function loadRunesMarket() {
    console.log('🪙🪙🪙 RUNE-SYMBOL-FIX-v1: Loading runes market...');
    
    const listContainer = document.getElementById('runes-market-list');
    listContainer.innerHTML = `
        <div class="loading-state">
            <div class="spinner"></div>
            <p>Loading listings...</p>
        </div>
    `;
    
    try {
        // Get connected wallet address
        let connectedAddress = '';
        try {
            const walletInfo = await sendMessage({ action: 'getWalletInfo' });
            connectedAddress = walletInfo?.data?.address || walletInfo?.address || '';
        } catch (e) {
            console.log('Could not get wallet address');
        }
        
        const res = await fetch(`${RUNES_MARKET_API}/?sort=price_asc&limit=50`);
        const data = await res.json();
        
        if (!data.success || !data.listings || data.listings.length === 0) {
            listContainer.innerHTML = `
                <div style="text-align: center; padding: 40px;">
                    <div style="font-size: 48px; margin-bottom: 16px;">⧈</div>
                    <div style="color: #888; font-size: 14px;">No runes for sale</div>
                    <div style="color: #666; font-size: 12px; margin-top: 8px;">Be the first to list!</div>
                </div>
            `;
            return;
        }
        
        listContainer.innerHTML = '';
        
        console.log('🪙 RAW API Response - listings:', JSON.stringify(data.listings.map(l => ({ name: l.rune_name, symbol: l.rune_symbol })), null, 2));
        
        data.listings.forEach(listing => {
            const card = document.createElement('div');
            card.className = 'market-listing-card rune-listing';
            
            const displayAmount = Number(listing.sell_amount) / Math.pow(10, listing.divisibility || 0);
            const pricePerToken = (listing.price_sats / displayAmount).toFixed(2);
            
            // Check if this listing belongs to connected user
            const isOwner = connectedAddress && listing.seller_payout_address === connectedAddress;
            
            // Build thumbnail URL using parent inscription endpoint (same as runes tab)
            // Remove trailing symbol from rune_name (e.g., "THE•WOJAK•RUNE ⍵" → "THE•WOJAK•RUNE")
            const cleanRuneName = (listing.rune_name || '').replace(/\s+\S*$/, '').trim() || listing.rune_id;
            const thumbUrl = `https://kraywallet-backend.onrender.com/api/rune-thumbnail/parent/${encodeURIComponent(cleanRuneName)}`;
            
            // Use rune_symbol from listing (same as runes tab uses rune.symbol)
            const runeSymbol = listing.rune_symbol || '⧈';
            console.log(`🪙 Market listing: ${listing.rune_name}, symbol from API: "${listing.rune_symbol}", using: "${runeSymbol}"`);
            
            card.innerHTML = `
                <div class="market-listing-thumb" style="position: relative; overflow: hidden;">
                    <div class="market-thumb-symbol" style="width: 100%; height: 100%; display: flex; align-items: center; justify-content: center; font-size: 28px; background: linear-gradient(135deg, #1a0f0a 0%, #0d0d0d 100%); position: absolute; top: 0; left: 0; z-index: 1;">${runeSymbol}</div>
                    <img src="${thumbUrl}" 
                         alt="${listing.rune_name}" 
                         class="market-thumb-img"
                         style="width: 100%; height: 100%; object-fit: cover; border-radius: 8px; position: absolute; top: 0; left: 0; z-index: 2;"
                         onerror="this.style.display='none';">
                </div>
                <div class="market-listing-info">
                    <div class="market-listing-name">${listing.rune_name || listing.rune_id}</div>
                    <div class="market-listing-amount">${formatRuneAmount(displayAmount)} tokens</div>
                </div>
                <div class="market-listing-price">
                    <div class="market-listing-price-value">${formatSatsShort(listing.price_sats)}</div>
                    <div class="market-listing-price-per">${pricePerToken} sat/token</div>
                    ${isOwner 
                        ? `<button class="market-listing-cancel-btn" style="background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);">Cancel</button>`
                        : `<button class="market-listing-buy-btn">Buy</button>`
                    }
                </div>
            `;
            
            if (isOwner) {
                card.querySelector('.market-listing-cancel-btn').addEventListener('click', (e) => {
                    e.stopPropagation();
                    cancelRuneListing(listing);
                });
            } else {
                card.querySelector('.market-listing-buy-btn').addEventListener('click', (e) => {
                    e.stopPropagation();
                    buyRuneListing(listing);
                });
            }
            
            listContainer.appendChild(card);
        });
        
    } catch (err) {
        console.error('Error loading runes market:', err);
        listContainer.innerHTML = `
            <div style="text-align: center; padding: 40px; color: #ff6b35;">
                Error loading listings
            </div>
        `;
    }
}

/**
 * Buy Inscription Listing
 */
async function buyInscriptionListing(listing) {
    console.log('🛒 Buying inscription:', listing.inscription_id);
    showNotification('Opening inscription purchase...', 'info');
    window.open(`https://kray.space/ordinals.html?inscription=${listing.inscription_id}`, '_blank');
}

/**
 * Cancel Inscription Listing
 */
async function cancelInscriptionListing(listing) {
    console.log('🗑️ Cancelling inscription listing:', listing.order_id);
    
    if (!confirm(`Cancel listing for inscription #${listing.inscription_number || listing.inscription_id.slice(0, 8)}?`)) {
        return;
    }
    
    try {
        showNotification('Cancelling listing...', 'info');
        
        const walletInfo = await sendMessage({ action: 'getWalletInfo' });
        const sellerAddress = walletInfo?.data?.address || walletInfo?.address;
        
        if (!sellerAddress) {
            throw new Error('Wallet not connected');
        }
        
        const res = await fetch(`${INSCRIPTIONS_MARKET_API}/${listing.order_id}`, {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ seller_address: sellerAddress })
        });
        
        const data = await res.json();
        
        if (data.success || res.ok) {
            showNotification('✅ Listing cancelled!', 'success');
            loadInscriptionsMarket(); // Reload listings
        } else {
            throw new Error(data.error || 'Failed to cancel');
        }
    } catch (err) {
        console.error('Cancel error:', err);
        showNotification(err.message, 'error');
    }
}

/**
 * Buy Rune Listing
 */
async function buyRuneListing(listing) {
    console.log('🛒 Buying rune:', listing.rune_name);
    showNotification('Opening rune purchase...', 'info');
    window.open(`https://kray.space/runes.html?rune=${listing.rune_id}`, '_blank');
}

/**
 * Cancel Rune Listing
 */
async function cancelRuneListing(listing) {
    console.log('🗑️ Cancelling rune listing:', listing.order_id);
    
    if (!confirm(`Cancel listing for ${listing.rune_name || listing.rune_id}?`)) {
        return;
    }
    
    try {
        showNotification('Cancelling listing...', 'info');
        
        const walletInfo = await sendMessage({ action: 'getWalletInfo' });
        const sellerAddress = walletInfo?.data?.address || walletInfo?.address;
        
        if (!sellerAddress) {
            throw new Error('Wallet not connected');
        }
        
        const res = await fetch(`${RUNES_MARKET_API}/${listing.order_id}`, {
            method: 'DELETE',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ seller_address: sellerAddress })
        });
        
        const data = await res.json();
        
        if (data.success || res.ok) {
            showNotification('✅ Listing cancelled!', 'success');
            loadRunesMarket(); // Reload listings
        } else {
            throw new Error(data.error || 'Failed to cancel');
        }
    } catch (err) {
        console.error('Cancel error:', err);
        showNotification(err.message, 'error');
    }
}

/**
 * Show List Inscription Modal - Displays user's inscriptions to select for listing
 */
async function showListInscriptionModal() {
    console.log('◉ Opening inscription selector for market listing...');
    
    const selectorScreen = document.createElement('div');
    selectorScreen.className = 'rune-details-screen';
    selectorScreen.id = 'inscription-selector-screen';
    
    selectorScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="inscription-selector-back">←</button>
            <div class="rune-details-title">📝 Select Inscription to List</div>
        </div>
        
        <div class="rune-details-content" style="padding: 20px;">
            <div id="inscription-selector-loading" style="text-align: center; padding: 40px;">
                <div class="loading-spinner" style="margin: 0 auto 16px;"></div>
                <div style="color: #888;">Loading your inscriptions...</div>
            </div>
            <div id="inscription-selector-list" style="display: none; max-height: 400px; overflow-y: auto; display: grid; grid-template-columns: repeat(3, 1fr); gap: 12px;"></div>
            <div id="inscription-selector-empty" style="display: none; text-align: center; padding: 40px;">
                <div style="font-size: 48px; margin-bottom: 16px;">◉</div>
                <div style="color: #888; font-size: 14px;">No inscriptions found in your wallet</div>
            </div>
        </div>
    `;
    
    document.body.appendChild(selectorScreen);
    
    // Back button
    document.getElementById('inscription-selector-back').addEventListener('click', () => {
        selectorScreen.remove();
    });
    
    // Load user's inscriptions
    try {
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const walletAddress = walletInfo?.data?.address || walletInfo?.address;
        
        if (!walletAddress) {
            throw new Error('Wallet not connected');
        }
        
        // Fetch inscriptions via background script
        const response = await chrome.runtime.sendMessage({
            action: 'getInscriptions',
            data: { address: walletAddress }
        });
        
        console.log('📦 Inscriptions for listing:', response);
        
        // Fetch current listings to identify already listed ones (by this user)
        let listedInscriptionsMap = new Map(); // inscription_id -> listing data
        try {
            const listingsRes = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/listings`);
            const listingsData = await listingsRes.json();
            console.log('📋 All listings from API:', listingsData);
            
            if (listingsData.success && listingsData.listings) {
                // Filter by: open status (case-insensitive) AND owned by current user
                listingsData.listings
                    .filter(l => l.status?.toUpperCase() === 'OPEN' && (l.seller_payout_address === walletAddress || l.creator_address === walletAddress))
                    .forEach(l => {
                        console.log(`   📌 User's listing: ${l.inscription_id} (order: ${l.order_id})`);
                        listedInscriptionsMap.set(l.inscription_id, l);
                    });
                console.log(`📋 Found ${listedInscriptionsMap.size} listings by this user`);
            }
        } catch (e) {
            console.warn('Could not fetch existing listings:', e);
        }
        
        document.getElementById('inscription-selector-loading').style.display = 'none';
        
        if (!response?.success || !response?.inscriptions || response.inscriptions.length === 0) {
            document.getElementById('inscription-selector-empty').style.display = 'block';
            return;
        }
        
        const listContainer = document.getElementById('inscription-selector-list');
        listContainer.style.display = 'grid';
        
        // Show ALL inscriptions (available + listed)
        response.inscriptions.forEach(inscription => {
            const isListed = listedInscriptionsMap.has(inscription.id);
            const listing = listedInscriptionsMap.get(inscription.id);
            
            const inscItem = document.createElement('div');
            inscItem.style.cssText = `
                cursor: pointer;
                border-radius: 12px;
                overflow: hidden;
                background: var(--color-surface, #1a1a2e);
                border: 2px solid ${isListed ? '#f59e0b' : 'transparent'};
                transition: all 0.2s ease;
                position: relative;
                ${isListed ? 'opacity: 0.7;' : ''}
            `;
            
            inscItem.addEventListener('mouseenter', () => {
                inscItem.style.borderColor = isListed ? '#ef4444' : '#8b5cf6';
                inscItem.style.transform = 'scale(1.02)';
                inscItem.style.opacity = '1';
            });
            inscItem.addEventListener('mouseleave', () => {
                inscItem.style.borderColor = isListed ? '#f59e0b' : 'transparent';
                inscItem.style.transform = 'scale(1)';
                inscItem.style.opacity = isListed ? '0.7' : '1';
            });
            
            const previewUrl = inscription.preview || `https://ordinals.com/content/${inscription.id}`;
            
            // Listed badge
            const listedBadge = isListed ? `
                <div style="position: absolute; top: 4px; right: 4px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: #000; font-size: 8px; font-weight: 700; padding: 2px 6px; border-radius: 4px; z-index: 1;">
                    LISTED
                </div>
            ` : '';
            
            // Cancel overlay for listed items
            const cancelOverlay = isListed ? `
                <div style="position: absolute; bottom: 0; left: 0; right: 0; background: linear-gradient(0deg, rgba(239,68,68,0.9) 0%, transparent 100%); padding: 8px 4px 4px; text-align: center;">
                    <span style="font-size: 10px; color: white; font-weight: 600;">❌ Cancel</span>
                </div>
            ` : '';
            
            inscItem.innerHTML = `
                ${listedBadge}
                <div style="aspect-ratio: 1; background: #0a0a0f; display: flex; align-items: center; justify-content: center; overflow: hidden; ${isListed ? 'filter: grayscale(50%);' : ''}">
                    <img src="${previewUrl}" 
                         style="width: 100%; height: 100%; object-fit: cover;"
                         onerror="this.style.display='none'; this.parentElement.innerHTML='<span style=\\'font-size: 24px;\\'>◉</span>';"
                    />
                </div>
                ${cancelOverlay}
                <div style="padding: 8px; text-align: center;">
                    <div style="font-size: 11px; color: ${isListed ? '#f59e0b' : '#888'}; white-space: nowrap; overflow: hidden; text-overflow: ellipsis;">
                        #${inscription.number || '?'}${isListed ? ` • ${(listing.price_sats / 100000000).toFixed(8)} BTC` : ''}
                    </div>
                </div>
            `;
            
            inscItem.addEventListener('click', async () => {
                selectorScreen.remove();
                if (isListed && listing) {
                    // Show inscription details with Edit/Cancel options
                    showInscriptionDetails(inscription);
                } else {
                    // List the inscription
                    showListMarketModal(inscription);
                }
            });
            
            listContainer.appendChild(inscItem);
        });
        
    } catch (error) {
        console.error('Error loading inscriptions:', error);
        document.getElementById('inscription-selector-loading').style.display = 'none';
        document.getElementById('inscription-selector-empty').style.display = 'block';
        document.getElementById('inscription-selector-empty').innerHTML = `
            <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
            <div style="color: #ff6b35; font-size: 14px;">${error.message}</div>
        `;
    }
}

/**
 * Show Rune Packages - Displays UTXO packages for a SPECIFIC rune
 * Called from rune details screen "List for Sale" button
 */
async function showRunePackages(rune) {
    console.log('📦 Opening packages for:', rune.name);
    
    const runeName = rune.displayName || rune.name;
    const utxos = rune.utxos || [];
    
    const packagesScreen = document.createElement('div');
    packagesScreen.className = 'rune-details-screen';
    packagesScreen.id = 'rune-packages-screen';
    
    packagesScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="packages-back">←</button>
            <div class="rune-details-title">📦 Select Package to List</div>
        </div>
        
        <div class="rune-details-content" style="padding: 20px;">
            <div id="packages-loading" style="text-align: center; padding: 40px;">
                <div class="loading-spinner" style="margin: 0 auto 16px;"></div>
                <div style="color: #888;">Loading packages...</div>
            </div>
            <div id="packages-list" style="display: none; max-height: 400px; overflow-y: auto;"></div>
        </div>
    `;
    
    document.body.appendChild(packagesScreen);
    
    // Back button
    document.getElementById('packages-back').addEventListener('click', () => {
        packagesScreen.remove();
        showRuneDetails(rune); // Go back to rune details
    });
    
    try {
        // Get wallet address
        const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const walletAddress = response?.data?.address || response?.address;
        
        // Fetch current listings for this rune
        let listedUtxosMap = new Map();
        try {
            const listingsRes = await fetch(`https://kraywallet-backend.onrender.com/api/runes-atomic-swap/?sort=price_asc&limit=500`);
            const listingsData = await listingsRes.json();
            
            console.log('📋 Fetched rune listings:', listingsData?.listings?.length || 0);
            
            if (listingsData.success && listingsData.listings) {
                // Normalize rune name for comparison (handle • vs other bullet chars)
                const normalizeRuneName = (name) => {
                    if (!name) return '';
                    return name.replace(/[\u2022\u2219\u00B7]/g, '•').trim().toUpperCase();
                };
                const targetRuneName = normalizeRuneName(rune.name || rune.displayName);
                
                listingsData.listings
                    .filter(l => {
                        if (l.status?.toUpperCase() !== 'OPEN') return false;
                        if (l.seller_payout_address !== walletAddress) return false;
                        // Compare normalized rune names
                        const listingRuneName = normalizeRuneName(l.rune_name);
                        return listingRuneName === targetRuneName || listingRuneName.includes(targetRuneName.split('•')[0]);
                    })
                    .forEach(l => {
                        const utxoKey = `${l.seller_txid}:${l.seller_vout}`;
                        console.log(`   📌 Found listed UTXO: ${utxoKey} @ ${l.price_sats} sats`);
                        listedUtxosMap.set(utxoKey, l);
                    });
                    
                console.log(`📋 User has ${listedUtxosMap.size} listed packages for ${rune.name}`);
            }
        } catch (e) {
            console.warn('Could not fetch listings:', e);
        }
        
        document.getElementById('packages-loading').style.display = 'none';
        
        const listContainer = document.getElementById('packages-list');
        listContainer.style.display = 'block';
        
        // Header with rune info
        const header = document.createElement('div');
        header.style.cssText = `
            padding: 12px;
            display: flex;
            align-items: center;
            gap: 12px;
            background: var(--color-surface, #1a1a2e);
            border-radius: 12px;
            margin-bottom: 12px;
        `;
        
        const thumbnailHtml = rune.thumbnail 
            ? `<img src="${rune.thumbnail}" style="width: 40px; height: 40px; object-fit: cover; border-radius: 8px;" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
               <span style="display: none; font-size: 24px;">${rune.symbol || '⧈'}</span>`
            : `<span style="font-size: 24px;">${rune.symbol || '⧈'}</span>`;
        
        header.innerHTML = `
            <div style="width: 40px; height: 40px; border-radius: 8px; background: #0a0a0f; display: flex; align-items: center; justify-content: center; overflow: hidden;">
                ${thumbnailHtml}
            </div>
            <div>
                <div style="font-weight: 600; color: #fff; font-size: 15px;">${runeName}</div>
                <div style="font-size: 12px; color: #888;">${utxos.length} pacote${utxos.length > 1 ? 's' : ''} disponíve${utxos.length > 1 ? 'is' : 'l'}</div>
            </div>
        `;
        listContainer.appendChild(header);
        
        // Show each UTXO as a package
        utxos.forEach((utxo, index) => {
            const utxoKey = `${utxo.txid}:${utxo.vout}`;
            const listing = listedUtxosMap.get(utxoKey);
            const isListed = !!listing;
            const utxoAmount = parseInt(utxo.amount) || 0;
            
            const utxoItem = document.createElement('div');
            utxoItem.style.cssText = `
                cursor: pointer;
                display: flex;
                align-items: center;
                gap: 12px;
                padding: 14px 16px;
                background: ${isListed ? 'rgba(245, 158, 11, 0.1)' : 'var(--color-surface, #1a1a2e)'};
                border-radius: 12px;
                margin-bottom: 8px;
                transition: all 0.2s ease;
                position: relative;
                border: 2px solid ${isListed ? '#f59e0b' : 'transparent'};
                opacity: ${isListed ? '0.7' : '1'};
            `;
            
            utxoItem.addEventListener('mouseenter', () => {
                utxoItem.style.background = isListed ? 'rgba(245, 158, 11, 0.2)' : 'var(--color-surface-hover, #252540)';
                utxoItem.style.transform = 'translateX(4px)';
                utxoItem.style.opacity = '1';
            });
            utxoItem.addEventListener('mouseleave', () => {
                utxoItem.style.background = isListed ? 'rgba(245, 158, 11, 0.1)' : 'var(--color-surface, #1a1a2e)';
                utxoItem.style.transform = 'translateX(0)';
                utxoItem.style.opacity = isListed ? '0.7' : '1';
            });
            
            const listedBadge = isListed ? `
                <div style="position: absolute; top: -4px; right: -4px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: #000; font-size: 8px; font-weight: 700; padding: 2px 6px; border-radius: 4px; z-index: 1;">
                    LISTED
                </div>
            ` : '';
            
            const priceInfo = isListed && listing.price_sats 
                ? `<div style="font-size: 11px; color: #f59e0b; margin-top: 2px;">${listing.price_sats.toLocaleString()} sats</div>`
                : '';
            
            utxoItem.innerHTML = `
                ${listedBadge}
                <div style="width: 44px; height: 44px; border-radius: 10px; background: linear-gradient(135deg, #ff6b35 0%, #f59e0b 100%); display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                    <span style="font-size: 18px; color: white;">📦</span>
                </div>
                <div style="flex: 1; min-width: 0;">
                    <div style="font-weight: 600; color: #fff; font-size: 14px;">
                        Pacote ${index + 1}: <span style="color: ${isListed ? '#f59e0b' : '#22c55e'};">${formatRuneAmount(utxoAmount, rune.divisibility || 0)} ${rune.symbol || ''}</span>
                    </div>
                    <div style="font-size: 10px; color: #666; font-family: monospace; margin-top: 2px;">
                        ${utxo.txid.slice(0, 10)}...${utxo.txid.slice(-8)}:${utxo.vout}
                    </div>
                    ${priceInfo}
                </div>
                <div style="font-size: 22px; color: ${isListed ? '#ef4444' : '#22c55e'};">
                    ${isListed ? '📋' : '→'}
                </div>
            `;
            
            utxoItem.addEventListener('click', () => {
                packagesScreen.remove();
                if (isListed && listing) {
                    // Already listed - go back to details
                    showNotification('This package is already listed. Cancel it first to re-list.', 'info');
                    showRuneDetails(rune);
                } else {
                    // Create rune object with specific UTXO
                    // Convert raw utxoAmount to display amount using divisibility
                    const displayUtxoAmount = utxoAmount / Math.pow(10, rune.divisibility || 0);
                    const runeWithUtxo = {
                        ...rune,
                        _selectedUtxo: utxo,
                        _utxoAmount: utxoAmount, // Keep raw for PSBT
                        _rawAmount: utxoAmount,  // Explicit raw amount
                        amount: displayUtxoAmount, // Display amount for UI
                        utxos: [utxo]
                    };
                    showListRuneScreen(runeWithUtxo, false);
                }
            });
            
            listContainer.appendChild(utxoItem);
        });
        
        // If no packages available
        if (utxos.length === 0) {
            listContainer.innerHTML += `
                <div style="text-align: center; padding: 40px; color: #888;">
                    <div style="font-size: 48px; margin-bottom: 16px;">📦</div>
                    <div>No packages available</div>
                </div>
            `;
        }
        
    } catch (error) {
        console.error('Error loading packages:', error);
        document.getElementById('packages-loading').style.display = 'none';
        document.getElementById('packages-list').innerHTML = `
            <div style="text-align: center; padding: 40px;">
                <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
                <div style="color: #ff6b35;">${error.message}</div>
            </div>
        `;
    }
}

/**
 * Show Rune Selector - Displays user's runes to select for listing
 */
async function showRuneSelector() {
    console.log('🪙 Opening rune selector for market listing...');
    
    const selectorScreen = document.createElement('div');
    selectorScreen.className = 'rune-details-screen';
    selectorScreen.id = 'rune-selector-screen';
    
    selectorScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="rune-selector-back">←</button>
            <div class="rune-details-title">📝 Select Rune to List</div>
        </div>
        
        <div class="rune-details-content" style="padding: 20px;">
            <div id="rune-selector-loading" style="text-align: center; padding: 40px;">
                <div class="loading-spinner" style="margin: 0 auto 16px;"></div>
                <div style="color: #888;">Loading your runes...</div>
            </div>
            <div id="rune-selector-list" style="display: none; max-height: 400px; overflow-y: auto;"></div>
            <div id="rune-selector-empty" style="display: none; text-align: center; padding: 40px;">
                <div style="font-size: 48px; margin-bottom: 16px;">🪙</div>
                <div style="color: #888; font-size: 14px;">No runes found in your wallet</div>
            </div>
        </div>
    `;
    
    document.body.appendChild(selectorScreen);
    
    // Back button
    document.getElementById('rune-selector-back').addEventListener('click', () => {
        selectorScreen.remove();
    });
    
    // Load user's runes
    try {
        const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const walletAddress = response?.data?.address || response?.address;
        
        if (!walletAddress) {
            throw new Error('Wallet not connected');
        }
        
        // Fetch runes from our API
        const runesRes = await fetch(`https://kraywallet-backend.onrender.com/api/runes/fast/${walletAddress}`);
        const runesData = await runesRes.json();
        
        // Fetch current rune listings by this user
        let listedRunesMap = new Map(); // rune_name -> { listing, listedAmount }
        try {
            const listingsRes = await fetch(`https://kraywallet-backend.onrender.com/api/runes-atomic-swap/?sort=price_asc&limit=100`);
            const listingsData = await listingsRes.json();
            console.log('📋 All rune listings from API:', listingsData);
            
            if (listingsData.success && listingsData.listings) {
                listingsData.listings
                    .filter(l => l.status?.toUpperCase() === 'OPEN' && l.seller_payout_address === walletAddress)
                    .forEach(l => {
                        const existing = listedRunesMap.get(l.rune_name) || { listings: [], totalListed: 0 };
                        existing.listings.push(l);
                        existing.totalListed += parseInt(l.sell_amount) || 0;
                        listedRunesMap.set(l.rune_name, existing);
                        console.log(`   📌 User's rune listing: ${l.rune_name} - ${l.sell_amount} listed (order: ${l.order_id})`);
                    });
                console.log(`📋 Found ${listedRunesMap.size} rune types listed by this user`);
            }
        } catch (e) {
            console.warn('Could not fetch rune listings:', e);
        }
        
        document.getElementById('rune-selector-loading').style.display = 'none';
        
        if (!runesData.success || !runesData.runes || runesData.runes.length === 0) {
            document.getElementById('rune-selector-empty').style.display = 'block';
            return;
        }
        
        const listContainer = document.getElementById('rune-selector-list');
        listContainer.style.display = 'block';
        
        // Build a map of listed UTXOs: txid:vout -> listing
        let listedUtxosMap = new Map();
        listedRunesMap.forEach((info, runeName) => {
            info.listings.forEach(listing => {
                const utxoKey = `${listing.seller_txid}:${listing.seller_vout}`;
                listedUtxosMap.set(utxoKey, listing);
            });
        });
        
        runesData.runes.forEach(rune => {
            const runeName = rune.displayName || rune.name;
            const utxos = rune.utxos || [];
            const totalBalance = parseInt(rune.amount || rune.balance || 0);
            
            // Create header for this rune type
            const runeHeader = document.createElement('div');
            runeHeader.style.cssText = `
                padding: 12px 8px 8px;
                display: flex;
                align-items: center;
                gap: 10px;
                border-bottom: 1px solid rgba(255,255,255,0.1);
                margin-bottom: 8px;
            `;
            
            // Thumbnail
            const thumbnailHtml = rune.thumbnail 
                ? `<img src="${rune.thumbnail}" style="width: 32px; height: 32px; object-fit: cover; border-radius: 6px;" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                   <span style="display: none; font-size: 20px;">${rune.symbol || '⧈'}</span>`
                : `<span style="font-size: 20px;">${rune.symbol || '⧈'}</span>`;
            
            runeHeader.innerHTML = `
                <div style="width: 32px; height: 32px; border-radius: 6px; background: #0a0a0f; display: flex; align-items: center; justify-content: center; overflow: hidden;">
                    ${thumbnailHtml}
                </div>
                <div style="flex: 1;">
                    <div style="font-weight: 600; color: #fff; font-size: 14px;">${runeName}</div>
                    <div style="font-size: 11px; color: #888;">${utxos.length} pacote${utxos.length > 1 ? 's' : ''} · ${formatRuneAmount(totalBalance)} total</div>
                </div>
            `;
            listContainer.appendChild(runeHeader);
            
            // Show each UTXO as a package
            utxos.forEach((utxo, index) => {
                const utxoKey = `${utxo.txid}:${utxo.vout}`;
                const listing = listedUtxosMap.get(utxoKey);
                const isListed = !!listing;
                const utxoAmount = parseInt(utxo.amount) || 0;
                
                const utxoItem = document.createElement('div');
                utxoItem.style.cssText = `
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    gap: 12px;
                    padding: 10px 12px;
                    margin-left: 16px;
                    background: ${isListed ? 'rgba(245, 158, 11, 0.1)' : 'var(--color-surface, #1a1a2e)'};
                    border-radius: 10px;
                    margin-bottom: 6px;
                    transition: all 0.2s ease;
                    position: relative;
                    border: 2px solid ${isListed ? '#f59e0b' : 'transparent'};
                    opacity: ${isListed ? '0.7' : '1'};
                `;
                
                utxoItem.addEventListener('mouseenter', () => {
                    utxoItem.style.background = isListed ? 'rgba(245, 158, 11, 0.2)' : 'var(--color-surface-hover, #252540)';
                    utxoItem.style.transform = 'translateX(4px)';
                    utxoItem.style.opacity = '1';
                });
                utxoItem.addEventListener('mouseleave', () => {
                    utxoItem.style.background = isListed ? 'rgba(245, 158, 11, 0.1)' : 'var(--color-surface, #1a1a2e)';
                    utxoItem.style.transform = 'translateX(0)';
                    utxoItem.style.opacity = isListed ? '0.7' : '1';
                });
                
                // Listed badge
                const listedBadge = isListed ? `
                    <div style="position: absolute; top: -4px; right: -4px; background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: #000; font-size: 8px; font-weight: 700; padding: 2px 6px; border-radius: 4px; z-index: 1;">
                        LISTED
                    </div>
                ` : '';
                
                // Price info for listed
                const priceInfo = isListed && listing.price_sats 
                    ? `<div style="font-size: 10px; color: #f59e0b;">${listing.price_sats.toLocaleString()} sats</div>`
                    : '';
                
                utxoItem.innerHTML = `
                    ${listedBadge}
                    <div style="width: 36px; height: 36px; border-radius: 8px; background: linear-gradient(135deg, #ff6b35 0%, #f59e0b 100%); display: flex; align-items: center; justify-content: center; flex-shrink: 0;">
                        <span style="font-size: 14px; color: white; font-weight: 700;">📦</span>
                    </div>
                    <div style="flex: 1; min-width: 0;">
                        <div style="font-weight: 600; color: #fff; font-size: 13px;">
                            Pacote ${index + 1}: <span style="color: ${isListed ? '#f59e0b' : '#22c55e'};">${formatRuneAmount(utxoAmount, rune.divisibility || 0)} ${rune.symbol || ''}</span>
                        </div>
                        <div style="font-size: 10px; color: #666; font-family: monospace;">
                            ${utxo.txid.slice(0, 8)}...${utxo.txid.slice(-6)}:${utxo.vout}
                        </div>
                        ${priceInfo}
                    </div>
                    <div style="font-size: 18px; color: ${isListed ? '#ef4444' : '#22c55e'};">
                        ${isListed ? '📋' : '→'}
                    </div>
                `;
                
                utxoItem.addEventListener('click', () => {
                    selectorScreen.remove();
                    if (isListed && listing) {
                        // Show rune details (can cancel from there)
                        showRuneDetails(rune);
                    } else {
                        // Create rune object with specific UTXO
                        // Convert raw utxoAmount to display amount using divisibility
                        const displayUtxoAmount = utxoAmount / Math.pow(10, rune.divisibility || 0);
                        const runeWithUtxo = {
                            ...rune,
                            _selectedUtxo: utxo,
                            _utxoAmount: utxoAmount, // Keep raw for PSBT
                            _rawAmount: utxoAmount,  // Explicit raw amount
                            amount: displayUtxoAmount, // Display amount for UI
                            utxos: [utxo] // Only include this UTXO
                        };
                        showListRuneScreen(runeWithUtxo, true);
                    }
                });
                
                listContainer.appendChild(utxoItem);
            });
        });
        
    } catch (error) {
        console.error('Error loading runes:', error);
        document.getElementById('rune-selector-loading').style.display = 'none';
        document.getElementById('rune-selector-empty').style.display = 'block';
        document.getElementById('rune-selector-empty').innerHTML = `
            <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
            <div style="color: #ff6b35; font-size: 14px;">${error.message}</div>
        `;
    }
}

/**
 * Show List Rune Screen - Creates a listing on the Runes Marketplace
 * @param {Object|null} rune - If null, shows rune selector first
 * @param {boolean} fromMarket - Whether called from market tab
 */
async function showListRuneScreen(rune = null, fromMarket = false) {
    // If no rune provided, show rune selector first
    if (!rune) {
        return showRuneSelector();
    }
    
    // Store if coming from market
    rune._fromMarket = fromMarket;
    
    console.log('📝 Listing rune for sale:', rune.name);
    
    const listScreen = document.createElement('div');
    listScreen.className = 'rune-details-screen';
    listScreen.id = 'list-rune-screen';
    
    const availableBalance = rune.amount || rune.balance || 0;
    const displayBalance = formatRuneAmount(availableBalance);
    
    listScreen.innerHTML = `
        <div class="rune-details-header">
            <button class="rune-details-back" id="list-rune-back">←</button>
            <div class="rune-details-title">📝 List ${rune.symbol || '⧈'} ${rune.displayName || rune.name}</div>
        </div>
        
        <div class="rune-details-content" style="padding: 20px;">
            <!-- Rune Preview with Thumbnail -->
            <div class="list-rune-preview" style="display: flex; align-items: center; gap: 16px; background: var(--color-bg-tertiary); padding: 16px; border-radius: 12px; margin-bottom: 20px;">
                <div class="list-rune-thumb" style="width: 56px; height: 56px; border-radius: 12px; background: #0a0a0f; display: flex; align-items: center; justify-content: center; overflow: hidden; flex-shrink: 0;">
                    ${rune.thumbnail 
                        ? `<img src="${rune.thumbnail}" style="width: 100%; height: 100%; object-fit: cover;" onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';">
                           <span style="display: none; font-size: 32px;">${rune.symbol || '⧈'}</span>`
                        : `<span style="font-size: 32px;">${rune.symbol || '⧈'}</span>`
                    }
                </div>
                <div class="list-rune-info">
                    <div style="font-weight: 600; font-size: 16px; color: white;">${rune.displayName || rune.name}</div>
                    <div style="font-size: 13px; color: #888; font-family: var(--font-mono);">Balance: ${displayBalance}</div>
                    ${rune.runeId ? `<div style="font-size: 11px; color: #f59e0b; font-family: var(--font-mono);">ID: ${rune.runeId}</div>` : ''}
                </div>
            </div>
            
            <!-- Amount Input -->
            <div class="form-group" style="margin-bottom: 20px;">
                <label style="display: block; font-size: 13px; color: #888; margin-bottom: 8px;">Amount to Sell</label>
                <div style="display: flex; gap: 8px;">
                    <input type="number" id="list-amount-input" class="input-field" placeholder="Enter amount" style="flex: 1;" step="any" max="${availableBalance}">
                    <button id="list-max-btn" class="btn btn-secondary" style="padding: 0 16px; font-size: 12px;">MAX</button>
                </div>
                <div style="font-size: 11px; color: #666; margin-top: 4px;">Available: ${displayBalance}</div>
            </div>
            
            <!-- Price Input -->
            <div class="form-group" style="margin-bottom: 20px;">
                <label style="display: block; font-size: 13px; color: #888; margin-bottom: 8px;">Price (sats)</label>
                <input type="number" id="list-price-input" class="input-field" placeholder="Total price in sats" min="546">
                <div id="price-per-token" style="font-size: 11px; color: #ff6b35; margin-top: 4px;">Price per token: --</div>
            </div>
            
            <!-- Summary -->
            <div id="list-summary" class="list-summary" style="background: var(--color-bg-tertiary); border-radius: 12px; padding: 16px; margin-bottom: 20px; display: none;">
                <div style="font-size: 14px; font-weight: 600; margin-bottom: 12px; color: white;">Listing Summary</div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                    <span style="color: #888;">Selling</span>
                    <span id="summary-amount" style="color: white; font-family: var(--font-mono);">--</span>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                    <span style="color: #888;">Price</span>
                    <span id="summary-price" style="color: #ff6b35; font-family: var(--font-mono);">--</span>
                </div>
                <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                    <span style="color: #888;">Per Token</span>
                    <span id="summary-per-token" style="color: white; font-family: var(--font-mono);">--</span>
                </div>
                <div style="display: flex; justify-content: space-between; border-top: 1px solid #333; padding-top: 8px; margin-top: 8px;">
                    <span style="color: #888;">Market Fee (2%)</span>
                    <span id="summary-fee" style="color: #666; font-family: var(--font-mono);">--</span>
                </div>
            </div>
            
            <!-- Status -->
            <div id="list-status" style="display: none; text-align: center; padding: 16px; background: var(--color-bg-tertiary); border-radius: 12px; margin-bottom: 20px;">
                <div class="loading-spinner" style="margin: 0 auto 12px;"></div>
                <div id="list-status-text" style="color: #888;">Processing...</div>
            </div>
            
            <!-- Actions -->
            <div style="display: flex; gap: 12px;">
                <button id="list-cancel-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="list-confirm-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #ff6b35 0%, #f59e0b 100%);">Create Listing</button>
            </div>
        </div>
    `;
    
    document.body.appendChild(listScreen);
    
    // Elements
    const amountInput = document.getElementById('list-amount-input');
    const priceInput = document.getElementById('list-price-input');
    const pricePerToken = document.getElementById('price-per-token');
    const listSummary = document.getElementById('list-summary');
    const listStatus = document.getElementById('list-status');
    const listStatusText = document.getElementById('list-status-text');
    
    // Update price per token
    function updateSummary() {
        const amount = parseFloat(amountInput.value) || 0;
        const price = parseInt(priceInput.value) || 0;
        
        if (amount > 0 && price >= 546) {
            const perToken = (price / amount).toFixed(4);
            const fee = Math.max(Math.floor(price * 0.02), 546);
            
            pricePerToken.textContent = `Price per token: ${perToken} sats`;
            
            document.getElementById('summary-amount').textContent = `${amount} ${rune.symbol || ''}`;
            document.getElementById('summary-price').textContent = `${price.toLocaleString()} sats`;
            document.getElementById('summary-per-token').textContent = `${perToken} sats`;
            document.getElementById('summary-fee').textContent = `${fee.toLocaleString()} sats`;
            
            listSummary.style.display = 'block';
        } else {
            pricePerToken.textContent = 'Price per token: --';
            listSummary.style.display = 'none';
        }
    }
    
    // Event Listeners
    amountInput.addEventListener('input', updateSummary);
    priceInput.addEventListener('input', updateSummary);
    
    document.getElementById('list-max-btn').addEventListener('click', () => {
        amountInput.value = availableBalance;
        updateSummary();
    });
    
    document.getElementById('list-rune-back').addEventListener('click', () => {
        listScreen.remove();
        if (rune._fromMarket) {
            loadRunesMarket();
        } else if (typeof showRuneDetails === 'function') {
            showRuneDetails(rune);
        }
    });
    
    document.getElementById('list-cancel-btn').addEventListener('click', () => {
        listScreen.remove();
        if (rune._fromMarket) {
            loadRunesMarket();
        } else if (typeof showRuneDetails === 'function') {
            showRuneDetails(rune);
        }
    });
    
    document.getElementById('list-confirm-btn').addEventListener('click', async () => {
        const amount = parseFloat(amountInput.value);
        const price = parseInt(priceInput.value);
        
        // Validations
        if (!amount || amount <= 0) {
            showNotification('Enter a valid amount', 'error');
            return;
        }
        if (amount > availableBalance) {
            showNotification('Amount exceeds balance', 'error');
            return;
        }
        if (!price || price < 546) {
            showNotification('Price must be at least 546 sats', 'error');
            return;
        }
        
        // Check UTXOs
        if (!rune.utxos || rune.utxos.length === 0) {
            showNotification('No UTXOs found for this rune', 'error');
            return;
        }
        
        // Show status
        listStatus.style.display = 'block';
        listStatusText.textContent = 'Creating listing...';
        document.getElementById('list-confirm-btn').disabled = true;
        
        try {
            // Get UTXO info
            const utxo = rune.utxos[0];
            
            // Fetch UTXO data from our API with mempool.space fallback
            listStatusText.textContent = 'Fetching UTXO data...';
            const outpoint = `${utxo.txid}:${utxo.vout}`;
            
            let scriptPubKey, utxoValue;
            
            try {
                const outputRes = await fetch(`https://kraywallet-backend.onrender.com/api/output/${outpoint}`);
                const outputData = await outputRes.json();
                
                if (outputData.success && outputData.script_pubkey) {
                    scriptPubKey = outputData.script_pubkey;
                    utxoValue = outputData.value;
                } else {
                    throw new Error('API returned no data');
                }
            } catch (apiError) {
                console.log('⚠️ Our API failed, falling back to mempool.space:', apiError.message);
                const txRes = await fetch(`https://mempool.space/api/tx/${utxo.txid}`);
                const txData = await txRes.json();
                scriptPubKey = txData.vout[utxo.vout].scriptpubkey;
                utxoValue = txData.vout[utxo.vout].value;
            }
            
            // Calculate raw amount based on divisibility
            const divisibility = rune.divisibility || 0;
            const rawSellAmount = Math.floor(amount * Math.pow(10, divisibility));
            const rawTotalAmount = rune.rawAmount || rawSellAmount;
            
            // Get wallet address
            const walletInfo = await sendMessage({ action: 'getWalletInfo' });
            const walletAddress = walletInfo?.data?.address || walletInfo?.address;
            
            console.log('📋 Wallet info for listing:', walletInfo);
            console.log('📋 Wallet address:', walletAddress);
            
            if (!walletAddress) {
                throw new Error('Wallet address not found. Please unlock your wallet.');
            }
            
            // Create listing on backend
            listStatusText.textContent = 'Creating listing...';
            
            const createRes = await fetch(RUNES_MARKET_API + '/', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    rune_id: rune.runeId || rune.id,
                    rune_name: rune.displayName || rune.name,
                    rune_symbol: rune.symbol || '⧈',
                    sell_amount: rawSellAmount.toString(),
                    total_amount: rawTotalAmount.toString(),
                    divisibility: divisibility,
                    seller_txid: utxo.txid,
                    seller_vout: utxo.vout,
                    seller_value: utxoValue,
                    seller_script_pubkey: scriptPubKey,
                    price_sats: price,
                    seller_payout_address: walletAddress
                })
            });
            
            const createData = await createRes.json();
            
            if (!createData.success) {
                throw new Error(createData.error || 'Failed to create listing');
            }
            
            // Sign PSBT with SIGHASH_SINGLE|ANYONECANPAY (0x83)
            listStatusText.textContent = 'Requesting signature...';
            
            // Save the PSBT data for the confirmation screen
            await chrome.storage.local.set({
                pendingPsbtRequest: {
                    psbt: createData.psbt_base64,
                    inputsToSign: [{ index: 0, sighashTypes: [0x83] }],
                    sighashType: 0x83,
                    autoFinalized: false,
                    type: 'listRuneAtomic',
                    order_id: createData.order_id,
                    timestamp: Date.now()
                }
            });
            
            // Hide list screen and show PSBT confirmation
            listScreen.style.display = 'none';
            
            // Show the PSBT confirmation screen
            showScreen('confirm-psbt');
            await showPsbtConfirmation();
            
            // Wait for the result
            const signResult = await new Promise((resolve, reject) => {
                const checkResult = async () => {
                    const storage = await chrome.storage.local.get(['psbtSignResult']);
                    if (storage.psbtSignResult) {
                        const result = storage.psbtSignResult;
                        await chrome.storage.local.remove(['psbtSignResult', 'pendingPsbtRequest']);
                        
                        if (result.success) {
                            resolve(result);
                        } else {
                            reject(new Error(result.error || 'Signing failed'));
                        }
                    }
                };
                
                // Check every 500ms for up to 2 minutes
                const interval = setInterval(checkResult, 500);
                const timeout = setTimeout(() => {
                    clearInterval(interval);
                    reject(new Error('Signing timeout'));
                }, 120000);
                
                // Store cleanup functions for cancel
                window._psbtSignCleanup = () => {
                    clearInterval(interval);
                    clearTimeout(timeout);
                    reject(new Error('User cancelled'));
                };
            });
            
            // NOTE: The actual submission to /sign endpoint is handled by handlePsbtSign()
            // when it processes the 'listRuneAtomic' type. We just need to wait for success.
            
            if (!signResult || !signResult.success) {
                throw new Error(signResult?.error || 'Signing failed');
            }
            
            // Success! handlePsbtSign already submitted to /sign and showed notification
            // Just clean up the list screen
            console.log('✅ Rune listing completed via handlePsbtSign');
            
            // Return to appropriate screen
            listScreen.remove();
            if (rune._fromMarket) {
                loadRunesMarket();
            } else if (typeof showRuneDetails === 'function') {
                showRuneDetails(rune);
            }
            
        } catch (error) {
            console.error('List rune error:', error);
            listStatus.style.display = 'none';
            document.getElementById('list-confirm-btn').disabled = false;
            showNotification(error.message, 'error');
        }
    });
}

// Setup event listeners para DEX
document.addEventListener('DOMContentLoaded', () => {
    const createPoolBtn = document.getElementById('create-pool-btn');
    if (createPoolBtn) {
        createPoolBtn.addEventListener('click', showCreatePoolScreen);
        console.log('✅ Create Pool button listener added');
    }

    // Quando a tab Market for clicada, carregar listings
    const marketTab = document.querySelector('[data-tab="market"]');
    if (marketTab) {
        marketTab.addEventListener('click', () => {
            loadMarketData();
        });
        console.log('✅ Market tab listener added');
    }
    
    // Market sub-tabs switching
    document.querySelectorAll('.market-subtab').forEach(subtab => {
        subtab.addEventListener('click', () => {
            // Update sub-tab active state
            document.querySelectorAll('.market-subtab').forEach(t => t.classList.remove('active'));
            subtab.classList.add('active');
            
            // Show/hide content
            const targetMarket = subtab.dataset.marketTab;
            document.querySelectorAll('.market-content').forEach(c => c.classList.add('hidden'));
            document.getElementById(`${targetMarket}-content`)?.classList.remove('hidden');
            
            // Load appropriate market
            if (targetMarket === 'inscriptions-market') {
                loadInscriptionsMarket();
            } else if (targetMarket === 'runes-market') {
                loadRunesMarket();
            }
        });
    });
    
    // List buttons in market tab
    const listInscriptionBtn = document.getElementById('market-list-inscription-btn');
    if (listInscriptionBtn) {
        listInscriptionBtn.addEventListener('click', showListInscriptionModal);
    }
    
    const listRuneMarketBtn = document.getElementById('list-rune-market-btn');
    if (listRuneMarketBtn) {
        listRuneMarketBtn.addEventListener('click', () => showListRuneScreen());
    }
    
    // Load autolock setting when settings screen is shown
    const settingsBtn = document.getElementById('settings-btn');
    if (settingsBtn) {
        const originalHandler = settingsBtn.onclick;
        settingsBtn.onclick = function(e) {
            if (originalHandler) originalHandler.call(this, e);
            loadAutolockSetting();
        };
    }
});

// ========================================
// SPLIT UTXOs FUNCTIONALITY
// ========================================

let splitState = {
    selectedUTXOs: [],
    outputs: [],
    utxosList: []
};

// Navegar para tela Split
async function showSplitUTXOsScreen() {
    console.log('✂️ ========== Opening Split UTXOs screen ==========');
    console.log('   Function called successfully!');
    
    try {
        // ✅ PRIMEIRO: Mostrar loading IMEDIATAMENTE
        console.log('   Creating loading overlay...');
        
        // Criar overlay de loading
        const loadingOverlay = document.createElement('div');
        loadingOverlay.id = 'split-loading-overlay';
        loadingOverlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: #0a0a0a;
            z-index: 99999;
            display: flex;
            align-items: center;
            justify-content: center;
        `;
        loadingOverlay.innerHTML = `
            <div class="loading-container">
                <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
                <div class="loading-spinner"></div>
                <p class="loading-text">Loading UTXOs...</p>
                <p class="loading-subtext" style="font-size: 12px; color: #888; margin-top: 8px;">This may take a few seconds</p>
            </div>
        `;
        document.body.appendChild(loadingOverlay);
        console.log('   ✅ Loading overlay displayed!');
        
        // ⏱️ Pequeno delay para garantir que o loading seja renderizado
        await new Promise(resolve => setTimeout(resolve, 50));
        
        // ✅ Carregar UTXOs ANTES de mostrar a tela
        await loadSplitUTXOs();
        
        // ✅ Adicionar 2 outputs padrão
        splitState.outputs = [];
        addSplitOutput(546); // Output 1: 546 sats (padrão para Inscription/Runes)
        addSplitOutput(546); // Output 2: 546 sats (padrão para Inscription/Runes)
        
        // ✅ Remover loading overlay
        if (loadingOverlay) {
            loadingOverlay.remove();
            console.log('   ✅ Loading overlay removed');
        }
        
        // ✅ AGORA SIM: Mostrar a tela Split (com dados já carregados)
        showScreen('split-utxos');
        console.log('   ✅ Split screen displayed');
        
    } catch (error) {
        console.error('❌ Error loading Split screen:', error);
        
        // Remover loading em caso de erro
        const overlay = document.getElementById('split-loading-overlay');
        if (overlay) {
            overlay.remove();
            console.log('   ✅ Loading overlay removed (error case)');
        }
        
        showNotification('Failed to load Split screen: ' + error.message, 'error');
    }
}

// Carregar UTXOs disponíveis
async function loadSplitUTXOs() {
    console.log('📦 Loading UTXOs for split...');
    
    const listContainer = document.getElementById('split-utxo-list');
    if (listContainer) {
        listContainer.innerHTML = `
            <div class="loading-container">
                <img src="../assets/logo.png" alt="MyWallet" class="logo-medium" />
                <div class="loading-spinner"></div>
                <p class="loading-text">Loading UTXOs...</p>
            </div>
        `;
    }
    
    try {
        // Obter endereço da carteira (do DOM)
        const addressElement = document.getElementById('wallet-address');
        let address = addressElement ? addressElement.getAttribute('data-full-address') : null;
        
        // Se não encontrou no DOM, tentar via message
        if (!address) {
            console.log('   Trying to get address via chrome.runtime...');
            const response = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
            if (response && response.success && response.address) {
                address = response.address;
            }
        }
        
        if (!address) {
            throw new Error('Could not get wallet address. Is wallet unlocked?');
        }
        
        console.log('   Address:', address);
        console.log('   Address length:', address.length);
        
        // Buscar UTXOs enriquecidos do backend
        const url = `https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`;
        console.log('   Fetching enriched UTXOs from backend...');
        console.log('   URL:', url);
        
        const response = await fetch(url);
        console.log('   Response status:', response.status);
        console.log('   Response ok:', response.ok);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('   Error response body:', errorText);
            throw new Error(`Backend API error: ${response.status}`);
        }
        
        const data = await response.json();
        console.log('   Response data:', data);
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to fetch UTXOs');
        }
        
        console.log(`   ✅ Found ${data.utxos?.length || 0} UTXOs`);
        console.log(`   - ${data.inscriptions?.length || 0} inscriptions`);
        console.log(`   - ${Object.keys(data.runes || {}).length} runes`);
        
        if (!data.utxos || data.utxos.length === 0) {
            if (listContainer) {
                listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;">No UTXOs available</div>';
            }
            return;
        }
        
        // UTXOs já vêm enriquecidos do backend
        splitState.utxosList = data.utxos.map(utxo => ({
            ...utxo,
            selected: false
        }));
        
        console.log('   ✅ UTXOs loaded, rendering...');
        renderSplitUTXOs();
        
    } catch (error) {
        console.error('❌ Error loading UTXOs:', error);
        if (listContainer) {
            listContainer.innerHTML = `
                <div style="color: #ef4444; text-align: center; padding: 20px;">
                    <div style="margin-bottom: 8px;">❌ Failed to load UTXOs</div>
                    <div style="font-size: 12px; color: #888;">${error.message}</div>
                </div>
            `;
        }
    }
}

// Renderizar lista de UTXOs
function renderSplitUTXOs() {
    const listContainer = document.getElementById('split-utxo-list');
    if (!listContainer) return;
    
    if (splitState.utxosList.length === 0) {
        listContainer.innerHTML = `<div style="color: #888; text-align: center; padding: 20px;">No UTXOs available</div>`;
        return;
    }
    
    listContainer.innerHTML = '';
    
    splitState.utxosList.forEach((utxo, index) => {
        const utxoItem = document.createElement('div');
        utxoItem.style.cssText = 'display: flex; align-items: center; padding: 12px; border-bottom: 1px solid #2a2a2a; cursor: pointer; transition: background 0.2s;';
        utxoItem.onmouseover = () => { utxoItem.style.background = '#1a1a1a'; };
        utxoItem.onmouseout = () => { utxoItem.style.background = 'transparent'; };
        utxoItem.onclick = () => toggleUTXOSelection(index);
        
        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.checked = utxo.selected;
        checkbox.style.cssText = 'margin-right: 12px; cursor: pointer; flex-shrink: 0;';
        checkbox.onclick = (e) => {
            e.stopPropagation();
            toggleUTXOSelection(index);
        };
        
        // Thumbnail (se tiver inscrição)
        if (utxo.hasInscription && utxo.inscription) {
            const thumbnail = document.createElement('img');
            thumbnail.src = utxo.inscription.preview;
            thumbnail.style.cssText = 'width: 48px; height: 48px; border-radius: 6px; margin-right: 12px; object-fit: cover; flex-shrink: 0; border: 1px solid #333;';
            thumbnail.onerror = () => {
                thumbnail.style.display = 'none';
            };
            utxoItem.appendChild(checkbox);
            utxoItem.appendChild(thumbnail);
        } else {
            utxoItem.appendChild(checkbox);
        }
        
        const info = document.createElement('div');
        info.style.cssText = 'flex: 1; min-width: 0;';
        
        const badges = [];
        if (utxo.hasInscription) badges.push('<span style="background: #8b5cf6; padding: 2px 8px; border-radius: 4px; font-size: 11px; margin-right: 4px;">📜 Inscription</span>');
        if (utxo.hasRunes) badges.push('<span style="background: #f59e0b; padding: 2px 8px; border-radius: 4px; font-size: 11px; margin-right: 4px;">🪙 Runes</span>');
        if (!utxo.hasInscription && !utxo.hasRunes) badges.push('<span style="background: #10b981; padding: 2px 8px; border-radius: 4px; font-size: 11px; margin-right: 4px;">💰 Pure BTC</span>');
        
        let detailsHtml = '';
        
        // Detalhes das Runes
        if (utxo.hasRunes && utxo.runes && utxo.runes.length > 0) {
            const runesInfo = utxo.runes.map(r => {
                const amount = parseInt(r.amount) || 0;
                return `<div style="color: #f59e0b; font-size: 11px; margin-top: 2px;">
                    🪙 ${r.name || 'Unknown Rune'}: ${amount.toLocaleString()} ${r.symbol || ''}
                </div>`;
            }).join('');
            detailsHtml += runesInfo;
        }
        
        // Detalhes da Inscrição
        if (utxo.hasInscription && utxo.inscription) {
            detailsHtml += `<div style="color: #8b5cf6; font-size: 11px; margin-top: 2px;">
                📜 #${utxo.inscription.number || 'Unknown'}
            </div>`;
        }
        
        info.innerHTML = `
            <div style="display: flex; align-items: center; margin-bottom: 4px; flex-wrap: wrap;">
                ${badges.join('')}
            </div>
            ${detailsHtml}
            <div style="color: #888; font-size: 11px; font-family: monospace; margin-top: 4px; overflow: hidden; text-overflow: ellipsis;">
                ${utxo.txid.substring(0, 16)}...${utxo.txid.substring(utxo.txid.length - 4)}:${utxo.vout}
            </div>
            <div style="color: #fff; font-weight: 600; margin-top: 4px;">
                ${utxo.value.toLocaleString()} sats
            </div>
        `;
        
        utxoItem.appendChild(info);
        listContainer.appendChild(utxoItem);
    });
}

// Toggle seleção de UTXO
function toggleUTXOSelection(index) {
    splitState.utxosList[index].selected = !splitState.utxosList[index].selected;
    
    if (splitState.utxosList[index].selected) {
        splitState.selectedUTXOs.push(splitState.utxosList[index]);
    } else {
        splitState.selectedUTXOs = splitState.selectedUTXOs.filter(u => u.id !== splitState.utxosList[index].id);
    }
    
    renderSplitUTXOs();
    updateSplitSummary();
}

// Adicionar output
function addSplitOutput(value = 546) {
    const output = { value };
    splitState.outputs.push(output);
    renderSplitOutputs();
    updateSplitSummary();
}

// Remover output
function removeSplitOutput(index) {
    splitState.outputs.splice(index, 1);
    renderSplitOutputs();
    updateSplitSummary();
}

// Atualizar valor de output
function updateSplitOutputValue(index, value) {
    splitState.outputs[index].value = parseInt(value) || 0;
    updateSplitSummary();
}

// Renderizar outputs
function renderSplitOutputs() {
    const outputsContainer = document.getElementById('split-outputs-list');
    if (!outputsContainer) return;
    
    if (splitState.outputs.length === 0) {
        outputsContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 12px;">No outputs configured. Click "+" to add.</div>';
        return;
    }
    
    outputsContainer.innerHTML = '';
    
    splitState.outputs.forEach((output, index) => {
        const outputItem = document.createElement('div');
        outputItem.style.cssText = 'display: flex; align-items: center; gap: 8px; margin-bottom: 8px; padding: 8px; background: #0a0a0a; border-radius: 6px; flex-wrap: nowrap;';
        
        const label = document.createElement('span');
        label.style.cssText = 'color: #888; min-width: 70px; flex-shrink: 0;';
        label.textContent = `Output ${index + 1}:`;
        
        const input = document.createElement('input');
        input.type = 'number';
        input.value = output.value;
        input.min = '546';
        input.step = '1';
        input.style.cssText = 'flex: 1; min-width: 80px; background: #1a1a1a; border: 1px solid #333; border-radius: 4px; padding: 8px; color: #fff; font-size: 14px;';
        input.addEventListener('change', (e) => {
            updateSplitOutputValue(index, e.target.value);
        });
        
        const satsLabel = document.createElement('span');
        satsLabel.style.cssText = 'color: #888; min-width: 35px; flex-shrink: 0;';
        satsLabel.textContent = 'sats';
        
        const removeBtn = document.createElement('button');
        removeBtn.textContent = '✕';
        removeBtn.style.cssText = 'background: #ef4444; color: #fff; border: none; border-radius: 4px; padding: 8px 10px; cursor: pointer; font-size: 14px; font-weight: bold; flex-shrink: 0; min-width: 32px; transition: background 0.2s;';
        removeBtn.addEventListener('click', () => {
            removeSplitOutput(index);
        });
        removeBtn.addEventListener('mouseover', () => {
            removeBtn.style.background = '#dc2626';
        });
        removeBtn.addEventListener('mouseout', () => {
            removeBtn.style.background = '#ef4444';
        });
        
        outputItem.appendChild(label);
        outputItem.appendChild(input);
        outputItem.appendChild(satsLabel);
        outputItem.appendChild(removeBtn);
        
        outputsContainer.appendChild(outputItem);
    });
}

// Atualizar sumário
function updateSplitSummary() {
    const totalInput = splitState.selectedUTXOs.reduce((sum, utxo) => sum + utxo.value, 0);
    const totalOutput = splitState.outputs.reduce((sum, out) => sum + out.value, 0);
    
    // Estimar fee (aproximado)
    const numInputs = splitState.selectedUTXOs.length;
    const numOutputs = splitState.outputs.length;
    const estimatedSize = Math.ceil(10.5 + (numInputs * 57.5) + (numOutputs * 43));
    const estimatedFee = estimatedSize * 1; // 1 sat/vB
    
    const canExecute = totalInput > 0 && totalOutput > 0 && (totalInput >= totalOutput + estimatedFee);
    
    // Atualizar UI
    const summaryDiv = document.getElementById('split-summary');
    const executeBtn = document.getElementById('execute-split-btn');
    
    if (summaryDiv && totalInput > 0) {
        summaryDiv.style.display = 'block';
        document.getElementById('split-total-input').textContent = `${totalInput.toLocaleString()} sats`;
        document.getElementById('split-total-output').textContent = `${totalOutput.toLocaleString()} sats`;
        document.getElementById('split-est-fee').textContent = `${estimatedFee} sats`;
    } else if (summaryDiv) {
        summaryDiv.style.display = 'none';
    }
    
    if (executeBtn) {
        executeBtn.disabled = !canExecute;
        if (!canExecute && totalInput > 0) {
            executeBtn.textContent = `⚠️ Insufficient Balance (need ${(totalOutput + estimatedFee - totalInput)} more sats)`;
        } else {
            executeBtn.textContent = '✂️ Execute Split';
        }
    }
}

// Executar split - Abre modal de confirmação com senha
async function executeSplit() {
    console.log('✂️ Starting split process...');
    
    if (splitState.selectedUTXOs.length === 0) {
        showNotification('⚠️ Please select at least one UTXO', 'error');
        return;
    }
    
    if (splitState.outputs.length === 0) {
        showNotification('⚠️ Please add at least one output', 'error');
        return;
    }
    
    // Mostrar modal de confirmação de senha
    showSplitConfirmModal();
}

// Modal de confirmação para Split
async function showSplitConfirmModal() {
    console.log('🔐 Showing split confirmation modal...');
    
    // Calcular totais para mostrar
    const totalInput = splitState.selectedUTXOs.reduce((sum, u) => sum + u.value, 0);
    const totalOutput = splitState.outputs.reduce((sum, o) => sum + o.value, 0);
    
    // Criar modal
    const modal = document.createElement('div');
    modal.id = 'split-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(16, 185, 129, 0.2);">
                <h3 style="color: #10b981;">✂️ Confirm Split Transaction</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 40px; margin-bottom: 12px;">🔀</div>
                
                <div style="background: rgba(255,255,255,0.05); border-radius: 12px; padding: 16px; margin-bottom: 16px; text-align: left;">
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Inputs:</span>
                        <span style="color: #fff; font-weight: 600;">${splitState.selectedUTXOs.length} UTXOs</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Total In:</span>
                        <span style="color: #fff; font-weight: 600;">${totalInput.toLocaleString()} sats</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Outputs:</span>
                        <span style="color: #10b981; font-weight: 600;">${splitState.outputs.length} UTXOs</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #888;">Total Out:</span>
                        <span style="color: #10b981; font-weight: 600;">${totalOutput.toLocaleString()} sats</span>
                    </div>
                </div>
                
                <!-- FEE RATE SELECTOR -->
                <div style="background: rgba(245, 158, 11, 0.1); border: 1px solid rgba(245, 158, 11, 0.3); border-radius: 12px; padding: 16px; margin-bottom: 16px; text-align: left;">
                    <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 12px;">
                        <span style="font-size: 18px;">⚡</span>
                        <span style="color: #f59e0b; font-weight: 600; font-size: 14px;">Network Fee Rate</span>
                    </div>
                    <select id="split-fee-rate" class="input-field" style="background: rgba(0,0,0,0.3); border: 1px solid rgba(255,255,255,0.1); margin-bottom: 8px;">
                        <option value="loading">⏳ Loading fees...</option>
                    </select>
                    <div id="split-fee-estimate" style="font-size: 12px; color: #888; text-align: center;">
                        Estimated fee: calculating...
                    </div>
                </div>
                
                <div style="text-align: left; margin-bottom: 8px;">
                    <label style="font-size: 13px; color: #888;">Enter password to confirm:</label>
                </div>
                <input type="password" id="split-confirm-password" class="input-field" placeholder="Password" autocomplete="off" style="margin-bottom: 16px;" />
                
                <div id="split-status" style="display: none; padding: 12px; border-radius: 8px; margin-bottom: 16px;"></div>
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-split-btn" class="btn btn-secondary btn-large" style="flex: 1;">Cancel</button>
                <button id="confirm-split-btn" class="btn btn-large" style="flex: 1; background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                    ✂️ Execute Split
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Focus no campo de senha
    setTimeout(() => document.getElementById('split-confirm-password')?.focus(), 100);
    
    // Event listeners
    document.getElementById('cancel-split-btn').addEventListener('click', closeSplitConfirmModal);
    document.getElementById('confirm-split-btn').addEventListener('click', confirmAndExecuteSplit);
    document.getElementById('split-confirm-password').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') confirmAndExecuteSplit();
    });
    modal.addEventListener('click', (e) => {
        if (e.target.id === 'split-confirm-modal') closeSplitConfirmModal();
    });
    
    // Load mempool fees
    await loadSplitMempoolFees();
    
    // Update fee estimate when selection changes
    document.getElementById('split-fee-rate')?.addEventListener('change', updateSplitFeeEstimate);
}

// Load mempool.space fee rates for Split
async function loadSplitMempoolFees() {
    console.log('⚡ Loading mempool fees for Split...');
    
    const feeSelect = document.getElementById('split-fee-rate');
    const feeEstimate = document.getElementById('split-fee-estimate');
    
    try {
        // Fetch from mempool.space
        const response = await fetch('https://mempool.space/api/v1/fees/recommended');
        const fees = await response.json();
        
        console.log('✅ Mempool fees loaded:', fees);
        
        if (feeSelect) {
            feeSelect.innerHTML = `
                <option value="${fees.fastestFee}">🚀 Fastest (${fees.fastestFee} sat/vB) ~10 min</option>
                <option value="${fees.halfHourFee}">⚡ Fast (${fees.halfHourFee} sat/vB) ~30 min</option>
                <option value="${fees.hourFee}" selected>💰 Standard (${fees.hourFee} sat/vB) ~1 hour</option>
                <option value="${fees.economyFee}">🐢 Economy (${fees.economyFee} sat/vB) ~2+ hours</option>
                <option value="${fees.minimumFee}">🐌 Minimum (${fees.minimumFee} sat/vB) ~day</option>
                <option value="custom">⚙️ Custom...</option>
            `;
        }
        
        // Update estimate
        updateSplitFeeEstimate();
        
    } catch (error) {
        console.error('❌ Error loading mempool fees:', error);
        
        // Fallback to static fees
        if (feeSelect) {
            feeSelect.innerHTML = `
                <option value="20">🚀 Fast (20 sat/vB)</option>
                <option value="10" selected>💰 Standard (10 sat/vB)</option>
                <option value="5">🐢 Economy (5 sat/vB)</option>
                <option value="1">🐌 Minimum (1 sat/vB)</option>
                <option value="custom">⚙️ Custom...</option>
            `;
        }
        updateSplitFeeEstimate();
    }
}

// Update fee estimate display
function updateSplitFeeEstimate() {
    const feeSelect = document.getElementById('split-fee-rate');
    const feeEstimate = document.getElementById('split-fee-estimate');
    const totalInput = splitState.selectedUTXOs.reduce((sum, u) => sum + u.value, 0);
    const totalOutput = splitState.outputs.reduce((sum, o) => sum + o.value, 0);
    
    if (!feeSelect || !feeEstimate) return;
    
    const selectedValue = feeSelect.value;
    
    // Handle custom fee option
    if (selectedValue === 'custom') {
        // Add custom input if not exists
        let customInput = document.getElementById('split-fee-custom');
        if (!customInput) {
            const customDiv = document.createElement('div');
            customDiv.id = 'split-fee-custom-container';
            customDiv.style.marginTop = '8px';
            customDiv.innerHTML = `
                <input type="number" id="split-fee-custom" class="input-field" 
                       placeholder="Enter sat/vB (e.g., 15)" min="1" 
                       style="background: rgba(0,0,0,0.3); border: 1px solid rgba(255,255,255,0.1);">
            `;
            feeSelect.parentNode.insertBefore(customDiv, feeEstimate);
            
            // Add listener for custom input
            document.getElementById('split-fee-custom')?.addEventListener('input', updateSplitFeeEstimate);
        }
        
        const customValue = document.getElementById('split-fee-custom')?.value;
        if (customValue && parseInt(customValue) > 0) {
            calculateAndShowSplitFee(parseInt(customValue), totalInput, totalOutput, feeEstimate);
        } else {
            feeEstimate.textContent = 'Enter custom fee rate';
        }
    } else if (selectedValue !== 'loading') {
        // Remove custom input if exists
        document.getElementById('split-fee-custom-container')?.remove();
        
        const feeRate = parseInt(selectedValue);
        calculateAndShowSplitFee(feeRate, totalInput, totalOutput, feeEstimate);
    }
}

function calculateAndShowSplitFee(feeRate, totalInput, totalOutput, feeEstimate) {
    // Estimate transaction size for split
    // P2TR inputs: ~57.5 vB each, P2TR outputs: ~43 vB each, overhead: ~10.5 vB
    const inputCount = splitState.selectedUTXOs.length;
    const outputCount = splitState.outputs.length;
    const estimatedSize = Math.ceil(10.5 + (inputCount * 57.5) + (outputCount * 43));
    
    const estimatedFeeSats = feeRate * estimatedSize;
    const remaining = totalInput - totalOutput - estimatedFeeSats;
    
    if (remaining < 0) {
        feeEstimate.innerHTML = `
            <span style="color: #ef4444;">⚠️ Fee too high! Reduce outputs or add more inputs</span>
        `;
    } else {
        feeEstimate.innerHTML = `
            <span style="color: #10b981;">Fee: ~${estimatedFeeSats.toLocaleString()} sats</span>
            <span style="color: #888;"> (${estimatedSize} vB × ${feeRate} sat/vB)</span>
        `;
    }
}

function closeSplitConfirmModal() {
    const modal = document.getElementById('split-confirm-modal');
    if (modal) {
        // 🔒 SECURITY: Clear password
        const passwordInput = document.getElementById('split-confirm-password');
        if (passwordInput) passwordInput.value = '';
        modal.remove();
    }
}

// Executar split com senha confirmada
async function confirmAndExecuteSplit() {
    const password = document.getElementById('split-confirm-password')?.value;
    const statusDiv = document.getElementById('split-status');
    const confirmBtn = document.getElementById('confirm-split-btn');
    
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    // Get selected fee rate
    const feeSelect = document.getElementById('split-fee-rate');
    let feeRate = 1;
    
    if (feeSelect) {
        if (feeSelect.value === 'custom') {
            const customFee = document.getElementById('split-fee-custom')?.value;
            feeRate = customFee ? parseInt(customFee) : 1;
        } else if (feeSelect.value !== 'loading') {
            feeRate = parseInt(feeSelect.value);
        }
    }
    
    if (!feeRate || feeRate < 1) {
        showNotification('Please select a valid fee rate', 'error');
        return;
    }
    
    try {
        // Disable button and show status
        if (confirmBtn) {
            confirmBtn.disabled = true;
            confirmBtn.textContent = '⏳ Building...';
        }
        
        if (statusDiv) {
            statusDiv.style.display = 'block';
            statusDiv.style.background = 'rgba(59, 130, 246, 0.1)';
            statusDiv.style.color = '#60a5fa';
            statusDiv.textContent = '🔨 Building transaction...';
        }
        
        // Obter endereço da carteira
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (!walletInfo?.success || !walletInfo?.data?.address) {
            throw new Error('Could not get wallet address');
        }
        const address = walletInfo.data.address;
        
        console.log('📦 Building Split PSBT...');
        console.log('   Address:', address);
        console.log('   Inputs:', splitState.selectedUTXOs.length);
        console.log('   Outputs:', splitState.outputs.length);
        console.log('   Fee Rate:', feeRate, 'sat/vB');
        
        // 1. Construir PSBT no backend
        const buildResponse = await fetch('https://kraywallet-backend.onrender.com/api/psbt/split', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address: address,
                inputs: splitState.selectedUTXOs.map(u => ({
                    txid: u.txid,
                    vout: u.vout,
                    value: u.value
                })),
                outputs: splitState.outputs,
                feeRate: feeRate
            })
        });
        
        const buildResult = await buildResponse.json();
        
        if (!buildResult.success) {
            throw new Error(buildResult.error || 'Failed to build PSBT');
        }
        
        console.log('✅ PSBT built successfully');
        console.log('   Fee:', buildResult.fee, 'sats');
        
        if (statusDiv) {
            statusDiv.textContent = '🔐 Signing transaction...';
        }
        if (confirmBtn) {
            confirmBtn.textContent = '🔐 Signing...';
        }
        
        // 2. Assinar PSBT com senha (via signPsbtWithPassword)
        const signResponse = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: buildResult.psbt,
                password: password,
                inputsToSign: splitState.selectedUTXOs.map((_, i) => i) // Sign all inputs
            }
        });
        
        // 🔒 SECURITY: Clear password from input immediately
        const passwordInput = document.getElementById('split-confirm-password');
        if (passwordInput) passwordInput.value = '';
        
        if (!signResponse?.success) {
            throw new Error(signResponse?.error || 'Failed to sign transaction');
        }
        
        console.log('✅ PSBT signed successfully');
        
        if (statusDiv) {
            statusDiv.textContent = '📡 Broadcasting transaction...';
        }
        if (confirmBtn) {
            confirmBtn.textContent = '📡 Broadcasting...';
        }
        
        // 3. Finalizar PSBT para obter hex
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                psbt: signResponse.signedPsbt
            })
        });
        
        const finalizeResult = await finalizeResponse.json();
        
        if (!finalizeResult.success) {
            throw new Error(finalizeResult.error || 'Failed to finalize transaction');
        }
        
        console.log('✅ PSBT finalized');
        console.log('   TXID:', finalizeResult.txid);
        
        // 4. Broadcast da transação
        const broadcastResponse = await fetch('https://kraywallet-backend.onrender.com/api/wallet/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                hex: finalizeResult.hex
            })
        });
        
        const broadcastResult = await broadcastResponse.json();
        
        if (!broadcastResult.success) {
            throw new Error(broadcastResult.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Transaction broadcast successfully!');
        console.log('   TXID:', finalizeResult.txid);
        
        // Show success with clickable TXID
        const txid = finalizeResult.txid;
        if (statusDiv) {
            statusDiv.style.display = 'block';
            statusDiv.style.background = 'rgba(16, 185, 129, 0.1)';
            statusDiv.style.color = '#10b981';
            statusDiv.innerHTML = `
                ✅ Success!<br>
                <span id="split-txid-link" style="color: #10b981; text-decoration: underline; font-size: 11px; cursor: pointer;">
                    TXID: ${txid.substring(0, 20)}... 🔗
                </span>
                <div style="font-size: 10px; opacity: 0.7; margin-top: 4px;">👆 Click to view on mempool.space</div>
            `;
            
            // Add click handler to open mempool.space
            document.getElementById('split-txid-link')?.addEventListener('click', () => {
                chrome.tabs.create({ url: `https://mempool.space/tx/${txid}` });
            });
        }
        
        // Also show notification banner (clickable)
        showNotification(`✅ Split successful! ${splitState.outputs.length} new UTXOs created`, 'success');
        
        // 💾 Save split txid to localStorage for Activity tab detection
        try {
            const splitTxids = JSON.parse(localStorage.getItem('splitTransactions') || '[]');
            splitTxids.push({
                txid: txid,
                timestamp: Date.now(),
                outputs: splitState.outputs.length
            });
            // Keep only last 50 split transactions
            if (splitTxids.length > 50) splitTxids.shift();
            localStorage.setItem('splitTransactions', JSON.stringify(splitTxids));
            console.log('💾 Split TXID saved for Activity detection:', txid);
        } catch (e) {
            console.warn('⚠️ Could not save split txid:', e);
        }
        
        // 🔄 Clear activity cache so new tx appears
        clearActivityCache();
        
        // Fechar modal e voltar para settings após 2 segundos
        setTimeout(() => {
            closeSplitConfirmModal();
            showScreen('settings');
        }, 2000);
        
    } catch (error) {
        console.error('❌ Error executing split:', error);
        
        // Show error
        if (statusDiv) {
            statusDiv.style.background = 'rgba(239, 68, 68, 0.1)';
            statusDiv.style.color = '#ef4444';
            statusDiv.textContent = `❌ ${error.message}`;
        }
        
        if (confirmBtn) {
            confirmBtn.disabled = false;
            confirmBtn.textContent = '✂️ Execute Split';
        }
        
        showNotification(`Split failed: ${error.message}`, 'error');
    }
}

// Event listeners para Split
document.addEventListener('DOMContentLoaded', () => {
    // Botão para abrir Split UTXOs
    const splitUtxosBtn = document.getElementById('split-utxos-btn');
    console.log('🔍 Looking for split-utxos-btn:', splitUtxosBtn);
    if (splitUtxosBtn) {
        splitUtxosBtn.addEventListener('click', () => {
            console.log('🔥 Split button clicked!');
            showSplitUTXOsScreen();
        });
        console.log('✅ Split UTXOs button listener added');
    } else {
        console.error('❌ split-utxos-btn not found in DOM!');
    }
    
    // Botão voltar da tela Split
    const backFromSplitBtn = document.getElementById('back-from-split-btn');
    if (backFromSplitBtn) {
        backFromSplitBtn.addEventListener('click', () => {
            showScreen('settings');
        });
    }
    
    // Botão adicionar output
    const addOutputBtn = document.getElementById('add-split-output-btn');
    if (addOutputBtn) {
        addOutputBtn.addEventListener('click', () => addSplitOutput(546));
    }
    
    // Botão executar split
    const executeSplitBtn = document.getElementById('execute-split-btn');
    if (executeSplitBtn) {
        executeSplitBtn.addEventListener('click', executeSplit);
    }
});

// Expor funções globalmente para uso inline no HTML
window.updateSplitOutputValue = updateSplitOutputValue;
window.removeSplitOutput = removeSplitOutput;


// ========================================
// MARKETPLACE - LIST ON MARKET
// ========================================

// Global variable to store current inscription being listed
let currentInscriptionToList = null;

/**
 * Show List on Market Screen (Full Screen como inscription details)
 */
function showListMarketModal(inscription) {
    console.log('📋 Opening List on Market screen...', inscription);
    
    // Store current inscription
    currentInscriptionToList = inscription;
    
    // Update screen content
    const contentUrl = inscription.preview || `https://ordinals.com/content/${inscription.id}`;
    document.getElementById('list-inscription-preview-large').src = contentUrl;
    
    document.getElementById('list-inscription-number-large').textContent = 
        `Inscription #${inscription.number || '?'}`;
    
    document.getElementById('list-inscription-id-large').textContent = 
        `ID: ${inscription.id}`;
    
    // Reset form
    document.getElementById('list-price-screen').value = '';
    document.getElementById('list-description-screen').value = '';
    
    // Reset character counter
    const counter = document.getElementById('char-count');
    if (counter) {
        counter.textContent = '0 / 500 characters';
        counter.style.color = '#888';
    }
    
    // Update summary
    updateListingSummary();
    
    // Show screen
    showScreen('list-market');
}

/**
 * Hide List on Market Screen (volta para wallet)
 */
function hideListMarketModal() {
    showScreen('wallet');
    currentInscriptionToList = null;
}

/**
 * Update Listing Summary
 */
function updateListingSummary() {
    const price = parseInt(document.getElementById('list-price-screen')?.value) || 0;
    
    // You receive = price (buyer pays ALL fees!)
    const youReceive = price;
    
    // Update UI - só mostrar o total
    const totalEl = document.getElementById('summary-total-screen');
    if (totalEl) totalEl.textContent = youReceive.toLocaleString() + ' sats';
}

/**
 * Create Market Listing (MAIN LOGIC)
 */
// Flag para evitar cliques duplos
let isCreatingListing = false;

async function createMarketListing() {
    console.log('🚀 Creating market listing...');
    
    // Evitar cliques duplos
    if (isCreatingListing) {
        console.log('⚠️ Already creating listing, ignoring duplicate click');
        return;
    }
    
    isCreatingListing = true;
    
    if (!currentInscriptionToList) {
        showNotification('❌ No inscription selected', 'error');
        isCreatingListing = false;
        return;
    }
    
    const price = parseInt(document.getElementById('list-price-screen').value);
    const description = document.getElementById('list-description-screen').value;
    const feeRate = 1; // Dummy value - buyer pays fee anyway!
    
    // Validation
    if (!price || price < 1000) {
        showNotification('❌ Price must be at least 1,000 sats', 'error');
        return;
    }
    
    try {
        // Show loading
        showLoading('Preparing listing...');
        
        // 🔐 Get wallet address first
        const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const sellerAddress = walletInfo?.address || 'unknown';
        
        // 🔐 STEP 1: Criar mensagem para assinar (prova de propriedade)
        const timestamp = Date.now();
        const messageToSign = `KraySpace Listing Authorization\n\nI authorize listing inscription ${currentInscriptionToList.id} for sale at ${price} sats.\n\nTimestamp: ${timestamp}\nSeller: ${sellerAddress}`;
        
        console.log('🔐 Requesting ownership signature...');
        console.log('   Message:', messageToSign);
        
        hideLoading();
        
        // 🔐 STEP 2: Pedir assinatura via popup de senha
        const signResult = await chrome.runtime.sendMessage({
            action: 'signMessageWithPassword',
            data: {
                message: messageToSign,
                purpose: 'listing_authorization'
            }
        });
        
        // Se precisa senha, mostrar popup de senha
        if (signResult.requiresPassword) {
            console.log('🔐 Password required for signature...');
            
            // Salvar dados pendentes
            await chrome.storage.local.set({
                pendingListingData: {
                inscriptionId: currentInscriptionToList.id,
                    inscriptionNumber: currentInscriptionToList.number || null,
                    priceSats: price,
                    description: description,
                    message: messageToSign,
                    timestamp: timestamp
            }
        });
        
            // Mostrar tela de confirmação de listagem com senha
            document.getElementById('list-market-screen')?.classList.add('hidden');
            showListingConfirmScreen(currentInscriptionToList, price, messageToSign);
            return;
        }
        
        if (!signResult.success || !signResult.signature) {
            throw new Error(signResult.error || 'Signature failed');
        }
        
        // 🛒 STEP 3: Criar listagem COM assinatura
        await createListingWithSignature(
            currentInscriptionToList.id,
            price,
            description,
            signResult.signature,
            messageToSign,
            timestamp
        )
        
    } catch (error) {
        console.error('❌ Error creating listing:', error);
        hideLoading();
        showNotification('❌ Failed to create listing: ' + error.message, 'error');
    } finally {
        // ✅ SEMPRE resetar flag no final
        isCreatingListing = false;
        console.log('🔓 isCreatingListing reset to false');
    }
}

// ========================================
// 🔐 LISTING CONFIRMATION WITH SIGNATURE
// ========================================

/**
 * Show listing confirmation screen with password input
 * Shows UTXO details like KrayWallet and other marketplaces
 */
function showListingConfirmScreen(inscription, price, message) {
    console.log('🔐 Showing listing confirmation screen...');
    console.log('   Inscription:', inscription);
    
    const container = document.getElementById('main-content') || document.body;
    
    // Extract UTXO info from inscription
    const utxoTxid = inscription.utxo?.split(':')[0] || inscription.location?.split(':')[0] || inscription.id?.slice(0, 64) || '?';
    const utxoVout = inscription.utxo?.split(':')[1] || inscription.location?.split(':')[1] || '0';
    const outputValue = inscription.outputValue || inscription.value || 555;
    const shortTxid = utxoTxid.slice(0, 8) + '...' + utxoTxid.slice(-8);
    
    // Create confirmation overlay
    const overlay = document.createElement('div');
    overlay.id = 'listing-confirm-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: var(--color-background, #0a0a0f);
        z-index: 9999;
        display: flex;
        flex-direction: column;
        padding: 20px;
        overflow-y: auto;
    `;
    
    overlay.innerHTML = `
        <div style="display: flex; align-items: center; gap: 12px; margin-bottom: 20px;">
            <button id="listing-confirm-back" style="
                background: transparent;
                border: none;
                color: var(--color-text);
                font-size: 20px;
                cursor: pointer;
                padding: 8px;
            ">←</button>
            <h2 style="margin: 0; font-size: 18px; color: var(--color-text);">🔐 Confirm Listing</h2>
        </div>
        
        <!-- Inscription Preview -->
        <div style="
            background: var(--color-surface);
            border: 1px solid var(--color-border);
            border-radius: 12px;
            padding: 16px;
            margin-bottom: 16px;
        ">
            <div style="display: flex; gap: 12px; align-items: center;">
                <img src="https://ordinals.com/content/${inscription.id}" 
                     style="width: 70px; height: 70px; border-radius: 8px; object-fit: cover;"
                     onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/></svg>'">
                <div style="flex: 1;">
                    <div style="font-weight: 700; font-size: 16px; color: var(--color-text);">Inscription #${inscription.number || '?'}</div>
                    <div style="font-size: 11px; color: var(--color-text-secondary); margin-top: 4px;">${inscription.content_type || 'image/webp'}</div>
                </div>
            </div>
        </div>
        
        <!-- UTXO Details (like KrayWallet) -->
        <div style="
            background: var(--color-surface);
            border: 1px solid var(--color-border);
            border-radius: 12px;
            padding: 16px;
            margin-bottom: 16px;
        ">
            <div style="font-size: 11px; color: var(--color-text-secondary); text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 12px;">
                📦 UTXO Details
            </div>
            
            <!-- Input UTXO with Thumbnail -->
            <div style="
                background: rgba(239, 68, 68, 0.1);
                border: 1px solid rgba(239, 68, 68, 0.3);
                border-radius: 8px;
                padding: 12px;
                margin-bottom: 8px;
            ">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="font-size: 10px; color: #ef4444; font-weight: 600; text-transform: uppercase;">📥 INPUT</span>
                    <span style="font-size: 12px; color: #ef4444; font-weight: 700;">${outputValue} sats</span>
                </div>
                <div style="display: flex; gap: 10px; align-items: center;">
                    <img src="https://ordinals.com/content/${inscription.id}" 
                         style="width: 40px; height: 40px; border-radius: 6px; object-fit: cover; border: 2px solid rgba(239, 68, 68, 0.5);"
                         onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/></svg>'">
                    <div style="flex: 1; min-width: 0;">
                        <div style="font-size: 12px; font-weight: 600; color: var(--color-text);">#${inscription.number || '?'}</div>
                        <div style="font-family: monospace; font-size: 10px; color: var(--color-text-secondary); word-break: break-all;">
                            ${shortTxid}:${utxoVout}
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Arrow -->
            <div style="text-align: center; color: var(--color-text-secondary); font-size: 16px; margin: 4px 0;">↓</div>
            
            <!-- Output (Buyer will receive) -->
            <div style="
                background: rgba(34, 197, 94, 0.1);
                border: 1px solid rgba(34, 197, 94, 0.3);
                border-radius: 8px;
                padding: 12px;
            ">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
                    <span style="font-size: 10px; color: #22c55e; font-weight: 600; text-transform: uppercase;">📤 OUTPUT</span>
                    <span style="font-size: 12px; color: #22c55e; font-weight: 700;">${outputValue} sats</span>
                </div>
                <div style="display: flex; gap: 10px; align-items: center;">
                    <img src="https://ordinals.com/content/${inscription.id}" 
                         style="width: 40px; height: 40px; border-radius: 6px; object-fit: cover; border: 2px solid rgba(34, 197, 94, 0.5); opacity: 0.7;"
                         onerror="this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect fill=%22%23333%22 width=%22100%22 height=%22100%22/></svg>'">
                    <div style="flex: 1; min-width: 0;">
                        <div style="font-size: 12px; font-weight: 600; color: #22c55e;">→ Buyer's Address</div>
                        <div style="font-size: 10px; color: var(--color-text-secondary);">
                            Inscription transfers when purchased
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Sale Price -->
        <div style="
            background: var(--color-surface);
            border: 1px solid var(--color-border);
            border-radius: 12px;
            padding: 16px;
            margin-bottom: 16px;
        ">
            <div style="font-size: 11px; color: var(--color-text-secondary); text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 8px;">
                💰 Sale Details
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <span style="color: var(--color-text);">Listing Price</span>
                <span style="font-weight: 700; font-size: 18px; color: #22c55e;">${price.toLocaleString()} sats</span>
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 8px; padding-top: 8px; border-top: 1px solid var(--color-border);">
                <span style="font-size: 12px; color: var(--color-text-secondary);">Market Fee (2%)</span>
                <span style="font-size: 12px; color: var(--color-text-secondary);">${Math.max(Math.floor(price * 0.02), 546)} sats</span>
            </div>
            <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 4px;">
                <span style="font-size: 12px; color: var(--color-text-secondary);">You Receive</span>
                <span style="font-size: 12px; font-weight: 600; color: var(--color-text);">~${(price - Math.max(Math.floor(price * 0.02), 546)).toLocaleString()} sats</span>
            </div>
        </div>
        
        <!-- Warning -->
        <div style="
            background: rgba(251, 191, 36, 0.1);
            border: 1px solid rgba(251, 191, 36, 0.3);
            border-radius: 8px;
            padding: 12px;
            margin-bottom: 16px;
            font-size: 11px;
            color: #fbbf24;
        ">
            ⚠️ <strong>Authentication Required</strong><br>
            Enter your password to prove ownership and authorize this listing.
        </div>
        
        <!-- Password Input -->
        <div style="margin-bottom: 16px;">
            <label style="display: block; font-size: 12px; color: var(--color-text-secondary); margin-bottom: 8px;">
                🔑 Password
            </label>
            <input type="password" id="listing-confirm-password" placeholder="Enter your wallet password" style="
                width: 100%;
                padding: 14px 16px;
                background: var(--color-surface);
                border: 1px solid var(--color-border);
                border-radius: 8px;
                color: var(--color-text);
                font-size: 14px;
                box-sizing: border-box;
            " autofocus>
        </div>
        
        <!-- Sign Button -->
        <button id="listing-confirm-btn" style="
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #22c55e, #16a34a);
            border: none;
            border-radius: 12px;
            color: white;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        ">
            🔐 Sign & List for Sale
        </button>
        
        <p style="
            text-align: center;
            font-size: 10px;
            color: var(--color-text-secondary);
            margin-top: 12px;
        ">
            Your signature proves you own this inscription.<br>
            No funds will be moved until a buyer purchases.
        </p>
    `;
    
    container.appendChild(overlay);
    
    // Event listeners
    document.getElementById('listing-confirm-back').addEventListener('click', () => {
        overlay.remove();
        document.getElementById('list-market-screen')?.classList.remove('hidden');
        isCreatingListing = false;
    });
    
    document.getElementById('listing-confirm-password').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            document.getElementById('listing-confirm-btn').click();
        }
    });
    
    document.getElementById('listing-confirm-btn').addEventListener('click', async () => {
        const password = document.getElementById('listing-confirm-password').value;
        
        if (!password) {
            showNotification('❌ Please enter your password', 'error');
            return;
        }
        
        const btn = document.getElementById('listing-confirm-btn');
        btn.disabled = true;
        btn.innerHTML = '⏳ Creating & Signing...';
        
        try {
            // Get pending listing data
            const storage = await chrome.storage.local.get('pendingListingData');
            const pendingData = storage.pendingListingData;
            
            if (!pendingData) {
                throw new Error('Listing data not found');
            }
            
            // 🔐 NEW FLOW: Sign LOCALLY in popup (WASM works here!)
            console.log('🔐 ===== LOCAL SIGNING IN POPUP =====');
            console.log('   LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
            
            // Step 1: Get wallet info
            const walletInfo = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
            console.log('   Wallet info response:', walletInfo);
            if (!walletInfo || !walletInfo.success || !walletInfo.data?.address) {
                throw new Error('Failed to get wallet info: ' + (walletInfo?.error || 'unknown error'));
            }
            const walletAddress = walletInfo.data.address;
            console.log('   Wallet:', walletAddress.slice(0, 12) + '...');
            
            // Step 2: Decrypt mnemonic locally
            console.log('   Decrypting wallet...');
            const decryptResult = await chrome.runtime.sendMessage({
                action: 'decryptWallet',
                data: { password: password }
            });
            
            if (!decryptResult.success || !decryptResult.mnemonic) {
                throw new Error(decryptResult.error || 'Failed to decrypt wallet - wrong password?');
            }
            
            let mnemonic = decryptResult.mnemonic;
            console.log('   ✅ Wallet decrypted');
            
            // Step 3: Create listing on backend (gets PSBT to sign)
            console.log('   Creating listing on backend...');
            const BACKEND_URL = 'https://kraywallet-backend.onrender.com';
            const createResponse = await fetch(`${BACKEND_URL}/api/atomic-swap/buy-now/list`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    inscription_id: pendingData.inscriptionId,
                    inscription_number: pendingData.inscriptionNumber || null,
                    price_sats: pendingData.priceSats,
                    seller_address: walletAddress,
                    description: pendingData.description || ''
                })
            });
            
            const createData = await createResponse.json();
            
            if (!createResponse.ok || createData.error) {
                if (createData.error === 'already_listed') {
                    btn.disabled = false;
                    btn.innerHTML = '🔐 Sign & List for Sale';
                    showAlreadyListedModal({
                        inscriptionId: pendingData.inscriptionId,
                        priceSats: pendingData.priceSats,
                        existingOrderId: createData.existing_order_id,
                        currentPrice: createData.current_price
                    });
                    mnemonic = null; // Clear mnemonic
                    return;
                }
                throw new Error(createData.error || 'Failed to create listing');
            }
            
            console.log('   ✅ Listing created, Order ID:', createData.order_id);
            console.log('   PSBT received, length:', createData.psbt_base64?.length || 0);
            
            // Step 4: Sign PSBT 100% LOCALLY - LEI PRIMORDIAL!
            console.log('   🔐 Signing PSBT 100% LOCALLY...');
            console.log('   ⚡ LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
            
            let signedPsbt = null;
            
            // Parse inputs to sign
            const inputsToSign = createData.toSignInputs?.map(i => typeof i === 'number' ? i : i.index) || [0];
            const sighashType = createData.toSignInputs?.[0]?.sighashTypes?.[0] || 0x82;
            
            console.log('   inputsToSign:', inputsToSign);
            console.log('   sighashType: 0x' + sighashType.toString(16));
            
            // Try TaprootSignerPure (PURE JS - NO WASM!)
            if (typeof TaprootSignerPure !== 'undefined' && TaprootSignerPure.signPsbtTaprootLocal) {
                console.log('   ✅ Using TaprootSignerPure (PURE JS - NO WASM!)');
                
                try {
                    const signResult = await TaprootSignerPure.signPsbtTaprootLocal(
                        mnemonic,
                        createData.psbt_base64,
                        {
                            inputsToSign: inputsToSign,
                            sighashType: sighashType,
                            autoFinalize: false
                        }
                    );
                    
                    if (signResult && signResult.signedPsbt) {
                        signedPsbt = signResult.signedPsbt;
                        console.log('   ✅ PSBT signed 100% LOCALLY! (Pure JS)');
                    }
                } catch (localErr) {
                    console.error('   ❌ Pure JS signing failed:', localErr.message);
                    throw new Error('Local signing failed: ' + localErr.message);
                }
            } 
            // Fallback to TaprootSigner (WASM version) if available
            else if (typeof TaprootSigner !== 'undefined' && TaprootSigner.signPsbtTaprootLocal) {
                console.log('   ⚠️ Using TaprootSigner (WASM fallback)');
                
                try {
                    const signResult = await TaprootSigner.signPsbtTaprootLocal(
                        mnemonic,
                        createData.psbt_base64,
                        {
                            inputsToSign: inputsToSign,
                            sighashType: sighashType,
                            autoFinalize: false
                        }
                    );
                    
                    if (signResult && signResult.signedPsbt) {
                        signedPsbt = signResult.signedPsbt;
                        console.log('   ✅ PSBT signed LOCALLY! (WASM)');
                    }
                } catch (wasmErr) {
                    console.error('   ❌ WASM signing failed:', wasmErr.message);
                    throw new Error('Local signing failed: ' + wasmErr.message);
                }
            } else {
                throw new Error('No local signer available! Cannot sign securely. LEI PRIMORDIAL violated!');
            }
            
            // 🗑️ CLEAR MNEMONIC IMMEDIATELY!
            mnemonic = null;
            console.log('   🗑️ Mnemonic cleared from memory');
            
            if (!signedPsbt) {
                throw new Error('Signing failed - no signed PSBT returned');
            }
            
            console.log('   ✅ PSBT signed successfully!');
            console.log('   Signed PSBT length:', signedPsbt.length);
            
            // Step 5: Submit signed PSBT to backend (same endpoint, with order_id + signed_psbt)
            console.log('   Submitting signed PSBT...');
            const submitResponse = await fetch(`${BACKEND_URL}/api/atomic-swap/buy-now/list`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    order_id: createData.order_id,
                    signed_psbt: signedPsbt
                })
            });
            
            const submitData = await submitResponse.json();
            
            if (!submitResponse.ok || submitData.error) {
                throw new Error(submitData.error || 'Failed to submit listing');
            }
            
            console.log('🎉 Listing created and signed successfully!');
            console.log('   Order ID:', createData.order_id);
            
            // Cleanup
            await chrome.storage.local.remove('pendingListingData');
            overlay.remove();
            
            // 🎉 Show success modal
            showListingSuccessModal(pendingData.inscriptionId, pendingData.priceSats);
            
        } catch (error) {
            console.error('❌ Error:', error);
            
            // Check if already listed - show special modal with options
            if (error.message.includes('already listed') || error.message.includes('already_listed')) {
                btn.disabled = false;
                btn.innerHTML = '🔐 Sign & List for Sale';
                
                // Get pendingData from storage again (it's out of scope here)
                const storageData = await chrome.storage.local.get('pendingListingData');
                if (storageData.pendingListingData) {
                    showAlreadyListedModal(storageData.pendingListingData);
                } else {
                    showNotification('⚠️ This inscription is already listed on the marketplace!', 'warning');
                }
                return;
            }
            
            showNotification('❌ ' + error.message, 'error');
            btn.disabled = false;
            btn.innerHTML = '🔐 Sign & List for Sale';
        }
    });
}

/**
 * Show modal when inscription is already listed
 * Gives options to cancel or update price
 */
function showAlreadyListedModal(listingData) {
    const existingOverlay = document.getElementById('already-listed-overlay');
    if (existingOverlay) existingOverlay.remove();
    
    // Also remove the listing confirm overlay
    document.getElementById('listing-confirm-overlay')?.remove();
    
    const currentPrice = listingData.currentPrice || listingData.priceSats || '?';
    
    const overlay = document.createElement('div');
    overlay.id = 'already-listed-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.95);
        z-index: 99999;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 20px;
    `;
    
    overlay.innerHTML = `
        <div style="
            background: var(--color-surface, #1a1a2e);
            border: 1px solid var(--color-border, #333);
            border-radius: 16px;
            padding: 24px;
            max-width: 350px;
            width: 100%;
            text-align: center;
        ">
            <div style="font-size: 48px; margin-bottom: 16px;">⚠️</div>
            <h3 style="color: var(--color-text, white); margin: 0 0 12px;">Already Listed</h3>
            <p style="color: var(--color-text-secondary, #888); font-size: 13px; margin: 0 0 8px; line-height: 1.5;">
                This inscription is already listed on the marketplace.
            </p>
            <p style="color: #10b981; font-size: 16px; font-weight: 600; margin: 0 0 20px;">
                Current price: ${Number(currentPrice).toLocaleString()} sats
            </p>
            
            <div style="display: flex; flex-direction: column; gap: 10px;">
                <button id="update-price-btn" style="
                    background: linear-gradient(135deg, #3b82f6, #1d4ed8);
                    border: none;
                    padding: 12px 20px;
                    border-radius: 10px;
                    color: white;
                    font-weight: 600;
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    gap: 8px;
                ">
                    💰 Update Price
                </button>
                
                <button id="cancel-listing-btn" style="
                    background: rgba(239, 68, 68, 0.2);
                    border: 1px solid rgba(239, 68, 68, 0.5);
                    padding: 12px 20px;
                    border-radius: 10px;
                    color: #ef4444;
                    font-weight: 600;
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    gap: 8px;
                ">
                    🗑️ Cancel Listing
                </button>
                
                <button id="close-modal-btn" style="
                    background: transparent;
                    border: 1px solid var(--color-border, #333);
                    padding: 10px 20px;
                    border-radius: 10px;
                    color: var(--color-text-secondary, #888);
                    cursor: pointer;
                    margin-top: 8px;
                ">
                    Close
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(overlay);
    
    // Close button - close modal AND popup extension
    document.getElementById('close-modal-btn').addEventListener('click', () => {
        overlay.remove();
        window.close(); // Close the extension popup entirely
    });
    
    // Update price button
    document.getElementById('update-price-btn').addEventListener('click', async () => {
        overlay.remove();
        showUpdatePriceModal(listingData);
    });
    
    // Cancel listing button
    document.getElementById('cancel-listing-btn').addEventListener('click', async () => {
        const confirmed = confirm('🗑️ Are you sure you want to cancel this listing?');
        if (!confirmed) return;
        
        try {
            showLoading('Cancelling listing...');
            
            // Use existingOrderId if available, otherwise use inscriptionId
            const orderId = listingData.existingOrderId;
            console.log('🗑️ Cancelling listing:', orderId || listingData.inscriptionId);
            
            const result = await chrome.runtime.sendMessage({
                action: 'cancelListing',
                data: { 
                    orderId: orderId,
                    inscriptionId: listingData.inscriptionId 
                }
            });
            
            hideLoading();
            
            if (result.success) {
                overlay.remove();
                showNotification('✅ Listing cancelled successfully!', 'success');
                // Reload wallet data
                await loadWalletData();
                showScreen('wallet');
            } else {
                showNotification('❌ ' + (result.error || 'Failed to cancel'), 'error');
            }
        } catch (error) {
            hideLoading();
            showNotification('❌ ' + error.message, 'error');
        }
    });
}

/**
 * Show modal to update listing price
 */
function showUpdatePriceModal(listingData) {
    const existingOverlay = document.getElementById('update-price-overlay');
    if (existingOverlay) existingOverlay.remove();
    
    const overlay = document.createElement('div');
    overlay.id = 'update-price-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.9);
        z-index: 99999;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 20px;
    `;
    
    overlay.innerHTML = `
        <div style="
            background: var(--color-surface, #1a1a2e);
            border: 1px solid var(--color-border, #333);
            border-radius: 16px;
            padding: 24px;
            max-width: 350px;
            width: 100%;
        ">
            <h3 style="color: var(--color-text, white); margin: 0 0 16px; text-align: center;">
                💰 Atualizar Preço
            </h3>
            
            <div style="margin-bottom: 16px;">
                <label style="display: block; color: var(--color-text-secondary, #888); font-size: 12px; margin-bottom: 6px;">
                    Novo Preço (sats)
                </label>
                <input type="number" id="new-price-input" min="546" placeholder="Ex: 10000" value="${listingData.priceSats || ''}" style="
                    width: 100%;
                    padding: 12px;
                    background: var(--color-background, #0a0a0f);
                    border: 1px solid var(--color-border, #333);
                    border-radius: 8px;
                    color: var(--color-text, white);
                    font-size: 16px;
                    box-sizing: border-box;
                ">
            </div>
            
            <div style="display: flex; gap: 10px;">
                <button id="confirm-update-btn" style="
                    flex: 1;
                    background: linear-gradient(135deg, #22c55e, #16a34a);
                    border: none;
                    padding: 12px;
                    border-radius: 10px;
                    color: white;
                    font-weight: 600;
                    cursor: pointer;
                ">
                    ✅ Confirmar
                </button>
                <button id="cancel-update-btn" style="
                    flex: 1;
                    background: transparent;
                    border: 1px solid var(--color-border, #333);
                    padding: 12px;
                    border-radius: 10px;
                    color: var(--color-text-secondary, #888);
                    cursor: pointer;
                ">
                    Cancelar
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(overlay);
    
    document.getElementById('new-price-input').focus();
    
    // Cancel button
    document.getElementById('cancel-update-btn').addEventListener('click', () => {
        overlay.remove();
    });
    
    // Confirm button
    document.getElementById('confirm-update-btn').addEventListener('click', async () => {
        const newPrice = parseInt(document.getElementById('new-price-input').value);
        
        if (!newPrice || newPrice < 546) {
            showNotification('❌ Preço mínimo é 546 sats', 'error');
            return;
        }
        
        try {
            showLoading('Atualizando preço...');
            
            const result = await chrome.runtime.sendMessage({
                action: 'updateListingPrice',
                data: {
                    inscriptionId: listingData.inscriptionId,
                    newPrice: newPrice
                }
            });
            
            hideLoading();
            
            if (result.success) {
                overlay.remove();
                document.getElementById('listing-confirm-overlay')?.remove();
                
                if (result.requires_signature) {
                    showNotification('✅ Preço atualizado! Assine novamente para ativar.', 'success');
                    // Re-trigger listing flow with new price
                    // The user will need to list again
                } else {
                    showNotification('✅ Preço atualizado para ' + newPrice.toLocaleString() + ' sats!', 'success');
                }
            } else {
                showNotification('❌ ' + (result.error || 'Erro ao atualizar'), 'error');
            }
        } catch (error) {
            hideLoading();
            showNotification('❌ ' + error.message, 'error');
        }
    });
    
    // Enter to confirm
    document.getElementById('new-price-input').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            document.getElementById('confirm-update-btn').click();
        }
    });
}

/**
 * Create listing with verified signature
 * 
 * KRAYWALLET MODEL:
 * 1. Backend returns PSBT for seller to sign
 * 2. Seller signs with SIGHASH_SINGLE|ANYONECANPAY
 * 3. PSBT sent back to backend → Listing LIVE!
 */
async function createListingWithSignature(inscriptionId, price, description, signature, message, timestamp) {
    console.log('🛒 Creating listing with signature...');
    showLoading('Creating listing...');
    
    try {
        // Get inscription number from current inscription if available
        const inscriptionNumber = currentInscriptionToList?.number || null;
        
        const createListingResponse = await chrome.runtime.sendMessage({
            action: 'createBuyNowListing',
            data: {
                inscriptionId: inscriptionId,
                inscriptionNumber: inscriptionNumber,
                priceSats: price,
                description: description,
                signature: signature,
                message: message,
                timestamp: timestamp
            }
        });
        
        if (!createListingResponse.success) {
            throw new Error(createListingResponse.error || 'Failed to create listing');
        }
        
        console.log('📋 Listing response:', createListingResponse);
        
        // 🔐 KRAYWALLET MODEL: If requiresSignature, show PSBT signing screen immediately
        if (createListingResponse.requiresSignature) {
            console.log('🔐 Listing requires PSBT signature to activate...');
            console.log('   Order ID:', createListingResponse.order_id);
            console.log('   Status:', createListingResponse.status);
            
            hideLoading();
            
            // Hide list screen
        document.getElementById('list-market-screen')?.classList.add('hidden');
            
            // 🔐 SHOW PSBT SIGNING SCREEN IMMEDIATELY!
            console.log('📱 Switching to PSBT signing screen...');
        showScreen('confirm-psbt');
        
            // Load PSBT confirmation data
        await showPsbtConfirmation();
        
            return;
        }
        
        // If already OPEN (no signature needed - backwards compatibility)
        console.log('✅ BUY NOW listing created!');
        console.log('   Order ID:', createListingResponse.order_id);
        hideLoading();
        
        // Hide list screen and show success
        document.getElementById('list-market-screen')?.classList.add('hidden');
        
        showListingSuccessScreen(
            inscriptionId, 
            price, 
            createListingResponse.order_id
        );
        
        showNotification('✅ Listing published successfully!', 'success');
        
    } catch (error) {
        hideLoading();
        throw error;
    } finally {
        isCreatingListing = false;
    }
}

// ========================================
// EVENT LISTENERS - MARKETPLACE
// ========================================

// Setup listeners when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    console.log('🎯 Setting up marketplace listeners...');
    
    // Back button (volta para wallet)
    document.getElementById('back-from-list-market-btn')?.addEventListener('click', hideListMarketModal);
    
    // Create listing button
    document.getElementById('list-create-btn-screen')?.addEventListener('click', createMarketListing);
    
    // Update summary on input change
    document.getElementById('list-price-screen')?.addEventListener('input', updateListingSummary);
    
    // Character counter for description (like Twitter!)
    document.getElementById('list-description-screen')?.addEventListener('input', (e) => {
        const charCount = e.target.value.length;
        const counter = document.getElementById('char-count');
        if (counter) {
            counter.textContent = `${charCount} / 500 characters`;
            
            // Change color based on length (like Twitter)
            if (charCount > 450) {
                counter.style.color = '#ef4444'; // Red
            } else if (charCount > 400) {
                counter.style.color = '#f59e0b'; // Orange
            } else {
                counter.style.color = '#888'; // Gray
            }
        }
    });
    
    console.log('✅ Marketplace listeners set up');
});

// Expose functions globally
window.showListMarketModal = showListMarketModal;
window.hideListMarketModal = hideListMarketModal;
window.updateListingSummary = updateListingSummary;
window.createMarketListing = createMarketListing;

// ========================================
// BUY MARKET LISTING - UNIFIED FUNCTION
// ========================================

/**
 * 🛒 Buy a market listing (unified function for all buy buttons)
 * Can be called from: extension popup, KrayScan, inscription profile, etc.
 * 
 * @param {string} orderId - The order ID of the listing
 * @param {number} priceSats - Price in satoshis (optional, will fetch if not provided)
 */
async function buyMarketListing(orderId, priceSats = null) {
    try {
        console.log('\n🛒 ===== BUY MARKET LISTING (UNIFIED) =====');
        console.log('   Order ID:', orderId);
        console.log('   Price:', priceSats, 'sats');
        
        showLoading('Preparing purchase...');
        
        // Get wallet info
        const walletInfo = await sendMessage({ action: 'getWalletInfo' });
        
        if (!walletInfo.unlocked) {
            hideLoading();
            showNotification('🔒 Please unlock your wallet first', 'error');
            showScreen('unlock');
            return;
        }
        
        const buyerAddress = walletInfo.address;
        console.log('   Buyer address:', buyerAddress);
        
        // If price not provided, fetch from API
        if (!priceSats) {
            const listingResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${orderId}`);
            const listingData = await listingResponse.json();
            if (!listingData.success) {
                throw new Error('Listing not found');
            }
            priceSats = listingData.listing.price_sats;
            console.log('   Fetched price:', priceSats, 'sats');
        }
        
        // Call background to prepare buy PSBT
        const result = await sendMessage({
            action: 'buyAtomicSwap',
            data: {
                orderId,
                priceSats,
                buyerAddress,
                buyerChangeAddress: buyerAddress // Same address for change
            }
        });
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to prepare purchase');
        }
        
        console.log('✅ Buy PSBT prepared, waiting for signature...');
        hideLoading();
        
        // The background will save pendingPsbtRequest
        // Show the PSBT signing screen
        showScreen('confirm-psbt');
        await loadPsbtConfirmation();
        
    } catch (error) {
        console.error('❌ Error buying listing:', error);
        hideLoading();
        showNotification('❌ ' + error.message, 'error');
    }
}

// Expose buy function globally
window.buyMarketListing = buyMarketListing;

// ========================================
// MY OFFERS - MARKETPLACE LISTINGS
// ========================================

/**
 * Show My Offers Screen
 */
async function showMyOffersScreen() {
    console.log('📋 Opening My Offers screen...');
    showScreen('my-offers');
    await loadMyOffers();
}

/**
 * Load My Offers from Backend
 */
async function loadMyOffers() {
    console.log('📋 Loading my atomic swap listings...');
    
    try {
        // Get address from background script
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const address = result?.data?.address || walletState.address;
        
        console.log('📍 User address:', address);
        
        if (!address) {
            showNotification('❌ Wallet not initialized', 'error');
            return;
        }
        
        // Show loading
        document.getElementById('my-offers-loading').classList.remove('hidden');
        document.getElementById('my-offers-empty').classList.add('hidden');
        document.getElementById('my-offers-list').classList.add('hidden');
        
        // Fetch atomic swap listings from backend
        const url = `https://kraywallet-backend.onrender.com/api/atomic-swap/?seller_address=${address}`;
        console.log('📡 Fetching atomic listings from:', url);
        
        const response = await fetch(url);
        
        const data = await response.json();
        
        console.log('📦 Atomic listings response:', data);
        console.log('   Total listings:', data.listings?.length || 0);
        
        // 🔥 FILTRAR: Mostrar APENAS listings ATIVAS (OPEN ou PENDING_SIGNATURES)
        // NÃO mostrar CANCELLED, FILLED, ou EXPIRED
        const activeListings = (data.listings || []).filter(l => 
            l.status === 'OPEN' || l.status === 'PENDING_SIGNATURES'
        );
        
        console.log(`   Active listings: ${activeListings.length}`);
        
        // Hide loading
        document.getElementById('my-offers-loading').classList.add('hidden');
        
        if (activeListings.length === 0) {
            // Show empty state
            document.getElementById('my-offers-empty').classList.remove('hidden');
            return;
        }
        
        // Render listings
        const container = document.getElementById('my-offers-list');
        container.innerHTML = '';
        container.classList.remove('hidden');
        
        for (const listing of activeListings) {
            const listingCard = createAtomicListingCard(listing);
            container.appendChild(listingCard);
        }
        
        console.log(`✅ Loaded ${activeListings.length} active atomic listings (${data.listings?.length || 0} total)`);
        
    } catch (error) {
        console.error('❌ Error loading atomic listings:', error);
        document.getElementById('my-offers-loading').classList.add('hidden');
        showNotification('Failed to load listings', 'error');
    }
}

/**
 * Create Offer Card Element
 */
function createOfferCard(offer) {
    const card = document.createElement('div');
    card.className = 'offer-card';
    card.style.cssText = 'display: flex; gap: 12px; padding: 16px; border: 1px solid var(--color-border); border-radius: 12px; background: var(--color-surface); margin-bottom: 12px; align-items: center;';
    
    // Preview image
    const preview = document.createElement('img');
    preview.className = 'offer-preview';
    preview.src = `https://ordinals.com/content/${offer.inscription_id}`;
    preview.alt = 'Inscription';
    preview.style.cssText = 'width: 80px; height: 80px; border-radius: 8px; object-fit: cover; flex-shrink: 0;';
    preview.onerror = () => {
        preview.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80"%3E%3Crect fill="%23333" width="80" height="80"/%3E%3Ctext x="50%25" y="50%25" dominant-baseline="middle" text-anchor="middle" fill="%23666" font-size="24"%3E📜%3C/text%3E%3C/svg%3E';
    };
    
    // Details
    const details = document.createElement('div');
    details.className = 'offer-details';
    details.style.cssText = 'flex: 1; min-width: 0;';
    
    const title = document.createElement('p');
    title.className = 'offer-title';
    title.style.cssText = 'font-weight: 600; font-size: 14px; margin: 0 0 4px 0; color: var(--color-text);';
    title.textContent = offer.inscription_number 
        ? `Inscription #${offer.inscription_number}`
        : `Inscription ${offer.inscription_id.substring(0, 10)}...`;
    
    const contentType = document.createElement('p');
    contentType.className = 'offer-content-type';
    contentType.style.cssText = 'font-size: 11px; color: var(--color-text-secondary); margin: 0 0 8px 0;';
    contentType.textContent = offer.content_type || 'unknown';
    
    const price = document.createElement('p');
    price.className = 'offer-price';
    price.style.cssText = 'font-weight: 600; font-size: 14px; color: var(--color-primary); margin: 0 0 4px 0;';
    price.textContent = `💰 ${offer.offer_amount.toLocaleString()} sats`;
    
    const date = document.createElement('p');
    date.className = 'offer-date';
    date.style.cssText = 'font-size: 11px; color: var(--color-text-secondary); margin: 0;';
    const createdDate = new Date(offer.created_at);
    date.textContent = `Listed: ${createdDate.toLocaleDateString()}`;
    
    details.appendChild(title);
    details.appendChild(contentType);
    details.appendChild(price);
    details.appendChild(date);
    
    // Actions
    const actions = document.createElement('div');
    actions.className = 'offer-actions';
    actions.style.cssText = 'display: flex; flex-direction: column; gap: 8px; flex-shrink: 0;';
    
    const shareBtn = document.createElement('button');
    shareBtn.className = 'btn-share-offer';
    shareBtn.style.cssText = 'padding: 8px 16px; background: var(--color-primary); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 12px; white-space: nowrap;';
    shareBtn.textContent = '📱 Share';
    shareBtn.onclick = () => showShareModal(offer);
    
    const cancelBtn = document.createElement('button');
    cancelBtn.className = 'btn-cancel-offer';
    cancelBtn.style.cssText = 'padding: 8px 16px; background: var(--color-error); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 12px; white-space: nowrap;';
    cancelBtn.textContent = '❌ Cancel';
    cancelBtn.onclick = () => cancelOffer(offer.id);
    
    actions.appendChild(shareBtn);
    actions.appendChild(cancelBtn);
    
    // Assemble card
    card.appendChild(preview);
    card.appendChild(details);
    card.appendChild(actions);
    
    return card;
}

/**
 * Create Atomic Listing Card Element
 */
function createAtomicListingCard(listing) {
    const card = document.createElement('div');
    card.className = 'atomic-listing-card';
    card.style.cssText = 'display: flex; gap: 12px; padding: 16px; border: 1px solid var(--color-border); border-radius: 12px; background: var(--color-surface); margin-bottom: 12px; align-items: center;';
    
    // Preview image
    const preview = document.createElement('img');
    preview.className = 'listing-preview';
    preview.src = `https://kraywallet-backend.onrender.com/api/ordinals/${listing.inscription_id}/content`;
    preview.alt = 'Inscription';
    preview.style.cssText = 'width: 80px; height: 80px; border-radius: 8px; object-fit: cover; flex-shrink: 0;';
    preview.onerror = () => {
        preview.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="80" height="80"%3E%3Crect fill="%23333" width="80" height="80"/%3E%3Ctext x="50%25" y="50%25" dominant-baseline="middle" text-anchor="middle" fill="%23666" font-size="24"%3E📜%3C/text%3E%3C/svg%3E';
    };
    
    // Details
    const details = document.createElement('div');
    details.className = 'listing-details';
    details.style.cssText = 'flex: 1; min-width: 0;';
    
    const title = document.createElement('p');
    title.className = 'listing-title';
    title.style.cssText = 'font-weight: 600; font-size: 14px; margin: 0 0 4px 0; color: var(--color-text);';
    title.textContent = listing.inscription_number 
        ? `Inscription #${listing.inscription_number}`
        : `Inscription ${listing.inscription_id.substring(0, 10)}...`;
    
    // Status badge
    const statusColors = {
        'PENDING_SIGNATURES': '#FFA500',
        'OPEN': '#4CAF50',
        'FILLED': '#666',
        'CANCELLED': '#999',
        'EXPIRED': '#999'
    };
    
    const statusBadge = document.createElement('span');
    statusBadge.style.cssText = `display: inline-block; padding: 2px 8px; border-radius: 4px; font-size: 10px; font-weight: 600; background: ${statusColors[listing.status] || '#666'}; color: white; margin: 0 0 8px 0;`;
    statusBadge.textContent = listing.status;
    
    const price = document.createElement('p');
    price.className = 'listing-price';
    price.style.cssText = 'font-weight: 600; font-size: 14px; color: var(--color-primary); margin: 8px 0 4px 0;';
    price.textContent = `💰 ${listing.price_sats.toLocaleString()} sats`;
    
    const date = document.createElement('p');
    date.className = 'listing-date';
    date.style.cssText = 'font-size: 11px; color: var(--color-text-secondary); margin: 0;';
    const createdDate = new Date(listing.created_at * 1000);
    date.textContent = `Listed: ${createdDate.toLocaleDateString()}`;
    
    details.appendChild(title);
    details.appendChild(statusBadge);
    details.appendChild(price);
    details.appendChild(date);
    
    // Actions
    const actions = document.createElement('div');
    actions.className = 'listing-actions';
    actions.style.cssText = 'display: flex; flex-direction: column; gap: 8px; flex-shrink: 0;';
    
    // Botões dinâmicos baseados no status
    if (listing.status === 'PENDING_SIGNATURES') {
        const updateBtn = document.createElement('button');
        updateBtn.className = 'btn-update-price';
        updateBtn.style.cssText = 'padding: 8px 16px; background: var(--color-primary); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 12px; white-space: nowrap;';
        updateBtn.textContent = '💰 Update Price';
        updateBtn.onclick = () => showUpdatePriceModal(listing);
        actions.appendChild(updateBtn);
        
        const cancelBtn = document.createElement('button');
        cancelBtn.className = 'btn-cancel-listing';
        cancelBtn.style.cssText = 'padding: 8px 16px; background: var(--color-error); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 12px; white-space: nowrap;';
        cancelBtn.textContent = '❌ Cancel';
        cancelBtn.onclick = () => cancelAtomicListing(listing.order_id);
        actions.appendChild(cancelBtn);
    } else if (listing.status === 'OPEN') {
        const cancelBtn = document.createElement('button');
        cancelBtn.className = 'btn-cancel-listing';
        cancelBtn.style.cssText = 'padding: 8px 16px; background: var(--color-error); color: white; border: none; border-radius: 8px; cursor: pointer; font-size: 12px; white-space: nowrap;';
        cancelBtn.textContent = '❌ Cancel';
        cancelBtn.onclick = () => cancelAtomicListing(listing.order_id);
        actions.appendChild(cancelBtn);
    }
    
    // Assemble card
    card.appendChild(preview);
    card.appendChild(details);
    card.appendChild(actions);
    
    return card;
}

/**
 * Cancel Atomic Listing (COM ASSINATURA - SEM CONFIRM)
 */
async function cancelAtomicListing(orderId) {
    try {
        showNotification('🔐 Sign message to cancel listing...', 'info');
        
        const result = await chrome.runtime.sendMessage({
            action: 'cancelListing',
            data: { orderId }
        });
        
        if (result.success) {
            showNotification('✅ Listing cancelled successfully!', 'success');
            
            // 🔄 Recarregar TUDO para atualizar UI
            await loadMyOffers(); // Reload "My Market Listings" tab
            await loadOrdinals(); // Reload "Ordinals" tab (para atualizar botões List/Send)
        } else {
            throw new Error(result.error || 'Failed to cancel listing');
        }
    } catch (error) {
        console.error('❌ Error cancelling listing:', error);
        showNotification(`❌ ${error.message}`, 'error');
    }
}

/**
 * Show Update Price Modal
 */
function showUpdatePriceModal(listing) {
    const newPrice = prompt(`Update price for Inscription #${listing.inscription_number || '...'}?\n\nCurrent price: ${listing.price_sats} sats\nEnter new price in sats:`, listing.price_sats);
    
    if (!newPrice || isNaN(newPrice) || parseInt(newPrice) < 546) {
        if (newPrice !== null) {
            showNotification('❌ Invalid price (minimum 546 sats)', 'error');
        }
        return;
    }
    
    updateListingPrice(listing.order_id, parseInt(newPrice));
}

/**
 * Update Listing Price
 */
async function updateListingPrice(orderId, newPrice) {
    try {
        showNotification('⏳ Updating price...', 'info');
        
        const result = await chrome.runtime.sendMessage({
            action: 'updateListingPrice',
            data: { orderId, newPrice }
        });
        
        if (result.success) {
            if (result.requiresSignature) {
                showNotification('✅ Price updated! Please sign the new PSBT.', 'success');
            } else {
                showNotification('✅ Price updated successfully!', 'success');
            }
            await loadMyOffers(); // Reload listings
        } else {
            throw new Error(result.error || 'Failed to update price');
        }
    } catch (error) {
        console.error('❌ Error updating price:', error);
        showNotification(`❌ ${error.message}`, 'error');
    }
}

/**
 * Cancel Offer (OLD SYSTEM - MANTIDO PARA COMPATIBILIDADE)
 */
async function cancelOffer(offerId) {
    try {
        showLoadingOverlay('Cancelling listing...');
        
        const response = await fetch(
            `https://kraywallet-backend.onrender.com/api/offers/${offerId}/cancel`,
            { method: 'PUT' }
        );
        
        const data = await response.json();
        
        hideLoadingOverlay();
        
        if (data.success) {
            showNotification('✅ Listing cancelled', 'success');
            loadMyOffers(); // Reload
        } else {
            throw new Error(data.error || 'Failed to cancel');
        }
        
    } catch (error) {
        console.error('❌ Error cancelling offer:', error);
        hideLoadingOverlay();
        showNotification('Failed to cancel listing', 'error');
    }
}

/**
 * Show Share Modal (Placeholder for Fase 6)
 */
function showShareModal(offer) {
    console.log('📱 Sharing offer:', offer);
    
    // Open offer page in new tab
    const offerUrl = `https://kraywallet-backend.onrender.com/offer.html?id=${offer.id}`;
    chrome.tabs.create({ url: offerUrl });
    
    showNotification('📱 Opening offer page...', 'info');
}

// ========================================
// MY PUBLIC PROFILE
// ========================================

/**
 * Show My Profile Screen
 */
async function showMyProfileScreen() {
    console.log('🎭 Opening My Profile screen...');
    showScreen('my-profile');
    await loadMyProfile();
}

/**
 * Fetch BTC Price and Update USD Values
 */
async function updateUSDPrices() {
    try {
        // Fetch BTC price from Mempool.space API
        const response = await fetch('https://mempool.space/api/v1/prices');
        const prices = await response.json();
        const btcPriceUSD = prices.USD;
        
        console.log('💵 BTC Price:', btcPriceUSD, 'USD');
        
        // Update all USD price elements
        document.querySelectorAll('.usd-price-target').forEach(element => {
            const sats = parseInt(element.dataset.sats);
            const btc = sats / 100000000;
            const usd = btc * btcPriceUSD;
            element.textContent = `≈ $${usd.toFixed(2)} USD`;
        });
        
    } catch (error) {
        console.error('❌ Error fetching BTC price:', error);
        // Keep default "≈ $0.00 USD" if fetch fails
    }
}

/**
 * Load My Profile Data
 */
async function loadMyProfile() {
    console.log('🎭 Loading my profile...');
    
    try {
        // Get address from background script
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        const address = result?.data?.address || walletState.address;
        
        console.log('📍 User address:', address);
        
        if (!address) {
            showNotification('❌ Wallet not initialized', 'error');
            return;
        }
        
        // Update profile header
        document.getElementById('profile-address').textContent = address;
        
        // Show loading
        document.getElementById('profile-loading').classList.remove('hidden');
        
        // Fetch profile data
        const [offersRes, inscriptionsRes, runesRes] = await Promise.all([
            fetch(`https://kraywallet-backend.onrender.com/api/offers?address=${address}&status=active`),
            chrome.runtime.sendMessage({ action: 'getInscriptions' }),
            chrome.runtime.sendMessage({ action: 'getRunes' })
        ]);
        
        const offersData = await offersRes.json();
        
        console.log('📦 Profile data loaded:', {
            offers: offersData.offers?.length || 0,
            inscriptions: inscriptionsRes?.inscriptions?.length || 0,
            runes: runesRes?.runes?.length || 0
        });
        
        // Update stats
        document.getElementById('profile-listings-count').textContent = offersData.offers?.length || 0;
        document.getElementById('profile-inscriptions-count').textContent = inscriptionsRes?.inscriptions?.length || 0;
        document.getElementById('profile-runes-count').textContent = runesRes?.runes?.length || 0;
        
        // Hide loading and show tabs
        document.getElementById('profile-loading').classList.add('hidden');
        document.getElementById('profile-tabs').classList.remove('hidden');
        
        // Separate offers by type
        const ordinalOffers = offersData.offers?.filter(o => o.type === 'inscription') || [];
        const runeOffers = offersData.offers?.filter(o => o.type === 'rune_swap') || [];
        const poolOffers = offersData.offers?.filter(o => o.type === 'liquidity_pool') || [];
        
        console.log('📊 Offers by type:', {
            ordinals: ordinalOffers.length,
            runes: runeOffers.length,
            pools: poolOffers.length
        });
        
        // Render Ordinals tab
        renderProfileTab('ordinals', ordinalOffers);
        
        // Render Runes tab
        renderProfileTab('runes', runeOffers);
        
        // Render Pools tab
        renderProfileTab('pools', poolOffers);
        
        // Update USD prices after rendering cards
        await updateUSDPrices();
        
    } catch (error) {
        console.error('❌ Error loading profile:', error);
        document.getElementById('profile-loading').classList.add('hidden');
        showNotification('Failed to load profile', 'error');
    }
}

/**
 * Render Profile Tab Content
 */
function renderProfileTab(tabName, offers) {
    const listContainer = document.getElementById(`profile-${tabName}-list`);
    const emptyState = document.getElementById(`profile-${tabName}-empty`);
    
    if (!offers || offers.length === 0) {
        // Show empty state
        emptyState.classList.remove('hidden');
        listContainer.classList.add('hidden');
        listContainer.innerHTML = '';
        return;
    }
    
    // Hide empty state and show list
    emptyState.classList.add('hidden');
    listContainer.classList.remove('hidden');
    listContainer.innerHTML = '';
    
    // Render up to 3 offers
    const displayOffers = offers.slice(0, 3);
    for (const offer of displayOffers) {
        const card = createMiniOfferCard(offer);
        listContainer.appendChild(card);
    }
}

/**
 * Create Mini Offer Card for Profile Preview
 */
function createMiniOfferCard(offer) {
    const card = document.createElement('div');
    card.style.cssText = 'display: flex; gap: 12px; padding: 12px; border: 1px solid var(--color-border); border-radius: 8px; background: var(--color-surface); margin-bottom: 8px; transition: all 0.2s; position: relative;';
    
    card.onmouseenter = () => {
        card.style.transform = 'translateY(-2px)';
        card.style.boxShadow = '0 4px 12px rgba(0,0,0,0.1)';
    };
    card.onmouseleave = () => {
        card.style.transform = 'translateY(0)';
        card.style.boxShadow = 'none';
    };
    
    // Preview image
    const preview = document.createElement('img');
    preview.src = `https://ordinals.com/content/${offer.inscription_id}`;
    preview.alt = 'Inscription';
    preview.style.cssText = 'width: 60px; height: 60px; border-radius: 6px; object-fit: cover; flex-shrink: 0; cursor: pointer;';
    preview.onerror = () => {
        preview.src = 'data:image/svg+xml,%3Csvg xmlns="http://www.w3.org/2000/svg" width="60" height="60"%3E%3Crect fill="%23333" width="60" height="60"/%3E%3Ctext x="50%25" y="50%25" dominant-baseline="middle" text-anchor="middle" fill="%23666" font-size="20"%3E📜%3C/text%3E%3C/svg%3E';
    };
    preview.onclick = () => {
        // Open offer page
        const offerUrl = `https://kraywallet-backend.onrender.com/offer.html?id=${offer.id}`;
        chrome.tabs.create({ url: offerUrl });
    };
    
    // Details
    const details = document.createElement('div');
    details.style.cssText = 'flex: 1; min-width: 0; display: flex; flex-direction: column; justify-content: space-between;';
    
    // Top part (title)
    const topPart = document.createElement('div');
    
    const title = document.createElement('p');
    title.style.cssText = 'font-weight: 600; font-size: 13px; margin: 0 0 4px 0; color: var(--color-text); cursor: pointer;';
    title.textContent = offer.inscription_number 
        ? `Inscription #${offer.inscription_number}`
        : `Inscription ${offer.inscription_id.substring(0, 10)}...`;
    title.onclick = () => {
        const offerUrl = `https://kraywallet-backend.onrender.com/offer.html?id=${offer.id}`;
        chrome.tabs.create({ url: offerUrl });
    };
    
    topPart.appendChild(title);
    
    // Bottom part (price + buy button)
    const bottomPart = document.createElement('div');
    bottomPart.style.cssText = 'display: flex; align-items: center; justify-content: space-between; gap: 8px;';
    
    // Price container
    const priceContainer = document.createElement('div');
    
    const price = document.createElement('p');
    price.style.cssText = 'font-weight: 600; font-size: 13px; color: var(--color-primary); margin: 0;';
    price.textContent = `💰 ${offer.offer_amount.toLocaleString()} sats`;
    
    // USD price (will be calculated)
    const usdPrice = document.createElement('p');
    usdPrice.style.cssText = 'font-size: 11px; color: var(--color-text-secondary); margin: 2px 0 0 0;';
    usdPrice.textContent = '≈ $0.00 USD';
    usdPrice.className = 'usd-price-target';
    usdPrice.dataset.sats = offer.offer_amount;
    
    priceContainer.appendChild(price);
    priceContainer.appendChild(usdPrice);
    
    // Buy button
    const buyBtn = document.createElement('button');
    buyBtn.style.cssText = 'padding: 6px 12px; background: var(--color-primary); color: white; border: none; border-radius: 6px; cursor: pointer; font-size: 11px; font-weight: 600; white-space: nowrap; transition: all 0.2s;';
    buyBtn.textContent = '🛒 Buy';
    buyBtn.onmouseenter = () => {
        buyBtn.style.opacity = '0.8';
    };
    buyBtn.onmouseleave = () => {
        buyBtn.style.opacity = '1';
    };
    buyBtn.onclick = (e) => {
        e.stopPropagation();
        // Redirect to ordinals.html with buy parameter
        const buyUrl = `https://kraywallet-backend.onrender.com/ordinals.html?buy=${offer.id}`;
        chrome.tabs.create({ url: buyUrl });
    };
    
    bottomPart.appendChild(priceContainer);
    bottomPart.appendChild(buyBtn);
    
    details.appendChild(topPart);
    details.appendChild(bottomPart);
    
    card.appendChild(preview);
    card.appendChild(details);
    
    return card;
}

// Event Listeners for My Profile
document.addEventListener('DOMContentLoaded', () => {
    // My Profile button in settings
    document.getElementById('my-profile-btn')?.addEventListener('click', showMyProfileScreen);
    
    // Back button
    document.getElementById('back-from-profile-btn')?.addEventListener('click', () => {
        showScreen('settings');
    });
    
    // Open External Profile button
    document.getElementById('open-external-profile-btn')?.addEventListener('click', async () => {
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (result && result.data && result.data.address) {
            const profileUrl = `https://kraywallet-backend.onrender.com/profile.html?address=${result.data.address}`;
            chrome.tabs.create({ url: profileUrl });
        } else {
            alert('❌ Please unlock your wallet first');
        }
    });
    
    // Share Profile button
    document.getElementById('share-profile-btn')?.addEventListener('click', async () => {
        const result = await chrome.runtime.sendMessage({ action: 'getWalletInfo' });
        if (result && result.data && result.data.address) {
            const profileUrl = `https://kraywallet-backend.onrender.com/profile.html?address=${result.data.address}`;
            
            try {
                await navigator.clipboard.writeText(profileUrl);
                showNotification('📋 Profile link copied!', 'success');
            } catch (err) {
                console.error('Failed to copy:', err);
                // Fallback: show in alert
                alert(`Your profile URL:\n\n${profileUrl}`);
            }
        } else {
            alert('❌ Please unlock your wallet first');
        }
    });
    
    // View All Listings button
    document.getElementById('view-all-ordinals-btn')?.addEventListener('click', () => {
        showMyOffersScreen();
    });
    
    // Profile tabs switching
    document.querySelectorAll('.profile-tab').forEach(tab => {
        tab.addEventListener('click', (e) => {
            const tabName = e.currentTarget.dataset.tab;
            switchProfileTab(tabName);
        });
    });
});

/**
 * Switch Profile Tab
 */
function switchProfileTab(tabName) {
    console.log('🔄 Switching to tab:', tabName);
    
    // Update tab buttons
    document.querySelectorAll('.profile-tab').forEach(tab => {
        if (tab.dataset.tab === tabName) {
            tab.classList.add('active');
            tab.style.borderBottomColor = 'var(--color-primary)';
            tab.style.color = 'var(--color-primary)';
        } else {
            tab.classList.remove('active');
            tab.style.borderBottomColor = 'transparent';
            tab.style.color = 'var(--color-text-secondary)';
        }
    });
    
    // Update tab content
    document.querySelectorAll('.profile-tab-content').forEach(content => {
        content.classList.add('hidden');
    });
    
    document.getElementById(`profile-tab-${tabName}`)?.classList.remove('hidden');
}

// Event Listeners for My Offers
document.addEventListener('DOMContentLoaded', () => {
    // My Offers button in settings
    document.getElementById('my-offers-btn')?.addEventListener('click', showMyOffersScreen);
    
    // Back button
    document.getElementById('back-from-my-offers-btn')?.addEventListener('click', () => {
        showScreen('settings');
    });
});

// ========================================
// PSBT SIGNING SCREEN EVENT LISTENERS
// ========================================
document.addEventListener('DOMContentLoaded', () => {
    console.log('🔐 Setting up PSBT signing listeners...');
    
    // Sign button
    const psbtSignBtn = document.getElementById('psbt-sign-btn');
    if (psbtSignBtn) {
        psbtSignBtn.addEventListener('click', async () => {
            console.log('✅ Sign button clicked');
            await handlePsbtSign();
        });
        console.log('   ✅ psbt-sign-btn listener added');
    } else {
        console.warn('   ⚠️  psbt-sign-btn not found');
    }
    
    // Cancel button
    const psbtCancelBtn = document.getElementById('psbt-cancel-btn');
    if (psbtCancelBtn) {
        psbtCancelBtn.addEventListener('click', () => {
            console.log('❌ Cancel button clicked');
            handlePsbtCancel();
        });
        console.log('   ✅ psbt-cancel-btn listener added');
    } else {
        console.warn('   ⚠️  psbt-cancel-btn not found');
    }
    
    // ⚡ Lightning Payment listeners
    console.log('⚡ Setting up Lightning payment listeners...');
    
    const lightningConfirmBtn = document.getElementById('lightning-payment-confirm-btn');
    if (lightningConfirmBtn) {
        lightningConfirmBtn.addEventListener('click', async () => {
            console.log('✅ Lightning payment confirm button clicked');
            await handleLightningPaymentConfirm();
        });
        console.log('   ✅ lightning-payment-confirm-btn listener added');
    } else {
        console.warn('   ⚠️  lightning-payment-confirm-btn not found');
    }
    
    const lightningCancelBtn = document.getElementById('lightning-payment-cancel-btn');
    if (lightningCancelBtn) {
        lightningCancelBtn.addEventListener('click', () => {
            console.log('❌ Lightning payment cancel button clicked');
            handleLightningPaymentCancel();
        });
        console.log('   ✅ lightning-payment-cancel-btn listener added');
    } else {
        console.warn('   ⚠️  lightning-payment-cancel-btn not found');
    }
    
    // Enter key para confirmar Lightning payment
    const lightningPasswordInput = document.getElementById('lightning-payment-password');
    if (lightningPasswordInput) {
        lightningPasswordInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                console.log('⚡ Enter pressed on Lightning payment password');
                handleLightningPaymentConfirm();
            }
        });
        console.log('   ✅ lightning-payment-password Enter listener added');
    }
});

// ==========================================
// 💝 SIGN MESSAGE (Para Likes)
// ==========================================

// Mostrar tela de sign message
// Now supports display metadata from L2 transactions
function showSignMessageScreen(message, displayMetadata = null) {
    console.log('✍️  Showing sign message screen for:', message);
    console.log('   Display metadata:', displayMetadata);
    
    let actionDetails;
    
    // If display metadata is provided (from signL2Transaction), use it directly
    if (displayMetadata && displayMetadata.title) {
        console.log('📋 Using provided display metadata');
        const amountNum = parseFloat(displayMetadata.amount) || 0;
        actionDetails = {
            title: `${displayMetadata.icon || '🔐'} ${displayMetadata.title}`,
            icon: displayMetadata.icon || '🔐',
            type: displayMetadata.type || 'L2 Transaction',
            cost: amountNum > 0 ? `Cost: ${displayMetadata.amount}` : 'Free',
            description: displayMetadata.description || 'Sign to authorize this action.',
            inscription: displayMetadata.recipient 
                ? `To: ${displayMetadata.recipient}\nAmount: ${displayMetadata.amount}`
                : message.substring(0, 50) + '...',
            bgColor: 'linear-gradient(135deg, rgba(139,92,246,0.2), rgba(59,130,246,0.1))',
            borderColor: 'rgba(139,92,246,0.4)'
        };
    } else {
        // Parse message to extract action details (fallback)
        actionDetails = parseActionMessage(message);
    }
    
    // Update UI with details
    document.getElementById('sign-screen-title').textContent = actionDetails.title;
    document.getElementById('action-icon').textContent = actionDetails.icon;
    document.getElementById('action-type').textContent = actionDetails.type;
    document.getElementById('action-cost').textContent = actionDetails.cost;
    document.getElementById('action-description').textContent = actionDetails.description;
    document.getElementById('action-inscription').textContent = actionDetails.inscription;
    document.getElementById('message-to-sign').textContent = message;
    
    // Update action summary box color based on action type
    const summaryBox = document.getElementById('action-summary-box');
    summaryBox.style.background = actionDetails.bgColor;
    summaryBox.style.borderColor = actionDetails.borderColor;
    
    // Mostrar tela
    showScreen('sign-message');
    
    // Focar no campo de senha
    setTimeout(() => {
        document.getElementById('message-sign-password')?.focus();
    }, 100);
}

// Parse action message to extract details for display
function parseActionMessage(message) {
    const msgLower = message.toLowerCase();
    let details = {
        title: '✍️ Sign Action',
        icon: '📝',
        type: 'Action',
        cost: 'Free',
        description: 'You are about to sign this message.',
        inscription: '...',
        bgColor: 'linear-gradient(135deg, rgba(247,147,26,0.15), rgba(239,68,68,0.1))',
        borderColor: 'rgba(247,147,26,0.3)'
    };
    
    // ═══════════════════════════════════════════════════════════════
    // L2 TRANSACTION FORMAT: from:to:amount:nonce:tx_type
    // ═══════════════════════════════════════════════════════════════
    const l2Parts = message.split(':');
    if (l2Parts.length >= 5 && l2Parts[0].startsWith('bc1')) {
        const [from, to, amount, nonce, txType] = l2Parts;
        const amountNum = parseFloat(amount) || 0;
        
        // Format addresses for display
        const fromShort = from.substring(0, 8) + '...' + from.slice(-6);
        const toShort = to.startsWith('bc1') ? to.substring(0, 8) + '...' + to.slice(-6) : to;
        
        // Detect L2 transaction type
        if (txType === 'AI_AVATAR') {
            details.title = '🎨 AI Avatar Generation';
            details.icon = '🎨';
            details.type = 'AI Avatar';
            details.cost = `Cost: ${amountNum} KRAY`;
            details.description = `Generate AI avatar image. ${amountNum} KRAY will be sent to AI Provider.`;
            details.inscription = `To: AI Provider\nAmount: ${amountNum} KRAY`;
            details.bgColor = 'linear-gradient(135deg, rgba(139,92,246,0.2), rgba(236,72,153,0.1))';
            details.borderColor = 'rgba(139,92,246,0.4)';
        } else if (txType === 'AI_GENERATE' || txType === 'AI_NFT') {
            details.title = '🤖 AI NFT Generation';
            details.icon = '🤖';
            details.type = 'AI Generate';
            details.cost = `Cost: ${amountNum} KRAY`;
            details.description = `Generate AI image for NFT. ${amountNum} KRAY will be sent to AI Provider.`;
            details.inscription = `To: AI Provider\nAmount: ${amountNum} KRAY`;
            details.bgColor = 'linear-gradient(135deg, rgba(16,185,129,0.2), rgba(59,130,246,0.1))';
            details.borderColor = 'rgba(16,185,129,0.4)';
        } else if (txType === 'collection_config') {
            details.title = '⚙️ Collection Settings';
            details.icon = '⚙️';
            details.type = 'Configuration';
            details.cost = 'Free (no transfer)';
            details.description = 'Save your collection settings. This proves you are the owner.';
            details.inscription = `Collection Owner: ${fromShort}`;
            details.bgColor = 'linear-gradient(135deg, rgba(59,130,246,0.2), rgba(139,92,246,0.1))';
            details.borderColor = 'rgba(59,130,246,0.4)';
        } else if (txType === 'NFT_MINT' || txType === 'nft_mint') {
            details.title = '🖼️ Mint NFT';
            details.icon = '🖼️';
            details.type = 'NFT Mint';
            details.cost = amountNum > 0 ? `Cost: ${amountNum} KRAY` : 'Gas: 1 KRAY';
            details.description = amountNum > 0 
                ? `Mint NFT for ${amountNum} KRAY (goes to developer).`
                : 'Mint NFT as developer. Only 1 KRAY gas fee.';
            details.inscription = `To: ${toShort}\nAmount: ${amountNum} KRAY`;
            details.bgColor = 'linear-gradient(135deg, rgba(245,158,11,0.2), rgba(247,147,26,0.1))';
            details.borderColor = 'rgba(245,158,11,0.4)';
        } else if (txType === 'transfer' || txType === 'TRANSFER') {
            details.title = '💸 L2 Transfer';
            details.icon = '💸';
            details.type = 'Transfer';
            details.cost = `Amount: ${amountNum} KRAY`;
            details.description = `Send ${amountNum} KRAY to another address.`;
            details.inscription = `To: ${toShort}\nAmount: ${amountNum} KRAY`;
            details.bgColor = 'linear-gradient(135deg, rgba(34,197,94,0.2), rgba(16,185,129,0.1))';
            details.borderColor = 'rgba(34,197,94,0.4)';
        } else {
            // Generic L2 transaction
            details.title = '🔐 L2 Transaction';
            details.icon = '🔐';
            details.type = txType || 'L2 Action';
            details.cost = amountNum > 0 ? `Amount: ${amountNum} KRAY` : 'Free';
            details.description = `L2 action: ${txType}. Verify details before signing.`;
            details.inscription = `Type: ${txType}\nAmount: ${amountNum} KRAY`;
            details.bgColor = 'linear-gradient(135deg, rgba(107,114,128,0.2), rgba(75,85,99,0.1))';
            details.borderColor = 'rgba(107,114,128,0.4)';
        }
        
        return details;
    }
    
    // Extract inscription ID if present
    const inscriptionMatch = message.match(/on ([a-f0-9]{64}i\d+)/i);
    if (inscriptionMatch) {
        details.inscription = inscriptionMatch[1].substring(0, 12) + '...' + inscriptionMatch[1].slice(-8);
    }
    
    // Detect action type (legacy social actions)
    if (msgLower.includes('post on') || msgLower.startsWith('post ')) {
        details.title = '📝 Create Post';
        details.icon = '📝';
        details.type = 'Social Post';
        details.cost = 'Cost: 1 KRAY';
        details.description = 'You are creating a new post on your inscription profile. This will be permanently recorded on L2.';
        details.bgColor = 'linear-gradient(135deg, rgba(139,92,246,0.15), rgba(239,68,68,0.1))';
        details.borderColor = 'rgba(139,92,246,0.3)';
    } else if (msgLower.includes('like post') || msgLower.includes('like on')) {
        details.title = '❤️ Like Post';
        details.icon = '❤️';
        details.type = 'Social Like';
        details.cost = 'Cost: 1 KRAY';
        details.description = 'You are liking this post. 1 KRAY will go to the post creator.';
        details.bgColor = 'linear-gradient(135deg, rgba(239,68,68,0.15), rgba(247,147,26,0.1))';
        details.borderColor = 'rgba(239,68,68,0.3)';
    } else if (msgLower.includes('repost')) {
        details.title = '🔄 Repost';
        details.icon = '🔄';
        details.type = 'Social Repost';
        details.cost = 'Earn KRAY!';
        details.description = 'You are reposting this content. You may earn KRAY from the repost pool!';
        details.bgColor = 'linear-gradient(135deg, rgba(16,185,129,0.15), rgba(247,147,26,0.1))';
        details.borderColor = 'rgba(16,185,129,0.3)';
    } else if (msgLower.includes('comment')) {
        details.title = '💬 Add Comment';
        details.icon = '💬';
        details.type = 'Comment';
        details.cost = 'Free';
        details.description = 'You are adding a comment to this post. Signature required for authentication.';
        details.bgColor = 'linear-gradient(135deg, rgba(59,130,246,0.15), rgba(139,92,246,0.1))';
        details.borderColor = 'rgba(59,130,246,0.3)';
    } else if (msgLower.includes('bio') || msgLower.includes('profile')) {
        details.title = '👤 Update Profile';
        details.icon = '👤';
        details.type = 'Profile Update';
        details.cost = 'Free';
        details.description = 'You are updating your inscription profile bio.';
        details.bgColor = 'linear-gradient(135deg, rgba(247,147,26,0.15), rgba(255,255,255,0.05))';
        details.borderColor = 'rgba(247,147,26,0.3)';
    } else if (msgLower.includes('offer') || msgLower.includes('share')) {
        details.title = '🎁 Share Offer';
        details.icon = '🎁';
        details.type = 'Share Offer';
        details.cost = 'Variable';
        details.description = 'You are creating a repost pool to reward users who share your content.';
        details.bgColor = 'linear-gradient(135deg, rgba(245,158,11,0.15), rgba(247,147,26,0.1))';
        details.borderColor = 'rgba(245,158,11,0.3)';
    }
    
    return details;
}

// Handle message sign
async function handleMessageSign() {
    const password = document.getElementById('message-sign-password').value;
    
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    try {
        showLoading('Signing message...');
        
        // Get pending message
        const storage = await chrome.storage.local.get(['pendingMessageRequest']);
        console.log('📦 Pending message storage:', storage);
        if (!storage.pendingMessageRequest) {
            throw new Error('No pending message found');
        }
        
        const message = storage.pendingMessageRequest.message;
        console.log('✍️  Message to sign:', message);
        
        // 🔥 Request background to sign with password
        console.log('🔄 Requesting background to sign with password...');
        
        const result = await chrome.runtime.sendMessage({
            action: 'signMessageWithPassword',
            data: { message, password }
        });
        
        console.log('📨 Background response:', result);
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to sign message');
        }
        
        // 🔒 SECURITY: Clear password field immediately
        clearAllPasswordFields();
        
        const { signature, address } = result;
        
        // Save result to storage
        await chrome.storage.local.set({
            messageSignResult: {
                success: true,
                signature: signature,
                address: address
            }
        });
        
        hideLoading();
        showNotification('✅ Message signed successfully!', 'success');
        
        // Close popup after short delay
        setTimeout(() => window.close(), 500);
        
    } catch (error) {
        hideLoading();
        console.error('Error signing message:', error);
        
        if (error.message.includes('Invalid password')) {
            showNotification('❌ Invalid password', 'error');
        } else {
            showNotification('❌ Error signing message: ' + error.message, 'error');
        }
    }
}

// Handle message cancel
async function handleMessageCancel() {
    // 🔒 SECURITY: Clear password field immediately
    clearAllPasswordFields();
    
    // Save cancelled result
    await chrome.storage.local.set({
        messageSignResult: {
            success: false,
            error: 'User cancelled'
        }
    });
    
    // Clear pending request
    await chrome.storage.local.remove('pendingMessageRequest');
    
    // Close popup
    window.close();
}

// Sign message locally (same logic as background)
async function signMessageLocal(message, mnemonic) {
    const network = bitcoin.networks.bitcoin;
    const seed = bip39.mnemonicToSeedSync(mnemonic);
    const root = bip32.BIP32Factory(ecc).fromSeed(seed, network);
    
    const child = root.derivePath("m/86'/0'/0'/0/0");
    const address = bitcoin.payments.p2tr({
        internalPubkey: child.publicKey.slice(1, 33),
        network
    }).address;
    
    const privateKey = child.privateKey;
    if (!privateKey) {
        throw new Error('Failed to derive private key');
    }
    
    const messageBuffer = new TextEncoder().encode(message);
    const messageHash = bitcoin.crypto.sha256(messageBuffer);
    const signatureBytes = ecc.sign(messageHash, privateKey);
    const signatureBase64 = btoa(String.fromCharCode.apply(null, signatureBytes));
    
    return {
        signature: signatureBase64,
        address: address
    };
}

// Event listeners for sign message
document.getElementById('message-sign-btn')?.addEventListener('click', handleMessageSign);
document.getElementById('message-cancel-btn')?.addEventListener('click', handleMessageCancel);

// ==========================================
// ⚡ LIGHTNING NETWORK UI (SEND/RECEIVE/OPEN CHANNEL)
// ==========================================

// Send Lightning - Mostrar tela
function showSendLightningScreen() {
    console.log('⚡ Opening Send Lightning screen...');
    showScreen('send-lightning');
    
    // Limpar campos
    document.getElementById('send-lightning-invoice').value = '';
    document.getElementById('send-lightning-decoded-info').classList.add('hidden');
    document.getElementById('send-lightning-pay-btn').disabled = true;
}

// Send Lightning - Decode invoice
async function handleDecodeLightningInvoice() {
    const invoice = document.getElementById('send-lightning-invoice').value.trim();
    
    if (!invoice) {
        showNotification('❌ Please enter a Lightning invoice', 'error');
        return;
    }
    
    try {
        console.log('🔍 Decoding invoice...');
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/lightning/decode', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ invoice })
        });
        
        const decoded = await response.json();
        
        if (!decoded.success) {
            throw new Error(decoded.error || 'Failed to decode invoice');
        }
        
        console.log('✅ Invoice decoded:', decoded);
        
        // Mostrar informações
        document.getElementById('send-lightning-amount').textContent = 
            `${decoded.amount?.toLocaleString() || '?'} sats`;
        document.getElementById('send-lightning-description').textContent = 
            decoded.description || 'No description';
        
        document.getElementById('send-lightning-decoded-info').classList.remove('hidden');
        document.getElementById('send-lightning-pay-btn').disabled = false;
        
        showNotification('✅ Invoice decoded successfully', 'success');
        
    } catch (error) {
        console.error('❌ Error decoding invoice:', error);
        showNotification('❌ Failed to decode invoice: ' + error.message, 'error');
    }
}

// Send Lightning - Pay invoice
async function handlePayLightningInvoice() {
    const invoice = document.getElementById('send-lightning-invoice').value.trim();
    
    if (!invoice) {
        showNotification('❌ Please enter a Lightning invoice', 'error');
        return;
    }
    
    try {
        console.log('⚡ Paying Lightning invoice...');
        
        // Usar window.krayWallet.sendPayment() que já implementamos!
        const result = await sendMessage({ 
            action: 'sendPayment', 
            invoice 
        });
        
        if (result.success) {
            console.log('✅ Payment successful!');
            console.log('   Preimage:', result.preimage);
            
            showNotification('✅ Lightning payment sent successfully!', 'success');
            
            // Voltar para wallet após 1 segundo
            setTimeout(() => {
                showScreen('wallet');
                loadWalletData();
            }, 1000);
        } else {
            throw new Error(result.error || 'Payment failed');
        }
        
    } catch (error) {
        console.error('❌ Error paying invoice:', error);
        showNotification('❌ Payment failed: ' + error.message, 'error');
    }
}

// Receive Lightning - Mostrar tela
function showReceiveLightningScreen() {
    console.log('⚡ Opening Receive Lightning screen...');
    showScreen('receive-lightning');
    
    // Limpar campos
    document.getElementById('receive-lightning-amount').value = '';
    document.getElementById('receive-lightning-description').value = '';
    document.getElementById('receive-lightning-invoice-display').classList.add('hidden');
    document.getElementById('receive-lightning-copy-btn').classList.add('hidden');
}

// Receive Lightning - Create invoice
async function handleCreateLightningInvoice() {
    const amount = parseInt(document.getElementById('receive-lightning-amount').value);
    const description = document.getElementById('receive-lightning-description').value.trim();
    
    if (!amount || amount <= 0) {
        showNotification('❌ Please enter a valid amount', 'error');
        return;
    }
    
    try {
        console.log('⚡ Creating Lightning invoice...');
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/lightning/invoice', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                amount,
                description: description || 'KrayWallet payment'
            })
        });
        
        const result = await response.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to create invoice');
        }
        
        console.log('✅ Invoice created:', result.invoice.substring(0, 50) + '...');
        
        // Mostrar invoice
        document.getElementById('receive-lightning-invoice-text').value = result.invoice;
        document.getElementById('receive-lightning-invoice-display').classList.remove('hidden');
        document.getElementById('receive-lightning-copy-btn').classList.remove('hidden');
        
        // TODO: Gerar QR Code
        // document.getElementById('receive-lightning-qr').innerHTML = `<img src="..." />`;
        
        showNotification('✅ Invoice created successfully!', 'success');
        
    } catch (error) {
        console.error('❌ Error creating invoice:', error);
        showNotification('❌ Failed to create invoice: ' + error.message, 'error');
    }
}

// Receive Lightning - Copy invoice
function handleCopyLightningInvoice() {
    const invoice = document.getElementById('receive-lightning-invoice-text').value;
    
    navigator.clipboard.writeText(invoice).then(() => {
        showNotification('✅ Invoice copied to clipboard!', 'success');
    }).catch(err => {
        console.error('Failed to copy invoice:', err);
        showNotification('❌ Failed to copy invoice', 'error');
    });
}

// Open Channel - Mostrar tela
function showOpenChannelScreen() {
    console.log('📡 Opening Open Channel screen...');
    showScreen('open-channel');
    
    // Limpar campos
    document.getElementById('open-channel-pubkey').value = '';
    document.getElementById('open-channel-amount').value = '';
    document.getElementById('open-channel-host').value = '';
}

// Open Channel - Confirm
async function handleOpenChannel() {
    const pubkey = document.getElementById('open-channel-pubkey').value.trim();
    const amount = parseInt(document.getElementById('open-channel-amount').value);
    const host = document.getElementById('open-channel-host').value.trim();
    
    if (!pubkey) {
        showNotification('❌ Please enter a node public key', 'error');
        return;
    }
    
    if (!amount || amount < 20000) {
        showNotification('❌ Minimum channel capacity is 20,000 sats', 'error');
        return;
    }
    
    try {
        console.log('📡 Opening Lightning channel...');
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/lightning/open-channel', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                nodePubkey: pubkey,
                localAmount: amount,
                host: host || undefined
            })
        });
        
        const result = await response.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to open channel');
        }
        
        console.log('✅ Channel opened:', result);
        
        showNotification('✅ Channel opened successfully!', 'success');
        
        // Voltar para wallet após 1 segundo
        setTimeout(() => {
            showScreen('wallet');
            loadWalletData();
        }, 1000);
        
    } catch (error) {
        console.error('❌ Error opening channel:', error);
        showNotification('❌ Failed to open channel: ' + error.message, 'error');
    }
}

// Event Listeners para botões Lightning
document.getElementById('send-lightning-btn')?.addEventListener('click', showSendLightningScreen);
document.getElementById('receive-lightning-btn')?.addEventListener('click', showReceiveLightningScreen);
document.getElementById('open-channel-btn')?.addEventListener('click', showOpenChannelScreen);

// Event Listeners para ações nas telas
document.getElementById('send-lightning-decode-btn')?.addEventListener('click', handleDecodeLightningInvoice);
document.getElementById('send-lightning-pay-btn')?.addEventListener('click', handlePayLightningInvoice);

document.getElementById('receive-lightning-create-btn')?.addEventListener('click', handleCreateLightningInvoice);
document.getElementById('receive-lightning-copy-btn')?.addEventListener('click', handleCopyLightningInvoice);

document.getElementById('open-channel-confirm-btn')?.addEventListener('click', handleOpenChannel);

// Event Listeners para botões de voltar
document.getElementById('back-from-send-lightning-btn')?.addEventListener('click', () => {
    showScreen('wallet');
    loadWalletData();
});
document.getElementById('back-from-receive-lightning-btn')?.addEventListener('click', () => {
    showScreen('wallet');
    loadWalletData();
});
document.getElementById('back-from-open-channel-btn')?.addEventListener('click', () => {
    showScreen('wallet');
    loadWalletData();
});

console.log('✅ Lightning UI functions loaded!');

// Expose functions
window.showMyOffersScreen = showMyOffersScreen;
window.loadMyOffers = loadMyOffers;
window.cancelOffer = cancelOffer;
window.showShareModal = showShareModal;


async function showSeparateAssetsScreen() {
    console.log('🔀 Opening Separate Assets screen...');
    showScreen('separate-assets');
    await loadSeparateAssets();
}

async function loadSeparateAssets() {
    const listContainer = document.getElementById('separate-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;"><div class="spinner"></div> Loading UTXOs...</div>';
    
    try {
        const address = await getWalletAddress();
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await response.json();
        
        if (!data.success || !data.utxos) {
            throw new Error('Failed to load UTXOs');
        }
        
        // Filter for mixed UTXOs (has BOTH inscription AND runes) and process raw amounts
        const mixedUtxos = data.utxos
            .filter(u => u.hasInscription && u.hasRunes)
            .map(utxo => {
                // Process runes to extract rawAmount and divisibility
                const processedRunes = (utxo.runes || []).map(r => {
                    let rawAmount = 0;
                    let divisibility = 0;
                    let displayAmount = 0;
                    
                    if (r.amount && typeof r.amount === 'object') {
                        // Structured amount: {amount, divisibility, symbol}
                        rawAmount = r.amount.amount || 0;
                        divisibility = r.amount.divisibility || 0;
                    } else {
                        // Simple amount (already raw or display)
                        rawAmount = r.amount || 0;
                    }
                    
                    // Calculate display amount
                    displayAmount = divisibility > 0 
                        ? rawAmount / Math.pow(10, divisibility) 
                        : rawAmount;
                    
                    return {
                        ...r,
                        rawAmount: rawAmount,           // Raw amount from blockchain
                        divisibility: divisibility,     // Divisibility from etching
                        amount: displayAmount,          // Display amount for UI
                        name: r.name || 'Unknown',
                        runeId: r.runeId || r.id
                    };
                });
                
                return {
                    ...utxo,
                    runes: processedRunes,
                    runeId: processedRunes[0]?.runeId || '',
                    rawRuneAmount: processedRunes[0]?.rawAmount || 0,
                    divisibility: processedRunes[0]?.divisibility || 0
                };
            });
        
        if (mixedUtxos.length === 0) {
            listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;">No mixed UTXOs found (Inscription + Rune in same UTXO)</div>';
            return;
        }
        
        separateState.utxosList = mixedUtxos;
        renderSeparateUtxos();
        
    } catch (error) {
        console.error('❌ Error loading UTXOs:', error);
        listContainer.innerHTML = '<div style="color: #ef4444; text-align: center; padding: 20px;">Failed to load UTXOs</div>';
    }
}

function renderSeparateUtxos() {
    const listContainer = document.getElementById('separate-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '';
    
    separateState.utxosList.forEach((utxo, index) => {
        const isSelected = separateState.selectedUtxo?.txid === utxo.txid && separateState.selectedUtxo?.vout === utxo.vout;
        const item = document.createElement('div');
        item.style.cssText = `
            display: flex; align-items: center; padding: 12px; margin-bottom: 8px;
            background: ${isSelected ? 'rgba(147, 51, 234, 0.2)' : 'rgba(255,255,255,0.03)'};
            border: 1px solid ${isSelected ? 'rgba(147, 51, 234, 0.5)' : '#333'};
            border-radius: 8px; cursor: pointer; transition: all 0.2s;
        `;
        
        const runeNames = utxo.runes?.map(r => r.name).join(', ') || 'Unknown';
        
        item.innerHTML = `
            <div style="flex: 1;">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                    <span style="color: #9333ea;">◉</span>
                    <span style="color: #f7931a;">⧈</span>
                    <span style="color: #fff; font-weight: 600;">Mixed UTXO</span>
                </div>
                <div style="font-size: 11px; color: #888;">Runes: ${runeNames}</div>
                <div style="font-size: 11px; color: #888;">${utxo.value} sats</div>
            </div>
            <div style="color: ${isSelected ? '#9333ea' : '#666'}; font-size: 20px;">
                ${isSelected ? '✓' : '○'}
            </div>
        `;
        
        item.addEventListener('click', () => {
            separateState.selectedUtxo = isSelected ? null : utxo;
            renderSeparateUtxos();
            updateSeparatePreview();
        });
        
        listContainer.appendChild(item);
    });
}

function updateSeparatePreview() {
    const preview = document.getElementById('separate-preview');
    const executeBtn = document.getElementById('execute-separate-btn');
    
    if (separateState.selectedUtxo) {
        preview.style.display = 'block';
        executeBtn.disabled = false;
    } else {
        preview.style.display = 'none';
        executeBtn.disabled = true;
    }
}

async function executeSeparateAssets() {
    if (!separateState.selectedUtxo) {
        showNotification('⚠️ Please select a mixed UTXO', 'error');
        return;
    }
    
    showSeparateConfirmModal();
}

async function showSeparateConfirmModal() {
    const modal = document.createElement('div');
    modal.id = 'separate-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(147, 51, 234, 0.2);">
                <h3 style="color: #9333ea;">🔀 Confirm Separation</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 40px; margin-bottom: 12px;">🔀</div>
                
                <p style="color: #ccc; margin-bottom: 16px;">
                    This will separate the inscription and runes into individual UTXOs.
                </p>
                
                <div style="background: rgba(255,255,255,0.05); border-radius: 8px; padding: 12px; margin-bottom: 16px; text-align: left;">
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Input:</span>
                        <span style="color: #fff;">${separateState.selectedUtxo.value} sats</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Inscription Output:</span>
                        <span style="color: #9333ea;">555 sats</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #888;">Rune Output:</span>
                        <span style="color: #f7931a;">546 sats</span>
                    </div>
                </div>
                
                <input type="password" id="separate-password" class="input-field" placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-separate-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-separate-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #9333ea 0%, #7c3aed 100%);">
                    🔀 Separate
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.getElementById('separate-password').focus();
    
    document.getElementById('cancel-separate-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-separate-btn').addEventListener('click', confirmSeparateAssets);
    document.getElementById('separate-password').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') confirmSeparateAssets();
    });
}

async function confirmSeparateAssets() {
    const password = document.getElementById('separate-password').value;
    if (!password) {
        showNotification('⚠️ Please enter your password', 'error');
        return;
    }
    
    const confirmBtn = document.getElementById('confirm-separate-btn');
    confirmBtn.disabled = true;
    confirmBtn.textContent = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const feeRate = parseInt(document.getElementById('separate-fee-rate').value) || 2;
        
        // Get pure UTXOs for fees
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const utxosData = await utxosResponse.json();
        const pureUtxos = utxosData.utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        
        // 🔥 Get raw rune amount for PSBT
        const rune = separateState.selectedUtxo.runes?.[0];
        const rawRuneAmount = separateState.selectedUtxo.rawRuneAmount || rune?.rawAmount || 0;
        const runeId = separateState.selectedUtxo.runeId || rune?.runeId || '';
        
        console.log('🔀 Separate Assets - Rune data:');
        console.log('   runeId:', runeId);
        console.log('   rawRuneAmount:', rawRuneAmount);
        
        // Call backend to create PSBT
        const response = await fetch('https://kraywallet-backend.onrender.com/api/psbt/separate-assets', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                mixedUtxo: {
                    txid: separateState.selectedUtxo.txid,
                    vout: separateState.selectedUtxo.vout,
                    value: separateState.selectedUtxo.value,
                    runeId: runeId,                    // 🔥 Rune ID
                    runeAmount: rawRuneAmount          // 🔥 RAW amount!
                },
                paymentUtxos: pureUtxos.slice(0, 3).map(u => ({ txid: u.txid, vout: u.vout, value: u.value })),
                feeRate
            })
        });
        
        const result = await response.json();
        if (!result.success) throw new Error(result.error || 'Failed to create PSBT');
        
        // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
        console.log('🔐 Signing PSBT locally...');
        // ✅ Sign ALL inputs (mixed UTXO + fee UTXOs)
        const inputCount = result.inputCount || 4; // Default to 4 if not provided
        const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
        console.log('   Inputs to sign:', inputsToSign);
        
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01, // SIGHASH_ALL
                inputsToSign: inputsToSign // ✅ Sign ALL inputs
            }
        });
        
        if (!signResult.success) throw new Error(signResult.error || 'Failed to sign');
        
        // Finalize
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeResult = await finalizeResponse.json();
        if (!finalizeResult.success) throw new Error(finalizeResult.error || 'Failed to finalize');
        
        showNotification(`✅ Assets separated! TX: ${finalizeResult.txid.slice(0, 8)}...`, 'success');
        
        document.getElementById('separate-confirm-modal')?.remove();
        showScreen('settings');
        
    } catch (error) {
        console.error('❌ Error:', error);
        showNotification('❌ Failed: ' + error.message, 'error');
        confirmBtn.disabled = false;
        confirmBtn.textContent = '🔀 Separate';
    }
}

// Event Listeners for Separate Assets
document.getElementById('separate-assets-btn')?.addEventListener('click', showSeparateAssetsScreen);
document.getElementById('back-from-separate-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('execute-separate-btn')?.addEventListener('click', executeSeparateAssets);


// ========================================
// 💎 EXTRACT SATS (Recover pure BTC)
// ========================================

let extractSatsState = {
    selectedItems: new Map(), // Map of "txid:vout" -> { utxo, targetSats }
    inscriptionUtxos: [],
    runeUtxos: [],
    pureUtxos: [],
    activeTab: 'inscriptions',
    feeRates: null,
    selectedFeeRate: 5
};

async function showExtractSatsScreen() {
    console.log('💎 Opening Extract Sats screen...');
    showScreen('extract-sats');
    extractSatsState.activeTab = 'inscriptions';
    await Promise.all([loadExtractSatsUtxos(), loadExtractFeeRates()]);
    updateExtractTabStyles();
    setupExtractFeeButtons();
}

async function loadExtractFeeRates() {
    try {
        const res = await fetch('https://mempool.space/api/v1/fees/recommended');
        const data = await res.json();
        extractSatsState.feeRates = data;
        extractSatsState.selectedFeeRate = data.halfHourFee;
        
        // Update UI
        document.getElementById('extract-fee-fast').textContent = data.fastestFee + ' sat/vB';
        document.getElementById('extract-fee-normal').textContent = data.halfHourFee + ' sat/vB';
        document.getElementById('extract-fee-slow').textContent = data.hourFee + ' sat/vB';
        
        console.log('💎 Fee rates loaded:', data);
    } catch (error) {
        console.error('Error loading fee rates:', error);
    }
}

function setupExtractFeeButtons() {
    document.querySelectorAll('.extract-fee-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const speed = btn.dataset.speed;
            const rates = extractSatsState.feeRates;
            if (!rates) return;
            
            // Update selected fee rate
            if (speed === 'fast') extractSatsState.selectedFeeRate = rates.fastestFee;
            else if (speed === 'normal') extractSatsState.selectedFeeRate = rates.halfHourFee;
            else if (speed === 'slow') extractSatsState.selectedFeeRate = rates.hourFee;
            
            // Update UI
            document.querySelectorAll('.extract-fee-btn').forEach(b => {
                b.style.background = '#1a1a1a';
                b.style.border = '1px solid #333';
                b.style.color = '#888';
                b.classList.remove('selected');
            });
            btn.style.background = 'rgba(247,147,26,0.2)';
            btn.style.border = '1px solid #f7931a';
            btn.style.color = '#f7931a';
            btn.classList.add('selected');
            
            // Clear custom input
            document.getElementById('extract-fee-custom').value = '';
        });
    });
    
    // Custom fee input
    document.getElementById('extract-fee-custom')?.addEventListener('input', (e) => {
        const rate = parseInt(e.target.value);
        if (rate > 0) {
            extractSatsState.selectedFeeRate = rate;
            // Deselect preset buttons
            document.querySelectorAll('.extract-fee-btn').forEach(b => {
                b.style.background = '#1a1a1a';
                b.style.border = '1px solid #333';
                b.style.color = '#888';
                b.classList.remove('selected');
            });
        }
    });
}

function updateExtractTabStyles() {
    const inscBtn = document.getElementById('extract-tab-inscriptions');
    const runeBtn = document.getElementById('extract-tab-runes');
    
    if (extractSatsState.activeTab === 'inscriptions') {
        inscBtn.style.background = 'rgba(16, 185, 129, 0.2)';
        inscBtn.style.border = '1px solid #10b981';
        runeBtn.style.background = '';
        runeBtn.style.border = '';
    } else {
        runeBtn.style.background = 'rgba(247, 147, 26, 0.2)';
        runeBtn.style.border = '1px solid #f7931a';
        inscBtn.style.background = '';
        inscBtn.style.border = '';
    }
}

async function loadExtractSatsUtxos() {
    const listContainer = document.getElementById('extract-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;"><div class="spinner"></div> Loading...</div>';
    
    try {
        const address = await getWalletAddress();
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await response.json();
        
        if (!data.success || !data.utxos) {
            listContainer.innerHTML = '<div style="color: #f87171; text-align: center; padding: 20px;">Failed to load UTXOs</div>';
            return;
        }
        
        extractSatsState.inscriptionUtxos = [];
        extractSatsState.runeUtxos = [];
        extractSatsState.pureUtxos = [];
        
        for (const utxo of data.utxos) {
            const hasInscription = utxo.hasInscription || (utxo.inscriptions && utxo.inscriptions.length > 0);
            const hasRunes = utxo.hasRunes || (utxo.runes && utxo.runes.length > 0);
            
            console.log(`💎 UTXO ${utxo.txid.slice(0,8)}:${utxo.vout} - value: ${utxo.value}, hasInscr: ${hasInscription}, hasRunes: ${hasRunes}`);
            
            // Pure BTC
            if (!hasInscription && !hasRunes) {
                extractSatsState.pureUtxos.push(utxo);
                continue;
            }
            
            // Inscription only (no rune) - can extract if >= 330 sats
            if (hasInscription && !hasRunes && utxo.value >= 330) {
                const insc = utxo.inscription || utxo.inscriptions?.[0];
                console.log('   📜 Inscription found:', insc);
                extractSatsState.inscriptionUtxos.push({
                    ...utxo,
                    type: 'inscription',
                    inscription: {
                        number: insc?.inscriptionNumber || insc?.number,
                        id: insc?.inscriptionId || insc?.id,
                        thumbnail: insc?.thumbnail || insc?.preview
                    }
                });
            }
            
            // Rune only (no inscription) - can extract if >= 546 sats
            if (hasRunes && !hasInscription && utxo.value >= 546) {
                const r = utxo.runes?.[0];
                console.log('   🪙 Rune found:', r);
                if (r) {
                    let rawAmount = 0;
                    let divisibility = 0;
                    let symbol = '';
                    
                    if (r.amount && typeof r.amount === 'object') {
                        rawAmount = r.amount.amount || 0;
                        divisibility = r.amount.divisibility || 0;
                        symbol = r.amount.symbol || '';
                    } else {
                        rawAmount = r.amount || 0;
                        symbol = r.symbol || '';
                    }
                    
                    extractSatsState.runeUtxos.push({
                        ...utxo,
                        type: 'rune',
                        rune: {
                            name: r.name || 'Unknown',
                            amount: divisibility > 0 ? rawAmount / Math.pow(10, divisibility) : rawAmount,
                            rawAmount,
                            divisibility,
                            symbol,
                            runeId: r.runeId || ''
                        }
                    });
                }
            }
        }
        
        console.log(`💎 Found: ${extractSatsState.inscriptionUtxos.length} inscriptions, ${extractSatsState.runeUtxos.length} runes, ${extractSatsState.pureUtxos.length} pure`);
        
        renderExtractUtxos();
        
    } catch (error) {
        console.error('Error loading UTXOs:', error);
        listContainer.innerHTML = `<div style="color: #f87171; text-align: center; padding: 20px;">Error: ${error.message}</div>`;
    }
}

function renderExtractUtxos() {
    const listContainer = document.getElementById('extract-utxo-list');
    if (!listContainer) return;
    
    const utxos = extractSatsState.activeTab === 'inscriptions' 
        ? extractSatsState.inscriptionUtxos 
        : extractSatsState.runeUtxos;
    
    if (utxos.length === 0) {
        const icon = extractSatsState.activeTab === 'inscriptions' ? '◉' : '⧈';
        listContainer.innerHTML = `<div style="color: #888; text-align: center; padding: 30px;">
            <span style="font-size: 32px; opacity: 0.5;">${icon}</span>
            <div style="margin-top: 8px;">No ${extractSatsState.activeTab} UTXOs with extractable sats</div>
        </div>`;
        return;
    }
    
    let html = '';
    for (const utxo of utxos) {
        const key = `${utxo.txid}:${utxo.vout}`;
        const isSelected = extractSatsState.selectedItems.has(key);
        const item = extractSatsState.selectedItems.get(key);
        const defaultTarget = utxo.type === 'rune' ? 546 : 330;
        const targetSats = item?.targetSats || defaultTarget;
        
        const cardStyle = isSelected 
            ? 'border: 1px solid #10b981; background: rgba(16, 185, 129, 0.1); border-bottom-left-radius: 0; border-bottom-right-radius: 0; margin-bottom: 0;'
            : 'border: 1px solid #333; margin-bottom: 8px;';
        
        const checkboxStyle = isSelected 
            ? 'width: 18px; height: 18px; border-radius: 4px; background: #10b981; border: none; display: flex; align-items: center; justify-content: center; color: #000; font-size: 12px; font-weight: bold;'
            : 'width: 18px; height: 18px; border-radius: 4px; background: transparent; border: 2px solid #444;';
        
        if (utxo.type === 'inscription') {
            html += `
                <div class="extract-utxo-item" data-key="${key}" data-type="inscription"
                     style="${cardStyle} border-radius: 8px; padding: 12px; cursor: pointer;">
                    <div style="display: flex; align-items: center; gap: 12px;">
                        <div style="${checkboxStyle}">${isSelected ? '✓' : ''}</div>
                        <div style="flex: 1;">
                            <div style="color: #fff; font-weight: 600;">◉ ${utxo.inscription?.number ? `Inscription #${utxo.inscription.number.toLocaleString()}` : `Inscription ${utxo.inscription?.id?.slice(0, 12) || ''}...`}</div>
                            <div style="color: #888; font-size: 12px;">${utxo.value.toLocaleString()} sats</div>
                            <div style="color: #10b981; font-size: 11px;">💰 Max recover: ${(utxo.value - 330).toLocaleString()} sats</div>
                        </div>
                    </div>
                </div>
            `;
        } else {
            html += `
                <div class="extract-utxo-item" data-key="${key}" data-type="rune"
                     style="${cardStyle} border-radius: 8px; padding: 12px; cursor: pointer;">
                    <div style="display: flex; align-items: center; gap: 12px;">
                        <div style="${checkboxStyle}">${isSelected ? '✓' : ''}</div>
                        <div style="flex: 1;">
                            <div style="color: #f7931a; font-weight: 600;">⧈ ${utxo.rune?.name}</div>
                            <div style="color: #888; font-size: 12px;">${utxo.rune?.amount?.toLocaleString()} ${utxo.rune?.symbol || ''}</div>
                            <div style="color: #888; font-size: 12px;">${utxo.value.toLocaleString()} sats</div>
                            <div style="color: #10b981; font-size: 11px;">💰 Max recover: ${(utxo.value - 330).toLocaleString()} sats</div>
                        </div>
                    </div>
                </div>
            `;
        }
        
        // Add individual slider if selected
        if (isSelected) {
            const saved = utxo.value - targetSats;
            html += `
                <div class="extract-item-slider" data-key="${key}"
                     style="background: rgba(16,185,129,0.1); border: 1px solid #10b981; border-top: none; border-bottom-left-radius: 8px; border-bottom-right-radius: 8px; padding: 10px; margin-bottom: 8px;">
                    <div style="display: flex; justify-content: space-between; margin-bottom: 4px;">
                        <span class="extract-keep-label" style="color: #888; font-size: 11px;">Keep: ${targetSats} sats</span>
                        <span class="extract-save-label" style="color: #10b981; font-size: 11px; font-weight: 600;">Save: ${saved.toLocaleString()} sats</span>
                    </div>
                    <input type="range" class="extract-sats-slider" data-key="${key}" 
                           min="1" max="${utxo.value - 1}" value="${targetSats}" 
                           style="width: 100%; accent-color: #f7931a; cursor: pointer;">
                </div>
            `;
        }
    }
    
    listContainer.innerHTML = html;
    
    // Add click listeners for toggling selection
    listContainer.querySelectorAll('.extract-utxo-item').forEach(item => {
        item.addEventListener('click', () => toggleExtractUtxoSelection(item.dataset.key, item.dataset.type));
    });
    
    // Add slider listeners (CSP-compliant - no inline handlers!)
    listContainer.querySelectorAll('.extract-sats-slider').forEach(slider => {
        slider.addEventListener('input', (e) => {
            const key = slider.dataset.key;
            const value = parseInt(e.target.value);
            
            // Update the labels in real-time
            const container = slider.closest('.extract-item-slider');
            if (container) {
                const item = extractSatsState.selectedItems.get(key);
                if (item) {
                    const saved = item.utxo.value - value;
                    container.querySelector('.extract-keep-label').textContent = `Keep: ${value.toLocaleString()} sats`;
                    container.querySelector('.extract-save-label').textContent = `Save: ${saved.toLocaleString()} sats`;
                }
            }
            
            // Update state
            updateExtractItemTargetSats(key, value);
        });
    });
    
    // Update summary
    updateExtractSummary();
}

function toggleExtractUtxoSelection(key, type) {
    const utxos = type === 'inscription' ? extractSatsState.inscriptionUtxos : extractSatsState.runeUtxos;
    const [txid, voutStr] = key.split(':');
    const vout = parseInt(voutStr);
    const utxo = utxos.find(u => u.txid === txid && u.vout === vout);
    
    if (!utxo) return;
    
    if (extractSatsState.selectedItems.has(key)) {
        extractSatsState.selectedItems.delete(key);
    } else {
        // Default to minimum viable: 546 for runes, 330 for inscriptions
        const defaultTarget = utxo.type === 'rune' ? 546 : 330;
        extractSatsState.selectedItems.set(key, { utxo, targetSats: defaultTarget });
    }
    
    renderExtractUtxos();
}

function updateExtractItemTargetSats(key, value) {
    const item = extractSatsState.selectedItems.get(key);
    if (item) {
        item.targetSats = parseInt(value);
        extractSatsState.selectedItems.set(key, item);
        // Only update summary, not re-render (slider updates labels in real-time)
        updateExtractSummary();
    }
}

function updateExtractSummary() {
    const count = extractSatsState.selectedItems.size;
    const previewSection = document.getElementById('extract-preview');
    const targetSection = document.getElementById('extract-target-section');
    const btn = document.getElementById('execute-extract-btn');
    
    if (count === 0) {
        if (previewSection) previewSection.style.display = 'none';
        if (targetSection) targetSection.style.display = 'none';
        if (btn) btn.disabled = true;
        return;
    }
    
    // Calculate totals
    let totalOriginal = 0;
    let totalTarget = 0;
    
    extractSatsState.selectedItems.forEach(item => {
        totalOriginal += item.utxo.value;
        totalTarget += item.targetSats;
    });
    
    const totalRecovered = totalOriginal - totalTarget;
    const feeRate = extractSatsState.selectedFeeRate || 5;
    const estFee = Math.ceil((10 + (count + 1) * 57 + (count + 2) * 43) * feeRate);
    
    // Update UI
    if (targetSection) targetSection.style.display = 'none'; // Hide old slider section
    if (previewSection) {
        previewSection.style.display = 'block';
        document.getElementById('extract-original').textContent = `${totalOriginal.toLocaleString()} sats (${count} items)`;
        document.getElementById('extract-new-value').textContent = `${totalTarget.toLocaleString()} sats`;
        document.getElementById('extract-recovered').textContent = `${totalRecovered.toLocaleString()} sats`;
        
        // Update target value display to show estimated fee
        const targetValueEl = document.getElementById('extract-target-value');
        if (targetValueEl) {
            targetValueEl.textContent = `Est. fee: ~${estFee.toLocaleString()} sats`;
        }
    }
    
    if (btn) {
        btn.disabled = false;
        btn.textContent = `💎 Extract ${totalRecovered.toLocaleString()} sats${count > 1 ? ` (${count} items)` : ''}`;
    }
}

async function executeExtractSats() {
    const count = extractSatsState.selectedItems.size;
    if (count === 0) {
        showNotification('Please select at least one UTXO', 'error');
        return;
    }
    
    const totalPure = extractSatsState.pureUtxos.reduce((sum, u) => sum + u.value, 0);
    const minRequired = 1000 + (count * 300);
    if (totalPure < minRequired) {
        showNotification(`Not enough pure BTC for fees. Need ${minRequired.toLocaleString()} sats`, 'error');
        return;
    }
    
    // Calculate totals for modal
    const itemsArray = Array.from(extractSatsState.selectedItems.values());
    let totalRecovered = 0;
    itemsArray.forEach(item => {
        totalRecovered += item.utxo.value - item.targetSats;
    });
    
    const feeRate = extractSatsState.selectedFeeRate || 5;
    
    // Show confirmation modal with password
    const modal = document.createElement('div');
    modal.className = 'modal-overlay';
    modal.id = 'extract-confirm-modal';
    modal.innerHTML = `
        <div class="modal-content" style="max-width: 400px;">
            <div class="modal-body">
                <h3 style="color: #f7931a; margin-bottom: 16px; text-align: center;">💎 Confirm Extract Sats</h3>
                <div style="background: rgba(0,0,0,0.3); border-radius: 8px; padding: 12px; margin-bottom: 16px;">
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">Items:</span>
                        <span style="color: #fff;">${count} UTXO${count > 1 ? 's' : ''}</span>
                    </div>
                    <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
                        <span style="color: #888;">To recover:</span>
                        <span style="color: #10b981; font-weight: 600;">~${totalRecovered.toLocaleString()} sats</span>
                    </div>
                    <div style="display: flex; justify-content: space-between;">
                        <span style="color: #888;">Fee rate:</span>
                        <span style="color: #fff;">${feeRate} sat/vB</span>
                    </div>
                </div>
                <input type="password" id="extract-confirm-password" class="input-field" 
                       placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-extract-confirm-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-extract-confirm-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #f7931a 0%, #e67e00 100%);">
                    💎 Extract
                </button>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    document.getElementById('extract-confirm-password').focus();
    
    document.getElementById('cancel-extract-confirm-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-extract-confirm-btn').addEventListener('click', confirmExtractSats);
    
    // Allow Enter key to confirm
    document.getElementById('extract-confirm-password').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') confirmExtractSats();
    });
}

// Execute extract after password confirmation
async function confirmExtractSats() {
    const password = document.getElementById('extract-confirm-password').value;
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    const modal = document.getElementById('extract-confirm-modal');
    const confirmBtn = document.getElementById('confirm-extract-confirm-btn');
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner"></span> Building...';
    
    const count = extractSatsState.selectedItems.size;
    const feeRate = extractSatsState.selectedFeeRate || 5;
    
    try {
        const address = await getWalletAddress();
        
        // Get payment UTXOs (need more for multi-extract)
        const sortedPure = [...extractSatsState.pureUtxos].sort((a, b) => a.value - b.value);
        const paymentUtxos = [];
        let paymentValue = 0;
        const minPayment = 1500 + (count * 300);
        
        while (paymentValue < minPayment && sortedPure.length > 0) {
            const pureUtxo = sortedPure.shift();
            paymentUtxos.push({
                txid: pureUtxo.txid,
                vout: pureUtxo.vout,
                value: pureUtxo.value
            });
            paymentValue += pureUtxo.value;
        }
        
        // Convert selectedItems to array
        const itemsArray = Array.from(extractSatsState.selectedItems.values());
        const isMulti = itemsArray.length > 1;
        const firstItem = itemsArray[0];
        
        let requestBody;
        let totalRecovered = 0;
        
        if (isMulti && firstItem.utxo.type === 'inscription') {
            // MULTI-EXTRACT MODE (inscriptions only)
            console.log(`🔀 Multi-extract: ${itemsArray.length} inscriptions`);
            
            itemsArray.forEach(item => {
                totalRecovered += item.utxo.value - item.targetSats;
            });
            
            requestBody = {
                address,
                assetUtxos: itemsArray.map(item => ({
                    txid: item.utxo.txid,
                    vout: item.utxo.vout,
                    value: item.utxo.value,
                    type: item.utxo.type,
                    targetSats: item.targetSats,
                    inscriptionId: item.utxo.inscription?.id,
                })),
                paymentUtxos,
                feeRate
            };
        } else {
            // SINGLE EXTRACT MODE
            const utxo = firstItem.utxo;
            const targetSats = firstItem.targetSats;
            totalRecovered = utxo.value - targetSats;
            
            requestBody = {
                address,
                assetUtxo: {
                    txid: utxo.txid,
                    vout: utxo.vout,
                    value: utxo.value,
                    type: utxo.type
                },
                targetSats,
                paymentUtxos,
                feeRate
            };
            
            // Add type-specific fields
            if (utxo.type === 'inscription') {
                requestBody.assetUtxo.inscriptionId = utxo.inscription?.id;
            } else {
                requestBody.assetUtxo.runeId = utxo.rune?.runeId;
                requestBody.assetUtxo.runeAmount = utxo.rune?.rawAmount;
            }
        }
        
        // 1. Create PSBT
        const res = await fetch('https://kraywallet-backend.onrender.com/api/psbt/extract-sats', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(requestBody)
        });
        
        const result = await res.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to create PSBT');
        }
        
        console.log('✅ PSBT created, fee:', result.fee, 'net profit:', result.summary?.netProfit || (totalRecovered - result.fee));
        console.log('   Inputs in PSBT:', result.inputs || 'unknown');
        confirmBtn.innerHTML = '<span class="spinner"></span> Signing...';
        
        // 2. Sign PSBT via chrome.runtime.sendMessage
        // Use inputCount from backend if available, otherwise sign all inputs
        const signData = {
            psbt: result.psbt_base64,
            password: password,
            sighashType: 0x01
        };
        
        // Only specify inputsToSign if backend tells us the count
        if (result.inputs) {
            signData.inputsToSign = Array.from({ length: result.inputs }, (_, i) => i);
            console.log('   Signing inputs:', signData.inputsToSign);
        } else {
            console.log('   Signing all inputs (no count from backend)');
        }
        
        console.log('📤 Sending sign request to background...');
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: signData
        });
        
        console.log('📥 Sign result received:', signResult?.success ? 'SUCCESS' : 'FAILED', signResult);
        
        if (!signResult || !signResult.success) {
            throw new Error(signResult?.error || 'Signing failed - invalid password');
        }
        
        console.log('✅ PSBT signed, length:', signResult.signedPsbt?.length);
        confirmBtn.innerHTML = '<span class="spinner"></span> Finalizing...';
        
        // 3. Finalize
        console.log('📤 Sending finalize request...');
        const finalizeRes = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeData = await finalizeRes.json();
        console.log('📥 Finalize result:', finalizeData?.success ? 'SUCCESS' : 'FAILED', finalizeData);
        
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize');
        }
        
        confirmBtn.innerHTML = '<span class="spinner"></span> Broadcasting...';
        
        // 4. Broadcast
        const broadcastRes = await fetch('https://kraywallet-backend.onrender.com/api/psbt/extract-sats/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: finalizeData.hex })
        });
        
        const broadcastData = await broadcastRes.json();
        if (!broadcastData.success) {
            throw new Error(broadcastData.error || 'Failed to broadcast');
        }
        
        // Use txid from broadcast response
        const confirmedTxid = broadcastData.txid || finalizeData.txid;
        console.log('✅ Extraction complete!', confirmedTxid, 'via', broadcastData.source || 'broadcast');
        
        // Save for Activity
        try {
            const splits = JSON.parse(localStorage.getItem('splitTransactions') || '[]');
            splits.push({ 
                txid: confirmedTxid, 
                outputs: itemsArray.length + 2, 
                timestamp: Date.now(),
                type: 'extract',
                recovered: totalRecovered,
                count: itemsArray.length
            });
            if (splits.length > 50) splits.shift();
            localStorage.setItem('splitTransactions', JSON.stringify(splits));
        } catch (e) { console.warn('Could not save extract txid:', e); }
        
        // Close confirmation modal
        modal.remove();
        
        // Update main button
        const btn = document.getElementById('execute-extract-btn');
        if (btn) {
            btn.innerHTML = '✅ Success!';
            btn.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
        }
        
        // Show success modal with links
        showExtractSuccessModal(confirmedTxid, totalRecovered);
        
    } catch (error) {
        console.error('❌ Extract error:', error);
        confirmBtn.disabled = false;
        confirmBtn.innerHTML = '💎 Extract';
        showNotification('❌ ' + (error.message || 'Extraction failed'), 'error');
    }
}

// Event listeners for Extract Sats
document.getElementById('extract-sats-btn')?.addEventListener('click', showExtractSatsScreen);
document.getElementById('back-from-extract-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('execute-extract-btn')?.addEventListener('click', executeExtractSats);

document.getElementById('extract-tab-inscriptions')?.addEventListener('click', () => {
    extractSatsState.activeTab = 'inscriptions';
    extractSatsState.selectedItems.clear();
    document.getElementById('extract-target-section').style.display = 'none';
    document.getElementById('extract-preview').style.display = 'none';
    document.getElementById('execute-extract-btn').disabled = true;
    document.getElementById('execute-extract-btn').textContent = '💎 Extract Sats';
    updateExtractTabStyles();
    renderExtractUtxos();
});

document.getElementById('extract-tab-runes')?.addEventListener('click', () => {
    extractSatsState.activeTab = 'runes';
    extractSatsState.selectedItems.clear();
    document.getElementById('extract-target-section').style.display = 'none';
    document.getElementById('extract-preview').style.display = 'none';
    document.getElementById('execute-extract-btn').disabled = true;
    document.getElementById('execute-extract-btn').textContent = '💎 Extract Sats';
    updateExtractTabStyles();
    renderExtractUtxos();
});

function showExtractSuccessModal(txid, recovered) {
    // Create overlay
    const overlay = document.createElement('div');
    overlay.id = 'extract-success-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0,0,0,0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 10000;
    `;
    
    overlay.innerHTML = `
        <div style="background: #1a1a1a; border-radius: 16px; padding: 24px; max-width: 320px; text-align: center;">
            <div style="font-size: 48px; margin-bottom: 12px;">✅</div>
            <h2 style="color: #10b981; margin: 0 0 8px 0; font-size: 20px;">Sats Extracted!</h2>
            <p style="color: #fff; margin: 0 0 16px 0; font-size: 16px;">
                💰 Recovered: <strong>${recovered.toLocaleString()} sats</strong>
            </p>
            <div style="background: #111; border-radius: 8px; padding: 12px; margin-bottom: 16px;">
                <div style="color: #888; font-size: 11px; margin-bottom: 4px;">TRANSACTION ID</div>
                <div style="color: #fff; font-size: 12px; font-family: monospace; word-break: break-all;">
                    ${txid.slice(0, 20)}...${txid.slice(-8)}
                </div>
            </div>
            <div style="display: flex; gap: 8px; margin-bottom: 16px;">
                <button onclick="window.open('https://kray.space/krayscan.html?txid=${txid}', '_blank')" 
                        style="flex: 1; padding: 10px; background: rgba(247,147,26,0.15); border: none; border-radius: 8px; color: #f7931a; font-weight: 600; cursor: pointer;">
                    🔍 KrayScan
                </button>
                <button onclick="window.open('https://mempool.space/tx/${txid}', '_blank')"
                        style="flex: 1; padding: 10px; background: rgba(247,147,26,0.15); border: none; border-radius: 8px; color: #f7931a; font-weight: 600; cursor: pointer;">
                    🌐 Mempool
                </button>
            </div>
            <button onclick="document.getElementById('extract-success-overlay').remove(); showScreen('settings');"
                    style="width: 100%; padding: 12px; background: #f7931a; border: none; border-radius: 8px; color: #000; font-weight: 700; cursor: pointer; font-size: 14px;">
                Done
            </button>
        </div>
    `;
    
    document.body.appendChild(overlay);
}


// ========================================
// 📦 SPLIT RUNES
// ========================================

let splitRunesState = {
    selectedUtxo: null,
    utxosList: []
};

async function showSplitRunesScreen() {
    console.log('📦 Opening Split Runes screen...');
    showScreen('split-runes');
    await loadSplitRunesUtxos();
}

async function loadSplitRunesUtxos() {
    const listContainer = document.getElementById('split-runes-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;"><div class="spinner"></div> Loading...</div>';
    
    try {
        const address = await getWalletAddress();
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await response.json();
        
        if (!data.success || !data.utxos) throw new Error('Failed to load UTXOs');
        
        // Filter for rune UTXOs without inscriptions and process raw amounts
        const runeUtxos = data.utxos
            .filter(u => u.hasRunes && !u.hasInscription)
            .map(utxo => {
                // Process runes to extract rawAmount and divisibility
                const processedRunes = (utxo.runes || []).map(r => {
                    let rawAmount = 0;
                    let divisibility = 0;
                    let displayAmount = 0;
                    
                    if (r.amount && typeof r.amount === 'object') {
                        // Structured amount: {amount, divisibility, symbol}
                        rawAmount = r.amount.amount || 0;
                        divisibility = r.amount.divisibility || 0;
                    } else {
                        // Simple amount (already raw or display)
                        rawAmount = r.amount || 0;
                    }
                    
                    // Calculate display amount
                    displayAmount = divisibility > 0 
                        ? rawAmount / Math.pow(10, divisibility) 
                        : rawAmount;
                    
                    return {
                        ...r,
                        rawAmount: rawAmount,           // Raw amount from blockchain
                        divisibility: divisibility,     // Divisibility from etching
                        amount: displayAmount,          // Display amount for UI
                        name: r.name || 'Unknown',
                        runeId: r.runeId || r.id
                    };
                });
                
                return {
                    ...utxo,
                    runes: processedRunes,
                    rawRuneAmount: processedRunes[0]?.rawAmount || 0,
                    divisibility: processedRunes[0]?.divisibility || 0
                };
            });
        
        if (runeUtxos.length === 0) {
            listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;">No pure Rune UTXOs found</div>';
            return;
        }
        
        splitRunesState.utxosList = runeUtxos;
        renderSplitRunesUtxos();
        
    } catch (error) {
        console.error('❌ Error:', error);
        listContainer.innerHTML = '<div style="color: #ef4444; text-align: center; padding: 20px;">Failed to load</div>';
    }
}

function renderSplitRunesUtxos() {
    const listContainer = document.getElementById('split-runes-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '';
    
    splitRunesState.utxosList.forEach((utxo) => {
        const isSelected = splitRunesState.selectedUtxo?.txid === utxo.txid;
        const item = document.createElement('div');
        item.style.cssText = `
            display: flex; align-items: center; padding: 12px; margin-bottom: 8px;
            background: ${isSelected ? 'rgba(247, 147, 26, 0.2)' : 'rgba(255,255,255,0.03)'};
            border: 1px solid ${isSelected ? 'rgba(247, 147, 26, 0.5)' : '#333'};
            border-radius: 8px; cursor: pointer;
        `;
        
        const rune = utxo.runes?.[0];
        item.innerHTML = `
            <div style="flex: 1;">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                    <span style="color: #f7931a;">⧈</span>
                    <span style="color: #fff; font-weight: 600;">${rune?.name || 'Unknown'}</span>
                </div>
                <div style="font-size: 12px; color: #f7931a;">${rune?.amount?.toLocaleString() || 0} runes</div>
            </div>
            <div style="color: ${isSelected ? '#f7931a' : '#666'}; font-size: 20px;">
                ${isSelected ? '✓' : '○'}
            </div>
        `;
        
        item.addEventListener('click', () => {
            splitRunesState.selectedUtxo = isSelected ? null : utxo;
            renderSplitRunesUtxos();
            updateSplitRunesSummary();
        });
        
        listContainer.appendChild(item);
    });
}

function updateSplitRunesSummary() {
    const summary = document.getElementById('split-runes-summary');
    const executeBtn = document.getElementById('execute-split-runes-btn');
    const amount = parseInt(document.getElementById('split-runes-amount')?.value) || 0;
    const count = parseInt(document.getElementById('split-runes-count')?.value) || 0;
    
    if (splitRunesState.selectedUtxo && amount > 0 && count >= 2) {
        summary.style.display = 'block';
        document.getElementById('split-runes-total').textContent = (amount * count).toLocaleString();
        document.getElementById('split-runes-outputs').textContent = `${count} x 546 sats`;
        document.getElementById('split-runes-fee').textContent = `~${count * 50 + 200} sats`;
        executeBtn.disabled = false;
    } else {
        summary.style.display = 'none';
        executeBtn.disabled = true;
    }
}

async function executeSplitRunes() {
    if (!splitRunesState.selectedUtxo) {
        showNotification('⚠️ Select a rune UTXO', 'error');
        return;
    }
    
    showSplitRunesConfirmModal();
}

async function showSplitRunesConfirmModal() {
    const amount = parseInt(document.getElementById('split-runes-amount').value) || 0;
    const count = parseInt(document.getElementById('split-runes-count').value) || 0;
    const rune = splitRunesState.selectedUtxo.runes?.[0];
    
    const modal = document.createElement('div');
    modal.id = 'split-runes-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(247, 147, 26, 0.2);">
                <h3 style="color: #f7931a;">📦 Confirm Split</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 40px; margin-bottom: 12px;">📦</div>
                
                <p style="color: #ccc; margin-bottom: 16px;">
                    Split <strong style="color: #f7931a;">${rune?.name}</strong> into ${count} packages of ${amount.toLocaleString()} each
                </p>
                
                <input type="password" id="split-runes-password" class="input-field" placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-split-runes-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-split-runes-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #f7931a 0%, #d97706 100%);">
                    📦 Split
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.getElementById('split-runes-password').focus();
    
    document.getElementById('cancel-split-runes-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-split-runes-btn').addEventListener('click', confirmSplitRunes);
}

async function confirmSplitRunes() {
    const password = document.getElementById('split-runes-password').value;
    if (!password) {
        showNotification('⚠️ Enter password', 'error');
        return;
    }
    
    const confirmBtn = document.getElementById('confirm-split-runes-btn');
    confirmBtn.disabled = true;
    confirmBtn.textContent = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const displayAmount = parseFloat(document.getElementById('split-runes-amount').value) || 0;
        const count = parseInt(document.getElementById('split-runes-count').value) || 0;
        const rune = splitRunesState.selectedUtxo.runes?.[0];
        
        // 🔥 Get divisibility and convert display → raw amounts
        const divisibility = splitRunesState.selectedUtxo.divisibility || rune?.divisibility || 0;
        const rawAmountPerSplit = Math.floor(displayAmount * Math.pow(10, divisibility));
        const rawRuneAmount = splitRunesState.selectedUtxo.rawRuneAmount || rune?.rawAmount || 0;
        
        console.log('📦 Split Runes - Amount conversion:');
        console.log('   Display amount:', displayAmount);
        console.log('   Divisibility:', divisibility);
        console.log('   Raw amount per split:', rawAmountPerSplit);
        console.log('   Total raw rune amount:', rawRuneAmount);
        
        // Get pure UTXOs for fees
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const utxosData = await utxosResponse.json();
        const pureUtxos = utxosData.utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/psbt/split-runes', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                runeUtxo: {
                    txid: splitRunesState.selectedUtxo.txid,
                    vout: splitRunesState.selectedUtxo.vout,
                    value: splitRunesState.selectedUtxo.value,
                    runeId: rune?.runeId || rune?.id,
                    runeAmount: rawRuneAmount  // 🔥 RAW amount!
                },
                packages: Array(count).fill({ amount: rawAmountPerSplit }),  // 🔥 RAW amounts!
                paymentUtxos: pureUtxos.slice(0, 5).map(u => ({ txid: u.txid, vout: u.vout, value: u.value })),
                feeRate: 2
            })
        });
        
        const result = await response.json();
        if (!result.success) throw new Error(result.error || 'Failed');
        
        // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
        console.log('🔐 Signing PSBT locally...');
        // ✅ Sign ALL inputs
        const inputCount = result.inputCount || 6; // Default to 6 if not provided
        const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
        console.log('   Inputs to sign:', inputsToSign);
        
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01, // SIGHASH_ALL
                inputsToSign: inputsToSign // ✅ Sign ALL inputs
            }
        });
        
        if (!signResult.success) throw new Error(signResult.error || 'Sign failed');
        
        // Finalize
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeResult = await finalizeResponse.json();
        if (!finalizeResult.success) throw new Error(finalizeResult.error || 'Finalize failed');
        
        showNotification(`✅ Runes split! TX: ${finalizeResult.txid.slice(0, 8)}...`, 'success');
        
        document.getElementById('split-runes-confirm-modal')?.remove();
        showScreen('settings');
        
    } catch (error) {
        console.error('❌ Error:', error);
        showNotification('❌ Failed: ' + error.message, 'error');
        confirmBtn.disabled = false;
        confirmBtn.textContent = '📦 Split';
    }
}

// Event Listeners for Split Runes
document.getElementById('split-runes-btn')?.addEventListener('click', showSplitRunesScreen);
document.getElementById('back-from-split-runes-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('execute-split-runes-btn')?.addEventListener('click', executeSplitRunes);
document.getElementById('split-runes-amount')?.addEventListener('input', updateSplitRunesSummary);
document.getElementById('split-runes-count')?.addEventListener('input', updateSplitRunesSummary);


// ========================================
// 🚀 BULK RUNE SENDER (Airdrop)
// ========================================

let bulkRunesState = {
    selectedUtxo: null,
    utxosList: []
};

async function showBulkRuneSenderScreen() {
    console.log('🚀 Opening Bulk Rune Sender screen...');
    showScreen('bulk-rune-sender');
    await loadBulkRunesUtxos();
    updateBulkRunesCount();
}

async function loadBulkRunesUtxos() {
    const listContainer = document.getElementById('bulk-runes-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;"><div class="spinner"></div> Loading...</div>';
    
    try {
        const address = await getWalletAddress();
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await response.json();
        
        if (!data.success || !data.utxos) throw new Error('Failed to load');
        
        // Filter for rune UTXOs without inscriptions and process raw amounts
        const runeUtxos = data.utxos
            .filter(u => u.hasRunes && !u.hasInscription)
            .map(utxo => {
                // Process runes to extract rawAmount and divisibility
                const processedRunes = (utxo.runes || []).map(r => {
                    let rawAmount = 0;
                    let divisibility = 0;
                    let displayAmount = 0;
                    
                    if (r.amount && typeof r.amount === 'object') {
                        // Structured amount: {amount, divisibility, symbol}
                        rawAmount = r.amount.amount || 0;
                        divisibility = r.amount.divisibility || 0;
                    } else {
                        // Simple amount (already raw or display)
                        rawAmount = r.amount || 0;
                    }
                    
                    // Calculate display amount
                    displayAmount = divisibility > 0 
                        ? rawAmount / Math.pow(10, divisibility) 
                        : rawAmount;
                    
                    return {
                        ...r,
                        rawAmount: rawAmount,           // Raw amount from blockchain
                        divisibility: divisibility,     // Divisibility from etching
                        amount: displayAmount,          // Display amount for UI
                        name: r.name || 'Unknown',
                        runeId: r.runeId || r.id
                    };
                });
                
                return {
                    ...utxo,
                    runes: processedRunes,
                    rawRuneAmount: processedRunes[0]?.rawAmount || 0,
                    divisibility: processedRunes[0]?.divisibility || 0
                };
            });
        
        if (runeUtxos.length === 0) {
            listContainer.innerHTML = '<div style="color: #888; text-align: center; padding: 20px;">No Rune UTXOs available</div>';
            return;
        }
        
        bulkRunesState.utxosList = runeUtxos;
        renderBulkRunesUtxos();
        
    } catch (error) {
        console.error('❌ Error:', error);
        listContainer.innerHTML = '<div style="color: #ef4444; text-align: center; padding: 20px;">Failed to load</div>';
    }
}

function renderBulkRunesUtxos() {
    const listContainer = document.getElementById('bulk-runes-utxo-list');
    if (!listContainer) return;
    
    listContainer.innerHTML = '';
    
    bulkRunesState.utxosList.forEach((utxo) => {
        const isSelected = bulkRunesState.selectedUtxo?.txid === utxo.txid;
        const item = document.createElement('div');
        item.style.cssText = `
            display: flex; align-items: center; padding: 12px; margin-bottom: 8px;
            background: ${isSelected ? 'rgba(16, 185, 129, 0.2)' : 'rgba(255,255,255,0.03)'};
            border: 1px solid ${isSelected ? 'rgba(16, 185, 129, 0.5)' : '#333'};
            border-radius: 8px; cursor: pointer;
        `;
        
        const rune = utxo.runes?.[0];
        item.innerHTML = `
            <div style="flex: 1;">
                <div style="display: flex; align-items: center; gap: 8px; margin-bottom: 4px;">
                    <span style="color: #f7931a;">⧈</span>
                    <span style="color: #fff; font-weight: 600;">${rune?.name || 'Unknown'}</span>
                </div>
                <div style="font-size: 12px; color: #10b981;">${rune?.amount?.toLocaleString() || 0} available</div>
            </div>
            <div style="color: ${isSelected ? '#10b981' : '#666'}; font-size: 20px;">
                ${isSelected ? '✓' : '○'}
            </div>
        `;
        
        item.addEventListener('click', () => {
            bulkRunesState.selectedUtxo = isSelected ? null : utxo;
            renderBulkRunesUtxos();
            updateBulkRunesSummary();
        });
        
        listContainer.appendChild(item);
    });
}

function updateBulkRunesCount() {
    const textarea = document.getElementById('bulk-runes-recipients');
    const countEl = document.getElementById('bulk-runes-count');
    
    if (textarea && countEl) {
        const lines = textarea.value.split('\n').filter(l => l.trim().startsWith('bc1'));
        countEl.textContent = lines.length;
        updateBulkRunesSummary();
    }
}

function updateBulkRunesSummary() {
    const summary = document.getElementById('bulk-runes-summary');
    const executeBtn = document.getElementById('execute-bulk-runes-btn');
    
    const amount = parseInt(document.getElementById('bulk-runes-amount')?.value) || 0;
    const lines = (document.getElementById('bulk-runes-recipients')?.value || '').split('\n').filter(l => l.trim().startsWith('bc1'));
    const recipientCount = lines.length;
    
    if (bulkRunesState.selectedUtxo && amount > 0 && recipientCount > 0) {
        summary.style.display = 'block';
        document.getElementById('bulk-summary-recipients').textContent = recipientCount;
        document.getElementById('bulk-summary-total-runes').textContent = (amount * recipientCount).toLocaleString();
        document.getElementById('bulk-summary-sats').textContent = `${(recipientCount * 546).toLocaleString()} sats`;
        document.getElementById('bulk-summary-fee').textContent = `~${recipientCount * 30 + 500} sats`;
        executeBtn.disabled = recipientCount > 500;
    } else {
        summary.style.display = 'none';
        executeBtn.disabled = true;
    }
}

async function executeBulkRunes() {
    if (!bulkRunesState.selectedUtxo) {
        showNotification('⚠️ Select a rune UTXO', 'error');
        return;
    }
    
    showBulkRunesConfirmModal();
}

async function showBulkRunesConfirmModal() {
    const amount = parseInt(document.getElementById('bulk-runes-amount').value) || 0;
    const lines = (document.getElementById('bulk-runes-recipients').value || '').split('\n').filter(l => l.trim().startsWith('bc1'));
    const rune = bulkRunesState.selectedUtxo.runes?.[0];
    
    const modal = document.createElement('div');
    modal.id = 'bulk-runes-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(16, 185, 129, 0.2);">
                <h3 style="color: #10b981;">🚀 Confirm Bulk Send</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 40px; margin-bottom: 12px;">🚀</div>
                
                <p style="color: #ccc; margin-bottom: 16px;">
                    Send <strong style="color: #f7931a;">${amount.toLocaleString()} ${rune?.name}</strong> to <strong style="color: #10b981;">${lines.length}</strong> recipients
                </p>
                
                <div style="background: rgba(245, 158, 11, 0.1); border: 1px solid rgba(245, 158, 11, 0.3); border-radius: 8px; padding: 12px; margin-bottom: 16px;">
                    <span style="color: #f59e0b; font-size: 12px;">
                        ⚠️ Total: ${(amount * lines.length).toLocaleString()} runes + ~${(lines.length * 546 + lines.length * 30 + 500).toLocaleString()} sats
                    </span>
                </div>
                
                <input type="password" id="bulk-runes-password" class="input-field" placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-bulk-runes-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-bulk-runes-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                    🚀 Send All
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.getElementById('bulk-runes-password').focus();
    
    document.getElementById('cancel-bulk-runes-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-bulk-runes-btn').addEventListener('click', confirmBulkRunes);
}

async function confirmBulkRunes() {
    const password = document.getElementById('bulk-runes-password').value;
    if (!password) {
        showNotification('⚠️ Enter password', 'error');
        return;
    }
    
    const confirmBtn = document.getElementById('confirm-bulk-runes-btn');
    confirmBtn.disabled = true;
    confirmBtn.textContent = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const displayAmount = parseFloat(document.getElementById('bulk-runes-amount').value) || 0;
        const lines = (document.getElementById('bulk-runes-recipients').value || '').split('\n').filter(l => l.trim().startsWith('bc1'));
        const rune = bulkRunesState.selectedUtxo.runes?.[0];
        
        // 🔥 Get divisibility and convert display → raw amounts
        const divisibility = bulkRunesState.selectedUtxo.divisibility || rune?.divisibility || 0;
        const rawAmountPerRecipient = Math.floor(displayAmount * Math.pow(10, divisibility));
        const rawRuneAmount = bulkRunesState.selectedUtxo.rawRuneAmount || rune?.rawAmount || 0;
        
        console.log('🚀 Bulk Runes - Amount conversion:');
        console.log('   Display amount per recipient:', displayAmount);
        console.log('   Divisibility:', divisibility);
        console.log('   Raw amount per recipient:', rawAmountPerRecipient);
        console.log('   Total raw rune amount:', rawRuneAmount);
        
        // 🔥 Recipients with RAW amounts
        const recipients = lines.map(addr => ({
            address: addr.trim(),
            amount: rawAmountPerRecipient  // 🔥 RAW amount!
        }));
        
        // Get pure UTXOs for fees
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const utxosData = await utxosResponse.json();
        const pureUtxos = utxosData.utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/psbt/bulk-runes', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                runeUtxo: {
                    txid: bulkRunesState.selectedUtxo.txid,
                    vout: bulkRunesState.selectedUtxo.vout,
                    value: bulkRunesState.selectedUtxo.value,
                    runeId: rune?.runeId || rune?.id,
                    runeAmount: rawRuneAmount  // 🔥 RAW amount!
                },
                recipients,  // 🔥 Already contains raw amounts
                paymentUtxos: pureUtxos.map(u => ({ txid: u.txid, vout: u.vout, value: u.value })),
                feeRate: 2
            })
        });
        
        const result = await response.json();
        if (!result.success) throw new Error(result.error || 'Failed');
        
        // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
        console.log('🔐 Signing bulk runes PSBT locally...');
        // ✅ Sign ALL inputs
        const inputCount = result.inputCount || 6; // Default to 6 if not provided
        const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
        console.log('   Inputs to sign:', inputsToSign);
        
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01, // SIGHASH_ALL
                inputsToSign: inputsToSign // ✅ Sign ALL inputs
            }
        });
        
        if (!signResult.success) throw new Error(signResult.error || 'Sign failed');
        
        // Finalize
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeResult = await finalizeResponse.json();
        if (!finalizeResult.success) throw new Error(finalizeResult.error || 'Finalize failed');
        
        // Remove modal before showing success
        document.getElementById('bulk-runes-confirm-modal')?.remove();
        
        // 🎉 Show success modal with mempool.space and KrayScan links
        showSuccessModal(
            finalizeResult.txid,
            'Bulk Runes Sent!',
            `Successfully sent to ${recipients.length} recipients.`
        );
        
    } catch (error) {
        console.error('❌ Error:', error);
        showNotification('❌ Failed: ' + error.message, 'error');
        confirmBtn.disabled = false;
        confirmBtn.textContent = '🚀 Send All';
    }
}

// Event Listeners for Bulk Rune Sender
document.getElementById('bulk-rune-sender-btn')?.addEventListener('click', showBulkRuneSenderScreen);
document.getElementById('back-from-bulk-runes-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('execute-bulk-runes-btn')?.addEventListener('click', executeBulkRunes);
document.getElementById('bulk-runes-amount')?.addEventListener('input', updateBulkRunesSummary);
document.getElementById('bulk-runes-recipients')?.addEventListener('input', updateBulkRunesCount);

console.log('✅ Advanced Tools (Separate Assets, Split Runes, Bulk Sender) loaded!');


// ========================================
// 📦 BULK TRANSFER (Multi-asset transfer)
// ========================================

let bulkTransferState = {
    activeTab: 'inscriptions',
    inscriptions: [],
    runes: [],
    pureUtxos: [],
    selectedItems: new Map(), // key: "txid:vout", value: { type, utxo, recipient }
    feeRates: null,
    selectedFeeRate: 5,
    useSameAddress: false
};

async function showBulkTransferScreen() {
    console.log('📦 Opening Bulk Transfer screen...');
    showScreen('bulk-transfer');
    bulkTransferState.activeTab = 'inscriptions';
    bulkTransferState.selectedItems.clear();
    await Promise.all([loadBulkTransferAssets(), loadBulkTransferFeeRates()]);
    updateBulkTransferTabStyles();
    updateBulkTransferSummary();
}

async function loadBulkTransferAssets() {
    try {
        const address = await getWalletAddress();
        console.log('📦 Loading assets for:', address?.slice(0, 10) + '...');
        
        // Use same enriched endpoint as Extract Sats
        const res = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const data = await res.json();
        
        if (!data.success || !data.utxos) {
            throw new Error('Failed to load UTXOs');
        }
        
        bulkTransferState.inscriptions = [];
        bulkTransferState.runes = [];
        bulkTransferState.pureUtxos = [];
        
        for (const utxo of data.utxos) {
            const hasInscription = utxo.hasInscription || (utxo.inscriptions && utxo.inscriptions.length > 0);
            const hasRunes = utxo.hasRunes || (utxo.runes && utxo.runes.length > 0);
            
            console.log(`📦 UTXO ${utxo.txid.slice(0,8)}:${utxo.vout} - value: ${utxo.value}, hasInscr: ${hasInscription}, hasRunes: ${hasRunes}`);
            
            // Pure BTC (no inscriptions, no runes)
            if (!hasInscription && !hasRunes && utxo.value >= 546) {
                bulkTransferState.pureUtxos.push({
                    type: 'btc',
                    txid: utxo.txid,
                    vout: utxo.vout,
                    value: utxo.value
                });
                continue;
            }
            
            // Inscription only (value >= 330 sats)
            if (hasInscription && !hasRunes && utxo.value >= 330) {
                const insc = utxo.inscription || utxo.inscriptions?.[0];
                console.log('   📜 Inscription found:', insc);
                bulkTransferState.inscriptions.push({
                    type: 'inscription',
                    txid: utxo.txid,
                    vout: utxo.vout,
                    value: utxo.value,
                    inscriptionId: insc?.inscriptionId || insc?.id,
                    number: insc?.inscriptionNumber || insc?.number,
                    contentType: insc?.contentType || insc?.content_type
                });
            }
            
            // Rune only (value >= 546 sats)
            if (hasRunes && !hasInscription && utxo.value >= 546) {
                const r = utxo.runes?.[0];
                console.log('   🪙 Rune found:', r);
                if (r) {
                    // runeId comes from utxoFilter.getRunesData()
                    bulkTransferState.runes.push({
                        type: 'rune',
                        txid: utxo.txid,
                        vout: utxo.vout,
                        value: utxo.value,
                        runeId: r.runeId || '',
                        runeName: r.name || 'Unknown',
                        runeAmount: r.amount || 0,
                        symbol: r.symbol || '⧈'
                    });
                }
            }
        }
        
        console.log('📦 Loaded:', bulkTransferState.inscriptions.length, 'inscriptions,', 
                    bulkTransferState.runes.length, 'runes,',
                    bulkTransferState.pureUtxos.length, 'pure UTXOs');
        
        renderBulkTransferAssets();
        
    } catch (error) {
        console.error('Error loading bulk transfer assets:', error);
        document.getElementById('bulk-transfer-asset-list').innerHTML = 
            '<div style="color: #f87171; text-align: center; padding: 20px;">Error loading assets</div>';
    }
}

async function loadBulkTransferFeeRates() {
    try {
        const res = await fetch('https://mempool.space/api/v1/fees/recommended');
        bulkTransferState.feeRates = await res.json();
        bulkTransferState.selectedFeeRate = bulkTransferState.feeRates.halfHourFee;
        
        document.getElementById('bulk-transfer-fee-fast').textContent = bulkTransferState.feeRates.fastestFee + ' sat/vB';
        document.getElementById('bulk-transfer-fee-normal').textContent = bulkTransferState.feeRates.halfHourFee + ' sat/vB';
        document.getElementById('bulk-transfer-fee-slow').textContent = bulkTransferState.feeRates.hourFee + ' sat/vB';
    } catch (error) {
        console.error('Error loading fee rates:', error);
    }
}

function renderBulkTransferAssets() {
    const listContainer = document.getElementById('bulk-transfer-asset-list');
    let assets = [];
    
    if (bulkTransferState.activeTab === 'inscriptions') {
        assets = bulkTransferState.inscriptions;
    } else if (bulkTransferState.activeTab === 'runes') {
        assets = bulkTransferState.runes;
    } else {
        assets = bulkTransferState.pureUtxos;
    }
    
    if (assets.length === 0) {
        listContainer.innerHTML = `<div style="color: #888; text-align: center; padding: 20px;">No ${bulkTransferState.activeTab} found</div>`;
        return;
    }
    
    const useSameAddress = bulkTransferState.useSameAddress;
    
    let html = '';
    for (const asset of assets) {
        const key = `${asset.txid}:${asset.vout}`;
        const isSelected = bulkTransferState.selectedItems.has(key);
        const selectedItem = bulkTransferState.selectedItems.get(key);
        
        let displayName = '';
        let displaySats = asset.value.toLocaleString() + ' sats';
        
        if (asset.type === 'inscription') {
            displayName = asset.number ? `Inscription #${asset.number.toLocaleString()}` : `${asset.inscriptionId?.slice(0, 12)}...`;
        } else if (asset.type === 'rune') {
            displayName = `${asset.runeName} (${asset.runeAmount?.toLocaleString() || '?'} ${asset.symbol || ''})`;
        } else {
            displayName = 'Pure BTC';
        }
        
        html += `
            <div class="bulk-transfer-item" data-key="${key}" style="display: flex; align-items: flex-start; gap: 12px; padding: 12px; margin-bottom: 8px; background: ${isSelected ? 'rgba(139,92,246,0.15)' : '#0d0d0d'}; border: 1px solid ${isSelected ? '#8b5cf6' : '#333'}; border-radius: 8px; cursor: pointer;">
                <input type="checkbox" ${isSelected ? 'checked' : ''} style="width: 20px; height: 20px; margin-top: 2px; cursor: pointer;">
                <div style="flex: 1;">
                    <div style="color: ${asset.type === 'rune' ? '#f7931a' : asset.type === 'btc' ? '#10b981' : '#fff'}; font-weight: 600; font-size: 13px;">
                        ${asset.type === 'inscription' ? '◉' : asset.type === 'rune' ? '⧈' : '₿'} ${displayName}
                    </div>
                    <div style="color: #888; font-size: 11px; margin-top: 4px;">${displaySats}</div>
                    ${!useSameAddress && isSelected ? `
                        <input type="text" class="bulk-transfer-recipient-input" data-key="${key}" 
                            placeholder="bc1p... recipient" 
                            value="${selectedItem?.recipient || ''}"
                            onclick="event.stopPropagation();"
                            style="width: 100%; margin-top: 8px; padding: 8px; background: #1a1a1a; border: 1px solid #444; border-radius: 6px; color: #fff; font-size: 12px; box-sizing: border-box;">
                    ` : ''}
                </div>
            </div>
        `;
    }
    
    listContainer.innerHTML = html;
    
    // Add event listeners
    listContainer.querySelectorAll('.bulk-transfer-item').forEach(item => {
        item.addEventListener('click', (e) => {
            if (e.target.tagName === 'INPUT' && e.target.type === 'text') return;
            const key = item.dataset.key;
            toggleBulkTransferItem(key);
        });
    });
    
    listContainer.querySelectorAll('.bulk-transfer-recipient-input').forEach(input => {
        input.addEventListener('input', (e) => {
            const key = e.target.dataset.key;
            updateBulkTransferRecipient(key, e.target.value);
        });
    });
}

function toggleBulkTransferItem(key) {
    const [txid, voutStr] = key.split(':');
    const vout = parseInt(voutStr);
    
    // Find the asset
    let asset = bulkTransferState.inscriptions.find(a => a.txid === txid && a.vout === vout) ||
                bulkTransferState.runes.find(a => a.txid === txid && a.vout === vout) ||
                bulkTransferState.pureUtxos.find(a => a.txid === txid && a.vout === vout);
    
    if (!asset) return;
    
    if (bulkTransferState.selectedItems.has(key)) {
        bulkTransferState.selectedItems.delete(key);
    } else {
        bulkTransferState.selectedItems.set(key, {
            type: asset.type,
            utxo: asset,
            recipient: ''
        });
    }
    
    renderBulkTransferAssets();
    updateBulkTransferSummary();
}

function updateBulkTransferRecipient(key, recipient) {
    const item = bulkTransferState.selectedItems.get(key);
    if (item) {
        item.recipient = recipient;
        bulkTransferState.selectedItems.set(key, item);
        updateBulkTransferSummary();
    }
}

function updateBulkTransferSummary() {
    const count = bulkTransferState.selectedItems.size;
    document.getElementById('bulk-transfer-selected-count').textContent = `(${count} selected)`;
    
    const summaryDiv = document.getElementById('bulk-transfer-summary');
    const contentDiv = document.getElementById('bulk-transfer-summary-content');
    const executeBtn = document.getElementById('execute-bulk-transfer-btn');
    
    if (count === 0) {
        summaryDiv.style.display = 'none';
        executeBtn.disabled = true;
        executeBtn.textContent = '📦 Transfer Selected Assets';
        return;
    }
    
    summaryDiv.style.display = 'block';
    
    let inscCount = 0, runeCount = 0, btcCount = 0;
    let totalValue = 0;
    let allHaveRecipients = true;
    const globalRecipient = document.getElementById('bulk-transfer-recipient-all')?.value || '';
    
    bulkTransferState.selectedItems.forEach(item => {
        if (item.type === 'inscription') inscCount++;
        else if (item.type === 'rune') runeCount++;
        else btcCount++;
        totalValue += item.utxo.value;
        
        const recipient = bulkTransferState.useSameAddress ? globalRecipient : item.recipient;
        if (!recipient || !recipient.startsWith('bc1')) {
            allHaveRecipients = false;
        }
    });
    
    contentDiv.innerHTML = `
        <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
            <span style="color: #888;">Inscriptions:</span>
            <span style="color: #fff;">${inscCount}</span>
        </div>
        <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
            <span style="color: #888;">Runes:</span>
            <span style="color: #fff;">${runeCount}</span>
        </div>
        <div style="display: flex; justify-content: space-between; margin-bottom: 8px;">
            <span style="color: #888;">Pure BTC:</span>
            <span style="color: #fff;">${btcCount}</span>
        </div>
        <div style="height: 1px; background: #333; margin: 12px 0;"></div>
        <div style="display: flex; justify-content: space-between;">
            <span style="color: #8b5cf6; font-weight: 600;">Total Value:</span>
            <span style="color: #8b5cf6; font-weight: 700;">${totalValue.toLocaleString()} sats</span>
        </div>
    `;
    
    executeBtn.disabled = !allHaveRecipients;
    executeBtn.textContent = allHaveRecipients 
        ? `📦 Transfer ${count} Asset${count > 1 ? 's' : ''}`
        : '📦 Enter all recipient addresses';
}

function updateBulkTransferTabStyles() {
    const tabs = ['inscriptions', 'runes', 'btc'];
    tabs.forEach(tab => {
        const btn = document.getElementById(`bulk-transfer-tab-${tab}`);
        if (btn) {
            if (tab === bulkTransferState.activeTab) {
                btn.style.background = 'rgba(139, 92, 246, 0.2)';
                btn.style.borderColor = '#8b5cf6';
                btn.style.color = '#8b5cf6';
            } else {
                btn.style.background = '#1a1a1a';
                btn.style.borderColor = '#333';
                btn.style.color = '#888';
            }
        }
    });
}

async function executeBulkTransfer() {
    const count = bulkTransferState.selectedItems.size;
    if (count === 0) {
        showNotification('Please select at least one asset', 'error');
        return;
    }
    
    // Show password modal
    const modal = document.createElement('div');
    modal.id = 'bulk-transfer-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 400px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(139, 92, 246, 0.3);">
                <h3 style="color: #8b5cf6;">📦 Confirm Bulk Transfer</h3>
            </div>
            <div class="modal-body" style="padding: 20px;">
                <p style="color: #ccc; font-size: 14px; text-align: center; margin-bottom: 16px;">
                    You are about to transfer <strong style="color: #8b5cf6;">${count} asset${count > 1 ? 's' : ''}</strong> in a single transaction.
                </p>
                <input type="password" id="bulk-transfer-password" class="input-field" placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-bulk-transfer-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-bulk-transfer-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                    📦 Transfer All
                </button>
            </div>
        </div>
    `;
    document.body.appendChild(modal);
    document.getElementById('bulk-transfer-password').focus();
    
    document.getElementById('cancel-bulk-transfer-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-bulk-transfer-btn').addEventListener('click', confirmBulkTransfer);
    
    document.getElementById('bulk-transfer-password').addEventListener('keypress', (e) => {
        if (e.key === 'Enter') confirmBulkTransfer();
    });
}

async function confirmBulkTransfer() {
    const password = document.getElementById('bulk-transfer-password').value;
    if (!password) {
        showNotification('Please enter your password', 'error');
        return;
    }
    
    const modal = document.getElementById('bulk-transfer-confirm-modal');
    const confirmBtn = document.getElementById('confirm-bulk-transfer-btn');
    confirmBtn.disabled = true;
    confirmBtn.innerHTML = '<span class="spinner"></span> Building...';
    
    try {
        const address = await getWalletAddress();
        const globalRecipient = document.getElementById('bulk-transfer-recipient-all')?.value || '';
        const feeRate = bulkTransferState.selectedFeeRate;
        
        // Build transfers array
        const transfers = [];
        bulkTransferState.selectedItems.forEach((item, key) => {
            const recipient = bulkTransferState.useSameAddress ? globalRecipient : item.recipient;
            
            const transfer = {
                type: item.type,
                txid: item.utxo.txid,
                vout: item.utxo.vout,
                value: item.utxo.value,
                recipient
            };
            
            if (item.type === 'inscription') {
                transfer.inscriptionId = item.utxo.inscriptionId;
            } else if (item.type === 'rune') {
                transfer.runeId = item.utxo.runeId;
                transfer.runeAmount = item.utxo.runeAmount;
            } else if (item.type === 'btc') {
                transfer.amount = item.utxo.value; // Send full amount
            }
            
            transfers.push(transfer);
        });
        
        // Get fee UTXOs (pure BTC only, not being transferred)
        const selectedKeys = new Set(bulkTransferState.selectedItems.keys());
        const feeUtxos = bulkTransferState.pureUtxos
            .filter(u => !selectedKeys.has(`${u.txid}:${u.vout}`))
            .sort((a, b) => b.value - a.value)
            .slice(0, 3)
            .map(u => ({ txid: u.txid, vout: u.vout, value: u.value }));
        
        if (feeUtxos.length === 0) {
            throw new Error('No pure BTC UTXOs available for fees. You need at least one pure BTC UTXO that is not being transferred.');
        }
        
        console.log('📦 Creating bulk transfer PSBT...');
        console.log('   Transfers:', transfers.length);
        console.log('   Fee UTXOs:', feeUtxos.length);
        
        // 1. Create PSBT
        const res = await fetch('https://kraywallet-backend.onrender.com/api/psbt/bulk-transfer', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                transfers,
                feeUtxos,
                feeRate
            })
        });
        
        const result = await res.json();
        
        if (!result.success) {
            throw new Error(result.error || 'Failed to create PSBT');
        }
        
        console.log('✅ PSBT created, fee:', result.fee);
        confirmBtn.innerHTML = '<span class="spinner"></span> Signing...';
        
        // 2. Sign PSBT
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01
            }
        });
        
        if (!signResult || !signResult.success) {
            throw new Error(signResult?.error || 'Signing failed - invalid password');
        }
        
        console.log('✅ PSBT signed');
        confirmBtn.innerHTML = '<span class="spinner"></span> Finalizing...';
        
        // 3. Finalize
        const finalizeRes = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeData = await finalizeRes.json();
        
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize');
        }
        
        confirmBtn.innerHTML = '<span class="spinner"></span> Broadcasting...';
        
        // 4. Broadcast
        const broadcastRes = await fetch('https://kraywallet-backend.onrender.com/api/psbt/bulk-transfer/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: finalizeData.hex })
        });
        
        const broadcastData = await broadcastRes.json();
        
        if (!broadcastData.success) {
            throw new Error(broadcastData.error || 'Broadcast failed');
        }
        
        console.log('✅ Bulk transfer broadcast:', broadcastData.txid);
        
        modal.remove();
        showNotification(`✅ Bulk transfer complete! ${transfers.length} assets transferred`, 'success');
        
        // Show success with links
        const successModal = document.createElement('div');
        successModal.className = 'modal-overlay';
        successModal.innerHTML = `
            <div class="modal-container" style="max-width: 400px;">
                <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(16, 185, 129, 0.3);">
                    <h3 style="color: #10b981;">✅ Transfer Complete!</h3>
                </div>
                <div class="modal-body" style="padding: 20px; text-align: center;">
                    <p style="color: #10b981; font-size: 48px; margin-bottom: 16px;">📦</p>
                    <p style="color: #ccc; font-size: 14px; margin-bottom: 16px;">
                        Successfully transferred <strong>${transfers.length}</strong> asset${transfers.length > 1 ? 's' : ''} in one transaction!
                    </p>
                    <div style="display: flex; flex-direction: column; gap: 8px; margin-top: 16px;">
                        <a href="https://mempool.space/tx/${broadcastData.txid}" target="_blank" style="display: flex; align-items: center; justify-content: center; gap: 8px; padding: 12px; background: #1a1a1a; border: 1px solid #333; border-radius: 8px; color: #10b981; text-decoration: none; font-size: 13px;">
                            🔗 View on Mempool.space
                        </a>
                        <a href="https://kray.space/krayscan.html?txid=${broadcastData.txid}" target="_blank" style="display: flex; align-items: center; justify-content: center; gap: 8px; padding: 12px; background: #1a1a1a; border: 1px solid #333; border-radius: 8px; color: #8b5cf6; text-decoration: none; font-size: 13px;">
                            🔍 View on KrayScan
                        </a>
                    </div>
                </div>
                <div class="modal-footer" style="padding: 16px;">
                    <button onclick="this.closest('.modal-overlay').remove(); showScreen('settings');" class="btn btn-primary" style="width: 100%; background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                        Done
                    </button>
                </div>
            </div>
        `;
        document.body.appendChild(successModal);
        
        // Refresh wallet
        await refreshWalletUI();
        
    } catch (error) {
        console.error('❌ Bulk transfer failed:', error);
        modal.remove();
        showNotification('❌ ' + (error.message || 'Transfer failed'), 'error');
    }
}

// Event Listeners for Bulk Transfer
document.getElementById('bulk-transfer-btn')?.addEventListener('click', showBulkTransferScreen);
document.getElementById('back-from-bulk-transfer-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('execute-bulk-transfer-btn')?.addEventListener('click', executeBulkTransfer);

document.getElementById('bulk-transfer-tab-inscriptions')?.addEventListener('click', () => {
    bulkTransferState.activeTab = 'inscriptions';
    updateBulkTransferTabStyles();
    renderBulkTransferAssets();
});

document.getElementById('bulk-transfer-tab-runes')?.addEventListener('click', () => {
    bulkTransferState.activeTab = 'runes';
    updateBulkTransferTabStyles();
    renderBulkTransferAssets();
});

document.getElementById('bulk-transfer-tab-btc')?.addEventListener('click', () => {
    bulkTransferState.activeTab = 'btc';
    updateBulkTransferTabStyles();
    renderBulkTransferAssets();
});

document.getElementById('bulk-transfer-same-address')?.addEventListener('change', (e) => {
    bulkTransferState.useSameAddress = e.target.checked;
    document.getElementById('bulk-transfer-global-address').style.display = e.target.checked ? 'block' : 'none';
    renderBulkTransferAssets();
    updateBulkTransferSummary();
});

document.getElementById('bulk-transfer-recipient-all')?.addEventListener('input', updateBulkTransferSummary);

// Fee button listeners
document.querySelectorAll('.bulk-transfer-fee-btn').forEach(btn => {
    btn.addEventListener('click', () => {
        document.querySelectorAll('.bulk-transfer-fee-btn').forEach(b => {
            b.classList.remove('selected');
            b.style.background = '#1a1a1a';
            b.style.borderColor = '#333';
            b.style.color = '#888';
        });
        btn.classList.add('selected');
        btn.style.background = 'rgba(139,92,246,0.2)';
        btn.style.borderColor = '#8b5cf6';
        btn.style.color = '#8b5cf6';
        
        const speed = btn.dataset.speed;
        if (speed === 'fast') bulkTransferState.selectedFeeRate = bulkTransferState.feeRates?.fastestFee || 20;
        else if (speed === 'normal') bulkTransferState.selectedFeeRate = bulkTransferState.feeRates?.halfHourFee || 10;
        else bulkTransferState.selectedFeeRate = bulkTransferState.feeRates?.hourFee || 5;
    });
});

document.getElementById('bulk-transfer-fee-custom')?.addEventListener('input', (e) => {
    const val = parseInt(e.target.value);
    if (val > 0) {
        bulkTransferState.selectedFeeRate = val;
        document.querySelectorAll('.bulk-transfer-fee-btn').forEach(b => {
            b.classList.remove('selected');
            b.style.background = '#1a1a1a';
            b.style.borderColor = '#333';
            b.style.color = '#888';
        });
    }
});

console.log('✅ Bulk Transfer loaded!');


// ========================================
// 💸 MULTI-SEND BTC (up to 5 recipients)
// ========================================
let multiSendState = {
    recipients: []
};

function initMultiSendUI() {
    console.log('🔀 ========== INIT MULTI-SEND UI ==========');
    
    // Toggle buttons
    const singleBtn = document.getElementById('send-mode-single');
    const multiBtn = document.getElementById('send-mode-multi');
    
    console.log('   Single button found:', !!singleBtn);
    console.log('   Multi button found:', !!multiBtn);
    
    if (singleBtn) {
        singleBtn.addEventListener('click', () => {
            console.log('🖱️ Single button clicked');
            setMultiSendMode('single');
        });
        console.log('   ✅ Single button listener added');
    }
    
    if (multiBtn) {
        multiBtn.addEventListener('click', () => {
            console.log('🖱️ Multi button clicked');
            setMultiSendMode('multi');
        });
        console.log('   ✅ Multi button listener added');
    }
    
    // Add recipient button
    const addBtn = document.getElementById('add-recipient-btn');
    if (addBtn) {
        addBtn.addEventListener('click', addMultiRecipient);
        console.log('   ✅ Add recipient button listener added');
    }
    
    // Multi send confirm
    const confirmBtn = document.getElementById('multi-send-confirm-btn');
    if (confirmBtn) {
        confirmBtn.addEventListener('click', handleMultiSend);
        console.log('   ✅ Multi-send confirm button listener added');
    }
    
    // Initialize with 2 recipients
    console.log('   Initializing with 2 empty recipients...');
    addMultiRecipient();
    addMultiRecipient();
    
    console.log('✅ Multi-Send UI initialized!');
}

function setMultiSendMode(mode) {
    console.log('🔀 setMultiSendMode called with:', mode);
    
    const singleBtn = document.getElementById('send-mode-single');
    const multiBtn = document.getElementById('send-mode-multi');
    const singleForm = document.getElementById('single-send-form');
    const multiForm = document.getElementById('multi-send-form');
    const singleConfirm = document.getElementById('send-confirm-btn');
    const multiConfirm = document.getElementById('multi-send-confirm-btn');
    
    console.log('   Elements found:', {
        singleBtn: !!singleBtn,
        multiBtn: !!multiBtn,
        singleForm: !!singleForm,
        multiForm: !!multiForm,
        singleConfirm: !!singleConfirm,
        multiConfirm: !!multiConfirm
    });
    
    if (!singleForm || !multiForm) {
        console.error('❌ Multi-send forms not found!');
        return;
    }
    
    if (mode === 'single') {
        if (singleBtn) {
            singleBtn.style.background = '#fff';
            singleBtn.style.color = '#000';
        }
        if (multiBtn) {
            multiBtn.style.background = 'transparent';
            multiBtn.style.color = '#888';
        }
        singleForm.style.display = 'block';
        multiForm.style.display = 'none';
        if (singleConfirm) singleConfirm.style.display = 'block';
        if (multiConfirm) multiConfirm.style.display = 'none';
    } else {
        if (multiBtn) {
            multiBtn.style.background = '#fff';
            multiBtn.style.color = '#000';
        }
        if (singleBtn) {
            singleBtn.style.background = 'transparent';
            singleBtn.style.color = '#888';
        }
        singleForm.style.display = 'none';
        multiForm.style.display = 'block';
        if (singleConfirm) singleConfirm.style.display = 'none';
        if (multiConfirm) multiConfirm.style.display = 'block';
    }
    
    console.log('✅ Multi-send mode set to:', mode);
}

function addMultiRecipient() {
    const container = document.getElementById('multi-recipients-container');
    const addBtn = document.getElementById('add-recipient-btn');
    
    if (multiSendState.recipients.length >= 5) {
        showNotification('⚠️ Maximum 5 recipients', 'error');
        return;
    }
    
    const index = multiSendState.recipients.length;
    multiSendState.recipients.push({ address: '', amount: 0 });
    
    const recipientDiv = document.createElement('div');
    recipientDiv.id = `recipient-${index}`;
    recipientDiv.style.cssText = 'background: rgba(255,255,255,0.03); border: 1px solid #333; border-radius: 8px; padding: 12px; margin-bottom: 12px;';
    
    recipientDiv.innerHTML = `
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 8px;">
            <span style="color: #888; font-size: 12px; font-weight: 600;">Recipient ${index + 1}</span>
            <button type="button" onclick="removeMultiRecipient(${index})" style="background: rgba(239, 68, 68, 0.2); color: #ef4444; border: none; border-radius: 4px; padding: 4px 8px; cursor: pointer; font-size: 11px;">Remove</button>
        </div>
        <input type="text" id="multi-address-${index}" placeholder="bc1p..." onchange="updateMultiRecipient(${index})" style="width: 100%; padding: 10px; background: #1a1a1a; border: 1px solid #333; border-radius: 6px; color: #fff; font-size: 13px; margin-bottom: 8px; box-sizing: border-box;" />
        <input type="number" id="multi-amount-${index}" placeholder="Amount (sats)" onchange="updateMultiRecipient(${index})" style="width: 100%; padding: 10px; background: #1a1a1a; border: 1px solid #333; border-radius: 6px; color: #fff; font-size: 13px; box-sizing: border-box;" />
    `;
    
    container.appendChild(recipientDiv);
    
    // Hide add button if max reached
    if (multiSendState.recipients.length >= 5) {
        addBtn.style.display = 'none';
    }
    
    updateMultiSendSummary();
}

function removeMultiRecipient(index) {
    const container = document.getElementById('multi-recipients-container');
    const recipientDiv = document.getElementById(`recipient-${index}`);
    const addBtn = document.getElementById('add-recipient-btn');
    
    if (recipientDiv) {
        recipientDiv.remove();
        multiSendState.recipients.splice(index, 1);
        
        // Re-render all recipients with correct indices
        container.innerHTML = '';
        const oldRecipients = [...multiSendState.recipients];
        multiSendState.recipients = [];
        
        oldRecipients.forEach((r, i) => {
            addMultiRecipient();
            document.getElementById(`multi-address-${i}`).value = r.address;
            document.getElementById(`multi-amount-${i}`).value = r.amount || '';
            multiSendState.recipients[i] = r;
        });
        
        // Show add button again
        addBtn.style.display = 'block';
    }
    
    updateMultiSendSummary();
}

function updateMultiRecipient(index) {
    const address = document.getElementById(`multi-address-${index}`)?.value || '';
    const amount = parseInt(document.getElementById(`multi-amount-${index}`)?.value) || 0;
    
    if (multiSendState.recipients[index]) {
        multiSendState.recipients[index] = { address, amount };
    }
    
    updateMultiSendSummary();
}

function updateMultiSendSummary() {
    const summary = document.getElementById('multi-send-summary');
    const validRecipients = multiSendState.recipients.filter(r => r.address && r.amount > 0);
    const totalAmount = validRecipients.reduce((sum, r) => sum + r.amount, 0);
    
    if (validRecipients.length > 0) {
        summary.style.display = 'block';
        document.getElementById('multi-summary-count').textContent = validRecipients.length;
        document.getElementById('multi-summary-total').textContent = totalAmount.toLocaleString() + ' sats';
        document.getElementById('multi-summary-fee').textContent = `~${validRecipients.length * 50 + 150} sats`;
    } else {
        summary.style.display = 'none';
    }
}

async function handleMultiSend() {
    const validRecipients = multiSendState.recipients.filter(r => r.address?.startsWith('bc1') && r.amount > 0);
    
    if (validRecipients.length === 0) {
        showNotification('⚠️ Add at least one valid recipient', 'error');
        return;
    }
    
    // Show password modal
    showMultiSendConfirmModal(validRecipients);
}

function showMultiSendConfirmModal(recipients) {
    const totalAmount = recipients.reduce((sum, r) => sum + r.amount, 0);
    const feeRate = parseInt(document.getElementById('multi-send-fee')?.value) || 2;
    
    const modal = document.createElement('div');
    modal.id = 'multi-send-confirm-modal';
    modal.className = 'modal-overlay';
    modal.innerHTML = `
        <div class="modal-container" style="max-width: 420px;">
            <div class="modal-header" style="justify-content: center; border-bottom: 1px solid rgba(247, 147, 26, 0.2);">
                <h3 style="color: #f7931a;">💸 Confirm Multi-Send</h3>
            </div>
            <div class="modal-body" style="text-align: center;">
                <div style="font-size: 40px; margin-bottom: 12px;">💸</div>
                
                <div style="background: rgba(255,255,255,0.05); border-radius: 8px; padding: 12px; margin-bottom: 16px; text-align: left; max-height: 150px; overflow-y: auto;">
                    ${recipients.map((r, i) => `
                        <div style="display: flex; justify-content: space-between; padding: 8px 0; border-bottom: 1px solid rgba(255,255,255,0.05);">
                            <span style="color: #888; font-size: 11px;">${r.address.slice(0, 10)}...${r.address.slice(-6)}</span>
                            <span style="color: #f7931a; font-size: 12px; font-weight: 600;">${r.amount.toLocaleString()} sats</span>
                        </div>
                    `).join('')}
                </div>
                
                <div style="display: flex; justify-content: space-between; margin-bottom: 16px; padding: 12px; background: rgba(247, 147, 26, 0.1); border-radius: 8px;">
                    <span style="color: #888;">Total:</span>
                    <span style="color: #f7931a; font-weight: 600;">${totalAmount.toLocaleString()} sats + fee</span>
                </div>
                
                <input type="password" id="multi-send-password" class="input-field" placeholder="Enter password" autocomplete="off" style="margin-bottom: 16px;" />
            </div>
            <div class="modal-footer" style="display: flex; gap: 12px;">
                <button id="cancel-multi-send-btn" class="btn btn-secondary" style="flex: 1;">Cancel</button>
                <button id="confirm-multi-send-btn" class="btn btn-primary" style="flex: 1; background: linear-gradient(135deg, #f7931a 0%, #d97706 100%);">
                    💸 Send All
                </button>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    document.getElementById('multi-send-password').focus();
    
    document.getElementById('cancel-multi-send-btn').addEventListener('click', () => modal.remove());
    document.getElementById('confirm-multi-send-btn').addEventListener('click', () => confirmMultiSend(recipients));
}

async function confirmMultiSend(recipients) {
    const password = document.getElementById('multi-send-password').value;
    if (!password) {
        showNotification('⚠️ Enter password', 'error');
        return;
    }
    
    const confirmBtn = document.getElementById('confirm-multi-send-btn');
    confirmBtn.disabled = true;
    confirmBtn.textContent = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const feeRate = parseInt(document.getElementById('multi-send-fee')?.value) || 2;
        
        // Get pure UTXOs
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${address}`);
        const utxosData = await utxosResponse.json();
        const pureUtxos = utxosData.utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        
        if (pureUtxos.length === 0) {
            throw new Error('No pure BTC UTXOs available');
        }
        
        // Call backend
        const response = await fetch('https://kraywallet-backend.onrender.com/api/psbt/multi-send', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                recipients,
                paymentUtxos: pureUtxos.map(u => ({ txid: u.txid, vout: u.vout, value: u.value })),
                feeRate
            })
        });
        
        const result = await response.json();
        if (!result.success) throw new Error(result.error || 'Failed');
        
        // 🔐 LEI PRIMORDIAL: Sign PSBT LOCALLY via background script
        console.log('🔐 Signing multi-send PSBT locally...');
        // ✅ Sign ALL inputs
        const inputCount = result.inputCount || 5; // Default to 5 if not provided
        const inputsToSign = Array.from({ length: inputCount }, (_, i) => i);
        console.log('   Inputs to sign:', inputsToSign);
        
        const signResult = await chrome.runtime.sendMessage({
            action: 'signPsbtWithPassword',
            data: {
                psbt: result.psbt_base64,
                password: password,
                sighashType: 0x01, // SIGHASH_ALL
                inputsToSign: inputsToSign // ✅ Sign ALL inputs
            }
        });
        
        if (!signResult.success) throw new Error(signResult.error || 'Sign failed');
        
        // Finalize
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeResult = await finalizeResponse.json();
        if (!finalizeResult.success) throw new Error(finalizeResult.error || 'Finalize failed');
        
        // Remove modal before showing success
        document.getElementById('multi-send-confirm-modal')?.remove();
        
        // 🎉 Show success modal with mempool.space and KrayScan links
        showSuccessModal(
            finalizeResult.txid,
            'Multi-Send Complete!',
            `Successfully sent to ${recipients.length} recipients.`
        );
        
    } catch (error) {
        console.error('❌ Error:', error);
        showNotification('❌ Failed: ' + error.message, 'error');
        confirmBtn.disabled = false;
        confirmBtn.textContent = '💸 Send All';
    }
}

// Expose functions globally
window.removeMultiRecipient = removeMultiRecipient;
window.updateMultiRecipient = updateMultiRecipient;

// Initialize when DOM ready
document.addEventListener('DOMContentLoaded', initMultiSendUI);

console.log('✅ Multi-Send BTC loaded!');


// ═══════════════════════════════════════════════════════════════════════════
// 🆕 INSCRIBE ORDINAL & ETCH RUNE (2025-12-17)
// ═══════════════════════════════════════════════════════════════════════════

let inscribeMode = 'file';
let etchDivisibility = 0;

// === INSCRIBE ORDINAL MODAL ===

function openInscribeModal() {
    console.log('💎 Opening Inscribe Modal');
    const modal = document.getElementById('inscribe-modal');
    if (modal) {
        modal.classList.remove('hidden');
        resetInscribeForm();
        // Fetch dynamic fee rates from mempool.space
        fetchInscribeFeeRates();
    }
}

function closeInscribeModal() {
    const modal = document.getElementById('inscribe-modal');
    if (modal) {
        modal.classList.add('hidden');
    }
}

function setInscribeMode(mode) {
    console.log('🔄 Setting inscribe mode to:', mode);
    inscribeMode = mode;
    
    // Update button states
    const fileBtn = document.getElementById('inscribe-file-btn');
    const textBtn = document.getElementById('inscribe-text-btn');
    const fileSection = document.getElementById('inscribe-file-section');
    const textSection = document.getElementById('inscribe-text-section');
    
    console.log('📦 Elements found:', {
        fileBtn: !!fileBtn,
        textBtn: !!textBtn,
        fileSection: !!fileSection,
        textSection: !!textSection
    });
    
    if (!fileBtn || !textBtn || !fileSection || !textSection) {
        console.error('❌ Inscribe modal elements not found!');
        return;
    }
    
    if (mode === 'file') {
        console.log('📁 Switching to FILE mode');
        // File mode active - orange gradient
        fileBtn.style.cssText = 'flex: 1; padding: 12px; border-radius: 10px; border: 2px solid #f7931a; background: linear-gradient(135deg, #f7931a, #ffb84d); color: #000; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 6px; font-weight: 600; font-size: 14px;';
        // Text inactive - dark
        textBtn.style.cssText = 'flex: 1; padding: 12px; border-radius: 10px; border: 2px solid #333; background: #1a1a1a; color: #fff; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 6px; font-weight: 600; font-size: 14px;';
        // Show/hide sections
        fileSection.style.cssText = 'display: block !important;';
        textSection.style.cssText = 'display: none !important;';
    } else {
        console.log('📝 Switching to TEXT mode');
        // Text mode active - orange gradient
        textBtn.style.cssText = 'flex: 1; padding: 12px; border-radius: 10px; border: 2px solid #f7931a; background: linear-gradient(135deg, #f7931a, #ffb84d); color: #000; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 6px; font-weight: 600; font-size: 14px;';
        // File inactive - dark
        fileBtn.style.cssText = 'flex: 1; padding: 12px; border-radius: 10px; border: 2px solid #333; background: #1a1a1a; color: #fff; cursor: pointer; display: flex; align-items: center; justify-content: center; gap: 6px; font-weight: 600; font-size: 14px;';
        // Show/hide sections
        textSection.style.cssText = 'display: block !important;';
        fileSection.style.cssText = 'display: none !important;';
    }
    
    // Reset fee estimate when switching modes
    currentInscribeContentSize = 0;
    const feeEstimate = document.getElementById('inscribe-fee-estimate');
    if (feeEstimate) feeEstimate.style.display = 'none';
    const sizeWarning = document.getElementById('inscribe-size-warning');
    if (sizeWarning) sizeWarning.innerHTML = '';
}

function resetInscribeForm() {
    setInscribeMode('file');
    currentInscribeContentSize = 0;
    
    const fileInput = document.getElementById('inscribe-file-input');
    const textInput = document.getElementById('inscribe-text-input');
    const feeRateSelect = document.getElementById('inscribe-fee-rate');
    const textSizeEl = document.getElementById('inscribe-text-size');
    
    if (fileInput) fileInput.value = '';
    if (textInput) textInput.value = '';
    if (feeRateSelect) feeRateSelect.value = '10';
    if (textSizeEl) textSizeEl.textContent = '0';
    
    document.getElementById('inscribe-fee-estimate')?.classList.add('hidden');
    document.getElementById('inscribe-size-warning').innerHTML = '';
}

// Track current content size for fee rate updates
let currentInscribeContentSize = 0;

// Initialize inscribe UI event listeners
document.addEventListener('DOMContentLoaded', () => {
    console.log('🚀 Inscribe UI initializing...');
    
    // Mode toggle buttons (File/Text)
    const fileBtn = document.getElementById('inscribe-file-btn');
    const textBtn = document.getElementById('inscribe-text-btn');
    
    console.log('🔍 Mode buttons found:', { fileBtn: !!fileBtn, textBtn: !!textBtn });
    
    if (fileBtn) {
        fileBtn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            console.log('📁 FILE BUTTON CLICKED!');
            setInscribeMode('file');
        });
        console.log('✅ File button listener attached');
    }
    
    if (textBtn) {
        textBtn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            console.log('📝 TEXT BUTTON CLICKED!');
            setInscribeMode('text');
        });
        console.log('✅ Text button listener attached');
    }
    
    // File input handler
    const fileInput = document.getElementById('inscribe-file-input');
    if (fileInput) {
        fileInput.addEventListener('change', async (e) => {
            const file = e.target.files[0];
            if (file) {
                // Check max size
                if (file.size > SIZE_THRESHOLDS.MAX) {
                    showNotification(`⚠️ Arquivo muito grande! Máximo: ${Math.floor(SIZE_THRESHOLDS.MAX / 1024)}KB`, 'error');
                    e.target.value = '';
                    return;
                }
                console.log('📁 File selected:', file.name, file.size, file.type);
                currentInscribeContentSize = file.size;
                await fetchInscribeFeeEstimate(file.size);
            }
        });
    }
    
    // Text input handler
    const textInput = document.getElementById('inscribe-text-input');
    if (textInput) {
        textInput.addEventListener('input', async (e) => {
            const size = new TextEncoder().encode(e.target.value).length;
            document.getElementById('inscribe-text-size').textContent = size.toLocaleString();
            currentInscribeContentSize = size;
            if (size > 0) {
                await fetchInscribeFeeEstimate(size);
            } else {
                document.getElementById('inscribe-fee-estimate')?.classList.add('hidden');
            }
        });
    }
    
    // Template buttons - use event delegation on document body
    document.body.addEventListener('click', (e) => {
        const btn = e.target.closest('.inscribe-template-btn');
        if (!btn) return;
        
        e.preventDefault();
        e.stopPropagation();
        
        const template = btn.dataset.template;
        const textInput = document.getElementById('inscribe-text-input');
        if (!textInput) {
            console.error('❌ Text input not found!');
            return;
        }
        
        console.log('📝 Template clicked:', template);
        
        switch (template) {
            case 'name':
                textInput.value = 'yourname.btc';
                setTimeout(() => {
                    textInput.setSelectionRange(0, 8); // Select "yourname"
                    textInput.focus();
                }, 50);
                break;
            case 'message':
                textInput.value = 'Hello from Bitcoin! 🟠';
                textInput.focus();
                break;
            case 'json':
                textInput.value = JSON.stringify({
                    "p": "brc-20",
                    "op": "deploy",
                    "tick": "KRAY",
                    "max": "21000000",
                    "lim": "1000"
                }, null, 2);
                textInput.focus();
                break;
            case 'clear':
                textInput.value = '';
                document.getElementById('inscribe-text-size').textContent = '0';
                document.getElementById('inscribe-fee-estimate')?.classList.add('hidden');
                const sizeWarning = document.getElementById('inscribe-size-warning');
                if (sizeWarning) sizeWarning.innerHTML = '';
                currentInscribeContentSize = 0;
                return; // Don't trigger input event for clear
        }
        
        // Trigger input event to update size and estimate
        textInput.dispatchEvent(new Event('input'));
    });
    
    // Fee rate selector
    const feeRateSelect = document.getElementById('inscribe-fee-rate');
    if (feeRateSelect) {
        feeRateSelect.addEventListener('change', updateInscribeFee);
    }
    
    // Submit button (must use event listener for CSP compliance)
    const submitBtn = document.getElementById('inscribe-submit-btn');
    if (submitBtn) {
        submitBtn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            console.log('💎 INSCRIBE SUBMIT CLICKED!');
            submitInscription();
        });
        console.log('✅ Inscribe submit button listener attached');
    }
    
    // Modal close buttons
    const inscribeCloseBtn = document.getElementById('inscribe-modal-close');
    if (inscribeCloseBtn) {
        inscribeCloseBtn.addEventListener('click', () => closeInscribeModal());
    }
    
    const etchCloseBtn = document.getElementById('etch-modal-close');
    if (etchCloseBtn) {
        etchCloseBtn.addEventListener('click', () => closeEtchRuneModal());
    }
    
    // Etch submit button
    const etchSubmitBtn = document.getElementById('etch-submit-btn');
    if (etchSubmitBtn) {
        etchSubmitBtn.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            console.log('✨ ETCH RUNE SUBMIT CLICKED!');
            submitEtchRune();
        });
        console.log('✅ Etch submit button listener attached');
    }
    
    // Divisibility buttons (event delegation)
    document.querySelectorAll('.div-btn').forEach(btn => {
        btn.addEventListener('click', () => {
            const div = parseInt(btn.dataset.div);
            console.log('🔢 Divisibility clicked:', div);
            setDivisibility(div);
        });
    });
});

// Update fee when rate changes
async function updateInscribeFee() {
    if (currentInscribeContentSize > 0) {
        await fetchInscribeFeeEstimate(currentInscribeContentSize);
    }
}

// Fetch dynamic fee rates from mempool.space
async function fetchInscribeFeeRates() {
    const select = document.getElementById('inscribe-fee-rate');
    const sourceEl = document.getElementById('inscribe-fee-source');
    
    if (!select) return;
    
    try {
        console.log('📡 Fetching fee rates from mempool.space...');
        if (sourceEl) sourceEl.innerHTML = '⏳ loading...';
        
        const res = await fetch('https://mempool.space/api/v1/fees/recommended');
        const fees = await res.json();
        
        console.log('✅ Fee rates:', fees);
        
        // Update select options with real fees
        select.innerHTML = `
            <option value="economy" data-fee="${fees.economyFee}">🐢 Economy (${fees.economyFee} sat/vB) ~1h+</option>
            <option value="slow" data-fee="${fees.hourFee}">🚶 Low Priority (${fees.hourFee} sat/vB) ~1h</option>
            <option value="medium" data-fee="${fees.halfHourFee}" selected>🚗 Medium (${fees.halfHourFee} sat/vB) ~30min</option>
            <option value="fast" data-fee="${fees.fastestFee}">🚀 Fast (${fees.fastestFee} sat/vB) ~10min</option>
        `;
        
        if (sourceEl) sourceEl.innerHTML = '<span style="color: #4ade80;">● live</span> mempool.space';
        
        // Store current fees globally
        window.currentMempoolFees = fees;
        
        // Update estimate if content is selected
        if (currentInscribeContentSize > 0) {
            await fetchInscribeFeeEstimate(currentInscribeContentSize);
        }
        
    } catch (error) {
        console.error('❌ Failed to fetch fee rates:', error);
        if (sourceEl) sourceEl.innerHTML = '<span style="color: #f87171;">● offline</span> using defaults';
        
        // Use fallback values
        select.innerHTML = `
            <option value="economy" data-fee="1">🐢 Economy (1 sat/vB) ~1h+</option>
            <option value="slow" data-fee="5">🚶 Low Priority (5 sat/vB) ~1h</option>
            <option value="medium" data-fee="10" selected>🚗 Medium (10 sat/vB) ~30min</option>
            <option value="fast" data-fee="20">🚀 Fast (20 sat/vB) ~10min</option>
        `;
    }
}

// Get current selected fee rate
function getInscribeFeeRate() {
    const select = document.getElementById('inscribe-fee-rate');
    if (!select) return 10;
    const option = select.options[select.selectedIndex];
    return parseInt(option.dataset.fee) || 10;
}

// Expose function globally
window.updateInscribeFee = updateInscribeFee;
window.fetchInscribeFeeRates = fetchInscribeFeeRates;
window.getInscribeFeeRate = getInscribeFeeRate;

// Size thresholds for warnings (bytes)
const SIZE_THRESHOLDS = {
    IDEAL: 10000,      // 10KB - cheap
    MODERATE: 50000,   // 50KB - moderate cost
    EXPENSIVE: 100000, // 100KB - expensive
    EXTREME: 200000,   // 200KB - very expensive
    MAX: 390000        // ~390KB max
};

function getSizeWarning(size) {
    if (size === 0) return { level: 'none', color: '#666', message: '', tip: '' };
    
    if (size <= SIZE_THRESHOLDS.IDEAL) {
        return {
            level: 'ideal',
            color: '#22c55e',
            message: '✅ Tamanho ideal - custo baixo',
            tip: ''
        };
    }
    if (size <= SIZE_THRESHOLDS.MODERATE) {
        return {
            level: 'moderate',
            color: '#f59e0b',
            message: '💰 Tamanho moderado',
            tip: '💡 Considere comprimir a imagem para reduzir custos'
        };
    }
    if (size <= SIZE_THRESHOLDS.EXPENSIVE) {
        return {
            level: 'expensive',
            color: '#ef4444',
            message: '⚠️ Arquivo grande - custo elevado',
            tip: '💡 Imagens WebP são menores que PNG/JPEG'
        };
    }
    if (size <= SIZE_THRESHOLDS.EXTREME) {
        return {
            level: 'extreme',
            color: '#dc2626',
            message: '🔥 Arquivo muito grande - custo extremo',
            tip: '💡 Reduza a resolução ou use formato WebP'
        };
    }
    return {
        level: 'maximum',
        color: '#7f1d1d',
        message: '💸 Tamanho máximo - custo muito alto',
        tip: '💡 Considere reduzir para < 100KB'
    };
}

async function fetchInscribeFeeEstimate(contentSize) {
    try {
        // Get the fee rate from the selected option's data attribute
        const feeRate = getInscribeFeeRate();
        
        const res = await fetch(`https://kraywallet-backend.onrender.com/api/inscription/estimate?contentSize=${contentSize}&feeRate=${feeRate}`);
        const data = await res.json();
        
        if (data.success) {
            const warning = getSizeWarning(contentSize);
            const sizeKB = (contentSize / 1024).toFixed(1);
            const barPercent = Math.min((contentSize / SIZE_THRESHOLDS.MAX) * 100, 100);
            
            // Update size warning
            const warningEl = document.getElementById('inscribe-size-warning');
            if (warningEl) {
                warningEl.innerHTML = `
                    <div style="border: 2px solid ${warning.color}; border-radius: 8px; padding: 12px; margin-bottom: 12px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <span style="color: ${warning.color}; font-weight: 600;">${warning.message}</span>
                            <span style="color: var(--color-text-secondary); font-weight: 600;">${sizeKB} KB</span>
                        </div>
                        ${warning.tip ? `<div style="font-size: 12px; color: var(--color-text-muted); margin-top: 8px; font-style: italic;">${warning.tip}</div>` : ''}
                    </div>
                    <div style="margin-bottom: 12px;">
                        <div style="height: 8px; background: var(--color-surface); border-radius: 4px; overflow: hidden;">
                            <div style="height: 100%; width: ${barPercent}%; background: ${warning.color}; border-radius: 4px;"></div>
                        </div>
                        <div style="display: flex; justify-content: space-between; margin-top: 4px; font-size: 10px; color: var(--color-text-muted);">
                            <span>10KB</span><span>50KB</span><span>100KB</span><span>200KB</span><span>390KB</span>
                        </div>
                    </div>
                `;
            }
            
            // Update cost breakdown
            document.getElementById('inscribe-content-size').textContent = `${sizeKB} KB`;
            document.getElementById('inscribe-fee-rate-display').textContent = `${feeRate} sat/vB`;
            document.getElementById('inscribe-commit-fee').textContent = `${data.estimate.commitFee.toLocaleString()} sats`;
            document.getElementById('inscribe-reveal-fee').textContent = `${data.estimate.revealFee.toLocaleString()} sats`;
            document.getElementById('inscribe-output-value').textContent = `${data.estimate.outputValue.toLocaleString()} sats`;
            document.getElementById('inscribe-total-cost').textContent = `${data.estimate.totalCost.toLocaleString()} sats${data.estimate.totalCost > 50000 ? ' ⚠️' : ''}`;
            
            // USD estimate (at ~$100k/BTC)
            const usdEstimate = ((data.estimate.totalCost / 100000000) * 100000).toFixed(2);
            document.getElementById('inscribe-usd-estimate').textContent = `≈ $${usdEstimate} USD`;
            
            document.getElementById('inscribe-fee-estimate').classList.remove('hidden');
        }
    } catch (error) {
        console.error('Failed to fetch fee estimate:', error);
    }
}

async function submitInscription() {
    const submitBtn = document.getElementById('inscribe-submit-btn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const publicKey = await getWalletPublicKey();
        
        if (!publicKey) {
            throw new Error('Public key not available. Please re-import your wallet.');
        }
        let content, contentType;
        
        if (inscribeMode === 'file') {
            const fileInput = document.getElementById('inscribe-file-input');
            const file = fileInput.files[0];
            if (!file) {
                throw new Error('Please select a file');
            }
            
            // Read file as base64
            const arrayBuffer = await file.arrayBuffer();
            content = btoa(String.fromCharCode(...new Uint8Array(arrayBuffer)));
            contentType = file.type || 'application/octet-stream';
        } else {
            const text = document.getElementById('inscribe-text-input').value;
            if (!text) {
                throw new Error('Please enter text to inscribe');
            }
            content = btoa(text);
            contentType = 'text/plain;charset=utf-8';
        }
        
        // Get selected fee rate from mempool.space dynamic rates
        const feeRate = getInscribeFeeRate();
        console.log('📊 Using fee rate:', feeRate, 'sat/vB');
        
        // Calculate content size for display
        const contentBuffer = Uint8Array.from(atob(content), c => c.charCodeAt(0));
        const contentSizeKB = (contentBuffer.length / 1024).toFixed(1);
        const contentSizeBytes = contentBuffer.length;
        
        // Fetch fee estimate from backend
        console.log('💰 Fetching fee estimate...');
        let feeEstimate = null;
        try {
            const estimateRes = await fetch(`https://kraywallet-backend.onrender.com/api/inscription/estimate?contentSize=${contentSizeBytes}&feeRate=${feeRate}`);
            const estimateData = await estimateRes.json();
            if (estimateData.success) {
                feeEstimate = estimateData.estimate;
                console.log('✅ Fee estimate:', feeEstimate);
            }
        } catch (e) {
            console.warn('Could not fetch fee estimate:', e);
        }
        
        // Show inscription fees in modal
        const feesContainer = document.getElementById('confirm-inscription-fees');
        if (feesContainer && feeEstimate) {
            feesContainer.style.display = 'block';
            document.getElementById('confirm-commit-fee').textContent = `${feeEstimate.commitFee.toLocaleString()} sats`;
            document.getElementById('confirm-reveal-fee').textContent = `${feeEstimate.revealFee.toLocaleString()} sats`;
            document.getElementById('confirm-output-value').textContent = `${feeEstimate.outputValue} sats`;
            document.getElementById('confirm-total-cost').textContent = `${feeEstimate.totalCost.toLocaleString()} sats`;
        }
        
        // Request password with inscription details
        const password = await requestPassword('💎 Confirm Inscription', {
            type: `📝 ${contentType.split('/')[0].toUpperCase()} Inscription`,
            contentSize: `${contentSizeKB} KB (${contentSizeBytes} bytes)`,
            feeRate: feeRate
        });
        
        if (!password) {
            // Hide fees container on cancel
            if (feesContainer) feesContainer.style.display = 'none';
            throw new Error('Password required');
        }
        
        showLoading('💎 Creating inscription...');
        
        const commitRes = await fetch('https://kraywallet-backend.onrender.com/api/inscription/commit', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ address, publicKey, content, contentType, feeRate })
        });
        
        const commitData = await commitRes.json();
        if (!commitData.success) {
            throw new Error(commitData.error || 'Failed to prepare commit');
        }
        
        // Sign commit
        const signedCommit = await signPsbtWithPassword(commitData.psbt, password);
        if (!signedCommit) {
            throw new Error('Failed to sign commit transaction');
        }
        
        // Finalize and broadcast commit
        const finalizeCommitRes = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signedCommit })
        });
        
        const finalizeCommitData = await finalizeCommitRes.json();
        if (!finalizeCommitData.success) {
            throw new Error(finalizeCommitData.error || 'Failed to finalize commit');
        }
        
        // Broadcast commit
        const broadcastCommitRes = await fetch('https://kraywallet-backend.onrender.com/api/wallet/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ txHex: finalizeCommitData.txHex })
        });
        
        const broadcastCommitData = await broadcastCommitRes.json();
        if (!broadcastCommitData.success) {
            throw new Error(broadcastCommitData.error || 'Failed to broadcast commit');
        }
        
        console.log('✅ Commit broadcast:', broadcastCommitData.txid);
        
        // Wait a bit
        await new Promise(r => setTimeout(r, 2000));
        
        // Step 2: Reveal
        showLoading('💎 Revealing inscription...');
        
        const revealRes = await fetch('https://kraywallet-backend.onrender.com/api/inscription/reveal', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                commitTxId: broadcastCommitData.txid,
                commitVout: 0,
                commitValue: commitData.commitValue,
                recipientAddress: address,
                revealData: commitData.revealData,
                feeRate
            })
        });
        
        const revealData = await revealRes.json();
        if (!revealData.success) {
            throw new Error(revealData.error || 'Failed to prepare reveal');
        }
        
        // Sign reveal
        const signedReveal = await signPsbtWithPassword(revealData.psbt, password);
        if (!signedReveal) {
            throw new Error('Failed to sign reveal transaction');
        }
        
        // Finalize and broadcast reveal
        const finalizeRevealRes = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signedReveal })
        });
        
        const finalizeRevealData = await finalizeRevealRes.json();
        if (!finalizeRevealData.success) {
            throw new Error(finalizeRevealData.error || 'Failed to finalize reveal');
        }
        
        // Broadcast reveal
        const broadcastRevealRes = await fetch('https://kraywallet-backend.onrender.com/api/wallet/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ txHex: finalizeRevealData.txHex })
        });
        
        const broadcastRevealData = await broadcastRevealRes.json();
        if (!broadcastRevealData.success) {
            throw new Error(broadcastRevealData.error || 'Failed to broadcast reveal');
        }
        
        hideLoading();
        
        const inscriptionId = `${broadcastRevealData.txid}i0`;
        showNotification(`🎉 Inscription created! ID: ${inscriptionId.slice(0, 16)}...`, 'success');
        
        closeInscribeModal();
        loadWalletData();
        
    } catch (error) {
        console.error('❌ Inscribe error:', error);
        hideLoading();
        showNotification('❌ ' + error.message, 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '💎 Inscribe';
    }
}

// === ETCH RUNE MODAL ===

function openEtchRuneModal() {
    console.log('✨ Opening Etch Rune Modal');
    const modal = document.getElementById('etch-rune-modal');
    if (modal) {
        modal.classList.remove('hidden');
        resetEtchRuneForm();
    }
}

function closeEtchRuneModal() {
    const modal = document.getElementById('etch-rune-modal');
    if (modal) {
        modal.classList.add('hidden');
    }
}

function setDivisibility(div) {
    etchDivisibility = div;
    
    document.querySelectorAll('.div-btn').forEach(btn => {
        if (parseInt(btn.dataset.div) === div) {
            btn.style.background = 'var(--color-primary)';
            btn.style.color = '#000';
        } else {
            btn.style.background = 'var(--color-surface)';
            btn.style.color = '';
        }
    });
}

function toggleMintOptions() {
    const checkbox = document.getElementById('etch-enable-mint');
    const options = document.getElementById('etch-mint-options');
    const slider = checkbox.parentElement.querySelector('.slider');
    
    if (checkbox.checked) {
        options.classList.remove('hidden');
        slider.style.background = 'var(--color-primary)';
    } else {
        options.classList.add('hidden');
        slider.style.background = 'var(--color-border)';
    }
}

function resetEtchRuneForm() {
    document.getElementById('etch-rune-name').value = '';
    document.getElementById('etch-rune-symbol').value = '';
    document.getElementById('etch-rune-premine').value = '';
    document.getElementById('etch-rune-cap').value = '';
    document.getElementById('etch-rune-amount').value = '';
    document.getElementById('etch-enable-mint').checked = false;
    document.getElementById('etch-mint-options').classList.add('hidden');
    setDivisibility(0);
}

// Name validation
document.addEventListener('DOMContentLoaded', () => {
    const nameInput = document.getElementById('etch-rune-name');
    if (nameInput) {
        nameInput.addEventListener('input', async (e) => {
            // Convert spaces to bullets (•) as per Runes specification
            // Using character-by-character processing for reliability
            const text = e.target.value;
            let result = '';
            let lastWasBullet = false;
            
            for (let i = 0; i < text.length; i++) {
                const char = text[i].toUpperCase();
                
                // Space, bullet, dot, dash, underscore → add bullet (if not double)
                if (char === ' ' || char === '•' || char === '.' || char === '-' || char === '_') {
                    if (!lastWasBullet && result.length > 0) {
                        result += '•';
                        lastWasBullet = true;
                    }
                }
                // Letter A-Z → add letter
                else if (char >= 'A' && char <= 'Z') {
                    result += char;
                    lastWasBullet = false;
                }
                // Ignore everything else
            }
            
            // Remove trailing bullet
            if (result.endsWith('•')) {
                result = result.slice(0, -1);
            }
            
            e.target.value = result;
            
            if (result.replace(/•/g, '').length >= 3) {
                await validateRuneName(result);
            } else {
                // Hide status box and show hint
                document.getElementById('etch-name-status-box').style.display = 'none';
                document.getElementById('etch-name-hint').style.display = 'block';
            }
        });
    }
});

async function validateRuneName(name) {
    try {
        const res = await fetch(`https://kraywallet-backend.onrender.com/api/runes/etch/validate?name=${encodeURIComponent(name)}`);
        const data = await res.json();
        
        const statusBox = document.getElementById('etch-name-status-box');
        const statusIcon = document.getElementById('etch-name-status-icon');
        const statusTitle = document.getElementById('etch-name-status-title');
        const statusText = document.getElementById('etch-name-status-text');
        const statusDetails = document.getElementById('etch-name-status-details');
        const hintEl = document.getElementById('etch-name-hint');
        
        statusBox.style.display = 'block';
        hintEl.style.display = 'none';
        
        if (data.success) {
            // Available
            statusBox.style.background = 'rgba(16, 185, 129, 0.1)';
            statusBox.style.border = '1px solid rgba(16, 185, 129, 0.3)';
            statusIcon.textContent = '✅';
            statusTitle.textContent = 'Available!';
            statusTitle.style.color = '#10b981';
            statusText.textContent = `This ${data.nameLength}-character name is ready to etch`;
            statusDetails.style.display = 'none';
        } else if (data.status === 'taken') {
            // Already taken
            statusBox.style.background = 'rgba(239, 68, 68, 0.1)';
            statusBox.style.border = '1px solid rgba(239, 68, 68, 0.3)';
            statusIcon.textContent = '❌';
            statusTitle.textContent = 'Already Taken';
            statusTitle.style.color = '#ef4444';
            statusText.textContent = data.error;
            statusDetails.style.display = 'none';
        } else if (data.status === 'locked' && data.details) {
            // Locked - show unlock details
            statusBox.style.background = 'rgba(245, 158, 11, 0.1)';
            statusBox.style.border = '1px solid rgba(245, 158, 11, 0.3)';
            statusIcon.textContent = '🔒';
            statusTitle.textContent = 'Name Locked';
            statusTitle.style.color = '#f59e0b';
            statusText.textContent = `${data.details.nameLength}-character names unlock at block ${data.details.unlockBlock?.toLocaleString()}`;
            statusDetails.style.display = 'block';
            document.getElementById('etch-detail-current-block').textContent = 
                `📊 Current block: ${data.details.currentBlock?.toLocaleString()}`;
            document.getElementById('etch-detail-blocks-remaining').textContent = 
                `⏳ Blocks remaining: ${data.details.blocksRemaining?.toLocaleString()}`;
            document.getElementById('etch-detail-time-estimate').textContent = 
                `🗓️ Estimated: ${data.details.timeEstimate} (${data.details.estimatedDate})`;
        } else {
            // Invalid
            statusBox.style.background = 'rgba(239, 68, 68, 0.1)';
            statusBox.style.border = '1px solid rgba(239, 68, 68, 0.3)';
            statusIcon.textContent = '⚠️';
            statusTitle.textContent = 'Invalid Name';
            statusTitle.style.color = '#ef4444';
            statusText.textContent = data.error;
            statusDetails.style.display = 'none';
        }
    } catch (error) {
        console.error('Failed to validate name:', error);
    }
}

async function submitEtchRune() {
    const submitBtn = document.getElementById('etch-submit-btn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '⏳ Processing...';
    
    try {
        const address = await getWalletAddress();
        const name = document.getElementById('etch-rune-name').value.toUpperCase();
        const symbol = document.getElementById('etch-rune-symbol').value || name.replace(/•/g, '')[0];
        const premine = document.getElementById('etch-rune-premine').value || '0';
        const enableMint = document.getElementById('etch-enable-mint').checked;
        const cap = enableMint ? document.getElementById('etch-rune-cap').value : undefined;
        const amountPerMint = enableMint ? document.getElementById('etch-rune-amount').value : undefined;
        
        if (!name || name.length < 3) {
            throw new Error('Rune name must be at least 3 characters');
        }
        
        // Request password
        const password = await requestPassword('✨ Confirm Etch Rune');
        if (!password) {
            throw new Error('Password required');
        }
        
        showLoading('✨ Etching rune...');
        
        // Prepare etch transaction
        const prepareRes = await fetch('https://kraywallet-backend.onrender.com/api/runes/etch/prepare', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                address,
                name,
                symbol,
                divisibility: etchDivisibility,
                premine,
                cap,
                amountPerMint,
                turbo: false,
                feeRate: 10
            })
        });
        
        const prepareData = await prepareRes.json();
        if (!prepareData.success) {
            throw new Error(prepareData.error || 'Failed to prepare etch transaction');
        }
        
        // Sign PSBT
        const signedPsbt = await signPsbtWithPassword(prepareData.psbt, password);
        if (!signedPsbt) {
            throw new Error('Failed to sign transaction');
        }
        
        // Finalize
        const finalizeRes = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signedPsbt })
        });
        
        const finalizeData = await finalizeRes.json();
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize transaction');
        }
        
        // Broadcast
        const broadcastRes = await fetch('https://kraywallet-backend.onrender.com/api/wallet/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ txHex: finalizeData.txHex })
        });
        
        const broadcastData = await broadcastRes.json();
        if (!broadcastData.success) {
            throw new Error(broadcastData.error || 'Failed to broadcast transaction');
        }
        
        hideLoading();
        
        showNotification(`🎉 Rune "${name}" etched! TX: ${broadcastData.txid.slice(0, 12)}...`, 'success');
        
        closeEtchRuneModal();
        loadWalletData();
        
    } catch (error) {
        console.error('❌ Etch error:', error);
        hideLoading();
        showNotification('❌ ' + error.message, 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.innerHTML = '✨ Etch Rune';
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 📖 ADDRESS BOOK FEATURE
// ═══════════════════════════════════════════════════════════════════════════

const ADDRESS_BOOK_KEY = 'addressBook';
let addressBookState = {
    contacts: [],
    editingContact: null,
    searchQuery: ''
};

// Load contacts from localStorage
function loadAddressBook() {
    try {
        const stored = localStorage.getItem(ADDRESS_BOOK_KEY);
        if (stored) {
            addressBookState.contacts = JSON.parse(stored);
        }
    } catch (error) {
        console.error('Error loading address book:', error);
        addressBookState.contacts = [];
    }
}

// Save contacts to localStorage
function saveAddressBook() {
    try {
        localStorage.setItem(ADDRESS_BOOK_KEY, JSON.stringify(addressBookState.contacts));
    } catch (error) {
        console.error('Error saving address book:', error);
    }
}

// Validate Bitcoin address
function isValidBitcoinAddress(address) {
    if (!address) return false;
    const taprootRegex = /^bc1p[a-zA-HJ-NP-Z0-9]{58}$/;
    const segwitRegex = /^bc1q[a-zA-HJ-NP-Z0-9]{38,58}$/;
    const legacyRegex = /^[13][a-km-zA-HJ-NP-Z1-9]{25,34}$/;
    return taprootRegex.test(address) || segwitRegex.test(address) || legacyRegex.test(address);
}

// Generate simple ID
function generateContactId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

// Show Address Book screen
function showAddressBookScreen() {
    loadAddressBook();
    addressBookState.searchQuery = '';
    renderAddressBook();
    showScreen('address-book');
}

// Render contacts list
function renderAddressBook() {
    const listEl = document.getElementById('contact-list');
    const emptyEl = document.getElementById('address-book-empty');
    const searchInput = document.getElementById('contact-search');
    
    if (!listEl) return;
    
    const query = addressBookState.searchQuery.toLowerCase();
    const filtered = addressBookState.contacts.filter(c =>
        c.name.toLowerCase().includes(query) ||
        c.address.toLowerCase().includes(query)
    );
    
    if (filtered.length === 0) {
        listEl.innerHTML = '';
        if (emptyEl) emptyEl.style.display = 'block';
        return;
    }
    
    if (emptyEl) emptyEl.style.display = 'none';
    
    listEl.innerHTML = filtered.map(contact => `
        <div class="address-book-contact" data-id="${contact.id}" style="
            display: flex;
            align-items: center;
            padding: 14px;
            background: rgba(255,255,255,0.03);
            border-radius: 12px;
            cursor: pointer;
            border: 1px solid #222;
        ">
            <div style="
                width: 42px;
                height: 42px;
                border-radius: 21px;
                background: #222;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-right: 12px;
                font-weight: 700;
                color: #fff;
                font-size: 16px;
            ">${contact.name.charAt(0).toUpperCase()}</div>
            <div style="flex: 1; min-width: 0;">
                <div style="color: #fff; font-weight: 600; font-size: 15px; margin-bottom: 2px;">${contact.name}</div>
                <div style="color: #888; font-size: 12px; font-family: monospace; overflow: hidden; text-overflow: ellipsis;">
                    ${contact.address.slice(0, 12)}...${contact.address.slice(-8)}
                </div>
            </div>
            <div style="display: flex; gap: 6px;">
                <button class="copy-contact-btn" data-address="${contact.address}" style="
                    width: 32px; height: 32px; border-radius: 8px;
                    background: rgba(255,255,255,0.05); border: none;
                    cursor: pointer; display: flex; align-items: center; justify-content: center;
                " title="Copy address">📋</button>
                <button class="edit-contact-btn" data-id="${contact.id}" style="
                    width: 32px; height: 32px; border-radius: 8px;
                    background: rgba(59,130,246,0.1); border: none;
                    cursor: pointer; display: flex; align-items: center; justify-content: center;
                " title="Edit">✏️</button>
                <button class="delete-contact-btn" data-id="${contact.id}" style="
                    width: 32px; height: 32px; border-radius: 8px;
                    background: rgba(239,68,68,0.1); border: none;
                    cursor: pointer; display: flex; align-items: center; justify-content: center;
                " title="Delete">🗑️</button>
            </div>
        </div>
    `).join('');
    
    // Add click listeners
    listEl.querySelectorAll('.copy-contact-btn').forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const address = btn.dataset.address;
            navigator.clipboard.writeText(address);
            showNotification('📋 Address copied!', 'success');
        });
    });
    
    listEl.querySelectorAll('.edit-contact-btn').forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const id = btn.dataset.id;
            const contact = addressBookState.contacts.find(c => c.id === id);
            if (contact) {
                openContactModal(contact);
            }
        });
    });
    
    listEl.querySelectorAll('.delete-contact-btn').forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const id = btn.dataset.id;
            const contact = addressBookState.contacts.find(c => c.id === id);
            if (contact && confirm(`Delete "${contact.name}"?`)) {
                addressBookState.contacts = addressBookState.contacts.filter(c => c.id !== id);
                saveAddressBook();
                renderAddressBook();
                showNotification('Contact deleted', 'success');
            }
        });
    });
}

// Open add/edit modal
function openContactModal(contact = null) {
    addressBookState.editingContact = contact;
    const modal = document.getElementById('contact-modal');
    const titleEl = document.getElementById('contact-modal-title');
    const nameInput = document.getElementById('contact-name-input');
    const addressInput = document.getElementById('contact-address-input');
    
    if (!modal) return;
    
    if (contact) {
        titleEl.textContent = 'Edit Contact';
        nameInput.value = contact.name;
        addressInput.value = contact.address;
    } else {
        titleEl.textContent = 'Add Contact';
        nameInput.value = '';
        addressInput.value = '';
    }
    
    modal.classList.remove('hidden');
    nameInput.focus();
}

// Close modal
function closeContactModal() {
    const modal = document.getElementById('contact-modal');
    if (modal) modal.classList.add('hidden');
    addressBookState.editingContact = null;
}

// Save contact
function saveContact() {
    const nameInput = document.getElementById('contact-name-input');
    const addressInput = document.getElementById('contact-address-input');
    
    const name = nameInput.value.trim();
    const address = addressInput.value.trim();
    
    if (!name) {
        showNotification('Please enter a name', 'error');
        return;
    }
    
    if (!isValidBitcoinAddress(address)) {
        showNotification('Please enter a valid Bitcoin address', 'error');
        return;
    }
    
    // Check for duplicate address (excluding current if editing)
    const isDuplicate = addressBookState.contacts.some(c => 
        c.address === address && c.id !== addressBookState.editingContact?.id
    );
    
    if (isDuplicate) {
        showNotification('This address is already saved', 'error');
        return;
    }
    
    if (addressBookState.editingContact) {
        // Edit existing
        const idx = addressBookState.contacts.findIndex(c => c.id === addressBookState.editingContact.id);
        if (idx !== -1) {
            addressBookState.contacts[idx].name = name;
            addressBookState.contacts[idx].address = address;
        }
    } else {
        // Add new
        addressBookState.contacts.push({
            id: generateContactId(),
            name,
            address,
            createdAt: Date.now()
        });
    }
    
    saveAddressBook();
    renderAddressBook();
    closeContactModal();
    showNotification(addressBookState.editingContact ? 'Contact updated!' : 'Contact added!', 'success');
}

// Get contacts (for use in Send screens)
function getAddressBookContacts() {
    loadAddressBook();
    return addressBookState.contacts;
}

// Event listeners for Address Book
document.getElementById('address-book-btn')?.addEventListener('click', showAddressBookScreen);
document.getElementById('back-from-address-book-btn')?.addEventListener('click', () => showScreen('settings'));
document.getElementById('add-contact-btn')?.addEventListener('click', () => openContactModal());
document.getElementById('close-contact-modal')?.addEventListener('click', closeContactModal);
document.getElementById('cancel-contact-btn')?.addEventListener('click', closeContactModal);
document.getElementById('save-contact-btn')?.addEventListener('click', saveContact);

document.getElementById('contact-search')?.addEventListener('input', (e) => {
    addressBookState.searchQuery = e.target.value;
    renderAddressBook();
});
