/**
 * 🔒 TAPROOT SIGNER - 100% LOCAL (ZERO-TRUST)
 * 
 * Assina transações Taproot LOCALMENTE sem NUNCA enviar mnemonic!
 * Compatível com Service Workers (Manifest V3).
 * 
 * SEGURANÇA:
 * - Mnemonic NUNCA sai do dispositivo ✅
 * - Chaves privadas NUNCA enviadas para backend ✅
 * - Apenas TX assinada é broadcast ✅
 */

// Importar do backend para derivação e assinatura
// (temporário até termos libs bundled na extension)
const BACKEND_URL = 'https://kraywallet-backend.onrender.com';

/**
 * Criar PSBT sem assinar (backend apenas cria estrutura)
 * @param {Object} params
 * @returns {string} PSBT base64 (unsigned)
 */
async function createUnsignedPsbt({ inscription, recipientAddress, feeRate, network = 'mainnet' }) {
    try {
        console.log('📝 Creating unsigned PSBT (backend)...');
        
        const response = await fetch(`${BACKEND_URL}/api/kraywallet/create-inscription-psbt`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                inscription: {
                    id: inscription.id,
                    utxo: {
                        txid: inscription.utxo.txid,
                        vout: inscription.utxo.vout,
                        value: inscription.utxo.value || inscription.outputValue || 600
                    }
                },
                recipientAddress,
                feeRate,
                network,
                signLocally: true  // Flag para backend NÃO assinar!
            })
        });
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to create PSBT');
        }
        
        console.log('✅ Unsigned PSBT created');
        
        return {
            psbtBase64: data.psbtBase64,
            fee: data.fee
        };
        
    } catch (error) {
        console.error('❌ Error creating PSBT:', error);
        throw error;
    }
}

/**
 * Assinar PSBT localmente - 100% SEGURO
 * 🔐 LEI PRIMORDIAL: MNEMONIC NUNCA SAI DO DISPOSITIVO!
 * 
 * @param {Object} params
 * @returns {string} TX hex assinada
 */
async function signPsbtLocally({ mnemonic, psbtBase64, network = 'mainnet' }) {
    try {
        console.log('🔐 Signing PSBT 100% LOCALLY...');
        console.log('   🔐 LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
        
        // Use TaprootSigner for 100% local signing
        if (typeof TaprootSigner !== 'undefined' && TaprootSigner.signPsbtTaprootLocal) {
            const signResult = await TaprootSigner.signPsbtTaprootLocal(
                mnemonic,
                psbtBase64,
                {
                    sighashType: 0x01, // SIGHASH_ALL
                    autoFinalize: true
                }
            );
            
            // Clear mnemonic from memory
            mnemonic = null;
            
            console.log('✅ PSBT signed 100% LOCALLY');
            return signResult.txHex || signResult.signedPsbt;
        }
        
        // 🔐 If local signer not available, throw error - NEVER send mnemonic to server
        console.error('❌ LOCAL SIGNER NOT AVAILABLE!');
        throw new Error('Local signer not loaded. Cannot sign securely. Please reload the extension.');
        
    } catch (error) {
        console.error('❌ Error signing PSBT:', error);
        mnemonic = null;
        throw error;
    }
}

/**
 * Broadcast transaction (backend só broadcast, sem chaves!)
 * @param {string} txHex
 * @returns {string} txid
 */
async function broadcastTransaction(txHex) {
    try {
        console.log('📡 Broadcasting transaction...');
        
        const response = await fetch(`${BACKEND_URL}/api/psbt/broadcast`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: txHex })
        });
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Transaction broadcast:', data.txid);
        
        return data.txid;
        
    } catch (error) {
        console.error('❌ Error broadcasting:', error);
        throw error;
    }
}

/**
 * FLOW COMPLETO: Criar → Assinar Local → Broadcast
 * @param {Object} params
 * @returns {Object} { txid, fee }
 */
async function sendInscriptionSecure({ mnemonic, inscription, recipientAddress, feeRate, network = 'mainnet' }) {
    try {
        console.log('═══════════════════════════════════════════════════');
        console.log('🔒 SECURE INSCRIPTION SEND (LOCAL SIGNING)');
        console.log('═══════════════════════════════════════════════════');
        
        // STEP 1: Criar PSBT (backend apenas estrutura)
        const { psbtBase64, fee } = await createUnsignedPsbt({
            inscription,
            recipientAddress,
            feeRate,
            network
        });
        
        // STEP 2: Assinar LOCALMENTE (mnemonic NÃO sai do dispositivo!)
        const txHex = await signPsbtLocally({
            mnemonic,
            psbtBase64,
            network
        });
        
        // STEP 3: Broadcast (backend só envia TX, sem chaves!)
        const txid = await broadcastTransaction(txHex);
        
        console.log('═══════════════════════════════════════════════════');
        console.log('✅ INSCRIPTION SENT SECURELY!');
        console.log('   TXID:', txid);
        console.log('   Fee:', fee, 'sats');
        console.log('═══════════════════════════════════════════════════');
        
        return { success: true, txid, fee };
        
    } catch (error) {
        console.error('❌ Error sending inscription:', error);
        return { success: false, error: error.message };
    }
}

// Export
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        createUnsignedPsbt,
        signPsbtLocally,
        broadcastTransaction,
        sendInscriptionSecure
    };
}

