/**
 * 🔒 LOCAL SIGNER - 100% SECURE (ZERO-TRUST)
 * 
 * Assina transações Bitcoin Taproot LOCALMENTE.
 * Mnemonic NUNCA sai do dispositivo do usuário!
 * 
 * SEGURANÇA:
 * ✅ Mnemonic permanece no dispositivo
 * ✅ Chaves privadas NUNCA enviadas
 * ✅ Apenas TX assinada é compartilhada
 * 
 * Este é o PADRÃO usado por:
 * - MetaMask
 * - Phantom  
 * - Unisat
 * - Ledger
 * - Todas as wallets sérias!
 */

/**
 * IMPORTANTE: Por enquanto, ainda usamos backend para assinatura
 * mas isolamos essa dependência neste arquivo.
 * 
 * TODO: Integrar bitcoinjs-lib bundled para assinatura 100% local
 */

const BACKEND_URL = 'https://kraywallet-backend.onrender.com';

/**
 * Criar PSBT para envio de inscription (backend apenas cria estrutura)
 */
async function createInscriptionPsbt({ inscription, recipientAddress, senderAddress, feeRate }) {
    try {
        console.log('📝 Creating PSBT structure (backend)...');
        console.log('   ✅ No mnemonic sent!');
        
        const response = await fetch(`${BACKEND_URL}/api/kraywallet/create-inscription-psbt`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                inscription: {
                    id: inscription.id,
                    utxo: {
                        txid: inscription.utxo.txid,
                        vout: inscription.utxo.vout,
                        value: inscription.utxo.value || inscription.outputValue || 600
                    }
                },
                recipientAddress,
                senderAddress,
                feeRate,
                network: 'mainnet'
            })
        });
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to create PSBT');
        }
        
        console.log('✅ PSBT structure created (unsigned)');
        
        return {
            psbtBase64: data.psbtBase64,
            fee: data.fee
        };
        
    } catch (error) {
        console.error('❌ Error creating PSBT:', error);
        throw error;
    }
}

/**
 * Assinar PSBT localmente - 100% SEGURO
 * 
 * 🔐 LEI PRIMORDIAL: MNEMONIC NUNCA SAI DO DISPOSITIVO!
 * Usa TaprootSigner.signPsbtTaprootLocal para assinatura 100% local
 */
async function signPsbtLocal({ mnemonic, psbtBase64, inputsToSign = null, sighashType = 0x01 }) {
    try {
        console.log('🔐 Signing PSBT 100% LOCALLY...');
        console.log('   ✅ LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
        
        // 🔐 Use TaprootSigner for 100% local signing
        if (typeof TaprootSigner !== 'undefined' && TaprootSigner.signPsbtTaprootLocal) {
            const signResult = await TaprootSigner.signPsbtTaprootLocal(
                mnemonic,
                psbtBase64,
                {
                    inputsToSign: inputsToSign,
                    sighashType: sighashType,
                    autoFinalize: true
                }
            );
            
            // 🗑️ Limpar mnemonic da memória IMEDIATAMENTE
            mnemonic = null;
            
            console.log('✅ PSBT signed LOCALLY');
            
            return signResult.txHex || signResult.signedPsbt;
        } else {
            // 🚨 NEVER fall back to server - throw error instead!
            throw new Error('Local signer not available. Cannot sign securely. Mnemonic protection ACTIVE!');
        }
        
    } catch (error) {
        console.error('❌ Error signing PSBT:', error);
        // 🗑️ Limpar mnemonic mesmo em erro
        mnemonic = null;
        throw error;
    }
}

/**
 * Broadcast transaction (backend apenas envia, sem chaves!)
 */
async function broadcastTransaction(txHex) {
    try {
        console.log('📡 Broadcasting transaction...');
        console.log('   ✅ Only sending signed TX (no keys!)');
        
        const response = await fetch(`${BACKEND_URL}/api/psbt/broadcast`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: txHex })
        });
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to broadcast');
        }
        
        console.log('✅ Transaction broadcast!');
        console.log('   TXID:', data.txid);
        
        return data.txid;
        
    } catch (error) {
        console.error('❌ Error broadcasting:', error);
        throw error;
    }
}

/**
 * FLOW COMPLETO SEGURO:
 * 1. Backend cria PSBT (sem mnemonic)
 * 2. Extension assina localmente (mnemonic não sai - TODO)
 * 3. Backend broadcast (apenas TX assinada)
 */
async function sendInscriptionSecure({ mnemonic, inscription, recipientAddress, senderAddress, feeRate }) {
    try {
        console.log('═══════════════════════════════════════════════════');
        console.log('🔒 SECURE SEND INSCRIPTION');
        console.log('═══════════════════════════════════════════════════');
        
        // STEP 1: Criar PSBT (backend SEM mnemonic!)
        const { psbtBase64, fee } = await createInscriptionPsbt({
            inscription,
            recipientAddress,
            senderAddress,
            feeRate
        });
        
        // STEP 2: Assinar localmente (mnemonic não deve sair!)
        const txHex = await signPsbtLocal({
            mnemonic,
            psbtBase64
        });
        
        // STEP 3: Broadcast (backend só envia TX)
        const txid = await broadcastTransaction(txHex);
        
        console.log('═══════════════════════════════════════════════════');
        console.log('✅ INSCRIPTION SENT!');
        console.log('   TXID:', txid);
        console.log('   Fee:', fee, 'sats');
        console.log('   View: https://mempool.space/tx/' + txid);
        console.log('═══════════════════════════════════════════════════');
        
        return {
            success: true,
            txid,
            fee
        };
        
    } catch (error) {
        console.error('❌ Error:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * TODO: Implementar assinatura 100% local
 * 
 * Próximos passos:
 * 1. Bundle bitcoinjs-lib na extension
 * 2. Implementar derivação BIP32/BIP86 local
 * 3. Implementar assinatura Schnorr local
 * 4. Remover COMPLETAMENTE dependência de backend para signing
 * 
 * Referência:
 * - bitcoinjs-lib: https://github.com/bitcoinjs/bitcoinjs-lib
 * - tiny-secp256k1: https://github.com/bitcoinjs/tiny-secp256k1
 * - bip32: https://github.com/bitcoinjs/bip32
 * - bip39: https://github.com/bitcoinjs/bip39
 */

// Export
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        createInscriptionPsbt,
        signPsbtLocal,
        broadcastTransaction,
        sendInscriptionSecure
    };
}

