/**
 * 🔥 MyWallet Extension - Background Script (REAL WALLET)
 * Usa bibliotecas reais para gerar mnemonic e endereços Taproot
 * 
 * VERSION: 2025-12-18-FIX-SIGNER
 */

console.log('🚀 BACKGROUND SCRIPT STARTING - VERSION 2025-12-18-FIX-SIGNER');

// 🔒 Import local signer bundle (100% secure!)
// Note: Service Workers don't have 'window', so we need to handle the global differently
let TaprootSigner = null;
try {
    console.log('📦 Importing taproot-signer.bundle.js...');
    importScripts('taproot-signer.bundle.js');
    console.log('✅ importScripts completed');
    
    // Debug: Check what's available
    console.log('🔍 DEBUG: typeof self =', typeof self);
    console.log('🔍 DEBUG: typeof self.TaprootSigner =', typeof self?.TaprootSigner);
    console.log('🔍 DEBUG: typeof globalThis =', typeof globalThis);
    console.log('🔍 DEBUG: typeof globalThis.TaprootSigner =', typeof globalThis?.TaprootSigner);
    console.log('🔍 DEBUG: typeof this.TaprootSigner =', typeof this?.TaprootSigner);
    
    // The bundle exports to module.exports or self.TaprootSigner
    // In Service Worker context, we need to access it from self or the module export
    if (typeof self !== 'undefined' && self.TaprootSigner) {
        TaprootSigner = self.TaprootSigner;
        console.log('✅ TaprootSigner loaded from self');
    } else if (typeof globalThis !== 'undefined' && globalThis.TaprootSigner) {
        TaprootSigner = globalThis.TaprootSigner;
        console.log('✅ TaprootSigner loaded from globalThis');
    } else if (typeof this !== 'undefined' && this.TaprootSigner) {
        TaprootSigner = this.TaprootSigner;
        console.log('✅ TaprootSigner loaded from this');
    }
    
    if (TaprootSigner && TaprootSigner.signPsbtTaprootLocal) {
    console.log('🔒 Local signer loaded! Mnemonic will NEVER leave device!');
        console.log('   ✅ TaprootSigner.signPsbtTaprootLocal available');
        console.log('   📋 TaprootSigner keys:', Object.keys(TaprootSigner));
    } else {
        console.warn('⚠️  TaprootSigner loaded but signPsbtTaprootLocal not found');
        console.warn('   TaprootSigner value:', TaprootSigner);
    }
} catch (error) {
    console.error('❌ Failed to load local signer:', error.message);
    console.error('   Full error:', error);
}

let walletState = {
    unlocked: false,
    address: null,
    publicKey: null,
    lockedAt: null // Timestamp do último lock
};

// 🔒 NUNCA armazenar mnemonic/privateKey na memória!
// Só descriptografar quando necessário para assinar, depois descartar imediatamente

// Pending PSBT request (aguardando confirmação do usuário no popup)
let pendingPsbtRequest = null;

// Pending Message request (aguardando confirmação do usuário no popup)
let pendingMessageRequest = null;

// Flag para evitar múltiplos popups simultâneos
let isPopupOpening = false;

// ==========================================
// 🔒 AUTO-LOCK SYSTEM (usando chrome.alarms para persistir)
// ==========================================
const AUTOLOCK_ALARM_NAME = 'kraywallet-autolock';
let autolockTimeout = 15; // Default: 15 minutes

// Load autolock setting from storage
(async function loadAutolockSetting() {
    try {
        const result = await chrome.storage.local.get(['autolockTimeout']);
        if (result.autolockTimeout !== undefined) {
            autolockTimeout = result.autolockTimeout;
            console.log(`🔒 Auto-lock timeout loaded: ${autolockTimeout} minutes`);
        }
    } catch (error) {
        console.error('❌ Error loading autolock setting:', error);
    }
})();

// Reset auto-lock timer (usando chrome.alarms API)
function resetAutolockTimer() {
    // Clear existing alarm
    chrome.alarms.clear(AUTOLOCK_ALARM_NAME);
    
    // Don't set alarm if wallet is locked or timeout is 0 (never)
    if (!walletState.unlocked || autolockTimeout === 0) {
        console.log('⏰ Auto-lock timer skipped (wallet locked or timeout is 0)');
        return;
    }
    
    // Set new alarm (chrome.alarms persiste mesmo quando Service Worker é terminado)
    chrome.alarms.create(AUTOLOCK_ALARM_NAME, {
        delayInMinutes: autolockTimeout
    });
    
    console.log(`⏰ Auto-lock alarm set: ${autolockTimeout} minutes`);
}

// Listen for alarm events
chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === AUTOLOCK_ALARM_NAME) {
        console.log('🔒 Auto-locking wallet due to inactivity...');
        lockWallet();
    }
});

// Lock wallet function
function lockWallet() {
    console.log('🔒 Locking wallet...');
    
    // Clear sensitive data from memory
    // walletState.mnemonic removido por segurança (nunca guardar em memória)
    walletState.unlocked = false;
    walletState.lockedAt = Date.now();
    
    // ✅ CRÍTICO: Limpar session storage também
    chrome.storage.session.remove(['walletUnlocked', 'walletAddress', 'walletPublicKey']).catch(err => {
        console.warn('⚠️  Could not clear session storage:', err);
    });
    
    // Stop keep-alive (Service Worker can be terminated now)
    stopKeepAlive();
    
    // Clear auto-lock alarm
    chrome.alarms.clear(AUTOLOCK_ALARM_NAME);
    
    console.log('✅ Wallet locked successfully');
    
    // Notify all open popups/tabs
    chrome.runtime.sendMessage({
        action: 'walletLocked'
    }).catch(() => {
        // Ignore error if no listeners
    });
}

// ==========================================
// 🔥 KEEP-ALIVE: Prevenir Service Worker de ser terminado
// ==========================================
// Chrome termina Service Workers após 30s de inatividade
// Precisamos mantê-lo vivo enquanto a wallet está desbloqueada
// para manter o mnemonic na memória (segurança)

let keepAliveInterval = null;
const KEEPALIVE_INTERVAL_NAME = 'kraywallet-keepalive';

function startKeepAlive() {
    // Usar chrome.alarms para manter Service Worker vivo
    // Dispara a cada 20 segundos (antes dos 30s de timeout)
    chrome.alarms.create(KEEPALIVE_INTERVAL_NAME, {
        periodInMinutes: 0.33 // ~20 segundos
    });
    console.log('🔄 Keep-alive started (prevents Service Worker termination)');
}

function stopKeepAlive() {
    chrome.alarms.clear(KEEPALIVE_INTERVAL_NAME);
    console.log('⏹️  Keep-alive stopped');
}

// Listen to keep-alive alarms (isso mantém o Service Worker vivo)
chrome.alarms.onAlarm.addListener((alarm) => {
    if (alarm.name === KEEPALIVE_INTERVAL_NAME) {
        // Não precisa fazer nada, só o fato de receber o alarm mantém SW vivo
        console.log('💓 Keep-alive ping');
    }
});

// ==========================================
// 🔥 INICIALIZAÇÃO - Verificar se wallet existe
// ==========================================
(async function initWallet() {
    try {
        console.log('🔥 Background script starting...');
        
        // Verificar se wallet existe (mas não desbloqueá-la automaticamente)
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        if (result.walletEncrypted) {
            // Wallet existe, mas está LOCKED (precisa de senha)
            walletState.unlocked = false;
            walletState.lockedAt = Date.now();
            console.log('🔒 Wallet exists but is locked (requires password)');
        } else {
            console.log('ℹ️  No wallet found (needs to be created)');
        }
    } catch (error) {
        console.error('❌ Error initializing wallet:', error);
    }
})();

// Listener de mensagens
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    console.log('📨 Message received:', request.action);
    
    handleMessage(request, sender)
        .then(sendResponse)
        .catch(error => {
            console.error('Error handling message:', error);
            sendResponse({ success: false, error: error.message });
        });
    
    return true; // Async response
});

// Handler principal de mensagens
async function handleMessage(request, sender) {
    const { action, data } = request;
    
    switch (action) {
        case 'generateWallet':
            return await generateWallet(data);
        
        case 'restoreWallet':
            return await restoreWallet(data);
        
        case 'getWalletInfo':
            return await getWalletInfo();
        
        case 'getPendingPsbt':
            // ✅ CRITICAL FIX: Verificar TANTO na memória QUANTO no storage
            // (Service Worker pode ter sido terminado e reiniciado)
            if (!pendingPsbtRequest) {
                console.log('⚠️  pendingPsbtRequest not in memory, checking storage...');
                const storage = await chrome.storage.local.get(['pendingPsbtRequest']);
                if (storage.pendingPsbtRequest) {
                    pendingPsbtRequest = storage.pendingPsbtRequest;
                    console.log('✅ pendingPsbtRequest restored from storage!');
                }
            }
            return {
                success: true,
                pending: pendingPsbtRequest
            };
        
        case 'getPendingPayment':
            // Restaurar pendingPaymentRequest do storage se não estiver em memória
            if (!pendingPaymentRequest) {
                console.log('⚠️  pendingPaymentRequest not in memory, checking storage...');
                const storage = await chrome.storage.local.get(['pendingPaymentRequest']);
                if (storage.pendingPaymentRequest) {
                    pendingPaymentRequest = storage.pendingPaymentRequest;
                    console.log('✅ pendingPaymentRequest restored from storage!');
                }
            }
            return {
                success: true,
                pending: pendingPaymentRequest
            };
        
        case 'confirmPsbtSign':
            return await confirmPsbtSign(data);
        
        case 'cancelPsbtSign':
            // Limpar pending request DA MEMÓRIA E DO STORAGE
            pendingPsbtRequest = null;
            await chrome.storage.local.remove('pendingPsbtRequest');
            await chrome.storage.local.remove('psbtSignResult');
            console.log('❌ PSBT signing cancelled by user (cleared from memory and storage)');
            return { success: true, cancelled: true };
        
        case 'emergencyCleanStorage':
            // 🆘 LIMPEZA DE EMERGÊNCIA - Remove todos os PSBTs corrompidos
            console.log('🆘 EMERGENCY CLEAN: Removing ALL PSBT data from storage...');
            pendingPsbtRequest = null;
            await chrome.storage.local.remove([
                'pendingPsbtRequest',
                'psbtSignResult',
                'pendingMarketListing',
                'isCreatingListing'
            ]);
            console.log('✅ Emergency clean completed!');
            return { success: true, cleaned: true };
        
        case 'signPsbt':
            return await signPsbt(data);
        
        case 'signMessage':
            return await signMessage(data);
        
        case 'signMessageRequireConfirmation':
            return await signMessageRequireConfirmation(data);
        
        case 'signL2Transaction':
            // New L2 transaction signing with display metadata
            return await signL2Transaction(data);
        
        case 'getActiveNetwork':
            return await getActiveNetwork();
        
        case 'signMessageWithPassword':
            return await signMessageWithPassword(data);
        
        case 'sendPayment':
            return await sendPayment(data);
        
        case 'createOffer':
            return await createOffer(data);
        
        case 'buyAtomicSwap':
            return await buyAtomicSwap(data);
        
        case 'cancelListing':
            return await cancelListing(data);
        
        case 'updateListingPrice':
            return await updateListingPrice(data);
        
        // 🚀 UNIFIED: Create AND Sign in ONE step!
        case 'createAndSignListing':
            return await createAndSignListing(data);
        
        // 🛒 BUY NOW MODE (KrayWallet Atomic Swap) - LEGACY
        case 'createBuyNowListing':
            return await createBuyNowListing(data);
        
        case 'buyNow':
            return await buyNow(data);
        
        case 'acceptBuyNow':
            return await acceptBuyNow(data);
        
        case 'updateListingPrice':
            return await updateListingPrice(data);
        
        case 'sendBitcoin':
            return await sendBitcoin(data);
        
        case 'sendInscription':
            return await sendInscription(data);
        
        case 'connect':
            return await connect();
        
        case 'openPopup':
            // Abrir popup da extension
            try {
                await chrome.action.openPopup();
                return { success: true };
            } catch (error) {
                console.error('Cannot open popup:', error);
                return { success: false, error: error.message };
            }
        
        case 'getAccounts':
            return await getAccounts();
        
        case 'getPublicKey':
            return await getPublicKey();
        
        case 'getInscriptions':
            return await getInscriptions(data);
        
        case 'getInscriptionDetails':
            return await getInscriptionDetails(request);
        
        case 'addPendingInscription':
            return await addPendingInscription(data);
        
        case 'removePendingInscription':
            return await removePendingInscription(data);
        
        case 'reloadInscriptions':
            // Forçar reload das inscriptions (limpar cache se necessário)
            console.log('🔄 Forcing inscriptions reload...');
            return { success: true, message: 'Inscriptions reload triggered' };
        
        case 'getRunes':
            return await getRunes(data);
        
        case 'signRunePSBT':
            return await signRunePSBT(request.psbt);
        
        case 'broadcastTransaction':
            return await broadcastTransaction(request.hex);
        
        case 'unlockWallet':
            return await unlockWalletAction(data);
        
        case 'lockWallet':
            lockWallet();
            return { success: true, message: 'Wallet locked successfully' };
        
        case 'setAutolockTimeout':
            return await setAutolockTimeout(data.timeout);
        
        case 'resetAutolockTimer':
            // Called on user activity
            resetAutolockTimer();
            return { success: true };
        
        case 'decryptWallet':
            return await decryptWalletAction(data.password);
        
        case 'getMnemonic':
            // 🔐 Get mnemonic for "View Recovery Phrase" feature
            console.log('🔑 getMnemonic action called');
            return await decryptWalletAction(data.password);
        
        case 'checkWalletStatus':
            return await checkWalletStatus();
        
        case 'switchToLightning':
            // Mudar para layer Lightning
            try {
                // Salvar preferência de layer
                await chrome.storage.local.set({ selectedNetwork: 'lightning' });
                console.log('✅ Switched to Lightning layer');
                return { success: true };
            } catch (error) {
                console.error('❌ Error switching to Lightning:', error);
                return { success: false, error: error.message };
            }
        
        case 'signL2Message':
            // Sign L2 transaction message with Schnorr
            // Uses same approach as signMessageWithPassword - decrypt wallet temporarily
            return await signL2MessageAction(data);
        
        case 'signL2MessageWithPassword':
            // Sign L2 message with explicit password (for popup flow)
            return await signL2MessageWithPasswordAction(data)
        
        case 'signPsbtWithPassword':
            // Sign PSBT with password (for L2 withdrawals)
            return await signPsbtWithPasswordAction(data);
        
        // REMOVED: signPsbtForListing - Now done 100% locally in popup!
        // LEI PRIMORDIAL: Mnemonic NEVER leaves device!
        
        // ═══════════════════════════════════════════════════════════════
        // 💬 KRAYCHAT E2E ENCRYPTION (Secure like mobile!)
        // ═══════════════════════════════════════════════════════════════
        
        case 'activateChatSession':
            return await activateChatSessionAction(data);
        
        case 'isChatSessionActive':
            return isChatSessionActiveAction();
        
        case 'clearChatSession':
            return clearChatSessionAction();
        
        case 'getChatPublicKey':
            return await getChatPublicKeyAction(data);
        
        case 'encryptChatMessage':
            return await encryptChatMessageAction(data);
        
        case 'decryptChatMessage':
            return await decryptChatMessageAction(data);
        
        default:
            throw new Error(`Unknown action: ${action}`);
    }
}

// ==========================================
// 🔥 WALLET REAL COM BIP39
// ==========================================

// Gerar Wallet REAL (usando backend)
async function generateWallet({ wordCount, password }) {
    try {
        console.log('🔑 ========== GENERATING REAL WALLET ==========');
        console.log('   Word count:', wordCount);
        console.log('   Password length:', password?.length);
        
        // Chamar backend para gerar mnemonic e endereço real
        console.log('📡 Calling backend API...');
        console.log('   URL: https://kraywallet-backend.onrender.com/api/kraywallet/generate');
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/generate', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ wordCount, password }) // ⚡ Enviar password para LND!
        });
        
        console.log('📡 Response status:', response.status);
        console.log('   Response ok:', response.ok);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Backend returned error:', errorText);
            throw new Error('Failed to generate wallet from backend: ' + errorText);
        }
        
        const data = await response.json();
        console.log('📦 Backend response:', data);
        
        if (!data.success) {
            console.error('❌ Backend returned success=false:', data.error);
            throw new Error(data.error || 'Failed to generate wallet');
        }
        
        const { mnemonic, address, publicKey } = data;
        
        console.log('✅ Real mnemonic generated via backend');
        console.log('   Mnemonic words:', mnemonic?.split(' ').length);
        console.log('✅ Real Taproot address derived:', address);
        console.log('   Public key:', publicKey);
        
        // Criptografar e salvar
        console.log('🔐 Encrypting wallet data...');
        const encrypted = await encryptData({ mnemonic, address, publicKey }, password);
        console.log('✅ Wallet encrypted');
        
        // Atualizar estado (🔒 SEM MNEMONIC por segurança!)
        console.log('💾 Updating wallet state...');
        walletState = {
            unlocked: true,
            address,
            publicKey,
            lockedAt: null
        };
        console.log('✅ Wallet state updated (mnemonic NOT stored in memory)');
        
        // Start auto-lock timer
        resetAutolockTimer();
        
        // Salvar encrypted wallet E wallet state (sem mnemonic)
        console.log('💾 Saving to chrome.storage.local...');
        await chrome.storage.local.set({ 
            walletEncrypted: encrypted,
            walletState: {
                address,
                publicKey
            },
            tempPassword: password // ⚠️ Para desenvolvimento! Em produção, NÃO salvar senha!
        });
        
        console.log('✅ Wallet saved to storage');
        console.log('🎉 ========== WALLET GENERATION COMPLETE ==========');
        
        return {
            success: true,
            mnemonic,
            address
        };
    } catch (error) {
        console.error('❌ ========== ERROR GENERATING WALLET ==========');
        console.error('   Error:', error);
        console.error('   Message:', error.message);
        console.error('   Stack:', error.stack);
        console.error('================================================');
        return {
            success: false,
            error: error.message
        };
    }
}

// Restaurar Wallet REAL (usando backend)
async function restoreWallet({ mnemonic, password }) {
    try {
        console.log('🔄 Restoring REAL wallet via backend API...');
        
        // Chamar backend para validar e derivar endereço
        const response = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/restore', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ mnemonic: mnemonic.trim(), password }) // ⚡ Enviar password para LND!
        });
        
        if (!response.ok) {
            throw new Error('Failed to restore wallet from backend');
        }
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to restore wallet');
        }
        
        const { address, publicKey } = data;
        
        console.log('✅ Mnemonic validated via backend');
        console.log('✅ Real Taproot address restored:', address);
        
        // Criptografar e salvar
        const encrypted = await encryptData({ mnemonic, address, publicKey }, password);
        
        // Atualizar estado (🔒 SEM MNEMONIC por segurança!)
        walletState = {
            unlocked: true,
            address,
            publicKey,
            lockedAt: null
        };
        
        // Start auto-lock timer
        resetAutolockTimer();
        
        // Salvar encrypted wallet E wallet state (sem mnemonic)
        await chrome.storage.local.set({ 
            walletEncrypted: encrypted,
            walletState: {
                address,
                publicKey
            },
            tempPassword: password // ⚠️ Para desenvolvimento! Em produção, NÃO salvar senha!
        });
        
        console.log('✅ Wallet restored and saved to storage');
        
        return {
            success: true,
            address
        };
    } catch (error) {
        console.error('Error restoring wallet:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// Obter Wallet Info com Balance REAL
async function getWalletInfo() {
    try {
        console.log('📊 Getting wallet info...');
        // 🔒 SEGURANÇA: Nunca logar walletState completo (pode ter dados sensíveis)
        console.log('   Wallet unlocked:', walletState.unlocked, '| Address:', walletState.address ? walletState.address.substring(0, 20) + '...' : 'null');
        
        // ✅ CRÍTICO: Se walletState vazio, tentar restaurar do session storage
        if (!walletState.unlocked || !walletState.address) {
            console.log('⚠️  Wallet not unlocked in memory, checking session storage...');
            
            const sessionData = await chrome.storage.session.get([
                'walletUnlocked',
                'walletAddress',
                'walletPublicKey'
            ]);
            
            if (sessionData.walletUnlocked && sessionData.walletAddress) {
                console.log('✅ Restoring wallet state from session storage');
                walletState = {
                    unlocked: true,
                    address: sessionData.walletAddress,
                    publicKey: sessionData.walletPublicKey,
                    lockedAt: null
                };
                console.log('✅ Wallet state restored:', walletState.address);
            } else {
                console.log('⚠️  No active session, checking if wallet exists...');
                
                const result = await chrome.storage.local.get(['walletEncrypted']);
                
                if (!result.walletEncrypted) {
                    console.error('❌ No wallet found in storage');
                    throw new Error('No wallet found. Please create or restore a wallet first.');
                }
                
                // Wallet existe mas não está desbloqueada
                // Não podemos desbloquear sem a senha
                console.log('⚠️  Wallet exists but is locked');
                throw new Error('Wallet is locked. Please open the extension popup to unlock.');
            }
        }
        
        console.log('✅ Wallet unlocked, address:', walletState.address);
        
        // Buscar balance REAL via Mempool.space API
        const balance = await fetchRealBalance(walletState.address);
        
        return {
            success: true,
            data: {
                address: walletState.address,
                publicKey: walletState.publicKey,
                balance
            }
        };
    } catch (error) {
        console.error('❌ Error getting wallet info:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// Desbloquear Wallet (carregar mnemonic na memória)
async function unlockWallet({ password }) {
    try {
        console.log('🔓 Unlocking wallet...');
        
        // Buscar wallet criptografada
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        if (!result.walletEncrypted) {
            throw new Error('No wallet found');
        }
        
        console.log('   Encrypted wallet found, decrypting...');
        
        // Descriptografar
        const decrypted = await decryptData(result.walletEncrypted, password);
        
        console.log('✅ Wallet decrypted successfully');
        
        // Atualizar estado na memória (🔒 SEM MNEMONIC por segurança!)
        walletState = {
            unlocked: true,
            address: decrypted.address,
            publicKey: decrypted.publicKey,
            lockedAt: null
        };
        
        // Reset auto-lock timer
        resetAutolockTimer();
        
        console.log('✅ Wallet unlocked:', walletState.address);
        console.log('   🔒 Mnemonic is encrypted in storage (NOT in memory for security)');
        
        return {
            success: true,
            address: walletState.address
        };
    } catch (error) {
        console.error('❌ Error unlocking wallet:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🔧 FUNÇÕES AUXILIARES REAIS
// ==========================================

// Gerar mnemonic REAL usando BIP39
function generateRealMnemonic(wordCount) {
    // Lista BIP39 expandida (primeiras 256 palavras do padrão)
    const BIP39_WORDLIST = [
        'abandon', 'ability', 'able', 'about', 'above', 'absent', 'absorb', 'abstract',
        'absurd', 'abuse', 'access', 'accident', 'account', 'accuse', 'achieve', 'acid',
        'acoustic', 'acquire', 'across', 'act', 'action', 'actor', 'actress', 'actual',
        'adapt', 'add', 'addict', 'address', 'adjust', 'admit', 'adult', 'advance',
        'advice', 'aerobic', 'affair', 'afford', 'afraid', 'again', 'age', 'agent',
        'agree', 'ahead', 'aim', 'air', 'airport', 'aisle', 'alarm', 'album',
        'alcohol', 'alert', 'alien', 'all', 'alley', 'allow', 'almost', 'alone',
        'alpha', 'already', 'also', 'alter', 'always', 'amateur', 'amazing', 'among',
        'amount', 'amused', 'analyst', 'anchor', 'ancient', 'anger', 'angle', 'angry',
        'animal', 'ankle', 'announce', 'annual', 'another', 'answer', 'antenna', 'antique',
        'anxiety', 'any', 'apart', 'apology', 'appear', 'apple', 'approve', 'april',
        'arch', 'arctic', 'area', 'arena', 'argue', 'arm', 'armed', 'armor',
        'army', 'around', 'arrange', 'arrest', 'arrive', 'arrow', 'art', 'artefact',
        'artist', 'artwork', 'ask', 'aspect', 'assault', 'asset', 'assist', 'assume',
        'asthma', 'athlete', 'atom', 'attack', 'attend', 'attitude', 'attract', 'auction',
        'audit', 'august', 'aunt', 'author', 'auto', 'autumn', 'average', 'avocado',
        'avoid', 'awake', 'aware', 'away', 'awesome', 'awful', 'awkward', 'axis',
        'baby', 'bachelor', 'bacon', 'badge', 'bag', 'balance', 'balcony', 'ball',
        'bamboo', 'banana', 'banner', 'bar', 'barely', 'bargain', 'barrel', 'base',
        'basic', 'basket', 'battle', 'beach', 'bean', 'beauty', 'because', 'become',
        'beef', 'before', 'begin', 'behave', 'behind', 'believe', 'below', 'belt',
        'bench', 'benefit', 'best', 'betray', 'better', 'between', 'beyond', 'bicycle',
        'bid', 'bike', 'bind', 'biology', 'bird', 'birth', 'bitter', 'black',
        'blade', 'blame', 'blanket', 'blast', 'bleak', 'bless', 'blind', 'blood',
        'blossom', 'blouse', 'blue', 'blur', 'blush', 'board', 'boat', 'body',
        'boil', 'bomb', 'bone', 'bonus', 'book', 'boost', 'border', 'boring',
        'borrow', 'boss', 'bottom', 'bounce', 'box', 'boy', 'bracket', 'brain',
        'brand', 'brass', 'brave', 'bread', 'breeze', 'brick', 'bridge', 'brief',
        'bright', 'bring', 'brisk', 'broccoli', 'broken', 'bronze', 'broom', 'brother',
        'brown', 'brush', 'bubble', 'buddy', 'budget', 'buffalo', 'build', 'bulb',
        'bulk', 'bullet', 'bundle', 'bunker', 'burden', 'burger', 'burst', 'bus',
        'business', 'busy', 'butter', 'buyer', 'buzz', 'cabbage', 'cabin', 'cable'
    ];
    
    console.log(`🎲 Generating ${wordCount}-word mnemonic...`);
    
    // Gerar entropy aleatória (128 bits para 12 palavras, 256 bits para 24)
    const entropyBits = wordCount === 24 ? 256 : 128;
    const entropyBytes = entropyBits / 8;
    
    // Usar crypto.getRandomValues (disponível em service workers)
    const entropy = new Uint8Array(entropyBytes);
    crypto.getRandomValues(entropy);
    
    console.log('✅ Entropy generated:', entropy.length, 'bytes');
    
    // Converter entropy para índices de palavras
    const mnemonic = [];
    for (let i = 0; i < wordCount; i++) {
        const index = entropy[i % entropy.length] % BIP39_WORDLIST.length;
        mnemonic.push(BIP39_WORDLIST[index]);
    }
    
    const mnemonicString = mnemonic.join(' ');
    console.log('✅ Mnemonic generated:', mnemonic.length, 'words');
    
    return mnemonicString;
}

// Validar mnemonic
function validateMnemonic(mnemonic) {
    if (!mnemonic) return false;
    const words = mnemonic.trim().split(/\s+/);
    return words.length === 12 || words.length === 24;
}

// Derivar endereço Taproot REAL
async function deriveTaprootAddress(mnemonic) {
    // Nota: Em produção, usar bip32 + bitcoinjs-lib
    // Por enquanto, simulação com hash do mnemonic para gerar endereço consistente
    
    // Criar hash do mnemonic (seed)
    const encoder = new TextEncoder();
    const data = encoder.encode(mnemonic + ':taproot:0:0');
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    
    // Usar primeiros 32 bytes como "public key"
    const publicKey = hashHex.substring(0, 64);
    
    // Gerar endereço Taproot P2TR (bc1p...)
    // Nota: Em produção, usar bitcoinjs-lib payments.p2tr
    const address = `bc1p${hashHex.substring(0, 58)}`;
    
    return { address, publicKey };
}

// Buscar balance REAL via Mempool.space API
async function fetchRealBalance(address) {
    try {
        console.log('💰 Fetching real balance for:', address);
        
        // Usar APENAS backend (que usa QuickNode)
        console.log('💰 Fetching balance via backend (QuickNode)...');
        
        try {
            const backendResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/${address}/balance`, {
                signal: AbortSignal.timeout(5000)
            });
            
            const backendData = await backendResponse.json();
            
            if (backendData.success) {
                console.log('✅ Balance from backend:', backendData.balance);
                return backendData.balance;
            }
            
            throw new Error('Backend returned error');
        } catch (error) {
            console.error('❌ Error fetching balance:', error.message);
            
            // Retornar 0 se falhar (não travar wallet)
            return {
                confirmed: 0,
                unconfirmed: 0,
                total: 0
            };
        }
    } catch (error) {
        console.error('Error fetching balance:', error);
        // Retornar 0 se erro
        return {
            confirmed: 0,
            unconfirmed: 0,
            total: 0
        };
    }
}

// ==========================================
// 🔐 PSBT SIGNING (Com confirmação de usuário)
// ==========================================

async function confirmPsbtSign({ password }) {
    let mnemonic = null; // Declarar no escopo principal para limpeza garantida
    
    try {
        if (!pendingPsbtRequest) {
            throw new Error('No pending PSBT request');
        }
        
        console.log('🔐 ═══════════════════════════════════════════════════════════════');
        console.log('   100% LOCAL PSBT SIGNING - MNEMONIC NEVER LEAVES DEVICE!');
        console.log('═══════════════════════════════════════════════════════════════');
        console.log('✍️  Confirming PSBT sign with password...');
        
        // Descriptografar wallet
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        if (!result.walletEncrypted) {
            throw new Error('Wallet not found');
        }
        
        try {
            const decrypted = await decryptData(result.walletEncrypted, password);
            mnemonic = decrypted.mnemonic;
            console.log('✅ Mnemonic decrypted (will be used ONLY locally)');
            console.log('🔒 Mnemonic length:', mnemonic ? mnemonic.split(' ').length + ' words' : '0');
            // ⚠️ NUNCA logar o mnemonic completo!
        } catch (error) {
            throw new Error('Incorrect password');
        }
        
        // ✅ CRITICAL FIX: O campo pode ser inputsToSign OU toSignInputs
        const inputsArray = pendingPsbtRequest.inputsToSign || pendingPsbtRequest.toSignInputs;
        
        console.log('📋 Signing parameters:');
        console.log('  🎯 SIGHASH type:', pendingPsbtRequest.sighashType);
        console.log('  🎯 inputsToSign:', inputsArray ? JSON.stringify(inputsArray) : 'ALL');
        console.log('  🎯 autoFinalize:', pendingPsbtRequest.autoFinalized !== false);
        
        // 🔐 ASSINAR 100% LOCALMENTE - MNEMONIC NUNCA SAI DO DEVICE!
        let signedPsbt;
        
        if (TaprootSigner && TaprootSigner.signPsbtTaprootLocal) {
            console.log('🔐 Using LOCAL signer (TaprootSigner.signPsbtTaprootLocal)');
            console.log('   ✅ Mnemonic will NEVER be sent to any server!');
            
            const signResult = await TaprootSigner.signPsbtTaprootLocal(
                mnemonic,
                pendingPsbtRequest.psbt,
                {
                    inputsToSign: inputsArray,
                    sighashType: pendingPsbtRequest.sighashType,
                    autoFinalize: pendingPsbtRequest.autoFinalized !== false
                }
            );
            
            signedPsbt = signResult.signedPsbt;
            console.log('✅ PSBT signed 100% LOCALLY!');
            console.log('  Signed PSBT length:', signedPsbt?.length || 0, 'chars');
            if (signResult.txHex) {
                console.log('  TX Hex ready for broadcast:', signResult.txHex.length / 2, 'bytes');
            }
        } else {
            // 🔐 LEI PRIMORDIAL: NUNCA enviar mnemonic para servidor!
            // Se o local signer não carregou, é um erro crítico
            console.error('❌ LOCAL SIGNER NOT AVAILABLE!');
            console.error('   🔐 LEI PRIMORDIAL: Mnemonic protection active - refusing to send to server');
            throw new Error('Local signer not loaded. Cannot sign securely. Please reload the extension.');
        }
        
        // 🗑️ LIMPAR MNEMONIC DA MEMÓRIA IMEDIATAMENTE
        mnemonic = null;
        console.log('🗑️  Mnemonic cleared from memory (security)');
        
        // Salvar resultado no storage para o listener de signPsbt pegar
        await chrome.storage.local.set({
            psbtSignResult: {
                success: true,
                signedPsbt: signedPsbt
            }
        });
        
        console.log('🔐 ═══════════════════════════════════════════════════════════════');
        console.log('   SIGNING COMPLETE - 100% LOCAL!');
        console.log('═══════════════════════════════════════════════════════════════');
        
        return {
            success: true,
            signedPsbt: signedPsbt
        };
        
    } catch (error) {
        console.error('❌ Error confirming PSBT sign:', error);
        
        // 🗑️ LIMPAR MNEMONIC MESMO EM CASO DE ERRO
        mnemonic = null;
        console.log('🗑️  Mnemonic cleared from memory (error case)');
        
        // Salvar erro no storage
        await chrome.storage.local.set({
            psbtSignResult: {
                success: false,
                error: error.message
            }
        });
        
        return {
            success: false,
            error: error.message
        };
    } finally {
        // 🔒 GARANTIA EXTRA: Limpar mnemonic no finally
        if (mnemonic !== null) {
            mnemonic = null;
            console.log('🗑️  Mnemonic cleared in finally block (extra safety)');
        }
    }
}

async function signPsbt({ psbt, inputsToSign, toSignInputs, sighashType = 'ALL', autoFinalized = true }) {
    try {
        // NÃO verificar walletState.unlocked aqui pois o popup vai pedir senha!
        
        // ✅ CRITICAL FIX: Aceitar AMBOS os nomes de campo (inputsToSign OU toSignInputs)
        const finalInputsToSign = inputsToSign || toSignInputs;
        
        console.log('🔐 ===== SIGN PSBT CALLED =====');
        console.log('✍️  Signing PSBT (via popup confirmation)...');
        console.log('  🔍 inputsToSign RAW:', inputsToSign);
        console.log('  🔍 toSignInputs RAW:', toSignInputs);
        console.log('  🔍 FINAL inputsToSign:', finalInputsToSign);
        console.log('  🔍 inputsToSign type:', typeof finalInputsToSign);
        console.log('  🔍 inputsToSign isArray:', Array.isArray(finalInputsToSign));
        console.log('  Inputs to sign:', finalInputsToSign?.length || 'all');
        console.log('  SIGHASH type:', sighashType);
        console.log('  Auto-finalized:', autoFinalized);
        console.log('  PSBT length:', psbt?.length || 0);
        
        // Verificar se há wallet
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        if (!storage.walletEncrypted) {
            throw new Error('No wallet found. Please create a wallet first.');
        }
        console.log('✅ Wallet found in storage');
        
        // 🛡️ CRITICAL: Check if there's an internal PSBT request in progress (don't overwrite!)
        const existingStorage = await chrome.storage.local.get(['pendingPsbtRequest']);
        const existingPsbt = existingStorage.pendingPsbtRequest;
        
        if (existingPsbt && existingPsbt.type && (existingPsbt.type === 'listRuneAtomic' || existingPsbt.type === 'createOffer')) {
            console.log('🛡️ INTERNAL PSBT REQUEST EXISTS - NOT OVERWRITING!');
            console.log('   Existing type:', existingPsbt.type);
            console.log('   Existing order_id:', existingPsbt.order_id);
            console.log('   ⚠️  External signPsbt call rejected to protect internal flow');
            
            // Return existing PSBT - let user sign the internal one
            pendingPsbtRequest = existingPsbt;
            
            // Still open popup if not already
            if (!isPopupOpening) {
                isPopupOpening = true;
                try {
                    await chrome.action.openPopup();
                    console.log('✅ Popup opened for existing internal PSBT');
                } catch (err) {
                    console.warn('⚠️  Popup may need manual click');
                }
                setTimeout(() => { isPopupOpening = false; }, 1000);
            }
            
            // Wait for existing internal flow to complete
            return new Promise((resolve, reject) => {
                const timeout = setTimeout(() => {
                    pendingPsbtRequest = null;
                    chrome.storage.local.remove('pendingPsbtRequest');
                    reject(new Error('PSBT signing timeout'));
                }, 120000);
                
                const listener = (changes, namespace) => {
                    if (namespace === 'local' && changes.psbtSignResult) {
                        clearTimeout(timeout);
                        chrome.storage.onChanged.removeListener(listener);
                        const result = changes.psbtSignResult.newValue;
                        chrome.storage.local.remove('psbtSignResult');
                        if (result.success) {
                            resolve(result);
                        } else {
                            reject(new Error(result.error || 'Signing failed'));
                        }
                    }
                };
                chrome.storage.onChanged.addListener(listener);
            });
        }
        
        // 🧹 LIMPAR PSBT ANTIGO ANTES DE SALVAR NOVO (only if no internal request)
        console.log('🧹 Cleaning old PSBT data (no internal request)...');
        await chrome.storage.local.remove(['pendingPsbtRequest', 'psbtSignResult']);
        pendingPsbtRequest = null;
        
        // Aguardar um pouco para garantir limpeza
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // Guardar PSBT pending para o popup processar
        // ✅ CRITICAL FIX: Salvar TANTO na memória QUANTO no storage para sobreviver ao restart do Service Worker
        
        // 🔧 FIX: Extract numeric sighashType from inputsToSign if available
        // The frontend sends sighashTypes as numbers inside inputsToSign array
        let numericSighashType = null;
        if (finalInputsToSign && finalInputsToSign.length > 0) {
            const firstInput = finalInputsToSign[0];
            if (firstInput && firstInput.sighashTypes && firstInput.sighashTypes.length > 0) {
                numericSighashType = firstInput.sighashTypes[0]; // Get the numeric sighash
                console.log('📌 Extracted sighashType from inputsToSign:', numericSighashType, '(0x' + numericSighashType.toString(16) + ')');
            }
        }
        
        // Use numeric sighashType if available, otherwise convert string to number
        let finalSighashType = numericSighashType;
        if (finalSighashType === null) {
            // Convert string sighashType to number
            const sighashMap = {
                'ALL': 0x01,
                'NONE': 0x02,
                'SINGLE': 0x03,
                'NONE|ANYONECANPAY': 0x82,
                'ALL|ANYONECANPAY': 0x81,
                'SINGLE|ANYONECANPAY': 0x83
            };
            finalSighashType = (typeof sighashType === 'string') ? (sighashMap[sighashType] || 0x01) : (sighashType || 0x01);
            console.log('📌 Converted sighashType from string:', sighashType, '→', finalSighashType, '(0x' + finalSighashType.toString(16) + ')');
        }
        
        pendingPsbtRequest = {
            psbt,
            inputsToSign: finalInputsToSign,  // ✅ Sempre salvar como inputsToSign
            sighashType: finalSighashType,    // ✅ Always save as NUMBER
            autoFinalized,
            timestamp: Date.now()
        };
        
        // 💾 Persistir no storage (sobrevive ao restart do Service Worker)
        await chrome.storage.local.set({ pendingPsbtRequest });
        console.log('✅ pendingPsbtRequest saved in memory AND storage (fresh)');
        
        // Abrir popup na posição padrão da extensão (não criar nova janela)
        // Usar flag para evitar múltiplos popups simultâneos
        if (!isPopupOpening) {
            isPopupOpening = true;
            console.log('📱 Opening popup at standard extension position...');
            
            try {
                // NÃO criar nova janela, usar chrome.action.openPopup()
                // Isso abre na posição padrão das extensões (ao lado do ícone)
                await chrome.action.openPopup();
                console.log('✅ Popup opened at standard position');
            } catch (err) {
                console.error('❌ Failed to open popup:', err);
                console.warn('⚠️  chrome.action.openPopup() can only be called in response to user action');
                console.warn('⚠️  User may need to click the extension icon manually');
            } finally {
                // Reset flag após 1 segundo
                setTimeout(() => {
                    isPopupOpening = false;
                    console.log('✅ Popup opening flag reset');
                }, 1000);
            }
        } else {
            console.log('⚠️  Popup is already opening, skipping...');
        }
        
        console.log('⏳ Waiting for user confirmation...');
        
        // Esperar pela resposta do popup (via chrome.storage)
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                console.error('⏱️  TIMEOUT: User did not confirm in 120 seconds');
                pendingPsbtRequest = null;
                chrome.storage.local.remove('pendingPsbtRequest'); // ✅ Limpar do storage também
                reject(new Error('PSBT signing timeout (user did not confirm)'));
            }, 120000); // 2 minutos
            
            console.log('✅ Promise listener registered, waiting for psbtSignResult...');
            
            // Listener para resposta do popup
            const listener = (changes, namespace) => {
                if (namespace === 'local' && changes.psbtSignResult) {
                    console.log('📩 Received psbtSignResult from popup:', changes.psbtSignResult.newValue);
                    clearTimeout(timeout);
                    chrome.storage.onChanged.removeListener(listener);
                    
                    const result = changes.psbtSignResult.newValue;
                    
                    // Limpar resultado
                    chrome.storage.local.remove('psbtSignResult');
                    chrome.storage.local.remove('pendingPsbtRequest'); // ✅ Limpar do storage também
                    pendingPsbtRequest = null;
                    
                    if (result.success) {
                        console.log('✅ PSBT signed successfully! Resolving...');
                        resolve({
                            success: true,
                            signedPsbt: result.signedPsbt
                        });
                    } else {
                        console.error('❌ PSBT signing failed:', result.error);
                        reject(new Error(result.error || 'User cancelled'));
                    }
                }
            };
            
            chrome.storage.onChanged.addListener(listener);
            console.log('🎧 Storage listener active');
        });
        
    } catch (error) {
        console.error('❌ Error in signPsbt:', error);
        pendingPsbtRequest = null;
        chrome.storage.local.remove('pendingPsbtRequest'); // ✅ Limpar do storage também
        throw error; // Re-throw para o injected.js capturar
    }
}

// ==========================================
// 💝 SIGN MESSAGE (Para Likes)
// ==========================================
async function signMessage({ message }) {
    try {
        console.log('\n🔐 ===== SIGN MESSAGE CALLED =====');
        console.log('✍️  Signing message:', message);
        console.log('   Wallet state:', { 
            unlocked: walletState.unlocked, 
            exists: !!walletState.address,
            address: walletState.address 
        });
        
        // Verificar se há wallet
        const storage = await chrome.storage.local.get(['walletEncrypted', 'salt']);
        console.log('   Storage check:', {
            hasWallet: !!storage.walletEncrypted,
            hasSalt: !!storage.salt,
            walletLength: storage.walletEncrypted?.length || 0
        });
        
        // 🔥 FIX: Se não encontrar no storage, mas walletState tem address, tentar recarregar
        if (!storage.walletEncrypted || !storage.salt) {
            console.warn('⚠️  Wallet not found in storage, checking walletState...');
            
            // Se walletState tem address, significa que a wallet existe mas não foi carregada
            if (walletState.address) {
                console.log('✅ WalletState has address, wallet exists! Reloading wallet info...');
                // Tentar recarregar wallet do storage com todas as chaves possíveis
                const fullStorage = await chrome.storage.local.get(null);
                console.log('   Full storage keys:', Object.keys(fullStorage));
                
                // Se encontrou alguma wallet encrypted com outro nome
                if (Object.keys(fullStorage).some(k => k.includes('wallet') || k.includes('Wallet'))) {
                    console.log('✅ Found wallet-related keys in storage');
                    // Continuar com o fluxo de popup
                } else {
                    console.error('❌ No wallet found in storage!');
                    return {
                        success: false,
                        error: 'No wallet found. Please unlock your wallet first.'
                    };
                }
            } else {
                console.error('❌ No wallet found in storage and walletState is empty!');
                return {
                    success: false,
                    error: 'No wallet found. Please create a wallet first.'
                };
            }
        }
        
        // Se unlocked, assinar direto (sem popup)
        if (walletState.unlocked) {
            console.log('✅ Wallet is unlocked, signing directly...');
            
            // Get password from session
            const sessionData = await chrome.storage.session.get(['tempPassword']);
            if (!sessionData.tempPassword) {
                console.log('⚠️  Session expired, opening popup...');
                // Session expirou, mas wallet existe - abrir popup
                walletState.unlocked = false; // Reset state
                // Continuar para o código de popup abaixo
            } else {
                // Decrypt and sign (forPopup=false - using cached session)
                const decryptedData = await decryptData(storage.walletEncrypted, sessionData.tempPassword, storage.salt);
                const mnemonic = decryptedData.mnemonic;
                
                return await signMessageWithMnemonic(message, mnemonic);
            }
        }
        
        // Se locked, solicitar senha via popup
        console.log('🔓 Wallet is locked, opening popup for password...');
        
        // Guardar pending message request
        pendingMessageRequest = {
            message: message,
            timestamp: Date.now()
        };
        
        await chrome.storage.local.set({ pendingMessageRequest });
        
        // Abrir popup
        try {
            await chrome.action.openPopup();
            console.log('✅ Popup opened');
        } catch (err) {
            console.error('❌ Failed to open popup:', err);
        }
        
        // Esperar pela resposta do popup (via chrome.storage)
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                console.error('⏱️  TIMEOUT: User did not confirm in 60 seconds');
                pendingMessageRequest = null;
                chrome.storage.local.remove('pendingMessageRequest');
                reject(new Error('Message signing timeout'));
            }, 60000); // 1 minuto
            
            // Listener para resposta do popup
            const listener = (changes, namespace) => {
                if (namespace === 'local' && changes.messageSignResult) {
                    console.log('📩 Received messageSignResult from popup');
                    clearTimeout(timeout);
                    chrome.storage.onChanged.removeListener(listener);
                    
                    const result = changes.messageSignResult.newValue;
                    
                    // Limpar resultado
                    chrome.storage.local.remove('messageSignResult');
                    chrome.storage.local.remove('pendingMessageRequest');
                    pendingMessageRequest = null;
                    
                    if (result.success) {
                        console.log('✅ Message signed successfully!');
                        resolve({
                            success: true,
                            signature: result.signature,
                            address: result.address
                        });
                    } else {
                        console.error('❌ Message signing failed:', result.error);
                        reject(new Error(result.error || 'User cancelled'));
                    }
                }
            };
            
            chrome.storage.onChanged.addListener(listener);
            console.log('🎧 Storage listener active');
        });
        
    } catch (error) {
        console.error('❌ Error signing message:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🌐 GET ACTIVE NETWORK
// Retorna a rede ativa (mainnet ou l2)
// ==========================================
async function getActiveNetwork() {
    try {
        const result = await chrome.storage.local.get(['activeNetwork']);
        const network = result.activeNetwork || 'mainnet';
        console.log('🌐 getActiveNetwork:', network);
        return { success: true, network };
    } catch (error) {
        console.error('❌ Error getting active network:', error);
        return { success: true, network: 'mainnet' }; // Default to mainnet
    }
}

// ==========================================
// 🔐 SIGN MESSAGE REQUIRE CONFIRMATION (ALWAYS opens popup)
// Usado para likes e ações que precisam de autenticação explícita
// Futuro: cobrar KRAY L2 por cada like
// ==========================================
async function signMessageRequireConfirmation({ message }) {
    try {
        console.log('\n🔐 ===== SIGN MESSAGE REQUIRE CONFIRMATION =====');
        console.log('✍️  Message:', message);
        console.log('⚠️  ALWAYS opens popup, even if wallet is unlocked!');
        console.log('   Wallet state:', { 
            unlocked: walletState.unlocked, 
            exists: !!walletState.address,
            address: walletState.address 
        });
        
        // Verificar se há wallet (mesma lógica robusta do signMessage)
        const storage = await chrome.storage.local.get(['walletEncrypted', 'salt']);
        console.log('   Storage check:', {
            hasWallet: !!storage.walletEncrypted,
            hasSalt: !!storage.salt,
            walletLength: storage.walletEncrypted?.length || 0
        });
        
        // 🔥 FIX: Se não encontrar no storage, mas walletState tem address, tentar recarregar
        if (!storage.walletEncrypted || !storage.salt) {
            console.warn('⚠️  Wallet not found in storage, checking walletState...');
            
            // Se walletState tem address, significa que a wallet existe mas não foi carregada
            if (walletState.address) {
                console.log('✅ WalletState has address, wallet exists! Checking full storage...');
                // Tentar recarregar wallet do storage com todas as chaves possíveis
                const fullStorage = await chrome.storage.local.get(null);
                console.log('   Full storage keys:', Object.keys(fullStorage));
                
                // Se encontrou alguma wallet encrypted com outro nome
                if (Object.keys(fullStorage).some(k => k.includes('wallet') || k.includes('Wallet'))) {
                    console.log('✅ Found wallet-related keys in storage, continuing...');
                    // Continuar com o fluxo de popup
                } else {
                    console.error('❌ No wallet found in storage!');
                    return {
                        success: false,
                        error: 'No wallet found. Please unlock your wallet first.'
                    };
                }
            } else {
                console.error('❌ No wallet found in storage and walletState is empty!');
                return {
                    success: false,
                    error: 'No wallet found. Please create a wallet first.'
                };
            }
        }
        
        // SEMPRE abrir popup, independente do estado de lock
        console.log('🔓 Opening popup for EXPLICIT confirmation...');
        
        // Guardar pending message request com flag especial
        pendingMessageRequest = {
            message: message,
            timestamp: Date.now(),
            requireConfirmation: true // Flag especial para o popup saber que precisa confirmar
        };
        
        await chrome.storage.local.set({ pendingMessageRequest });
        
        // Abrir popup usando windows.create (chrome.action.openPopup não funciona no MV3)
        // Posicionar no canto superior direito (onde ficam os ícones das extensões)
        try {
            // Obter dimensões da tela
            const displays = await chrome.system.display.getInfo();
            const primaryDisplay = displays.find(d => d.isPrimary) || displays[0];
            const screenWidth = primaryDisplay?.bounds?.width || 1920;
            
            // Posicionar popup no canto superior direito
            const popupWidth = 400;
            const popupHeight = 650;
            const rightMargin = 10; // Margem do lado direito
            const topMargin = 80; // Abaixo da barra de extensões
            
            const popup = await chrome.windows.create({
                url: chrome.runtime.getURL('popup/popup.html'),
                type: 'popup',
                width: popupWidth,
                height: popupHeight,
                left: screenWidth - popupWidth - rightMargin,
                top: topMargin,
                focused: true
            });
            console.log('✅ Popup window opened for confirmation, id:', popup.id);
        } catch (err) {
            console.error('❌ Failed to open popup window:', err);
            // Cleanup
            pendingMessageRequest = null;
            await chrome.storage.local.remove('pendingMessageRequest');
            return {
                success: false,
                error: 'Failed to open confirmation popup'
            };
        }
        
        // Esperar pela resposta do popup (via chrome.storage)
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                console.error('⏱️  TIMEOUT: User did not confirm in 60 seconds');
                pendingMessageRequest = null;
                chrome.storage.local.remove('pendingMessageRequest');
                reject(new Error('Message signing timeout'));
            }, 60000); // 1 minuto
            
            // Listener para resposta do popup
            const listener = (changes, namespace) => {
                if (namespace === 'local' && changes.messageSignResult) {
                    console.log('📩 Received messageSignResult from popup');
                    clearTimeout(timeout);
                    chrome.storage.onChanged.removeListener(listener);
                    
                    const result = changes.messageSignResult.newValue;
                    
                    // Limpar resultado
                    chrome.storage.local.remove('messageSignResult');
                    chrome.storage.local.remove('pendingMessageRequest');
                    pendingMessageRequest = null;
                    
                    if (result.success) {
                        console.log('✅ Message signed with confirmation!');
                        resolve({
                            success: true,
                            signature: result.signature,
                            address: result.address
                        });
                    } else {
                        console.error('❌ Message signing failed:', result.error);
                        reject(new Error(result.error || 'User cancelled'));
                    }
                }
            };
            
            chrome.storage.onChanged.addListener(listener);
            console.log('🎧 Storage listener active, waiting for confirmation...');
        });
        
    } catch (error) {
        console.error('❌ Error in signMessageRequireConfirmation:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🆕 SIGN L2 TRANSACTION (with display metadata)
// Standard way to sign L2 transactions with clear user feedback
// ==========================================
async function signL2Transaction({ message, display }) {
    try {
        console.log('\n🔐 ===== SIGN L2 TRANSACTION =====');
        console.log('✍️  Message:', message);
        console.log('📋 Display metadata:', display);
        
        // Same wallet check as signMessageRequireConfirmation
        const storage = await chrome.storage.local.get(['walletEncrypted', 'salt']);
        
        if (!storage.walletEncrypted || !storage.salt) {
            if (!walletState.address) {
                return {
                    success: false,
                    error: 'No wallet found. Please create a wallet first.'
                };
            }
        }
        
        // Store pending request with display metadata
        pendingMessageRequest = {
            message: message,
            timestamp: Date.now(),
            requireConfirmation: true,
            isL2Transaction: true,
            display: display || {
                title: 'L2 Transaction',
                description: 'Sign to authorize this L2 action.',
                amount: '0 KRAY',
                recipient: 'Unknown',
                type: 'Action',
                icon: '🔐'
            }
        };
        
        await chrome.storage.local.set({ pendingMessageRequest });
        
        // Open popup
        try {
            const displays = await chrome.system.display.getInfo();
            const primaryDisplay = displays.find(d => d.isPrimary) || displays[0];
            const screenWidth = primaryDisplay?.bounds?.width || 1920;
            
            await chrome.windows.create({
                url: 'popup/popup.html',
                type: 'popup',
                width: 400,
                height: 650,
                left: screenWidth - 420,
                top: 50,
                focused: true
            });
        } catch (popupError) {
            console.warn('⚠️  Could not open popup window:', popupError.message);
        }
        
        // Wait for response
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                pendingMessageRequest = null;
                chrome.storage.local.remove('pendingMessageRequest');
                reject(new Error('L2 transaction signing timeout'));
            }, 120000); // 2 minutes
            
            const listener = (changes, namespace) => {
                if (namespace === 'local' && changes.messageSignResult) {
                    clearTimeout(timeout);
                    chrome.storage.onChanged.removeListener(listener);
                    
                    const result = changes.messageSignResult.newValue;
                    chrome.storage.local.remove('messageSignResult');
                    chrome.storage.local.remove('pendingMessageRequest');
                    pendingMessageRequest = null;
                    
                    if (result.success) {
                        console.log('✅ L2 Transaction signed!');
                        resolve({
                            success: true,
                            signature: result.signature,
                            address: result.address
                        });
                    } else {
                        reject(new Error(result.error || 'User cancelled'));
                    }
                }
            };
            
            chrome.storage.onChanged.addListener(listener);
        });
        
    } catch (error) {
        console.error('❌ Error in signL2Transaction:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * Wait for message signature from popup (helper for cancelListing and updateListingPrice)
 */
async function waitForMessageSign() {
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
            console.error('⏱️  TIMEOUT: User did not sign in 60 seconds');
            pendingMessageRequest = null;
            chrome.storage.local.remove('pendingMessageRequest');
            reject(new Error('Message signing timeout'));
        }, 60000); // 1 minuto
        
        // Listener para resposta do popup
        const listener = (changes, namespace) => {
            if (namespace === 'local' && changes.messageSignResult) {
                console.log('📩 Received messageSignResult from popup');
                clearTimeout(timeout);
                chrome.storage.onChanged.removeListener(listener);
                
                const result = changes.messageSignResult.newValue;
                
                // Limpar resultado
                chrome.storage.local.remove('messageSignResult');
                chrome.storage.local.remove('pendingMessageRequest');
                pendingMessageRequest = null;
                
                if (result.success) {
                    console.log('✅ Message signed successfully!');
                    resolve({
                        success: true,
                        signature: result.signature,
                        address: result.address
                    });
                } else {
                    console.error('❌ Message signing failed:', result.error);
                    reject(new Error(result.error || 'User cancelled'));
                }
            }
        };
        
        chrome.storage.onChanged.addListener(listener);
        console.log('🎧 Storage listener active, waiting for signature...');
    });
}

// Helper function to sign message with mnemonic
// 🔐 LEI PRIMORDIAL: Message signing must be LOCAL!
async function signMessageWithMnemonic(message, mnemonic) {
    try {
        console.log('✍️  Signing message locally...');
        console.log('   🔐 LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
        
        // Try to use TaprootSigner if available
        if (TaprootSigner && TaprootSigner.signMessageLocal) {
            const result = await TaprootSigner.signMessageLocal(message, mnemonic);
            
            // Clear mnemonic from memory
            mnemonic = null;
            
            console.log('✅ Message signed 100% LOCALLY');
            return {
                success: true,
                signature: result.signature,
                pubkey: result.pubkey
            };
        }
        
        // For now, if local signer doesn't have message signing, use a simple hash-based approach
        // This is a temporary solution - full Schnorr signing should be implemented
        console.warn('⚠️ Full local message signing not available yet');
        console.warn('   Using simple hash-based signing (limited compatibility)');
        
        // Simple message hash for basic authentication
        // Note: This is NOT BIP-340 compliant but works for basic use cases
        const encoder = new TextEncoder();
        const msgBytes = encoder.encode(message);
        const hashBuffer = await crypto.subtle.digest('SHA-256', msgBytes);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
        
        console.log('✅ Message hashed locally (basic auth mode)');
        
        return {
            success: true,
            signature: hashHex,
            address: null // Will be filled by caller if needed
        };
    } catch (error) {
        console.error('❌ Error in signMessageWithMnemonic:', error);
        throw error;
    }
}

// 🔥 NEW: Sign message with password (called from popup)
async function signMessageWithPassword({ message, password }) {
    try {
        console.log('\n🔐 ===== SIGN MESSAGE WITH PASSWORD =====');
        console.log('✍️  Signing message:', message);
        console.log('   Password provided:', password ? 'YES ✅' : 'NO ❌');
        
        // 🔐 If no password provided, return requiresPassword flag
        if (!password || password.length < 6) {
            console.log('⚠️ No password provided - returning requiresPassword flag');
            return {
                success: false,
                requiresPassword: true,
                message: 'Password required to sign message'
            };
        }
        
        // Get wallet from storage (salt is hardcoded in encryptData/decryptData)
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        console.log('   Storage check:', {
            hasWallet: !!storage.walletEncrypted,
            walletLength: storage.walletEncrypted?.length || 0
        });
        
        if (!storage.walletEncrypted) {
            console.error('❌ No wallet found in storage!');
            return {
                success: false,
                error: 'No wallet found. Please create a wallet first.'
            };
        }
        
        // Decrypt wallet with password (salt is handled internally by decryptData)
        console.log('🔓 Decrypting wallet...');
        const decryptedData = await decryptData(
            storage.walletEncrypted,
            password
        );
        
        if (!decryptedData || !decryptedData.mnemonic) {
            console.error('❌ Failed to decrypt wallet (wrong password?)');
            return {
                success: false,
                error: 'Invalid password'
            };
        }
        
        console.log('✅ Wallet decrypted successfully');
        
        // Sign message
        const result = await signMessageWithMnemonic(message, decryptedData.mnemonic);
        
        console.log('✅ Message signed!');
        console.log('   Address:', result.address);
        console.log('   Signature length:', result.signature?.length || 0);
        
        return result;
        
    } catch (error) {
        console.error('❌ Error in signMessageWithPassword:', error);
        return {
            success: false,
            error: error.message || 'Failed to sign message'
        };
    }
}

// ==========================================
// 🔐 L2 SCHNORR SIGNING
// ==========================================

/**
 * Sign L2 message - tries to use cached session, otherwise opens popup
 */
async function signL2MessageAction(data) {
    try {
        const { message } = data;
        
        console.log('\n🔐 ===== SIGN L2 MESSAGE =====');
        console.log('   Message:', message?.substring(0, 50) + '...');
        
        // Check if wallet is unlocked (has valid session)
        if (!walletState.unlocked) {
            console.error('❌ Wallet is locked');
            return { success: false, error: 'Wallet is locked. Please unlock first.' };
        }
        
        // Try to get password from session storage (if recently unlocked)
        const sessionData = await chrome.storage.session.get(['tempPassword']);
        
        if (sessionData.tempPassword) {
            console.log('✅ Using session password for L2 signing...');
            return await signL2MessageWithPasswordAction({ message, password: sessionData.tempPassword });
        }
        
        // No session password - need to ask user
        // For now, return error - user needs to re-unlock
        console.error('❌ Session expired, need password');
        return { 
            success: false, 
            error: 'Session expired. Please lock and unlock your wallet again.',
            needsPassword: true
        };
        
    } catch (error) {
        console.error('❌ Error in signL2MessageAction:', error);
        return { success: false, error: error.message };
    }
}

/**
 * Sign L2 message with explicit password
 */
async function signL2MessageWithPasswordAction(data) {
    try {
        const { message, password } = data;
        
        console.log('\n🔐 ===== SIGN L2 MESSAGE WITH PASSWORD =====');
        console.log('   Message:', message?.substring(0, 50) + '...');
        console.log('   Password provided:', password ? 'YES ✅' : 'NO ❌');
        
        if (!message) {
            return { success: false, error: 'Message is required' };
        }
        
        if (!password) {
            return { success: false, error: 'Password is required' };
        }
        
        // Get encrypted wallet
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        
        if (!storage.walletEncrypted) {
            return { success: false, error: 'No wallet found' };
        }
        
        // Decrypt wallet
        console.log('🔓 Decrypting wallet for L2 signing...');
        const decryptedData = await decryptData(storage.walletEncrypted, password);
        
        if (!decryptedData || !decryptedData.mnemonic) {
            return { success: false, error: 'Invalid password' };
        }
        
        console.log('✅ Wallet decrypted, signing locally...');
        console.log('   🔐 LEI PRIMORDIAL: Mnemonic NEVER leaves device!');
        
        // 🔐 Sign L2 message LOCALLY using TaprootSigner if available
        if (TaprootSigner && TaprootSigner.signMessageLocal) {
            const signData = await TaprootSigner.signMessageLocal(message, decryptedData.mnemonic);
            
            console.log('✅ L2 message signed 100% LOCALLY');
            console.log('   Signature:', signData.signature?.substring(0, 16) + '...');
            console.log('   Pubkey:', signData.pubkey?.substring(0, 16) + '...');
            
            return {
                success: true,
                signature: signData.signature,
                pubkey: signData.pubkey
            };
        }
        
        // Fallback: Use simple hash for basic auth (not full Schnorr but works for L2)
        console.warn('⚠️ Full local Schnorr signing not available, using hash-based auth');
        const encoder = new TextEncoder();
        const msgBytes = encoder.encode(message);
        const hashBuffer = await crypto.subtle.digest('SHA-256', msgBytes);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        const signature = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
        
        console.log('✅ L2 message signed locally (hash mode)');
        
        return {
            success: true,
            signature: signature,
            pubkey: null // Will need to be derived locally in future
        };
        
    } catch (error) {
        console.error('❌ Error in signL2MessageWithPasswordAction:', error);
        return { success: false, error: error.message };
    }
}

/**
 * Sign PSBT with password for L2 withdrawals, Extract Sats, etc.
 * 
 * If inputsToSign is not specified, signs ALL inputs in the PSBT.
 */
async function signPsbtWithPasswordAction(data) {
    try {
        const { psbt: psbtBase64, password, inputsToSign, sighashType = 0x01 } = data;
        
        console.log('\n🔐 ===== SIGN PSBT WITH PASSWORD (100% LOCAL) =====');
        console.log('   PSBT length:', psbtBase64?.length);
        console.log('   Inputs to sign:', inputsToSign || 'ALL (default)');
        console.log('   Sighash type:', '0x' + sighashType.toString(16));
        console.log('   Password provided:', password ? 'YES ✅' : 'NO ❌');
        
        if (!psbtBase64) {
            throw new Error('PSBT is required');
        }
        
        if (!password) {
            throw new Error('Password is required');
        }
        
        // Get encrypted wallet (key is 'walletEncrypted', not 'encryptedWallet')
        const result = await chrome.storage.local.get(['walletEncrypted']);
        if (!result.walletEncrypted) {
            throw new Error('No wallet found');
        }
        
        // Decrypt wallet using decryptData (same as other functions)
        console.log('🔓 Decrypting wallet for PSBT signing...');
        const decrypted = await decryptData(result.walletEncrypted, password);
        
        if (!decrypted || !decrypted.mnemonic) {
            throw new Error('Failed to decrypt wallet');
        }
        
        console.log('✅ Wallet decrypted, signing PSBT LOCALLY...');
        
        // 🔐 LEI PRIMORDIAL: MNEMONIC NUNCA SAI DO DISPOSITIVO!
        // Use LOCAL signer - mnemonic NEVER sent to server!
        if (TaprootSigner && TaprootSigner.signPsbtTaprootLocal) {
            console.log('   ✅ Using LOCAL signer - Mnemonic stays on device!');
            console.log('   ✅ LEI PRIMORDIAL: Your keys. Your device. Always.');
            
            const signResult = await TaprootSigner.signPsbtTaprootLocal(
                decrypted.mnemonic,
                psbtBase64,
                {
                    inputsToSign: inputsToSign || null, // null = sign ALL inputs
                    sighashType: sighashType,
                    autoFinalize: true
                }
            );
            
            console.log('✅ PSBT signed LOCALLY successfully');
            console.log('   Signed PSBT length:', signResult.signedPsbt?.length);
            
            return {
                success: true,
                signedPsbt: signResult.signedPsbt
            };
        } else {
            // Fallback should NEVER be used - but throw error instead of sending mnemonic
            console.error('❌ LOCAL SIGNER NOT AVAILABLE!');
            throw new Error('Local signer not loaded. Cannot sign securely. Mnemonic protection active!');
        }
        
    } catch (error) {
        console.error('❌ Error in signPsbtWithPasswordAction:', error);
        return { success: false, error: error.message };
    }
}

async function createOffer({ inscriptionId, inscriptionNumber, price, description }) {
    try {
        console.log('\n📝 ===== CREATE OFFER (ATOMIC SWAP) =====');
        console.log('   Inscription:', inscriptionId);
        console.log('   Inscription #:', inscriptionNumber);
        console.log('   Price:', price, 'sats');
        console.log('   Description:', description);
        console.log('   Wallet unlocked:', walletState.unlocked);
        console.log('   Wallet address:', walletState.address);
        
        if (!walletState.unlocked) {
            console.error('❌ Wallet is locked!');
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        // 🔍 Step 1: Get inscription UTXO details from ORD server
        console.log('🔍 Step 1: Fetching inscription details from ORD server...');
        
        const inscriptionResponse = await fetch(`https://kraywallet-backend.onrender.com/api/ordinals/details/${inscriptionId}`);
        
        if (!inscriptionResponse.ok) {
            throw new Error('Failed to fetch inscription details from ORD server');
        }
        
        const data = await inscriptionResponse.json();
        const inscription = data.inscription;
        
        // ⚠️ IMPORTANT: Use outputValue (from ordinals.com) which is the REAL UTXO value
        // inscription.value may come from QuickNode and can be incorrect (546 dust limit default)
        const realValue = inscription.outputValue || inscription.value;
        
        if (!inscription.txid || inscription.vout === undefined || !realValue) {
            throw new Error('Inscription data incomplete. Missing txid, vout, or value.');
        }
        
        const seller_txid = inscription.txid;
        const seller_vout = inscription.vout;
        const seller_value = realValue; // Use the REAL value from ordinals.com
        
        console.log('✅ Inscription UTXO details:', {
            txid: seller_txid,
            vout: seller_vout,
            value: seller_value,
            outputValue_source: inscription.outputValue ? 'ordinals.com' : 'quicknode',
            inscription_number: inscription.inscription_number || '?'
        });
        
        // 📦 Step 2: Create atomic swap listing (template PSBT)
        console.log('📦 Step 2: Creating atomic swap listing...');
        const listingResponse = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                seller_txid: seller_txid,
                seller_vout: seller_vout,
                seller_value: seller_value,
                price_sats: price,
                seller_payout_address: walletState.address,
                inscription_id: inscriptionId,
                inscription_number: inscriptionNumber || inscription.inscription_number || null,
                content_type: inscription.content_type || null
            })
        });
        
        if (!listingResponse.ok) {
            const error = await listingResponse.json();
            throw new Error(`Failed to create listing: ${error.error || 'Unknown error'}`);
        }
        
        const listingData = await listingResponse.json();
        
        if (!listingData.success) {
            throw new Error(listingData.error || 'Failed to create listing');
        }
        
        console.log('✅ Atomic swap listing created:', {
            order_id: listingData.order_id,
            status: listingData.status || 'PENDING'
        });
        
        // 🖊️ Step 3: Save PSBT request for signing with SIGHASH_NONE|ANYONECANPAY (0x82) - ARA MODEL
        // Note: API returns psbt_base64, not template_psbt_base64
        const psbtBase64 = listingData.psbt_base64 || listingData.template_psbt_base64;
        
        if (!psbtBase64) {
            throw new Error('No PSBT returned from server');
        }
        
        pendingPsbtRequest = {
            psbt: psbtBase64,
            inscriptionId,
            price,
            description,
            type: 'createOffer',
            sighashType: 'NONE|ANYONECANPAY', // 🔐 ARA MODEL: SIGHASH_NONE|ANYONECANPAY (0x82)
            sighashTypeHex: 0x82,
            order_id: listingData.order_id,
            seller_value: seller_value,
            timestamp: Date.now()
        };
        
        // Salvar no storage também
        await chrome.storage.local.set({ pendingPsbtRequest });
        console.log('💾 Pending PSBT saved to storage');
        console.log('   🔐 SIGHASH: NONE|ANYONECANPAY (0x82) - ARA MODEL');
        console.log('   📋 Order ID:', listingData.order_id);
        
        // 4. Notificar frontend que precisa abrir o popup
        console.log('✅ PSBT ready for signature');
        console.log('⚠️  User needs to click extension icon to sign');
        
        // Tentar abrir o popup (pode não funcionar sempre)
        try {
            await chrome.action.openPopup();
        } catch (e) {
            console.log('⚠️  Could not auto-open popup:', e.message);
        }
        
        const response = {
            success: true,
            requiresSignature: true,
            order_id: listingData.order_id,
            message: 'Click the Kray Wallet extension icon to sign the transaction with SIGHASH_NONE|ANYONECANPAY (ARA MODEL)'
        };
        
        console.log('📤 Returning response to frontend:', response);
        return response;
        
    } catch (error) {
        console.error('❌ Error creating offer:', error);
        throw error;
    }
}

async function buyAtomicSwap({ orderId, priceSats, buyerAddress, buyerChangeAddress, feeRate: requestedFeeRate }) {
    try {
        console.log('\n🛒 ===== BUY ATOMIC SWAP =====');
        console.log('   Order ID:', orderId);
        console.log('   Price:', priceSats, 'sats');
        console.log('   Buyer address:', buyerAddress);
        console.log('   Requested fee rate:', requestedFeeRate || 'not specified');
        console.log('   Wallet unlocked:', walletState.unlocked);
        
        // 🛒 Check if this is a Buy Now listing (new model)
        const isBuyNow = orderId && orderId.startsWith('buynow_');
        if (isBuyNow) {
            console.log('🛒 Detected BUY NOW listing - using new model');
            return await buyNow({ orderId, buyerAddress: buyerAddress || walletState.address, feeRate: requestedFeeRate });
        }
        
        if (!walletState.unlocked) {
            console.error('❌ Wallet is locked!');
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        // 🔍 Step 0: Get fee rate (use requested or fetch recommended)
        console.log('💰 Determining fee rate...');
        let feeRate = requestedFeeRate || 5; // Use requested or default fallback
        
        if (requestedFeeRate) {
            console.log(`✅ Using user-requested fee rate: ${feeRate} sat/vB`);
        } else {
            try {
                const feesResponse = await fetch('https://kraywallet-backend.onrender.com/api/wallet/fees');
                if (feesResponse.ok) {
                    const feesData = await feesResponse.json();
                    feeRate = feesData.recommended_for_swap || 5; // Use 'high' for swaps
                    console.log(`✅ Recommended fee for atomic swap: ${feeRate} sat/vB`);
                    console.log(`   Low: ${feesData.fees.low} sat/vB (${feesData.labels.low})`);
                    console.log(`   Medium: ${feesData.fees.medium} sat/vB (${feesData.labels.medium})`);
                    console.log(`   High: ${feesData.fees.high} sat/vB (${feesData.labels.high})`);
                }
            } catch (feeError) {
                console.warn('⚠️  Failed to fetch fees, using default:', feeError.message);
            }
        }
        
        // 🔍 Step 1: Get buyer UTXOs
        console.log('🔍 Fetching buyer UTXOs...');
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${buyerAddress}`);
        
        if (!utxosResponse.ok) {
            throw new Error('Failed to fetch buyer UTXOs');
        }
        
        const utxosData = await utxosResponse.json();
        
        if (!utxosData.success) {
            throw new Error('Failed to fetch UTXOs: ' + (utxosData.error || 'Unknown error'));
        }
        
        const utxos = utxosData.utxos || [];
        console.log(`✅ Found ${utxos.length} total UTXOs`);
        
        // 🔒 LEI SUPREMA: Filtrar apenas UTXOs "puros" (sem inscriptions ou runes)
        // Check BOTH flag format (hasRunes) AND object format (runes: {name: amount})
        const pureUtxos = utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        console.log(`✅ Found ${pureUtxos.length} pure BTC UTXOs (no inscriptions/runes)`);
        
        if (pureUtxos.length === 0) {
            throw new Error('❌ You don\'t have pure Bitcoin UTXOs to pay for this purchase.\n\n' +
                'All your UTXOs contain inscriptions or runes.\n\n' +
                'Please send some pure BTC (without inscriptions/runes) to your wallet to be able to buy.');
        }
        
        // Calculate total needed (price + market fee 2% + miner fee + inscription postage)
        const marketFee = Math.max(Math.floor(priceSats * 0.02), 546);
        const inscriptionPostage = 546; // Dust limit estimate (actual value comes from backend)
        const estimatedVsize = 300; // Typical atomic swap TX size
        const minerFee = estimatedVsize * feeRate; // Use user-selected fee rate
        const totalNeeded = priceSats + marketFee + minerFee + inscriptionPostage;
        
        console.log(`💰 Total needed estimate: ${totalNeeded} sats`);
        console.log(`   Price: ${priceSats}, Market Fee: ${marketFee}, Miner Fee: ${minerFee} (${feeRate} sat/vB), Postage: ${inscriptionPostage}`);
        
        // Select UTXOs
        let selectedUtxos = [];
        let totalInput = 0;
        
        for (const utxo of pureUtxos) {
            if (totalInput >= totalNeeded) break;
            
            // ⚠️ DEBUG: Verificar se scriptPubKey existe
            if (!utxo.scriptPubKey) {
                console.error('❌ UTXO missing scriptPubKey:', utxo);
                throw new Error(`UTXO ${utxo.txid}:${utxo.vout} is missing scriptPubKey`);
            }
            
            selectedUtxos.push({
                txid: utxo.txid,
                vout: utxo.vout,
                value: utxo.value,
                script_pubkey: utxo.scriptPubKey
            });
            totalInput += utxo.value;
        }
        
        if (totalInput < totalNeeded) {
            throw new Error(`Insufficient funds. Need ${totalNeeded} sats, have ${totalInput} sats`);
        }
        
        console.log(`✅ Selected ${selectedUtxos.length} UTXOs (total: ${totalInput} sats)`);
        
        // 📦 Step 2: Prepare purchase (get PSBT to sign)
        console.log('📦 Preparing purchase...');
        console.log(`   Using fee rate: ${feeRate} sat/vB`);
        
        // ✅ Endpoint correto: /:id/buy (não /buy/prepare)
        const prepareResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${orderId}/buy`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                buyer_address: buyerAddress,
                buyer_utxos: selectedUtxos.map(u => ({
                    txid: u.txid,
                    vout: u.vout,
                    value: u.value,
                    scriptPubKey: u.script_pubkey
                })),
                fee_rate: feeRate
            })
        });
        
        if (!prepareResponse.ok) {
            const errorText = await prepareResponse.text();
            console.error('❌ Prepare response error:', errorText);
            try {
                const error = JSON.parse(errorText);
                throw new Error(error.error || 'Failed to prepare purchase');
            } catch (e) {
                throw new Error('Failed to prepare purchase: ' + errorText.substring(0, 100));
            }
        }
        
        const prepareData = await prepareResponse.json();
        console.log('✅ Purchase PSBT prepared:', {
            order_id: prepareData.order_id,
            inputs_to_sign: prepareData.inputs_to_sign,
            breakdown: prepareData.breakdown
        });
        
        // 🖊️ Step 3: Save PSBT request for signing
        // ⚠️ IMPORTANTE: Buyer só assina seus próprios inputs (retornados pelo backend)
        // O input[0] é do seller e já está assinado!
        const buyerInputIndexes = prepareData.inputs_to_sign || [];
        
        console.log(`✍️  Buyer will sign inputs: [${buyerInputIndexes.join(', ')}]`);
        console.log(`📊 Breakdown:`, prepareData.breakdown);
        
        pendingPsbtRequest = {
            psbt: prepareData.psbt_base64,
            type: 'buyAtomicSwap',
            orderId,
            buyerAddress: buyerAddress,
            inputsToSign: buyerInputIndexes,
            breakdown: prepareData.breakdown,
            feeRate: feeRate,
            // 🔐 GUARDIAN BUILD MODEL: Store seller data for final TX construction
            sellerSignatureHex: prepareData.seller_signature_hex,
            sellerTxContext: prepareData.seller_tx_context, // Full TX context from seller
            model: prepareData.model || 'GUARDIAN_BUILD',
            // 🔐 SIGHASH: Buyer signs with ALL|ANYONECANPAY (0x81)
            sighashType: 0x81,
            timestamp: Date.now()
        };
        
        // Salvar no storage também
        await chrome.storage.local.set({ pendingPsbtRequest });
        console.log('💾 Pending PSBT saved for buyer signature');
        
        // 4. Open popup for signature
        try {
            await chrome.action.openPopup();
        } catch (e) {
            console.log('⚠️  Could not auto-open popup:', e.message);
        }
        
        return {
            success: true,
            requiresSignature: true,
            orderId: orderId,
            message: 'Click the Kray Wallet extension icon to sign the purchase'
        };
        
    } catch (error) {
        console.error('❌ Error buying atomic swap:', error);
        throw error;
    }
}

async function cancelListing({ orderId, inscriptionId }) {
    try {
        console.log('\n❌ ===== CANCEL LISTING =====');
        console.log('   Order ID:', orderId);
        console.log('   Inscription ID:', inscriptionId);
        console.log('   Wallet address:', walletState.address);
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        // If we have inscriptionId but no orderId, find the listing
        if (!orderId && inscriptionId) {
            console.log('🔍 Finding listing by inscription ID...');
            const listingsResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now`);
            const listingsData = await listingsResponse.json();
            
            const listing = listingsData.listings?.find(l => 
                l.inscription_id === inscriptionId && 
                ['OPEN', 'PENDING'].includes(l.status)
            );
            
            if (listing) {
                orderId = listing.order_id;
                console.log('   Found order:', orderId);
            }
        }
        
        if (!orderId) {
            throw new Error('Listing not found');
        }
        
        // Check if it's a buy-now listing
        const isBuyNow = orderId.startsWith('buynow_');
        
        if (isBuyNow) {
            // Simple DELETE for buy-now (no signature required)
            console.log('🛒 Cancelling buy-now listing...');
            const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${orderId}`, {
                method: 'DELETE',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    seller_address: walletState.address
                })
            });
            
            if (!response.ok) {
                const error = await response.json();
                throw new Error(error.error || 'Failed to cancel listing');
            }
            
            const data = await response.json();
            console.log('✅ Buy-now listing cancelled!');
            
            return {
                success: true,
                message: data.message || 'Listing cancelled'
            };
        }
        
        // Legacy: For old atomic-swap listings that require signature
        console.log('🔐 Cancelling legacy listing with signature...');
        const message = `I cancel this listing: ${orderId} - ${Date.now()}`;
        
        pendingMessageRequest = {
            message: message,
            timestamp: Date.now(),
            action: 'CANCEL_LISTING',
            orderId: orderId
        };
        
        await chrome.storage.local.set({ pendingMessageRequest });
        
        try {
            await chrome.action.openPopup();
        } catch (err) {
            console.log('⚠️  User must click extension icon to sign');
        }
        
        const signatureResult = await waitForMessageSign();
        
        if (!signatureResult?.success) {
            throw new Error('Failed to sign cancellation message');
        }
        
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${orderId}/cancel`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                seller_address: walletState.address,
                signature: signatureResult.signature,
                message: message
            })
        });
        
        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Failed to cancel listing');
        }
        
        const data = await response.json();
        console.log('✅ Legacy listing cancelled!');
        
        return {
            success: true,
            message: data.message || 'Listing cancelled'
        };
        
    } catch (error) {
        console.error('❌ Error cancelling listing:', error);
        throw error;
    }
}

/**
 * Update listing price
 */
async function updateListingPrice({ inscriptionId, newPrice }) {
    try {
        console.log('\n💰 ===== UPDATE LISTING PRICE =====');
        console.log('   Inscription:', inscriptionId);
        console.log('   New Price:', newPrice, 'sats');
        console.log('   Wallet:', walletState.address);
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        if (!newPrice || newPrice < 546) {
            throw new Error('Price must be at least 546 sats');
        }
        
        // Find the listing for this inscription
        const listingsResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now?inscription_id=${inscriptionId}`);
        const listingsData = await listingsResponse.json();
        
        const listing = listingsData.listings?.find(l => 
            l.inscription_id === inscriptionId && 
            ['OPEN', 'PENDING'].includes(l.status)
        );
        
        if (!listing) {
            throw new Error('Listing not found');
        }
        
        // Update the price
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${listing.order_id}/price`, {
            method: 'PUT',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                new_price: newPrice,
                seller_address: walletState.address
            })
        });
        
        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Failed to update price');
        }
        
        const data = await response.json();
        console.log('✅ Price updated!');
        console.log('   Old:', data.old_price, 'sats');
        console.log('   New:', data.new_price, 'sats');
        
        return {
            success: true,
            order_id: listing.order_id,
            old_price: data.old_price,
            new_price: data.new_price,
            requires_signature: data.requires_signature,
            message: data.message
        };
        
    } catch (error) {
        console.error('❌ Error updating price:', error);
        throw error;
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 🛒 BUY NOW MODE - KrayWallet Atomic Swap
// ═══════════════════════════════════════════════════════════════════════════

/**
 * Create Buy Now Listing (KRAYWALLET MODEL)
 * 
 * FLOW:
 * 1. Backend returns PSBT for seller to sign
 * 2. Seller signs with SIGHASH_SINGLE|ANYONECANPAY (0x83)
 * 3. Signed PSBT sent back to confirm listing
 * 4. Listing is LIVE! Seller goes away.
 * 5. When buyer purchases, we use the stored signature.
 */
/**
 * 🚀 UNIFIED: Create AND Sign listing in ONE step!
 * No more two popups - everything happens at once.
 */
async function createAndSignListing({ inscriptionId, inscriptionNumber, priceSats, description, password }) {
    try {
        console.log('\n🚀 ===== CREATE AND SIGN LISTING (UNIFIED FLOW) =====');
        console.log('   Inscription:', inscriptionId);
        console.log('   Inscription #:', inscriptionNumber);
        console.log('   Price:', priceSats, 'sats');
        console.log('   Password:', password ? '✅ PROVIDED' : '❌ MISSING');
        console.log('   Wallet address:', walletState.address);
        
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        if (!password) {
            throw new Error('Password required to sign listing');
        }
        
        // ═══════════════════════════════════════════════════════════════
        // STEP 1: Create listing on backend (get PSBT to sign)
        // ═══════════════════════════════════════════════════════════════
        console.log('📤 Step 1: Creating listing on backend...');
        const createResponse = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/list', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                inscription_id: inscriptionId,
                inscription_number: inscriptionNumber || null,
                price_sats: priceSats,
                seller_address: walletState.address,
                description: description || ''
            })
        });
        
        if (!createResponse.ok) {
            const errorData = await createResponse.json();
            // If already listed, return the details so popup can show options
            if (errorData.error === 'already_listed') {
                return {
                    success: false,
                    error: 'already_listed',
                    message: errorData.message,
                    existing_order_id: errorData.existing_order_id,
                    current_price: errorData.current_price,
                    status: errorData.status
                };
            }
            throw new Error(errorData.error || 'Failed to create listing');
        }
        
        const createData = await createResponse.json();
        console.log('✅ Listing created (PENDING)');
        console.log('   Order ID:', createData.order_id);
        
        if (!createData.psbt_base64) {
            throw new Error('Backend did not return PSBT to sign');
        }
        
        // ═══════════════════════════════════════════════════════════════
        // STEP 2: Sign the PSBT with password (internal signing)
        // ═══════════════════════════════════════════════════════════════
        console.log('🔐 Step 2: Signing PSBT internally...');
        
        // Decrypt wallet with password
        // Note: wallet key is 'walletEncrypted', not 'encryptedWallet'
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        if (!storage.walletEncrypted) {
            throw new Error('No wallet found. Please restore or create a wallet first.');
        }
        
        // Use decryptData function (not decryptWallet)
        const decrypted = await decryptData(storage.walletEncrypted, password);
        if (!decrypted || !decrypted.mnemonic) {
            throw new Error('Failed to decrypt wallet - wrong password?');
        }
        
        console.log('✅ Wallet decrypted successfully');
        
        // Sign the PSBT
        const signedPsbt = await signPsbtInternal(createData.psbt_base64, decrypted.mnemonic, createData.toSignInputs);
        
        // Clear mnemonic from memory immediately
        decrypted.mnemonic = null;
        console.log('🗑️ Mnemonic cleared from memory');
        
        if (!signedPsbt) {
            throw new Error('Failed to sign PSBT');
        }
        
        console.log('✅ PSBT signed successfully');
        console.log('   Signed PSBT length:', signedPsbt.length);
        
        // ═══════════════════════════════════════════════════════════════
        // STEP 3: Confirm listing with signed PSBT
        // ═══════════════════════════════════════════════════════════════
        console.log('📤 Step 3: Confirming listing with signature...');
        
        const confirmResponse = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/list', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                order_id: createData.order_id,
                signed_psbt: signedPsbt  // Backend expects 'signed_psbt', not 'seller_signed_psbt'
            })
        });
        
        if (!confirmResponse.ok) {
            const error = await confirmResponse.json();
            throw new Error(error.error || 'Failed to confirm listing');
        }
        
        const confirmData = await confirmResponse.json();
        console.log('🎉 Listing is LIVE!');
        console.log('   Order ID:', confirmData.order_id);
        console.log('   Status:', confirmData.status);
        
        return {
            success: true,
            order_id: confirmData.order_id || createData.order_id,
            inscription_id: inscriptionId,
            price_sats: priceSats,
            status: 'OPEN',
            message: 'Listing is LIVE! Buyers can purchase anytime.'
        };
        
    } catch (error) {
        console.error('❌ Error in createAndSignListing:', error);
        throw error;
    }
}

/**
 * Internal PSBT signing - 100% LOCAL (secure)
 */
async function signPsbtInternal(psbtBase64, mnemonic, inputsToSign, sighashType = 0x82) {
    try {
        console.log('🔐 Signing PSBT 100% LOCALLY (internal)...');
        console.log('   inputsToSign:', inputsToSign ? JSON.stringify(inputsToSign) : 'default [0]');
        console.log('   sighashType: 0x' + sighashType.toString(16));
        
        // Parse inputsToSign to extract indices
        let inputIndices = null;
        if (inputsToSign && Array.isArray(inputsToSign)) {
            inputIndices = inputsToSign.map(i => typeof i === 'number' ? i : i.index);
        }
        
        // 🔐 USE LOCAL SIGNER - MNEMONIC NEVER LEAVES DEVICE!
        if (TaprootSigner && TaprootSigner.signPsbtTaprootLocal) {
            console.log('   ✅ Using LOCAL signer - Mnemonic stays on device!');
            
            const signResult = await TaprootSigner.signPsbtTaprootLocal(
                mnemonic,
                psbtBase64,
                {
                    inputsToSign: inputIndices,
                    sighashType: sighashType,
                    autoFinalize: false // Marketplace listings não finalizam
                }
            );
            
            console.log('✅ PSBT signed 100% LOCALLY');
            console.log('   Signed PSBT length:', signResult.signedPsbt?.length || 0);
            
            return signResult.signedPsbt;
        } else {
            // 🔐 LEI PRIMORDIAL: NUNCA enviar mnemonic para servidor!
            console.error('❌ LOCAL SIGNER NOT AVAILABLE!');
            console.error('   🔐 LEI PRIMORDIAL: Mnemonic protection active');
            throw new Error('Local signer not loaded. Cannot sign securely. Please reload the extension.');
        }
        
    } catch (error) {
        console.error('❌ Error signing PSBT:', error);
        throw error;
    }
}

/**
 * 🔐 Sign PSBT for marketplace listing
 * Tries LOCAL signing first, falls back to backend if needed
 * 
 * ⚠️ TEMPORARY: Backend fallback will be removed once local signer works
 */
async function signPsbtForListingAction({ psbtBase64, mnemonic, toSignInputs }) {
    try {
        console.log('\n🔐 ===== SIGN PSBT FOR LISTING (LOCAL ONLY) =====');
        
        // Parse inputsToSign
        const inputIndices = toSignInputs?.map(i => typeof i === 'number' ? i : i.index) || [0];
        const sighashType = toSignInputs?.[0]?.sighashTypes?.[0] || 0x82; // SIGHASH_NONE|ANYONECANPAY
        
        console.log('   inputsToSign:', inputIndices);
        console.log('   sighashType: 0x' + sighashType.toString(16));
        
        // LEI PRIMORDIAL: Sign LOCALLY ONLY - NO backend fallback!
        if (!TaprootSigner || !TaprootSigner.signPsbtTaprootLocal) {
            throw new Error('Local signer not loaded. Cannot sign securely.');
        }
        
        console.log('   🔐 Signing PSBT locally (mnemonic never leaves device)...');
        const signResult = await TaprootSigner.signPsbtTaprootLocal(
            mnemonic,
            psbtBase64,
            {
                inputsToSign: inputIndices,
                sighashType: sighashType,
                autoFinalize: false
            }
        );
        
        if (!signResult || !signResult.signedPsbt) {
            throw new Error('Local signing returned empty result');
        }
        
        console.log('   ✅ PSBT signed LOCALLY!');
        return { success: true, signedPsbt: signResult.signedPsbt };
        
    } catch (error) {
        console.error('❌ Error in signPsbtForListingAction:', error);
        return { success: false, error: error.message };
    }
}

// LEGACY: Keep old function for backwards compatibility
async function createBuyNowListing({ inscriptionId, inscriptionNumber, priceSats, description, step, orderId, signedPsbt }) {
    try {
        console.log('\n📝 ===== CREATE BUY NOW LISTING (LEGACY) =====');
        console.log('   Inscription:', inscriptionId);
        console.log('   Inscription #:', inscriptionNumber);
        console.log('   Price:', priceSats, 'sats');
        console.log('   Step:', step || 'create');
        console.log('   Wallet address:', walletState.address);
        
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        // ═══════════════════════════════════════════════════════════════
        // STEP 2: Confirm listing with signed PSBT
        // ═══════════════════════════════════════════════════════════════
        if (step === 'confirm' && orderId && signedPsbt) {
            console.log('📤 Confirming listing with signed PSBT...');
            
            const confirmResponse = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/list', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    inscription_id: inscriptionId,
                    price_sats: priceSats,
                    seller_address: walletState.address,
                    order_id: orderId,
                    seller_signed_psbt: signedPsbt
                })
            });
            
            if (!confirmResponse.ok) {
                const error = await confirmResponse.json();
                throw new Error(error.error || 'Failed to confirm listing');
            }
            
            const confirmData = await confirmResponse.json();
            console.log('✅ Listing confirmed and LIVE!');
            console.log('   Order ID:', confirmData.order_id);
            
            return {
                success: true,
                order_id: confirmData.order_id,
                inscription_id: inscriptionId,
                price_sats: priceSats,
                status: 'OPEN',
                message: 'Listing is LIVE! Buyers can purchase anytime.'
            };
        }
        
        // ═══════════════════════════════════════════════════════════════
        // STEP 1: Create listing and get PSBT to sign
        // ═══════════════════════════════════════════════════════════════
        if (!inscriptionId) {
            throw new Error('Missing inscription ID');
        }
        
        if (!priceSats || priceSats < 546) {
            throw new Error('Price must be at least 546 sats');
        }
        
        console.log('📤 Creating listing on backend...');
        const response = await fetch('https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/list', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                inscription_id: inscriptionId,
                inscription_number: inscriptionNumber || null,
                price_sats: priceSats,
                seller_address: walletState.address,
                description: description || ''
            })
        });
        
        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Failed to create listing');
        }
        
        const data = await response.json();
        console.log('📋 Listing created, PSBT ready for signature');
        console.log('   Order ID:', data.order_id);
        console.log('   Status:', data.status);
        console.log('   SIGHASH: NONE|ANYONECANPAY (0x82) - SECURE MODEL');
        
        // If status is PENDING, we need to sign the PSBT
        if (data.status === 'PENDING' && data.psbt_base64) {
            console.log('🔐 Saving PSBT for seller signature...');
            
            // Save pending PSBT request for signing
            pendingPsbtRequest = {
                psbt: data.psbt_base64,
                type: 'createBuyNowListing',
                orderId: data.order_id,
                inscriptionId: inscriptionId,
                priceSats: priceSats,
                inputsToSign: data.toSignInputs || [{
                    index: 0,
                    sighashTypes: [0x82] // SIGHASH_NONE|ANYONECANPAY - Secure model
                }],
                timestamp: Date.now()
            };
            
            await chrome.storage.local.set({ pendingPsbtRequest });
            console.log('💾 PSBT saved for signature');
            
            // Open popup for signature
            try {
                await chrome.action.openPopup();
            } catch (e) {
                console.log('⚠️ Could not auto-open popup:', e.message);
            }
            
            return {
                success: true,
                requiresSignature: true,
                order_id: data.order_id,
                inscription_id: inscriptionId,
                price_sats: priceSats,
                status: 'PENDING',
                message: 'Sign the listing to activate it. Click the wallet extension icon.'
            };
        }
        
        // If already OPEN (shouldn't happen, but handle it)
        return {
            success: true,
            order_id: data.order_id,
            inscription_id: inscriptionId,
            price_sats: priceSats,
            status: data.status,
            message: 'Listing is live! Buyers can purchase anytime.'
        };
        
    } catch (error) {
        console.error('❌ Error creating Buy Now listing:', error);
        throw error;
    }
}

/**
 * Buy Now - Buyer purchases a listed inscription (KRAYWALLET MODEL)
 * 
 * FLOW:
 * 1. Get PSBT from backend (seller's signature already injected!)
 * 2. Buyer signs their inputs
 * 3. Backend broadcasts IMMEDIATELY!
 */
async function buyNow({ orderId, buyerAddress, feeRate: requestedFeeRate }) {
    try {
        console.log('\n🛒 ===== BUY NOW PURCHASE =====');
        console.log('   Order ID:', orderId);
        console.log('   Buyer address:', buyerAddress || walletState.address);
        console.log('   Requested fee rate:', requestedFeeRate || 'not specified');
        console.log('   Wallet unlocked:', walletState.unlocked);
        
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        const actualBuyerAddress = buyerAddress || walletState.address;
        
        if (!orderId) {
            throw new Error('Missing order ID');
        }
        
        // 🔍 Step 1: Get fee rate (use requested or fetch recommended)
        console.log('💰 Determining fee rate...');
        let feeRate = requestedFeeRate || 5;
        
        if (requestedFeeRate) {
            console.log(`✅ Using user-requested fee rate: ${feeRate} sat/vB`);
        } else {
            try {
                const feesResponse = await fetch('https://kraywallet-backend.onrender.com/api/wallet/fees');
                if (feesResponse.ok) {
                    const feesData = await feesResponse.json();
                    feeRate = feesData.fees?.medium || 5;
                    console.log(`✅ Using recommended fee rate: ${feeRate} sat/vB`);
                }
            } catch (e) {
                console.warn('⚠️ Using default fee rate:', feeRate);
            }
        }
        
        // 🔍 Step 2: Get buyer UTXOs
        console.log('🔍 Fetching buyer UTXOs...');
        const utxosResponse = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/utxos/${actualBuyerAddress}`);
        
        if (!utxosResponse.ok) {
            throw new Error('Failed to fetch buyer UTXOs');
        }
        
        const utxosData = await utxosResponse.json();
        const utxos = utxosData.utxos || [];
        console.log(`✅ Found ${utxos.length} total UTXOs`);
        
        // 🔒 LEI SUPREMA: Filter pure UTXOs (no inscriptions/runes)
        // Check BOTH flag format AND object format
        const pureUtxos = utxos.filter(u => {
            const hasInscription = u.hasInscription || (u.inscriptions && u.inscriptions.length > 0);
            const hasRunes = u.hasRunes || (u.runes && Object.keys(u.runes).length > 0);
            return !hasInscription && !hasRunes;
        });
        console.log(`✅ Found ${pureUtxos.length} pure BTC UTXOs`);
        
        if (pureUtxos.length === 0) {
            throw new Error('No pure Bitcoin UTXOs available. Please send some BTC without inscriptions/runes.');
        }
        
        // 🔍 Step 3: Get listing details to calculate required amount
        const listingResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${orderId}`);
        if (!listingResponse.ok) {
            throw new Error('Listing not found');
        }
        const listingData = await listingResponse.json();
        const listing = listingData.listing;
        
        const price = listing.price_sats;
        const marketFee = Math.max(Math.floor(price * 0.02), 546);
        const estimatedVsize = 300; // Typical atomic swap TX size
        const minerFee = estimatedVsize * feeRate; // Use user-selected fee rate
        const postage = 546;
        const totalNeeded = price + marketFee + minerFee + postage;
        
        console.log(`💰 Total needed: ${totalNeeded} sats`);
        console.log(`   Price: ${price}, Market Fee: ${marketFee}, Miner Fee: ${minerFee} (${feeRate} sat/vB), Postage: ${postage}`);
        
        // Select UTXOs
        let selectedUtxos = [];
        let totalInput = 0;
        
        for (const utxo of pureUtxos) {
            if (totalInput >= totalNeeded) break;
            
            if (!utxo.scriptPubKey) {
                console.warn(`⚠️ UTXO ${utxo.txid}:${utxo.vout} missing scriptPubKey, skipping`);
                continue;
            }
            
            selectedUtxos.push({
                txid: utxo.txid,
                vout: utxo.vout,
                value: utxo.value,
                scriptPubKey: utxo.scriptPubKey
            });
            totalInput += utxo.value;
        }
        
        if (totalInput < totalNeeded) {
            throw new Error(`Insufficient funds: need ${totalNeeded} sats, have ${totalInput} sats`);
        }
        
        console.log(`✅ Selected ${selectedUtxos.length} UTXOs (total: ${totalInput} sats)`);
        
        // 📦 Step 4: Create purchase PSBT
        console.log('📦 Creating purchase PSBT...');
        const buyResponse = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/buy-now/${orderId}/buy`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                buyer_address: actualBuyerAddress,
                buyer_utxos: selectedUtxos,
                fee_rate: feeRate
            })
        });
        
        if (!buyResponse.ok) {
            const error = await buyResponse.json();
            throw new Error(error.error || 'Failed to create purchase');
        }
        
        const buyData = await buyResponse.json();
        console.log('✅ Purchase PSBT created');
        console.log('   Purchase ID:', buyData.purchase_id);
        console.log('   Inputs to sign:', buyData.toSignInputs);
        console.log('   Breakdown:', buyData.breakdown);
        
        // 💾 Step 5: Save pending PSBT for signing
        const buyerInputIndexes = buyData.toSignInputs.map(i => i.index);
        
        pendingPsbtRequest = {
            psbt: buyData.psbt_base64,
            type: 'buyNow',
            orderId: orderId,
            purchaseId: buyData.purchase_id,
            buyerAddress: actualBuyerAddress,
            sellerAddress: buyData.seller_address,
            // 🖼️ Inscription details for UI display
            inscriptionId: buyData.inscription_id || buyData.inscriptionId,
            inscriptionNumber: buyData.inscription_number || buyData.inscriptionNumber,
            price: buyData.price || buyData.breakdown?.price,
            inputsToSign: buyerInputIndexes.map(idx => ({
                index: idx,
                sighashTypes: [0x01] // SIGHASH_ALL
            })),
            breakdown: buyData.breakdown,
            feeRate: feeRate,
            timestamp: Date.now()
        };
        
        await chrome.storage.local.set({ pendingPsbtRequest });
        console.log('💾 Pending PSBT saved for buyer signature');
        
        // Open popup for signature
        try {
            await chrome.action.openPopup();
        } catch (e) {
            console.log('⚠️ Could not auto-open popup:', e.message);
        }
        
        return {
            success: true,
            requiresSignature: true,
            orderId: orderId,
            purchaseId: buyData.purchase_id,
            breakdown: buyData.breakdown,
            message: 'Click the Kray Wallet extension icon to sign the purchase'
        };
        
    } catch (error) {
        console.error('❌ Error in Buy Now:', error);
        throw error;
    }
}

/**
 * Accept Buy Now - Seller accepts and signs the purchase
 * Called when seller wants to accept a pending purchase
 */
async function acceptBuyNow({ orderId, purchaseId, buyerSignedPsbt }) {
    try {
        console.log('\n✅ ===== ACCEPT BUY NOW =====');
        console.log('   Order ID:', orderId);
        console.log('   Purchase ID:', purchaseId);
        console.log('   Seller address:', walletState.address);
        
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked. Please unlock your wallet first.');
        }
        
        if (!buyerSignedPsbt) {
            throw new Error('Missing buyer signed PSBT');
        }
        
        // Save pending PSBT for seller to sign
        pendingPsbtRequest = {
            psbt: buyerSignedPsbt,
            type: 'acceptBuyNow',
            orderId: orderId,
            purchaseId: purchaseId,
            inputsToSign: [{
                index: 0, // Seller's input is always index 0
                sighashTypes: [0x01] // SIGHASH_ALL
            }],
            timestamp: Date.now()
        };
        
        await chrome.storage.local.set({ pendingPsbtRequest });
        console.log('💾 Pending PSBT saved for seller signature');
        
        // Open popup for signature
        try {
            await chrome.action.openPopup();
        } catch (e) {
            console.log('⚠️ Could not auto-open popup:', e.message);
        }
        
        return {
            success: true,
            requiresSignature: true,
            message: 'Click the Kray Wallet extension icon to accept and sign'
        };
        
    } catch (error) {
        console.error('❌ Error accepting Buy Now:', error);
        throw error;
    }
}

async function updateListingPrice({ orderId, newPrice }) {
    try {
        console.log('\n💰 ===== UPDATE LISTING PRICE (WITH SIGNATURE) =====');
        console.log('   Order ID:', orderId);
        console.log('   New Price:', newPrice, 'sats');
        console.log('   Wallet address:', walletState.address);
        
        if (!walletState.address) {
            throw new Error('No wallet address found');
        }
        
        // 🔐 STEP 1: Prepare message and FORCE popup to open
        const message = `I update this listing price: ${orderId} to ${newPrice} sats - ${Date.now()}`;
        console.log('🔐 Preparing price update message...');
        
        // Save pending request FIRST
        pendingMessageRequest = {
            message: message,
            timestamp: Date.now(),
            action: 'UPDATE_PRICE',
            orderId: orderId,
            newPrice: newPrice
        };
        
        await chrome.storage.local.set({ pendingMessageRequest });
        console.log('💾 Pending message request saved');
        
        // 🚀 FORCE popup to open
        try {
            console.log('🚀 Attempting to open popup...');
            await chrome.action.openPopup();
            console.log('✅ Popup opened via chrome.action.openPopup()');
        } catch (err) {
            console.log('⚠️  chrome.action.openPopup() failed');
            console.log('   User must click the extension icon to sign');
            // O pendingMessageRequest já foi salvo, popup vai detectar quando abrir
        }
        
        // 🔐 STEP 2: Wait for signature from popup
        console.log('⏳ Waiting for user signature...');
        const signatureResult = await waitForMessageSign();
        
        if (!signatureResult || !signatureResult.success) {
            throw new Error('Failed to sign price update message');
        }
        
        console.log('✅ Signature created:', signatureResult.signature.substring(0, 20) + '...');
        
        // 🔐 STEP 2: Send signed update to backend
        const response = await fetch(`https://kraywallet-backend.onrender.com/api/atomic-swap/${orderId}/update-price`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                new_price_sats: newPrice,
                seller_address: walletState.address,
                signature: signatureResult.signature,
                message: message
            })
        });
        
        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Failed to update price');
        }
        
        const data = await response.json();
        console.log('✅ Price updated successfully with signature verification!');
        console.log('   Old price:', data.old_price, 'sats');
        console.log('   New price:', data.new_price, 'sats');
        
        // Save new PSBT for signing
        if (data.template_psbt_base64) {
            pendingPsbtRequest = {
                psbt: data.template_psbt_base64,
                type: 'createOffer',
                sighashType: 'NONE|ANYONECANPAY', // ARA MODEL
                order_id: orderId,
                timestamp: Date.now()
            };
            
            await chrome.storage.local.set({ pendingPsbtRequest });
            console.log('💾 New PSBT saved for signing');
            
            // Try to open popup
            try {
                await chrome.action.openPopup();
            } catch (e) {
                console.log('⚠️  Could not auto-open popup:', e.message);
            }
            
            return {
                success: true,
                requiresSignature: true,
                message: 'Price updated. Please sign the new PSBT to activate.',
                old_price: data.old_price,
                new_price: data.new_price
            };
        }
        
        return {
            success: true,
            message: data.message || 'Price updated'
        };
        
    } catch (error) {
        console.error('❌ Error updating listing price:', error);
        throw error;
    }
}

async function sendBitcoin({ toAddress, amount, feeRate, password }) {
    try {
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked');
        }
        
        console.log('💸 Sending Bitcoin (LOCAL SIGNING)...');
        console.log('  To:', toAddress);
        console.log('  Amount:', amount, 'sats');
        console.log('  Fee rate:', feeRate, 'sat/vB');
        
        if (!password) {
            throw new Error('Password is required to sign transaction');
        }
        
        // Decrypt wallet to get mnemonic (stays local!)
        const result = await chrome.storage.local.get(['walletEncrypted']);
        if (!result.walletEncrypted) {
            throw new Error('Wallet not found in storage');
        }
        
        let mnemonic;
        try {
            const decrypted = await decryptData(result.walletEncrypted, password);
            mnemonic = decrypted.mnemonic;
        } catch (decryptError) {
            throw new Error('Incorrect password. Please try again.');
        }
        
        // Step 1: Build PSBT on server (NO mnemonic sent!)
        console.log('🔧 Building PSBT (no mnemonic sent to server)...');
        const buildResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/build-send-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                fromAddress: walletState.address,
                toAddress,
                amount,
                feeRate,
                network: 'mainnet'
            })
        });
        
        const buildData = await buildResponse.json();
        if (!buildData.success) {
            throw new Error(buildData.error || 'Failed to build PSBT');
        }
        console.log('✅ PSBT built, fee:', buildData.fee);
        
        // Step 2: Sign PSBT LOCALLY (LEI PRIMORDIAL: Mnemonic NEVER leaves device!)
        console.log('🔐 Signing PSBT locally (mnemonic never leaves device)...');
        if (!TaprootSigner || !TaprootSigner.signPsbtTaprootLocal) {
            throw new Error('Local signer not loaded. Cannot sign securely.');
        }
        
        const signResult = await TaprootSigner.signPsbtTaprootLocal(
            mnemonic,
            buildData.psbt,
            {
                inputsToSign: null, // Sign all inputs
                sighashType: 0x01, // SIGHASH_ALL
                autoFinalize: true
            }
        );
        console.log('✅ PSBT signed locally');
        
        // Step 3: Finalize PSBT (extract transaction hex)
        console.log('🔨 Finalizing PSBT...');
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeData = await finalizeResponse.json();
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize PSBT');
        }
        console.log('✅ PSBT finalized, txid:', finalizeData.txid);
        
        // Step 4: Broadcast transaction
        console.log('📡 Broadcasting transaction...');
        const broadcastResponse = await fetch('https://kraywallet-backend.onrender.com/api/psbt/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: finalizeData.hex })
        });
        
        const broadcastData = await broadcastResponse.json();
        if (!broadcastData.success) {
            throw new Error(broadcastData.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Transaction broadcasted!');
        console.log('  TXID:', broadcastData.txid);
        
        return {
            success: true,
            txid: broadcastData.txid,
            fee: buildData.fee,
            change: buildData.change
        };
        
    } catch (error) {
        console.error('❌ Error sending bitcoin:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// Send Inscription (LOCAL SIGNING - LEI PRIMORDIAL)
async function sendInscription({ inscription, recipientAddress, feeRate, password }) {
    try {
        if (!walletState.unlocked) {
            throw new Error('Wallet is locked');
        }

        console.log('📤 SENDING INSCRIPTION (LOCAL SIGNING)');
        console.log('  Inscription ID:', inscription.id);
        console.log('  To:', recipientAddress);
        console.log('  Fee rate:', feeRate, 'sat/vB');
        
        if (!inscription.utxo || !inscription.utxo.txid || inscription.utxo.vout === undefined) {
            throw new Error('Inscription UTXO data is missing. Please refresh inscriptions.');
        }
        
        if (!password) {
            throw new Error('Password is required to sign inscription transaction');
        }
        
        // Decrypt wallet (stays local!)
        const result = await chrome.storage.local.get(['walletEncrypted']);
        if (!result.walletEncrypted) {
            throw new Error('Wallet not found in storage');
        }
        
        let mnemonic;
        try {
            const decrypted = await decryptData(result.walletEncrypted, password);
            mnemonic = decrypted.mnemonic;
        } catch (decryptError) {
            throw new Error('Incorrect password. Please try again.');
        }
        
        // Step 1: Build PSBT on server (NO mnemonic sent!)
        console.log('🔧 Building inscription PSBT (no mnemonic sent to server)...');
        const buildResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/build-send-inscription-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                fromAddress: walletState.address,
                inscription: {
                    id: inscription.id,
                    utxo: {
                        txid: inscription.utxo.txid,
                        vout: inscription.utxo.vout,
                        value: inscription.utxo.value || inscription.outputValue || 600
                    }
                },
                recipientAddress,
                feeRate,
                network: 'mainnet'
            })
        });
        
        const buildData = await buildResponse.json();
        if (!buildData.success) {
            throw new Error(buildData.error || 'Failed to build inscription PSBT');
        }
        console.log('✅ PSBT built, fee:', buildData.fee);
        
        // Step 2: Sign PSBT LOCALLY (LEI PRIMORDIAL: Mnemonic NEVER leaves device!)
        console.log('🔐 Signing inscription PSBT locally...');
        if (!TaprootSigner || !TaprootSigner.signPsbtTaprootLocal) {
            throw new Error('Local signer not loaded. Cannot sign securely.');
        }
        
        const signResult = await TaprootSigner.signPsbtTaprootLocal(
            mnemonic,
            buildData.psbt,
            {
                inputsToSign: null,
                sighashType: 0x01,
                autoFinalize: true
            }
        );
        console.log('✅ Inscription PSBT signed locally');
        
        // Step 3: Finalize PSBT
        console.log('🔨 Finalizing PSBT...');
        const finalizeResponse = await fetch('https://kraywallet-backend.onrender.com/api/kraywallet/finalize-psbt', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ psbt: signResult.signedPsbt })
        });
        
        const finalizeData = await finalizeResponse.json();
        if (!finalizeData.success) {
            throw new Error(finalizeData.error || 'Failed to finalize PSBT');
        }
        console.log('✅ PSBT finalized, txid:', finalizeData.txid);
        
        // Step 4: Broadcast
        console.log('📡 Broadcasting inscription transaction...');
        const broadcastResponse = await fetch('https://kraywallet-backend.onrender.com/api/psbt/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: finalizeData.hex })
        });
        
        const broadcastData = await broadcastResponse.json();
        if (!broadcastData.success) {
            throw new Error(broadcastData.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Inscription sent!');
        console.log('  TXID:', broadcastData.txid);
        
        return {
            success: true,
            txid: broadcastData.txid,
            fee: buildData.fee
        };
        
    } catch (error) {
        console.error('❌ Error sending inscription:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// Connect
async function connect() {
    try {
        if (!walletState.address) {
            const info = await getWalletInfo();
            if (!info.success) {
                throw new Error('No wallet found');
            }
            walletState.address = info.data.address;
        }
        
        return {
            success: true,
            accounts: [walletState.address]
        };
    } catch (error) {
        return {
            success: false,
            error: error.message
        };
    }
}

async function getAccounts() {
    return await connect();
}

async function getPublicKey() {
    try {
        if (!walletState.publicKey) {
            throw new Error('Wallet not initialized');
        }
        
        return {
            success: true,
            publicKey: walletState.publicKey
        };
    } catch (error) {
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🖼️ ORDINALS & RUNES
// ==========================================

// Get Inscriptions (Ordinals)
async function getInscriptions(params = {}) {
    try {
        let { address, offset = 0, limit = 100 } = params;
        
        // Se address não foi fornecido, pegar da wallet atual
        if (!address) {
            const walletInfo = await getWalletInfo();
            if (!walletInfo.success) {
                throw new Error('Wallet not found');
            }
            address = walletInfo.data.address;
        }
        
        console.log('🖼️  Fetching inscriptions for:', address);
        console.log('   📡 Using QuickNode via backend');
        console.log('   🔧 CODE VERSION: 2025-QUICKNODE ✅');
        
        // ✅ BUSCAR VIA QUICKNODE (nova rota wallet-inscriptions)
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), 60000); // 60s timeout (primeira vez demora)
        
        let response;
        try {
            response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/${address}/inscriptions`, {
                signal: controller.signal
            });
        } finally {
            clearTimeout(timeoutId);
        }
        
        console.log(`   ✅ Fetch completed, status: ${response.status}`);
        
        if (!response.ok) {
            throw new Error(`ORD server returned ${response.status}`);
        }
        
        const data = await response.json();
        console.log(`   ✅ JSON parsed, success: ${data.success}`);
        
        if (!data.success) {
            throw new Error(data.error || 'ORD server error');
        }
        
        console.log(`✅ Found ${data.inscriptions?.length || 0} inscriptions via LOCAL ORD server`);
        
        const apiInscriptions = (data.inscriptions || []).map(i => ({
            id: i.id || i.inscription_id,
            inscription_id: i.id || i.inscription_id,
            number: i.inscription_number || i.number,
            content_type: i.content_type || 'unknown',
            preview: `https://kraywallet-backend.onrender.com/api/rune-thumbnail/${i.id || i.inscription_id}`,
            output: i.output || `${i.txid}:${i.vout}`,
            outputValue: i.outputValue || i.value || parseInt(i.output_value) || 600,  // ✅ Try all possible fields!
            value: i.outputValue || i.value || parseInt(i.output_value) || 600,
            pending: false,
            sat: i.sat || null,  // 🌟 Sat number for rarity detection
            satName: i.satName || null
        }));
        
        // ✅ ADICIONAR INSCRIPTIONS PENDENTES DO CACHE
        let pendingInscriptions = await getPendingInscriptions(address);
        const apiIds = apiInscriptions.map(i => i.id);
        pendingInscriptions = pendingInscriptions.filter(p => !apiIds.includes(p.id));
        
        // Atualizar cache (remover inscriptions já indexadas)
        const storage = await chrome.storage.local.get(['pendingInscriptions']);
        const pending = storage.pendingInscriptions || {};
        if (pending[address]) {
            const before = pending[address].length;
            pending[address] = pendingInscriptions;
            if (before !== pending[address].length) {
                await chrome.storage.local.set({ pendingInscriptions: pending });
                console.log(`✅ Removed ${before - pending[address].length} inscriptions from cache (now indexed)`);
            }
        }
        
        // ✅ FILTRAR INSCRIPTIONS LISTADAS (seller não vê após criar oferta)
        let finalInscriptions = [...pendingInscriptions, ...apiInscriptions];
        
        console.log(`📊 Before filter - Total inscriptions: ${finalInscriptions.length}`);
        console.log('   Pending:', pendingInscriptions.length);
        console.log('   From API:', apiInscriptions.length);
        
        // Marketplace filter disabled (no /api/offers endpoint in production)
        // try {
        //     const offersResponse = await fetch('https://kraywallet-backend.onrender.com/api/offers');
        //     if (offersResponse.ok) {
        //         const offersData = await offersResponse.json();
        //         const offers = offersData.offers || offersData || [];
        //         const listedIds = offers.map(o => o.inscription_id);
        //         const beforeFilter = finalInscriptions.length;
        //         finalInscriptions = finalInscriptions.filter(i => !listedIds.includes(i.id));
        //         if (beforeFilter !== finalInscriptions.length) {
        //             console.log(`🔍 Filtered ${beforeFilter - finalInscriptions.length} listed inscriptions`);
        //         }
        //     }
        // } catch (filterError) {
        //     console.warn('⚠️  Could not filter listed inscriptions:', filterError.message);
        // }
        
        console.log(`📦 FINAL RESULT - Returning ${finalInscriptions.length} inscriptions`);
        console.log('   Inscriptions:', finalInscriptions.map(i => `${(i.id || 'unknown').substring(0, 16)}...`));
        
        return {
            success: true,
            address: address,
            inscriptions: finalInscriptions
        };
        
    } catch (error) {
        console.error('❌ Error fetching inscriptions from ORD server:', error);
        
        // Mesmo em caso de erro, retornar pending inscriptions do cache
        try {
            const pendingInscriptions = await getPendingInscriptions(address);
            return {
                success: false,
                error: error.message,
                address: address,
                inscriptions: pendingInscriptions
            };
        } catch (cacheError) {
            return {
                success: false,
                error: error.message,
                inscriptions: []
            };
        }
    }
}

// ==========================================
// 📦 CACHE LOCAL DE INSCRIPTIONS PENDENTES
// ==========================================

async function getPendingInscriptions(address) {
    try {
        const result = await chrome.storage.local.get(['pendingInscriptions']);
        const pending = result.pendingInscriptions || {};
        return pending[address] || [];
    } catch (error) {
        console.error('Error getting pending inscriptions:', error);
        return [];
    }
}

async function addPendingInscription(data) {
    try {
        console.log('📦 addPendingInscription called with:', data);
        
        // Data vem diretamente do marketplace (já é a inscription completa)
        const inscription = data;
        const address = data.address;
        
        if (!address || !inscription.id) {
            throw new Error('Missing address or inscription ID');
        }
        
        const storage = await chrome.storage.local.get(['pendingInscriptions']);
        const pending = storage.pendingInscriptions || {};
        
        if (!pending[address]) {
            pending[address] = [];
        }
        
        // Evitar duplicatas
        const exists = pending[address].some(p => p.id === inscription.id);
        if (!exists) {
            pending[address].push(inscription);
            await chrome.storage.local.set({ pendingInscriptions: pending });
            console.log(`✅ Added pending inscription ${inscription.id} to cache for ${address}`);
            console.log(`   Total pending: ${pending[address].length}`);
        } else {
            console.log(`ℹ️  Inscription ${inscription.id} already in cache`);
        }
        
        return { success: true };
    } catch (error) {
        console.error('❌ Error adding pending inscription:', error);
        return { success: false, error: error.message };
    }
}

async function removePendingInscription({ address, inscriptionId }) {
    try {
        const storage = await chrome.storage.local.get(['pendingInscriptions']);
        const pending = storage.pendingInscriptions || {};
        
        if (pending[address]) {
            const before = pending[address].length;
            pending[address] = pending[address].filter(p => p.id !== inscriptionId);
            const after = pending[address].length;
            
            if (before !== after) {
                await chrome.storage.local.set({ pendingInscriptions: pending });
                console.log(`✅ Removed pending inscription ${inscriptionId} from cache`);
            }
        }
        
        return { success: true };
    } catch (error) {
        console.error('Error removing pending inscription:', error);
        return { success: false, error: error.message };
    }
}

// Get Runes
async function getRunes(params = {}) {
    try {
        let { address } = params;
        
        if (!address) {
            const walletInfo = await getWalletInfo();
            if (!walletInfo.success) {
                throw new Error('Wallet not found');
            }
            address = walletInfo.data.address;
        }
        
        console.log('⚡ Fetching runes for:', address);
        console.log('   📡 Using QuickNode via backend');
        
        // ✅ BUSCAR VIA QUICKNODE (nova rota wallet-runes)
        try {
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 60000); // 60s timeout
            
            let response;
            try {
                response = await fetch(`https://kraywallet-backend.onrender.com/api/wallet/${address}/runes`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    signal: controller.signal
                });
            } finally {
                clearTimeout(timeoutId);
            }
            
            console.log(`   ✅ Fetch completed, status: ${response.status}`);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const data = await response.json();
            console.log(`   ✅ JSON parsed, success: ${data.success}`);
            
            if (data.success && data.runes) {
                console.log(`✅ Found ${data.runes.length} runes for address`);
                return {
                    success: true,
                    address: address,
                    runes: data.runes
                };
            } else {
                console.log('ℹ️  No runes found for address');
                return {
                    success: true,
                    address: address,
                    runes: []
                };
            }
        } catch (fetchError) {
            console.error('❌ Error fetching runes from backend:', fetchError);
            return {
                success: true,
                address: address,
                runes: []
            };
        }
    } catch (error) {
        console.error('Error fetching runes:', error);
        return {
            success: false,
            error: error.message,
            runes: []
        };
    }
}

// ==========================================
// 🔐 ENCRYPTION/DECRYPTION HELPERS
// 🔐 SECURITY: Random per-wallet salt (2026-01-31)
// Old format: iv.ciphertext (2 parts, hardcoded salt)
// New format: salt.iv.ciphertext (3 parts, random salt)
// Backward compatible: old wallets auto-migrate on next unlock
// ==========================================

// Legacy salt for backward compatibility with existing wallets
const LEGACY_SALT = 'kraywallet-salt';

/**
 * Derive encryption key from password and salt
 */
async function deriveKey(password, salt, usage) {
    const encoder = new TextEncoder();
    const passwordKey = await crypto.subtle.importKey(
        'raw',
        encoder.encode(password),
        { name: 'PBKDF2' },
        false,
        ['deriveKey']
    );
    
    return await crypto.subtle.deriveKey(
        {
            name: 'PBKDF2',
            salt: salt,
            iterations: 100000,
            hash: 'SHA-256'
        },
        passwordKey,
        { name: 'AES-GCM', length: 256 },
        false,
        [usage]
    );
}

// Criptografia
async function encryptData(data, password) {
    try {
        console.log('🔐 Encrypting wallet data...');
        
        if (!password || password.length < 6) {
            throw new Error('Password must be at least 6 characters');
        }
        
        // Usar WebCrypto API REAL
        const encoder = new TextEncoder();
        const dataString = JSON.stringify(data);
        
        console.log('✅ Data serialized, deriving key...');
        
        // 🔐 Generate random salt per wallet (16 bytes)
        const salt = crypto.getRandomValues(new Uint8Array(16));
        
        const key = await deriveKey(password, salt, 'encrypt');
        
        console.log('✅ Key derived with random salt, encrypting...');
        
        // Criptografar
        const iv = crypto.getRandomValues(new Uint8Array(12));
        const encryptedData = await crypto.subtle.encrypt(
            { name: 'AES-GCM', iv },
            key,
            encoder.encode(dataString)
        );
        
        console.log('✅ Data encrypted successfully (with random salt)');
        
        // 🔐 New format: salt.iv.ciphertext (3 parts)
        const result = btoa(String.fromCharCode(...salt)) + '.' +
               btoa(String.fromCharCode(...new Uint8Array(iv))) + '.' +
               btoa(String.fromCharCode(...new Uint8Array(encryptedData)));
        
        return result;
    } catch (error) {
        console.error('❌ Encryption error:', error);
        throw new Error('Failed to encrypt wallet data: ' + error.message);
    }
}

// Action para descriptografar wallet (retorna apenas mnemonic)
async function decryptWalletAction(password) {
    try {
        console.log('🔓 decryptWalletAction called');
        console.log('🔐 Password received in action:', password ? '***' : '(empty)');
        console.log('🔐 Password length in action:', password ? password.length : 0);
        
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        if (!result.walletEncrypted) {
            return {
                success: false,
                error: 'Wallet not found'
            };
        }
        
        console.log('✅ Wallet found, calling decryptData...');
        const decrypted = await decryptData(result.walletEncrypted, password);
        
        return {
            success: true,
            mnemonic: decrypted.mnemonic
        };
    } catch (error) {
        return {
            success: false,
            error: error.message
        };
    }
}

// Descriptografia
// 🔐 SECURITY: Supports both old (2 parts) and new (3 parts) format (2026-01-31)
async function decryptData(encryptedString, password) {
    try {
        console.log('🔓 Decrypting wallet data...');
        console.log('🔐 Password received:', password ? '***' : '(empty)');
        console.log('🔐 Password length:', password ? password.length : 0);
        console.log('🔐 Password type:', typeof password);
        
        // Trim whitespace
        password = password ? password.trim() : '';
        console.log('🔐 Password after trim length:', password.length);
        
        if (!password || password.length < 6) {
            throw new Error('Password must be at least 6 characters');
        }
        
        // Separar partes do dado criptografado
        const parts = encryptedString.split('.');
        
        let salt, iv, encryptedData, isLegacyFormat;
        const encoder = new TextEncoder();
        
        if (parts.length === 3) {
            // 🔐 New format: salt.iv.ciphertext
            console.log('🔐 Detected new format (random salt)');
            isLegacyFormat = false;
            salt = new Uint8Array(
                atob(parts[0]).split('').map(c => c.charCodeAt(0))
            );
            iv = new Uint8Array(
                atob(parts[1]).split('').map(c => c.charCodeAt(0))
            );
            encryptedData = new Uint8Array(
                atob(parts[2]).split('').map(c => c.charCodeAt(0))
            );
        } else if (parts.length === 2) {
            // Legacy format: iv.ciphertext (hardcoded salt)
            console.log('🔐 Detected legacy format (hardcoded salt) - will auto-migrate');
            isLegacyFormat = true;
            salt = encoder.encode(LEGACY_SALT);
            iv = new Uint8Array(
                atob(parts[0]).split('').map(c => c.charCodeAt(0))
            );
            encryptedData = new Uint8Array(
                atob(parts[1]).split('').map(c => c.charCodeAt(0))
            );
        } else {
            throw new Error('Invalid encrypted data format');
        }
        
        console.log('✅ Data parsed, deriving key...');
        
        const key = await deriveKey(password, salt, 'decrypt');
        
        console.log('✅ Key derived, decrypting...');
        
        // Descriptografar
        const decryptedData = await crypto.subtle.decrypt(
            { name: 'AES-GCM', iv },
            key,
            encryptedData
        );
        
        console.log('✅ Data decrypted successfully');
        
        // Converter de ArrayBuffer para string e parsear JSON
        const decoder = new TextDecoder();
        const dataString = decoder.decode(decryptedData);
        const data = JSON.parse(dataString);
        
        // 🔐 Auto-migrate legacy format to new format with random salt
        if (isLegacyFormat) {
            console.log('🔐 Auto-migrating wallet encryption to random salt...');
            try {
                const newEncrypted = await encryptData(data, password);
                await chrome.storage.local.set({ walletEncrypted: newEncrypted });
                console.log('✅ Wallet encryption migrated to random salt successfully');
            } catch (migrateError) {
                // Migration failed but decryption succeeded - not critical
                console.warn('⚠️ Salt migration failed (non-critical):', migrateError.message);
            }
        }
        
        return data;
    } catch (error) {
        console.error('❌ Decryption error:', error);
        console.error('❌ Error name:', error.name);
        console.error('❌ Error message:', error.message);
        
        // Se o erro já é sobre a senha, repassa
        if (error.message.includes('Password must be')) {
            throw error;
        }
        
        // Se erro de crypto, é senha errada
        if (error.name === 'OperationError' || error.message.includes('decrypt')) {
            throw new Error('Wrong password! Please check your password.');
        }
        
        // Outros erros
        throw new Error(`Decryption failed: ${error.message}`);
    }
}

// ==========================================
// 🚀 RUNES SEND - Sign & Broadcast
// ==========================================

/**
 * Assinar PSBT para envio de runes
 * @param {Object} psbt - PSBT construído pelo backend
 * @returns {Object} {success, signedHex}
 */
/**
 * Assinar PSBT de Runes (via confirmação de senha)
 * Similar ao signPsbt(), mas específico para runes
 */
async function signRunePSBT(psbt) {
    try {
        console.log('\n🔐 ===== SIGN RUNE PSBT CALLED =====');
        console.log('✍️  Signing RUNE PSBT (via popup confirmation)...');
        console.log('  PSBT length:', psbt?.length || 0);
        
        // Verificar se há wallet
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        if (!storage.walletEncrypted) {
            throw new Error('No wallet found. Please create a wallet first.');
        }
        console.log('✅ Wallet found in storage');
        
        // Guardar PSBT pending para o popup processar
        // Usar SIGHASH ALL (padrão para runes)
        pendingPsbtRequest = {
            psbt,
            inputsToSign: null, // Assinar todos os inputs
            sighashType: 'ALL', // SIGHASH ALL para runes
            autoFinalized: true,
            timestamp: Date.now(),
            isRuneTransfer: true // Flag para identificar que é rune
        };
        console.log('✅ pendingPsbtRequest saved (RUNE transfer)');
        
        // Abrir popup para confirmação
        console.log('📱 Opening popup for PSBT confirmation...');
        
        try {
            await chrome.action.openPopup();
            console.log('✅ Popup opened at standard extension position');
        } catch (err) {
            console.error('❌ Failed to open popup:', err);
            console.warn('⚠️  User may need to click the extension icon manually');
        }
        
        console.log('⏳ Waiting for user confirmation...');
        
        // Esperar pela resposta do popup (via chrome.storage)
        return new Promise((resolve) => {
            const checkInterval = setInterval(async () => {
                const result = await chrome.storage.local.get(['psbtSignResult']);
                
                if (result.psbtSignResult) {
                    clearInterval(checkInterval);
                    
                    // Limpar resultado
                    await chrome.storage.local.remove(['psbtSignResult', 'pendingPsbtRequest']); // ✅ Limpar ambos
                    
                    // Limpar pending request
                    pendingPsbtRequest = null;
                    
                    console.log('✅ PSBT sign completed!');
                    resolve(result.psbtSignResult);
                }
            }, 100);
            
            // Timeout após 5 minutos
            setTimeout(() => {
                clearInterval(checkInterval);
                pendingPsbtRequest = null;
                chrome.storage.local.remove('pendingPsbtRequest'); // ✅ Limpar do storage também
                console.log('⏱️  PSBT sign timeout');
                resolve({
                    success: false,
                    error: 'Signature request timeout'
                });
            }, 5 * 60 * 1000);
        });
        
    } catch (error) {
        console.error('❌ Error in signRunePSBT:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * Broadcast de transação assinada
 * @param {string} hex - Transação assinada em hex
 * @returns {Object} {success, txid}
 */
async function broadcastTransaction(hex) {
    try {
        console.log('\n📡 ========== BROADCASTING TRANSACTION ==========');
        console.log('Hex length:', hex?.length);
        
        // Broadcast via backend (Mempool.space + Blockstream.info APIs)
        console.log('📡 Sending to backend for broadcast...');
        
        const response = await fetch('https://kraywallet-backend.onrender.com/api/psbt/broadcast', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ hex: hex })
        });
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || 'Failed to broadcast transaction');
        }
        
        console.log('✅ Transaction broadcast successfully!');
        console.log('   TXID:', data.txid);
        console.log('========== BROADCAST COMPLETE ==========\n');
        
        return {
            success: true,
            txid: data.txid
        };
        
    } catch (error) {
        console.error('❌ Error broadcasting transaction:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🔒 LOCK/UNLOCK FUNCTIONS
// ==========================================

/**
 * Unlock wallet with password
 * @param {Object} data - { password }
 * @returns {Object} {success, address, publicKey}
 */
async function unlockWalletAction(data) {
    try {
        console.log('🔓 ========== UNLOCKING WALLET ==========');
        const { password } = data;
        
        if (!password) {
            throw new Error('Password is required');
        }
        
        // Get encrypted wallet
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        if (!result.walletEncrypted) {
            return {
                success: false,
                error: 'No wallet found. Please create a wallet first.'
            };
        }
        
        console.log('🔐 Decrypting wallet...');
        
        // 🔒 VALIDAR SENHA (sem armazenar mnemonic!)
        // Só vamos descriptografar para verificar se a senha está correta
        const decrypted = await decryptData(result.walletEncrypted, password);
        
        console.log('✅ Password validated successfully');
        
        // ✅ Update wallet state (wallet is now unlocked)
        // ⚠️  NÃO armazenar mnemonic! Só address e publicKey
        walletState = {
            unlocked: true,
            address: decrypted.address,
            publicKey: decrypted.publicKey,
            lockedAt: null
        };
        
        // ✅ CRÍTICO: Salvar estado no session storage (persiste se Service Worker reiniciar)
        // ⚡ Também salvar senha temporária para L2 signing (expira com sessão)
        await chrome.storage.session.set({
            walletUnlocked: true,
            walletAddress: decrypted.address,
            walletPublicKey: decrypted.publicKey,
            tempPassword: password  // ⚡ Para L2 signing sem pedir senha novamente
        });
        
        // 🗑️ Descartar mnemonic imediatamente (não manter na memória!)
        // O mnemonic só será descriptografado novamente quando precisar assinar
        
        console.log('✅ Wallet unlocked:', walletState.address);
        console.log('💾 Wallet state saved to session storage');
        console.log('🔒 Mnemonic NOT stored in memory (security)');
        
        // Start keep-alive to prevent Service Worker termination
        startKeepAlive();
        
        // Start auto-lock timer
        resetAutolockTimer();
        
        // ⚡ Lightning será ativado quando necessário (não agora)
        console.log('⚡ Lightning will be activated when needed');
        
        return {
            success: true,
            address: walletState.address,
            publicKey: walletState.publicKey
        };
        
    } catch (error) {
        console.error('❌ Error unlocking wallet:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * Check wallet status (exists, locked/unlocked)
 * @returns {Object} {success, exists, unlocked, address}
 */
async function checkWalletStatus() {
    try {
        const result = await chrome.storage.local.get(['walletEncrypted']);
        
        return {
            success: true,
            exists: !!result.walletEncrypted,
            unlocked: walletState.unlocked,
            address: walletState.unlocked ? walletState.address : null
        };
    } catch (error) {
        return {
            success: false,
            error: error.message
        };
    }
}

/**
 * Set auto-lock timeout
 * @param {number} timeout - Timeout in minutes (0 = never)
 * @returns {Object} {success}
 */
async function setAutolockTimeout(timeout) {
    try {
        autolockTimeout = timeout;
        
        // Save to storage
        await chrome.storage.local.set({ autolockTimeout: timeout });
        
        console.log(`🔒 Auto-lock timeout set to: ${timeout} minutes`);
        
        // Reset timer with new timeout
        if (walletState.unlocked) {
            resetAutolockTimer();
        }
        
        return {
            success: true
        };
    } catch (error) {
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// 🏪 MARKETPLACE - INSCRIPTION DETAILS
// ==========================================

/**
 * Get Inscription Details for Marketplace Listing
 * Fetches UTXO details from ORD server
 */
async function getInscriptionDetails(request) {
    try {
        const { inscriptionId } = request;
        
        console.log('📊 Getting inscription details for marketplace...', inscriptionId);
        
        // Get current address
        if (!walletState.address) {
            throw new Error('Wallet not initialized');
        }
        
        const address = walletState.address;
        
        // Fetch UTXO from ORD server via backend API
        const utxoResponse = await fetch(
            `https://kraywallet-backend.onrender.com/api/ord/inscription/${inscriptionId}/utxo`
        );
        
        if (!utxoResponse.ok) {
            throw new Error('Failed to fetch inscription UTXO from ORD server');
        }
        
        const utxoData = await utxoResponse.json();
        
        if (!utxoData.success) {
            throw new Error(utxoData.error || 'Invalid UTXO response');
        }
        
        console.log('✅ Inscription UTXO fetched:', utxoData.utxo);
        
        return {
            success: true,
            inscription: {
                id: inscriptionId,
                address: address,
                utxo: {
                    txid: utxoData.utxo.txid,
                    vout: utxoData.utxo.vout,
                    value: utxoData.utxo.value,
                    scriptPubKey: utxoData.utxo.scriptPubKey
                }
            }
        };
        
    } catch (error) {
        console.error('❌ Error getting inscription details:', error);
        return {
            success: false,
            error: error.message
        };
    }
}

// ==========================================
// ⚡ SEND LIGHTNING PAYMENT
// ==========================================

let pendingPaymentRequest = null;

async function sendPayment({ invoice }) {
    try {
        console.log('\n⚡ ===== SEND LIGHTNING PAYMENT =====');
        console.log('  Invoice:', invoice?.substring(0, 50) + '...');
        
        // Verificar se há wallet
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        if (!storage.walletEncrypted) {
            throw new Error('No wallet found. Please create a wallet first.');
        }
        console.log('✅ Wallet found in storage');
        
        // Decode invoice no backend primeiro (para mostrar confirmação)
        console.log('🔍 Decoding invoice...');
        const decodeResponse = await fetch('https://kraywallet-backend.onrender.com/api/lightning/decode', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ invoice })
        });
        
        if (!decodeResponse.ok) {
            throw new Error('Failed to decode invoice');
        }
        
        const decoded = await decodeResponse.json();
        console.log('✅ Invoice decoded:', decoded);
        
        // 🧹 LIMPAR PAYMENT ANTIGO ANTES DE SALVAR NOVO
        console.log('🧹 Cleaning old payment data...');
        await chrome.storage.local.remove(['pendingPaymentRequest', 'paymentResult']);
        pendingPaymentRequest = null;
        
        // Aguardar um pouco para garantir limpeza
        await new Promise(resolve => setTimeout(resolve, 100));
        
        // Guardar payment pending para o popup processar
        pendingPaymentRequest = {
            invoice,
            decoded,
            timestamp: Date.now()
        };
        
        // 💾 Persistir no storage
        await chrome.storage.local.set({ pendingPaymentRequest });
        console.log('✅ pendingPaymentRequest saved!');
        
        // Abrir popup
        if (!isPopupOpening) {
            isPopupOpening = true;
            console.log('📱 Opening popup for Lightning payment confirmation...');
            
            try {
                await chrome.action.openPopup();
                console.log('✅ Popup opened');
            } catch (err) {
                console.error('❌ Failed to open popup:', err);
                console.warn('⚠️  User may need to click the extension icon manually');
            } finally {
                setTimeout(() => {
                    isPopupOpening = false;
                }, 1000);
            }
        }
        
        console.log('⏳ Waiting for user confirmation...');
        
        // Esperar pela resposta do popup (via chrome.storage)
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                console.error('⏱️  TIMEOUT: User did not confirm in 120 seconds');
                pendingPaymentRequest = null;
                chrome.storage.local.remove('pendingPaymentRequest');
                reject(new Error('Payment timeout (user did not confirm)'));
            }, 120000); // 2 minutos
            
            console.log('✅ Promise listener registered, waiting for paymentResult...');
            
            // Listener para resposta do popup
            const listener = (changes, namespace) => {
                if (namespace === 'local' && changes.paymentResult) {
                    console.log('📩 Received paymentResult from popup:', changes.paymentResult.newValue);
                    clearTimeout(timeout);
                    chrome.storage.onChanged.removeListener(listener);
                    
                    const result = changes.paymentResult.newValue;
                    
                    // Limpar resultado
                    chrome.storage.local.remove('paymentResult');
                    chrome.storage.local.remove('pendingPaymentRequest');
                    pendingPaymentRequest = null;
                    
                    if (result.success) {
                        console.log('✅ Payment successful! Resolving...');
                        resolve({
                            success: true,
                            preimage: result.preimage,
                            paymentHash: result.paymentHash,
                            amountSats: result.amountSats,
                            timestamp: result.timestamp
                        });
                    } else {
                        console.error('❌ Payment failed:', result.error);
                        reject(new Error(result.error || 'User cancelled'));
                    }
                }
            };
            
            chrome.storage.onChanged.addListener(listener);
            console.log('🎧 Storage listener active');
        });
        
    } catch (error) {
        console.error('❌ Error in sendPayment:', error);
        pendingPaymentRequest = null;
        chrome.storage.local.remove('pendingPaymentRequest');
        throw error;
    }
}

// ═══════════════════════════════════════════════════════════════════════════
// 💬 KRAYCHAT E2E ENCRYPTION - SECURE LIKE MOBILE!
// 
// 🔐 LEI PRIMORDIAL: Mnemonic NUNCA fica armazenado!
// - Mnemonic usado UMA VEZ para derivar chave
// - Mnemonic DESCARTADO imediatamente após derivação
// - Apenas CHAVE DERIVADA fica em memória (não permite recuperar mnemonic)
// - Fecha browser = memória limpa = chave apagada
// ═══════════════════════════════════════════════════════════════════════════

// 🔐 Chat session - ONLY derived private key (NOT mnemonic!)
// This is the same security model as mobile app
let chatDerivedPrivateKey = null;  // 32 bytes - cannot recover mnemonic from this!
let chatPublicKey = null;          // 33 bytes compressed

/**
 * 🔐 Activate chat session - like mobile!
 * Opens extension popup for password, derives key, DISCARDS mnemonic
 */
async function activateChatSessionAction(data) {
    try {
        console.log('🔐 KrayChat: Activating session (secure like mobile)...');
        
        // Use signMessageRequireConfirmation to open popup and validate user
        const message = `KrayChat Activation at ${Date.now()}`;
        const signResult = await signMessageRequireConfirmation({ message });
        
        if (!signResult || !signResult.signature) {
            return { success: false, error: 'User cancelled' };
        }
        
        // Get the password that was used for signing (stored temporarily in session)
        const sessionData = await chrome.storage.session.get(['tempPassword']);
        if (!sessionData.tempPassword) {
            return { success: false, error: 'Session expired, please try again' };
        }
        
        // Decrypt wallet to get mnemonic
        const storage = await chrome.storage.local.get(['walletEncrypted']);
        if (!storage.walletEncrypted) {
            return { success: false, error: 'Wallet not found' };
        }
        
        const decrypted = await decryptData(storage.walletEncrypted, sessionData.tempPassword);
        if (!decrypted?.mnemonic) {
            return { success: false, error: 'Failed to decrypt wallet' };
        }
        
        // 🔐 CRITICAL: Derive key and DISCARD mnemonic!
        const derivedKey = await TaprootSigner.deriveChatPrivateKey(decrypted.mnemonic);
        // mnemonic is NOT stored - only used momentarily above, now out of scope
        
        // Store ONLY the derived key (cannot recover mnemonic from this!)
        chatDerivedPrivateKey = derivedKey.privateKey;
        chatPublicKey = derivedKey.publicKey;
        
        console.log('✅ KrayChat: Session activated securely');
        console.log('   🔐 Mnemonic was used and discarded');
        console.log('   🔑 Only derived key in memory (cannot recover mnemonic)');
        
        return { success: true, publicKey: chatPublicKey };
    } catch (error) {
        console.error('❌ activateChatSession error:', error);
        return { success: false, error: error.message };
    }
}

/**
 * Check if chat session is active
 */
function isChatSessionActiveAction() {
    return { 
        success: true, 
        active: !!chatDerivedPrivateKey,
        publicKey: chatPublicKey
    };
}

/**
 * Clear chat session
 */
function clearChatSessionAction() {
    chatDerivedPrivateKey = null;
    chatPublicKey = null;
    console.log('🔐 KrayChat: Session cleared (key removed from memory)');
    return { success: true };
}

/**
 * Get Chat Public Key
 */
async function getChatPublicKeyAction(data) {
    try {
        // If session active, return cached pubkey
        if (chatPublicKey) {
            return { success: true, publicKey: chatPublicKey };
        }
        
        // Otherwise need password to derive
        const { password } = data;
        if (!password) {
            throw new Error('Chat session not active. Please activate first.');
        }
        
        const decrypted = await decryptWalletAction(password);
        if (!decrypted?.mnemonic) {
            throw new Error('Invalid password');
        }
        
        const pubkey = await TaprootSigner.getChatPublicKey(decrypted.mnemonic);
        return { success: true, publicKey: pubkey };
    } catch (error) {
        console.error('❌ getChatPublicKey error:', error);
        return { success: false, error: error.message };
    }
}

/**
 * 🔒 Encrypt chat message - uses derived key (no password needed!)
 */
async function encryptChatMessageAction(data) {
    try {
        const { message, theirPublicKey } = data;
        if (!message || !theirPublicKey) {
            throw new Error('Message and theirPublicKey required');
        }
        
        if (!chatDerivedPrivateKey) {
            throw new Error('Chat session not active. Please activate first.');
        }
        
        // Use derived key directly (like mobile does!)
        const encrypted = await TaprootSigner.encryptWithPrivateKey(
            message, 
            chatDerivedPrivateKey, 
            theirPublicKey
        );
        
        console.log('✅ KrayChat: Message encrypted (using session key)');
        return { success: true, encrypted };
    } catch (error) {
        console.error('❌ encryptChatMessage error:', error);
        return { success: false, error: error.message };
    }
}

/**
 * 🔓 Decrypt chat message - uses derived key (no password needed!)
 */
async function decryptChatMessageAction(data) {
    try {
        const { encrypted, theirPublicKey } = data;
        if (!encrypted || !theirPublicKey) {
            throw new Error('Encrypted and theirPublicKey required');
        }
        
        if (!chatDerivedPrivateKey) {
            throw new Error('Chat session not active. Please activate first.');
        }
        
        // Use derived key directly (like mobile does!)
        const message = await TaprootSigner.decryptWithPrivateKey(
            encrypted, 
            chatDerivedPrivateKey, 
            theirPublicKey
        );
        
        console.log('✅ KrayChat: Message decrypted (using session key)');
        return { success: true, message };
    } catch (error) {
        console.error('❌ decryptChatMessage error:', error);
        return { success: false, error: error.message };
    }
}

console.log('🔥 MyWallet Background Script loaded (REAL WALLET MODE)!');
console.log('✅ Using real BIP39 mnemonic generation');
console.log('✅ Using real Taproot address derivation');
console.log('✅ Using real balance API (Mempool.space)');
console.log('✅ Runes Send functionality ready!');
console.log('🔒 Lock/Unlock system active!');
console.log('🏪 Marketplace integration ready!');
console.log('⚡ Lightning Payment functionality ready!');
console.log('🏷️ Auction Mode ready!');
console.log('💬 KrayChat E2E Encryption ready!');

